jQuery(document).ready(($) => {
  // Brand colors for consistent styling
  const brandColors = {
    primary: '#6101fe', // Brand purple
    primaryLight: 'rgba(97, 1, 254, 0.08)',
    primaryGradientStart: '#6101fe',
    primaryGradientEnd: '#7c3aed',
    secondary: '#10b981', // Emerald green for secondary metrics
    secondaryLight: 'rgba(16, 185, 129, 0.08)',
    text: '#1d2327',
    textMuted: '#646970',
    border: '#e0e0e3',
    gridLines: '#f0f0f1',
  };

  // Set Chart.js defaults with brand styling
  Chart.defaults.font.family =
    '-apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,Oxygen-Sans,Ubuntu,Cantarell,"Helvetica Neue",sans-serif';
  Chart.defaults.color = brandColors.textMuted;

  // Check if Chart.js and data are available
  if (typeof Chart === 'undefined') {
    console.error('Chart.js is not loaded');
    return;
  }

  // Check if we're on the dashboard tab before requiring chart data
  const urlParams = new URLSearchParams(window.location.search);
  const currentTab = urlParams.get('tab') || 'dashboard';

  if (currentTab === 'dashboard' && typeof intucartChartData === 'undefined') {
    console.error('Chart data is not available for dashboard tab');
    return;
  }

  // Skip chart initialization if not on dashboard tab
  const shouldInitCharts = currentTab === 'dashboard' && typeof intucartChartData !== 'undefined';

  // Store chart instances
  let searchChart = null;
  let chatChart = null;

  // Create gradient for chart backgrounds
  function createGradient(ctx, colorStart, colorEnd) {
    const gradient = ctx.createLinearGradient(0, 0, 0, 400);
    gradient.addColorStop(0, colorStart);
    gradient.addColorStop(1, colorEnd);
    return gradient;
  }

  // Function to initialize search trends chart
  function initSearchTrendsChart() {
    const canvas = document.getElementById('searchTrendsChart');
    if (!canvas) {
      return;
    }

    if (!intucartChartData.search || !intucartChartData.search.labels || !intucartChartData.search.data) {
      return;
    }

    // Destroy existing chart if it exists
    if (searchChart) {
      searchChart.destroy();
    }

    const ctx = canvas.getContext('2d');
    const gradient = createGradient(ctx, 'rgba(97, 1, 254, 0.15)', 'rgba(97, 1, 254, 0.02)');

    searchChart = new Chart(canvas, {
      type: 'line',
      data: {
        labels: intucartChartData.search.labels,
        datasets: [
          {
            label: 'Searches',
            data: intucartChartData.search.data,
            borderColor: brandColors.primary,
            backgroundColor: gradient,
            borderWidth: 2.5,
            tension: 0.4,
            fill: true,
            pointBackgroundColor: brandColors.primary,
            pointBorderColor: '#ffffff',
            pointBorderWidth: 2,
            pointRadius: 4,
            pointHoverRadius: 6,
            pointHoverBackgroundColor: brandColors.primary,
            pointHoverBorderColor: '#ffffff',
            pointHoverBorderWidth: 3,
          },
        ],
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        interaction: {
          intersect: false,
          mode: 'index',
        },
        plugins: {
          legend: {
            display: false,
          },
          tooltip: {
            backgroundColor: brandColors.text,
            titleColor: '#ffffff',
            bodyColor: '#ffffff',
            borderColor: brandColors.primary,
            borderWidth: 1,
            cornerRadius: 8,
            padding: 12,
            displayColors: false,
            titleFont: {
              weight: '600',
            },
          },
        },
        scales: {
          x: {
            grid: {
              color: brandColors.gridLines,
              drawBorder: false,
            },
            ticks: {
              color: brandColors.textMuted,
              font: {
                size: 11,
              },
            },
          },
          y: {
            beginAtZero: true,
            grid: {
              color: brandColors.gridLines,
              drawBorder: false,
            },
            ticks: {
              precision: 0,
              color: brandColors.textMuted,
              font: {
                size: 11,
              },
            },
          },
        },
      },
    });
  }

  // Function to initialize chat trends chart
  function initChatTrendsChart() {
    const canvas = document.getElementById('chatTrendsChart');
    if (!canvas) {
      return;
    }

    if (!intucartChartData.chat || !intucartChartData.chat.labels || !intucartChartData.chat.data) {
      return;
    }

    if (chatChart) {
      chatChart.destroy();
    }

    const ctx = canvas.getContext('2d');
    const gradient = createGradient(ctx, 'rgba(16, 185, 129, 0.15)', 'rgba(16, 185, 129, 0.02)');

    chatChart = new Chart(canvas, {
      type: 'line',
      data: {
        labels: intucartChartData.chat.labels,
        datasets: [
          {
            label: 'Chats',
            data: intucartChartData.chat.data,
            borderColor: brandColors.secondary,
            backgroundColor: gradient,
            borderWidth: 2.5,
            tension: 0.4,
            fill: true,
            pointBackgroundColor: brandColors.secondary,
            pointBorderColor: '#ffffff',
            pointBorderWidth: 2,
            pointRadius: 4,
            pointHoverRadius: 6,
            pointHoverBackgroundColor: brandColors.secondary,
            pointHoverBorderColor: '#ffffff',
            pointHoverBorderWidth: 3,
          },
        ],
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        interaction: {
          intersect: false,
          mode: 'index',
        },
        plugins: {
          legend: { display: false },
          tooltip: {
            backgroundColor: brandColors.text,
            titleColor: '#ffffff',
            bodyColor: '#ffffff',
            borderColor: brandColors.secondary,
            borderWidth: 1,
            cornerRadius: 8,
            padding: 12,
            displayColors: false,
            titleFont: {
              weight: '600',
            },
          },
        },
        scales: {
          x: {
            grid: {
              color: brandColors.gridLines,
              drawBorder: false,
            },
            ticks: {
              color: brandColors.textMuted,
              font: {
                size: 11,
              },
            },
          },
          y: {
            beginAtZero: true,
            grid: {
              color: brandColors.gridLines,
              drawBorder: false,
            },
            ticks: {
              precision: 0,
              color: brandColors.textMuted,
              font: {
                size: 11,
              },
            },
          },
        },
      },
    });
  }

  // Initialize charts only on dashboard tab
  if (shouldInitCharts) {
    initSearchTrendsChart();
    initChatTrendsChart();
  }

  // Conversations Tab Functionality
  if (typeof intucartAnalytics !== 'undefined') {
    // Try immediate initialization
    initConversationsTab();

    // Also set up a delayed initialization in case DOM isn't ready
    setTimeout(() => {
      initConversationsTab();
    }, 100);
  } else {
    console.error('intucartAnalytics is not defined');
  }

  /**
   * Initialize conversations tab functionality
   */
  function initConversationsTab() {
    // Prevent duplicate initialization
    if (window.conversationsTabInitialized) {
      return;
    }

    // Load initial analytics and conversations when tab loads
    if ($('.intucart-conversations-tab').length > 0) {
      window.conversationsTabInitialized = true;
      loadConversationAnalytics();
      loadConversations();
    } else {
      // Set up tab switching handler to load conversations when tab becomes active
      $('.nav-tab').on('click', function () {
        const href = $(this).attr('href');
        if (href?.includes('tab=conversations')) {
          // Data will load after page refresh
        }
      });
    }

    // Handle conversation search form
    $('#conversation-search-form').on('submit', (e) => {
      e.preventDefault();
      loadConversations();
    });

    // Handle clear filters
    $('#clear-filters').on('click', () => {
      $('#conversation-search-form')[0].reset();
      loadConversations();
    });

    // Handle export conversations
    $('#export-conversations').on('click', () => {
      exportConversations();
    });

    // Handle view conversation buttons (delegated)
    $(document).on('click', '.view-conversation', function () {
      const conversationId = $(this).data('id');
      viewConversation(conversationId);
    });

    // Handle delete conversation buttons (delegated)
    $(document).on('click', '.delete-conversation', function () {
      const conversationId = $(this).data('id');
      const $button = $(this);

      if (confirm('Are you sure you want to delete this conversation? This action cannot be undone.')) {
        deleteConversation(conversationId, $button);
      }
    });
  }

  /**
   * Load conversation analytics
   */
  function loadConversationAnalytics() {
    $.ajax({
      url: intucartAnalytics.ajaxUrl,
      type: 'POST',
      data: {
        action: 'intucart_get_conversation_analytics',
        nonce: intucartAnalytics.nonce,
      },
      success: (response) => {
        if (response.success) {
          $('#conversation-analytics-content').html(response.data.html);
        } else {
          $('#conversation-analytics-content').html(
            '<div class="notice notice-error"><p>Failed to load analytics: ' +
            (response.data.message || 'Unknown error') +
            '</p></div>',
          );
        }
      },
      error: (xhr, status, error) => {
        console.error('Analytics AJAX error:', xhr, status, error);
        $('#conversation-analytics-content').html(
          '<div class="notice notice-error"><p>Failed to load analytics due to server error.</p></div>',
        );
      },
    });
  }

  /**
   * Load conversations based on current filters
   */
  function loadConversations() {
    const formData = $('#conversation-search-form').serialize();

    $('#conversations-table-container').html('<div class="loading-spinner">Loading conversations...</div>');

    $.ajax({
      url: intucartAnalytics.ajaxUrl,
      type: 'POST',
      data: `${formData}&action=intucart_search_conversations&nonce=${intucartAnalytics.nonce}`,
      success: (response) => {
        if (response.success) {
          $('#conversations-table-container').html(response.data.html);
          updatePagination(response.data.pagination);
        } else {
          $('#conversations-table-container').html(
            '<div class="notice notice-error"><p>Failed to load conversations: ' +
            (response.data.message || 'Unknown error') +
            '</p></div>',
          );
        }
      },
      error: (xhr, status, error) => {
        console.error('Conversations AJAX error:', xhr, status, error);
        $('#conversations-table-container').html(
          '<div class="notice notice-error"><p>Failed to load conversations due to server error.</p></div>',
        );
      },
    });
  }

  /**
   * Export conversations
   */
  function exportConversations() {
    const formData = $('#conversation-search-form').serialize();

    // Show loading toast
    const toastId = window.showToast('loading', 'Validating export parameters...', {
      id: 'export-toast',
      persistent: true,
    });

    // First, validate the export request via AJAX
    $.ajax({
      url: intucartAnalytics.ajaxUrl,
      type: 'POST',
      data: `${formData}&action=intucart_validate_export&nonce=${intucartAnalytics.nonce}`,
      success: (response) => {
        if (response.success) {
          // Update toast to show export progress
          window.updateToast(toastId, 'loading', 'Starting export...');

          // If validation passes, proceed with file download
          downloadExportFile(formData, toastId);
        } else {
          // Show error toast
          window.updateToast(toastId, 'error', response.data.message || 'Export validation failed', {
            autoDismiss: true,
          });
        }
      },
      error: (xhr) => {
        let errorMessage = 'Export failed';
        try {
          const response = JSON.parse(xhr.responseText);
          errorMessage = response.data?.message || errorMessage;
        } catch (_e) {
          errorMessage = xhr.responseText || errorMessage;
        }
        window.updateToast(toastId, 'error', errorMessage, {
          autoDismiss: true,
        });
      },
    });
  }

  /**
   * Download export file after validation
   */
  function downloadExportFile(formData, toastId) {
    // Create a temporary form to submit for download
    const $form = $('<form>', {
      method: 'POST',
      action: intucartAnalytics.ajaxUrl,
    });

    // Add form data
    $form.append($('<input>', { type: 'hidden', name: 'action', value: 'intucart_export_conversations' }));
    $form.append($('<input>', { type: 'hidden', name: 'nonce', value: intucartAnalytics.nonce }));

    // Add search parameters
    const params = new URLSearchParams(formData);
    for (const [key, value] of params) {
      $form.append($('<input>', { type: 'hidden', name: key, value: value }));
    }

    $form.appendTo('body').submit().remove();

    // Show success toast
    window.updateToast(toastId, 'success', 'Export started - your download should begin shortly', {
      autoDismiss: true,
    });
  }

  /**
   * View conversation details
   */
  function viewConversation(conversationId) {
    $.ajax({
      url: intucartAnalytics.ajaxUrl,
      type: 'POST',
      data: {
        action: 'intucart_get_conversation_details',
        nonce: intucartAnalytics.nonce,
        threadId: conversationId,
      },
      success: (response) => {
        if (response.success) {
          // Create and show modal with conversation details
          showConversationModal(response.data.html);
        } else {
          alert(`Error: ${response.data.message || 'Failed to load conversation details'}`);
        }
      },
      error: () => {
        alert('Failed to load conversation details');
      },
    });
  }

  /**
   * Show conversation details in a modal
   */
  function showConversationModal(html) {
    // Remove existing modal if any
    $('.conversation-modal-overlay').remove();

    // Create modal overlay
    const modalHtml = `
      <div class="conversation-modal-overlay">
        <div class="conversation-modal">
          <div class="conversation-modal-header">
            <h3>Conversation Details</h3>
            <button class="close-modal" aria-label="Close">&times;</button>
          </div>
          <div class="conversation-modal-content">
            ${html}
          </div>
        </div>
      </div>
    `;

    // Add modal to page
    $('body').append(modalHtml);

    // Update message count in header
    updateMessageCountInHeader();

    // Close modal handlers
    $('.conversation-modal-overlay .close-modal, .conversation-modal-overlay').click(function (e) {
      if (e.target === this) {
        $('.conversation-modal-overlay').remove();
      }
    });

    // Prevent modal content clicks from closing modal
    $('.conversation-modal').click((e) => {
      e.stopPropagation();
    });

    // Close on escape key
    $(document).on('keydown.conversationModal', (e) => {
      if (e.keyCode === 27) {
        $('.conversation-modal-overlay').remove();
        $(document).off('keydown.conversationModal');
      }
    });
  }

  /**
   * Delete conversation
   */
  function deleteConversation(conversationId, $button) {
    const originalText = $button.text();
    $button.prop('disabled', true).text('Deleting...');

    $.ajax({
      url: intucartAnalytics.ajaxUrl,
      type: 'POST',
      data: {
        action: 'intucart_delete_conversation',
        nonce: intucartAnalytics.nonce,
        threadId: conversationId,
      },
      success: (response) => {
        if (response.success) {
          // Remove the row from the table
          $button.closest('tr').fadeOut(300, function () {
            $(this).remove();
          });

          // Reload analytics to update counts
          loadConversationAnalytics();
        } else {
          alert(`Failed to delete conversation: ${response.data.message || 'Unknown error'}`);
          $button.prop('disabled', false).text(originalText);
        }
      },
      error: () => {
        alert('Failed to delete conversation due to server error.');
        $button.prop('disabled', false).text(originalText);
      },
    });
  }

  /**
   * Update pagination with navigation controls
   */
  function updatePagination(paginationData) {
    const paginationContainer = document.querySelector('#conversations-pagination .tablenav-pages');
    if (!paginationContainer) return;

    // Always show pagination for consistency
    if (paginationData && paginationData.total !== undefined) {
      const total = paginationData.total || 0;
      const limit = paginationData.limit || 10;
      const offset = paginationData.offset || 0;
      const currentPage = Math.floor(offset / limit) + 1;
      const totalPages = Math.ceil(total / limit) || 1;

      let paginationHTML = '<span class="displaying-num">';
      paginationHTML += `${total} item${total !== 1 ? 's' : ''}`;
      paginationHTML += '</span>';

      // Add navigation controls
      paginationHTML += '<span class="pagination-links">';

      // First page link
      if (currentPage > 1) {
        paginationHTML += `<a class="first-page button" href="#" data-page="1">`;
        paginationHTML += `<span class="screen-reader-text">First page</span>`;
        paginationHTML += `<span aria-hidden="true">«</span>`;
        paginationHTML += `</a>`;
      } else {
        paginationHTML += `<span class="tablenav-pages-navspan button disabled" aria-hidden="true">«</span>`;
      }

      // Previous page link
      if (currentPage > 1) {
        paginationHTML += `<a class="prev-page button" href="#" data-page="${currentPage - 1}">`;
        paginationHTML += `<span class="screen-reader-text">Previous page</span>`;
        paginationHTML += `<span aria-hidden="true">‹</span>`;
        paginationHTML += `</a>`;
      } else {
        paginationHTML += `<span class="tablenav-pages-navspan button disabled" aria-hidden="true">‹</span>`;
      }

      // Page info
      paginationHTML += `<span class="paging-input">`;
      paginationHTML += `<label for="current-page-selector" class="screen-reader-text">Current Page</label>`;
      paginationHTML += `<input class="current-page" id="current-page-selector" type="text" `;
      paginationHTML += `name="paged" value="${currentPage}" size="1" aria-describedby="table-paging" />`;
      paginationHTML += `<span class="tablenav-paging-text"> of `;
      paginationHTML += `<span class="total-pages">${totalPages}</span></span>`;
      paginationHTML += `</span>`;

      // Next page link
      if (currentPage < totalPages) {
        paginationHTML += `<a class="next-page button" href="#" data-page="${currentPage + 1}">`;
        paginationHTML += `<span class="screen-reader-text">Next page</span>`;
        paginationHTML += `<span aria-hidden="true">›</span>`;
        paginationHTML += `</a>`;
      } else {
        paginationHTML += `<span class="tablenav-pages-navspan button disabled" aria-hidden="true">›</span>`;
      }

      // Last page link
      if (currentPage < totalPages) {
        paginationHTML += `<a class="last-page button" href="#" data-page="${totalPages}">`;
        paginationHTML += `<span class="screen-reader-text">Last page</span>`;
        paginationHTML += `<span aria-hidden="true">»</span>`;
        paginationHTML += `</a>`;
      } else {
        paginationHTML += `<span class="tablenav-pages-navspan button disabled" aria-hidden="true">»</span>`;
      }

      paginationHTML += '</span>';

      paginationContainer.innerHTML = paginationHTML;

      // Add event listeners for pagination controls
      addPaginationEventListeners();

      // Show the pagination container
      document.getElementById('conversations-pagination').style.display = 'block';
    } else {
      // Hide pagination if no data
      document.getElementById('conversations-pagination').style.display = 'none';
    }
  }

  /**
   * Update the message count in the modal header based on actual parsed messages
   */
  function updateMessageCountInHeader() {
    const messageItems = document.querySelectorAll('.conversation-modal .message-item');
    const messageCount = messageItems.length;
    const placeholder = document.querySelector('.conversation-modal .message-count-placeholder');

    if (placeholder && messageCount > 0) {
      placeholder.textContent = `💬 ${messageCount} message${messageCount !== 1 ? 's' : ''}`;
    } else if (placeholder) {
      placeholder.textContent = '💬 No messages';
    }
  }

  /**
   * Add event listeners for pagination controls
   */
  function addPaginationEventListeners() {
    const paginationLinks = document.querySelectorAll('#conversations-pagination .pagination-links a');
    const currentPageInput = document.querySelector('#conversations-pagination .current-page');

    // Handle pagination link clicks
    paginationLinks.forEach((link) => {
      link.addEventListener('click', function (e) {
        e.preventDefault();
        const page = parseInt(this.getAttribute('data-page'), 10);
        if (page && page > 0) {
          loadConversationsPage(page);
        }
      });
    });

    // Handle direct page input
    if (currentPageInput) {
      currentPageInput.addEventListener('keypress', function (e) {
        if (e.key === 'Enter') {
          e.preventDefault();
          const page = parseInt(this.value, 10);
          if (page && page > 0) {
            loadConversationsPage(page);
          }
        }
      });

      currentPageInput.addEventListener('blur', function () {
        const page = parseInt(this.value, 10);
        if (page && page > 0) {
          loadConversationsPage(page);
        }
      });
    }
  }

  /**
   * Load conversations for a specific page
   */
  function loadConversationsPage(page) {
    const limit = 10; // Default limit
    const offset = (page - 1) * limit;

    // Get current search form data
    const form = document.getElementById('conversation-search-form');
    if (!form) return;

    const formData = new FormData(form);
    formData.set('limit', limit);
    formData.set('offset', offset);
    formData.set('action', 'intucart_search_conversations');
    formData.set('nonce', intucartAnalytics.nonce);

    // Show loading
    const tableContainer = document.getElementById('conversations-table-container');
    if (tableContainer) {
      tableContainer.innerHTML = '<div class="loading-spinner">Loading conversations...</div>';
    }

    // Make AJAX request
    fetch(intucartAnalytics.ajaxUrl, {
      method: 'POST',
      body: formData,
    })
      .then((response) => response.json())
      .then((data) => {
        if (data.success) {
          tableContainer.innerHTML = data.data.html;
          updatePagination(data.data.pagination);
        } else {
          tableContainer.innerHTML =
            '<div class="no-conversations">Error loading conversations: ' +
            (data.data?.message || 'Unknown error') +
            '</div>';
        }
      })
      .catch((error) => {
        console.error('Error loading conversations page:', error);
        tableContainer.innerHTML = '<div class="no-conversations">Error loading conversations.</div>';
      });
  }
});
