/**
 * Chatbot Styling Settings
 * Handles all JavaScript functionality for the styling settings page:
 * - AI Color Generator
 * - Font family toggles
 * - Background mode toggles
 * - Reset styling settings
 */

(($) => {
    /**
     * Initialize all styling settings functionality
     */
    function init() {
        initColorGenerator();
        initFontToggles();
        initBackgroundToggle();
        initResetButton();
    }

    // ============================================
    // AI COLOR GENERATOR
    // ============================================

    /**
     * Initialize color generator functionality
     */
    function initColorGenerator() {
        // Toggle secondary color inputs
        $('#intucart-use-secondary-color')
            .on('change', function () {
                const $secondaryInput = $('[name="intucart_ai_secondary_color"]');
                const $wpPickerContainer = $secondaryInput.closest('.wp-picker-container');
                const $wpColorButton = $wpPickerContainer.find('.wp-color-result');

                if ($(this).is(':checked')) {
                    // Enable the input and button
                    $secondaryInput.prop('disabled', false);
                    $wpColorButton.prop('disabled', false).css({
                        'pointer-events': 'auto',
                        cursor: 'pointer',
                    });
                    $wpPickerContainer.css('opacity', '1');
                } else {
                    // Disable the input and button
                    $secondaryInput.prop('disabled', true);
                    $wpColorButton.prop('disabled', true).css({
                        'pointer-events': 'none',
                        cursor: 'not-allowed',
                    });
                    $wpPickerContainer.css('opacity', '0.5');
                }
            })
            .trigger('change');

        // Generate color scheme button
        $('#intucart-generate-colors').on('click', handleColorGeneration);
    }

    /**
     * Handle color scheme generation
     */
    function handleColorGeneration(e) {
        e.preventDefault();

        const $button = $(this);
        const $status = $('#intucart-color-generator-status');
        const originalText = $button.html();

        const primaryColor = $('[name="intucart_ai_primary_color"]').val();
        const secondaryColor = $('#intucart-use-secondary-color').is(':checked')
            ? $('[name="intucart_ai_secondary_color"]').val()
            : '';

        // Validate hex color format
        if (!isValidHexColor(primaryColor)) {
            showStatus($status, 'error', 'Please enter a valid primary color in hex format (e.g., #0084ff)');
            return;
        }

        if (secondaryColor && !isValidHexColor(secondaryColor)) {
            showStatus($status, 'error', 'Please enter a valid secondary color in hex format (e.g., #ffffff)');
            return;
        }

        // Disable button and show loading
        $button
            .prop('disabled', true)
            .html(
                '<span class="dashicons dashicons-update" style="animation: rotation 1s infinite linear; vertical-align: middle;"></span> Generating...',
            );
        $status.hide();

        // Make AJAX request
        $.ajax({
            url: intucartAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'intucart_generate_color_scheme',
                nonce: intucartAdmin.nonce,
                primaryColor: primaryColor,
                secondaryColor: secondaryColor || undefined,
            },
            success: (response) => {
                if (response.success && response.data.colors) {
                    handleColorSuccess(response.data.colors, $status);
                } else {
                    showStatus($status, 'error', response.data?.message || 'Failed to generate color scheme');
                }
                $button.prop('disabled', false).html(originalText);
            },
            error: () => {
                showStatus($status, 'error', 'Failed to communicate with server. Please try again.');
                $button.prop('disabled', false).html(originalText);
            },
        });
    }

    /**
     * Handle successful color generation
     */
    function handleColorSuccess(colors, $status) {
        // SVG icon (same as AI Color Generator title)
        const sparkleIcon = '<span style="display: inline-block; width: 20px; height: 20px; vertical-align: middle; margin-right: 6px; background-image: url(\'data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHZpZXdCb3g9IjAgMCAyNCAyNCIgd2lkdGg9IjIwIiBoZWlnaHQ9IjIwIiBmaWxsPSIjMTU1NzI0Ij4KPHBhdGggZD0iTTE5IDlsMS4yNS0yLjc1TDIzIDVsLTIuNzUtMS4yNUwxOSAxbC0xLjI1IDIuNzVMMTUgNWwyLjc1IDEuMjVMMTkgOXptLTcuNS41TDkgNCA2LjUgOS41IDEgMTJsNS41IDIuNUw5IDIwbDIuNS01LjVMMTcgMTJsLTUuNS0yLjV6TTE5IDE1bC0xLjI1IDIuNzVMMTUgMTlsMi43NSAxLjI1TDE5IDIzbDEuMjUtMi43NUwyMyAxOWwtMi43NS0xLjI1TDE5IDE1eiIvPgo8L3N2Zz4g\'); background-size: contain; background-repeat: no-repeat; background-position: center;"></span>';

        // Show success message with reasoning
        showStatus(
            $status,
            'success',
            `<strong>${sparkleIcon}Success!</strong><br>${colors.reasoning || 'Color scheme generated successfully!'}`,
        );

        // Populate light mode colors
        if (colors.light) {
            populateLightModeColors(colors.light);
        }

        // Populate dark mode colors
        if (colors.dark) {
            populateDarkModeColors(colors.dark);
        }

        // Visual feedback for updated colors
        animateColorUpdate();
    }

    /**
     * Populate light mode color fields
     */
    function populateLightModeColors(lightColors) {
        const colorMap = {
            itf_chat_primary_color: lightColors.primary,
            itf_chat_assistant_bubble_bg: lightColors.assistantBubbleBg,
            itf_chat_header_background: lightColors.headerBackground,
            itf_chat_accent_color: lightColors.accent,
            itf_chat_send_button_color: lightColors.sendButton,
            itf_chat_link_color: lightColors.link,
            itf_chat_trigger_button_color: lightColors.triggerButton,
            itf_chat_trigger_button_hover_color: lightColors.triggerButtonHover,
            itf_chat_trigger_button_border_color: lightColors.triggerButtonBorder,
            itf_chat_input_background: lightColors.inputBackground,
            itf_chat_input_border: lightColors.inputBorder,
            itf_chat_widget_background_custom: lightColors.widgetBackground,
            itf_chat_header_title_color: lightColors.headerTitle,
            itf_chat_header_subtitle_color: lightColors.headerSubtitle,
            itf_chat_header_icon_color: lightColors.headerIcon,
            itf_chat_card_background: lightColors.cardBackground,
            itf_chat_card_border: lightColors.cardBorder,
            itf_chat_card_hover_bg: lightColors.cardHover,
            itf_chat_card_title_color: lightColors.cardTitle,
            itf_chat_card_meta_color: lightColors.cardMeta,
            itf_chat_card_desc_color: lightColors.cardDesc,
            itf_chat_prompt_chip_background: lightColors.promptChipBackground,
            itf_chat_prompt_chip_text_color: lightColors.promptChipText,
            itf_chat_prompt_chip_border: lightColors.promptChipBorder,
            itf_chat_prompt_chip_hover_bg: lightColors.promptChipHover,
        };

        updateColorFields(colorMap);
    }

    /**
     * Populate dark mode color fields
     */
    function populateDarkModeColors(darkColors) {
        const colorMap = {
            itf_chat_dark_primary_color: darkColors.primary,
            itf_chat_dark_assistant_bubble_bg: darkColors.assistantBubbleBg,
            itf_chat_dark_header_background: darkColors.headerBackground,
            itf_chat_dark_accent_color: darkColors.accent,
            itf_chat_dark_send_button_color: darkColors.sendButton,
            itf_chat_dark_link_color: darkColors.link,
            itf_chat_dark_trigger_button_color: darkColors.triggerButton,
            itf_chat_dark_trigger_button_hover_color: darkColors.triggerButtonHover,
            itf_chat_dark_trigger_button_border_color: darkColors.triggerButtonBorder,
            itf_chat_dark_input_background: darkColors.inputBackground,
            itf_chat_dark_input_border: darkColors.inputBorder,
            itf_chat_dark_widget_background_custom: darkColors.widgetBackground,
            itf_chat_dark_header_title_color: darkColors.headerTitle,
            itf_chat_dark_header_subtitle_color: darkColors.headerSubtitle,
            itf_chat_dark_header_icon_color: darkColors.headerIcon,
            itf_chat_dark_card_background: darkColors.cardBackground,
            itf_chat_dark_card_border: darkColors.cardBorder,
            itf_chat_dark_card_hover_bg: darkColors.cardHover,
            itf_chat_dark_card_title_color: darkColors.cardTitle,
            itf_chat_dark_card_meta_color: darkColors.cardMeta,
            itf_chat_dark_card_desc_color: darkColors.cardDesc,
            itf_chat_dark_prompt_chip_background: darkColors.promptChipBackground,
            itf_chat_dark_prompt_chip_text_color: darkColors.promptChipText,
            itf_chat_dark_prompt_chip_border: darkColors.promptChipBorder,
            itf_chat_dark_prompt_chip_hover_bg: darkColors.promptChipHover,
        };

        updateColorFields(colorMap);
    }

    /**
     * Update color fields with new values
     */
    function updateColorFields(colorMap) {
        Object.keys(colorMap).forEach((fieldName) => {
            const value = colorMap[fieldName];
            if (value) {
                // Target the itf-color-picker input by name
                const $input = $(`.itf-color-picker[name="${fieldName}"]`);
                if ($input.length) {
                    console.log(`Updating ${fieldName} to ${value}, input found:`, $input.length);
                    $input.wpColorPicker('color', value);
                } else {
                    console.warn(`Color picker not found for field: ${fieldName}`);
                }
            }
        });
    }

    /**
     * Animate color update with visual feedback
     */
    function animateColorUpdate() {
        $('.itf-color-picker').each(function () {
            // Find the WordPress color picker button associated with this input
            const $button = $(this).closest('.wp-picker-container').find('.wp-color-result');
            if ($button.length) {
                $button.css({
                    'box-shadow': '0 0 0 3px rgba(40, 167, 69, 0.3)',
                    transition: 'box-shadow 0.3s',
                });
                setTimeout(() => {
                    $button.css('box-shadow', '');
                }, 2000);
            }
        });
    }

    /**
     * Show status message
     */
    function showStatus($status, type, message) {
        const colors = {
            success: {
                bg: '#d4edda',
                color: '#155724',
                border: '#c3e6cb',
            },
            error: {
                bg: '#f8d7da',
                color: '#721c24',
                border: '#f5c6cb',
            },
        };

        const style = colors[type] || colors.error;

        $status.html(message).css({
            background: style.bg,
            color: style.color,
            border: `1px solid ${style.border}`,
            display: 'block',
            padding: '12px',
            'border-radius': '4px',
            'margin-top': '12px',
        });
    }

    /**
     * Validate hex color format
     */
    function isValidHexColor(color) {
        return /^#[0-9A-Fa-f]{6}$/.test(color);
    }

    // ============================================
    // FONT FAMILY TOGGLES
    // ============================================

    /**
     * Initialize font family toggles
     */
    function initFontToggles() {
        // Main font family toggle
        $('[name="intucart_chat_font_family"]').on('change', function () {
            $('#custom_font_wrapper').toggle($(this).val() === 'custom');
        });

        // Message font family toggle
        $('[name="intucart_chat_message_font_family"]').on('change', function () {
            $('#message_custom_font_wrapper').toggle($(this).val() === 'custom');
        });
    }

    // ============================================
    // BACKGROUND MODE TOGGLE
    // ============================================

    /**
     * Initialize background mode toggle
     */
    function initBackgroundToggle() {
        $('[name="intucart_chat_widget_background_mode"]').on('change', function () {
            $('#widget_bg_custom_wrapper').toggle($(this).val() === 'custom');
        });
    }

    // ============================================
    // RESET STYLING SETTINGS
    // ============================================

    /**
     * Initialize reset button
     */
    function initResetButton() {
        $('#intucart-reset-styling').on('click', handleResetStyling);
    }

    /**
     * Handle reset styling settings
     */
    function handleResetStyling(e) {
        e.preventDefault();

        if (
            !confirm(
                'Are you sure you want to reset ALL styling settings to defaults? This will:\n\n• Reset all colors to theme defaults\n• Remove all custom avatars\n• Clear custom CSS\n• Reset border radius and background settings\n\nThis action cannot be undone!',
            )
        ) {
            return;
        }

        const $button = $(this);
        const $status = $('#intucart-reset-status');
        const originalText = $button.html();

        // Disable button and show loading
        $button
            .prop('disabled', true)
            .html('<span class="dashicons dashicons-update" style="animation: rotation 1s infinite linear;"></span> Resetting...');
        $status.hide();

        $.ajax({
            url: intucartAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'intucart_reset_styling_settings',
                nonce: intucartAdmin.nonce,
            },
            success: (response) => {
                if (response.success) {
                    $status.html(`<span style="color: #28a745;">✓ ${response.data.message}</span>`).show();

                    // Reload page after 2 seconds to show reset values
                    setTimeout(() => {
                        window.location.reload();
                    }, 2000);
                } else {
                    $status.html(`<span style="color: #dc3545;">✗ ${response.data.message || 'Reset failed'}</span>`).show();
                    $button.prop('disabled', false).html(originalText);
                }
            },
            error: () => {
                $status.html('<span style="color: #dc3545;">✗ Error communicating with server</span>').show();
                $button.prop('disabled', false).html(originalText);
            },
        });
    }

    // ============================================
    // INITIALIZE
    // ============================================

    // Initialize on document ready
    $(document).ready(() => {
        init();
    });
})(jQuery);

