jQuery(document).ready(($) => {
  // Initialize WordPress color pickers with animation (styles in common.css)
  $('.itf-color-picker').wpColorPicker({
    // Hide picker by default (shows on input focus)
    hide: true,
    // Show palette of recent/common colors
    palettes: true,
  });

  // Handle trial button click
  $('#start-trial').on('click', (e) => {
    e.preventDefault();
    const button = $(this);
    button.prop('disabled', true).text('Starting trial...');

    $.ajax({
      url: intucartAdmin.ajaxUrl,
      type: 'POST',
      data: {
        action: 'intucart_start_trial',
        nonce: intucartAdmin.nonce,
      },
      success: (response) => {
        if (response.success) {
          location.reload();
        } else {
          alert(response.data.message || 'Failed to start trial. Please try again.');
        }
      },
      error: () => {
        alert('An error occurred. Please try again.');
        button.prop('disabled', false).text('Start Free Trial');
      },
    });
  });

  /**
   * Show a toast notification with loading support
   * @param {string} type Toast type (success, error, warning, info, loading)
   * @param {string} message Message to display
   * @param {object} [options] Optional: {id: string, autoDismiss: boolean, dismissDelay: number, persistent: boolean}
   */
  window.showToast = (type, message, options = {}) => {
    const toastId = options.id || 'intucart-toast';

    // Remove existing toast with same ID
    $(`.intucart-toast[data-id="${toastId}"]`).remove();

    // Create toast container if it doesn't exist
    if (!$('.intucart-toast-container').length) {
      $('body').append(`
        <div class="intucart-toast-container" style="
          position: fixed;
          top: 50px;
          right: 20px;
          z-index: 100000;
          min-width: 300px;
          max-width: 400px;
        "></div>
      `);
    }

    // Create loading spinner HTML
    const spinner =
      type === 'loading'
        ? `
      <div class="intucart-spinner" style="
        display: inline-block;
        width: 18px;
        height: 18px;
        min-width: 18px;
        min-height: 18px;
        border: 2px solid #f3f3f3;
        border-top: 2px solid #0073aa;
        border-radius: 50%;
        animation: intucart-spin 1s linear infinite;
        margin-right: 12px;
        vertical-align: middle;
        flex-shrink: 0;
      "></div>
    `
        : '';

    // Create toast element
    const $toast = $(`
      <div class="intucart-toast" data-id="${toastId}" style="
        background: white;
        border-left: 4px solid ${getToastColor(type)};
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        border-radius: 4px;
        padding: 16px;
        margin-bottom: 10px;
        min-width: 280px;
        display: flex;
        align-items: center;
        justify-content: space-between;
        transform: translateX(100%);
        transition: transform 0.3s ease;
        font-size: 14px;
        line-height: 1.4;
      ">
        <div class="toast-content" style="display: flex; align-items: center; flex: 1; min-width: 0;">
          ${spinner}
          <span class="toast-message" style="flex: 1; word-wrap: break-word;">${message}</span>
        </div>
        ${!options.persistent ? '<button class="toast-dismiss" style="background: none; border: none; font-size: 18px; cursor: pointer; color: #666; margin-left: 10px;">×</button>' : ''}
      </div>
    `);

    // Add CSS animation keyframes if not already added
    if (!$('#intucart-toast-styles').length) {
      $('head').append(`
        <style id="intucart-toast-styles">
          @keyframes intucart-spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
          }
        </style>
      `);
    }

    // Add toast to container
    $('.intucart-toast-container').append($toast);

    // Animate in
    setTimeout(() => {
      $toast.css('transform', 'translateX(0)');
    }, 10);

    // Handle dismiss button
    $toast.find('.toast-dismiss').on('click', () => {
      dismissToast($toast);
    });

    // Auto-dismiss for non-loading toasts
    const autoDismiss = options.autoDismiss !== undefined ? options.autoDismiss : type === 'success';
    const dismissDelay = options.dismissDelay || 5000;

    if (autoDismiss && type !== 'loading') {
      setTimeout(() => {
        dismissToast($toast);
      }, dismissDelay);
    }

    return toastId;
  };

  /**
   * Update an existing toast
   * @param {string} id Toast ID
   * @param {string} type New toast type
   * @param {string} message New message
   * @param {object} [options] Optional: {autoDismiss: boolean, dismissDelay: number}
   */
  window.updateToast = (id, type, message, options = {}) => {
    const $toast = $(`.intucart-toast[data-id="${id}"]`);
    if (!$toast.length) return;

    // Update border color
    $toast.css('border-left-color', getToastColor(type));

    // Update content
    const $content = $toast.find('.toast-content');
    const spinner =
      type === 'loading'
        ? `
      <div class="intucart-spinner" style="
        display: inline-block;
        width: 18px;
        height: 18px;
        min-width: 18px;
        min-height: 18px;
        border: 2px solid #f3f3f3;
        border-top: 2px solid #0073aa;
        border-radius: 50%;
        animation: intucart-spin 1s linear infinite;
        margin-right: 12px;
        vertical-align: middle;
        flex-shrink: 0;
      "></div>
    `
        : '';

    $content.html(`${spinner}<span class="toast-message" style="flex: 1; word-wrap: break-word;">${message}</span>`);

    // Add dismiss button if it doesn't exist and it's not a loading toast
    if (type !== 'loading' && !$toast.find('.toast-dismiss').length) {
      $toast.append(
        '<button class="toast-dismiss" style="background: none; border: none; font-size: 18px; cursor: pointer; color: #666; margin-left: 10px;">×</button>',
      );
      $toast.find('.toast-dismiss').on('click', () => {
        dismissToast($toast);
      });
    }

    // Auto-dismiss for non-loading toasts
    const autoDismiss = options.autoDismiss !== undefined ? options.autoDismiss : type === 'success';
    const dismissDelay = options.dismissDelay || 5000;

    if (autoDismiss && type !== 'loading') {
      setTimeout(() => {
        dismissToast($toast);
      }, dismissDelay);
    }
  };

  /**
   * Dismiss a toast
   * @param {jQuery} $toast Toast element
   */
  function dismissToast($toast) {
    $toast.css('transform', 'translateX(100%)');
    setTimeout(() => {
      $toast.remove();
    }, 300);
  }

  /**
   * Get color for toast type
   * @param {string} type Toast type
   * @returns {string} Color hex code
   */
  function getToastColor(type) {
    const colors = {
      success: '#46b450',
      error: '#dc3232',
      warning: '#ffb900',
      info: '#0073aa',
      loading: '#0073aa',
    };
    return colors[type] || '#0073aa';
  }

  // Handle focus border settings toggle
  const $focusBorderCheckbox = $('input[name="intucart_chat_show_focus_border"]');
  const $focusBorderColorPicker = $('input[name="intucart_chat_focus_border_color"]');
  const $focusBorderWidthInput = $('input[name="intucart_chat_focus_border_width"]');

  if ($focusBorderCheckbox.length && ($focusBorderColorPicker.length || $focusBorderWidthInput.length)) {
    // Initial state
    toggleFocusBorderControls();

    // Handle checkbox change
    $focusBorderCheckbox.on('change', toggleFocusBorderControls);

    function toggleFocusBorderControls() {
      const isEnabled = $focusBorderCheckbox.prop('checked');

      if ($focusBorderColorPicker.length) {
        $focusBorderColorPicker.prop('disabled', !isEnabled);
        // Update wpColorPicker container opacity
        $focusBorderColorPicker.closest('.wp-picker-container').css('opacity', isEnabled ? '1' : '0.5');
      }

      if ($focusBorderWidthInput.length) {
        $focusBorderWidthInput.prop('disabled', !isEnabled);
      }
    }
  }

  // Handle auto-sync checkbox for license activation
  const $autoSyncCheckbox = $('#intucart-auto-sync-checkbox');
  const $autoSyncHidden = $('#intucart-auto-sync-hidden');

  if ($autoSyncCheckbox.length && $autoSyncHidden.length) {
    // Update hidden field when checkbox changes
    $autoSyncCheckbox.on('change', () => {
      $autoSyncHidden.val($autoSyncCheckbox.prop('checked') ? '1' : '0');
    });

    // Set initial value
    $autoSyncHidden.val($autoSyncCheckbox.prop('checked') ? '1' : '0');
  }

  /**
   * License deactivation confirmation dialog
   * Shows comprehensive warning about data deletion
   * @returns {boolean} True if user confirms, false otherwise
   */
  window.confirmLicenseDeactivation = () => {
    const message = [
      '⚠️ IMPORTANT: License Deactivation Warning',
      '',
      'This action will:',
      '• First, automatically delete ALL your cloud AI data',
      '• Remove all posts, products, and categories from our AI servers',
      '• Delete all configured webhooks and their settings',
      '• Delete all custom chat prompts and configurations',
      '• Delete all user feedback and analytics data',
      '• Then deactivate your license for this site',
      '• Stop all AI-powered features (search, chat, recommendations)',
      '',
      'Your WordPress site data will remain safe and unchanged.',
      '',
      'This action CANNOT be undone. You would need to re-sync all content, reconfigure webhooks, and recreate prompts after reactivating.',
      '',
      'Are you absolutely sure you want to proceed?',
    ].join('\n');

    return confirm(message);
  };

  /**
   * Generic utility to toggle color picker with "None" checkbox
   * Used for all styling options that have optional color fields
   * @param {HTMLElement} checkbox The checkbox element
   * @param {string} fieldName The field name for the color input
   * @param {string} defaultColor The default color to use when unchecked
   */
  window.toggleColorPicker = (checkbox, fieldName, defaultColor) => {
    const colorInput = checkbox.closest('div').querySelector('.itf-color-picker');

    if (checkbox.checked) {
      // "None" is checked - disable color picker and remove from form submission
      $(colorInput).closest('.wp-picker-container').css('opacity', '0.5');
      colorInput.disabled = true;
      colorInput.name = ''; // Remove from form submission to send empty value
    } else {
      // "None" is unchecked - enable color picker and set to default color
      $(colorInput).closest('.wp-picker-container').css('opacity', '1');
      colorInput.disabled = false;
      colorInput.name = fieldName; // Add back to form submission
      $(colorInput).wpColorPicker('color', defaultColor); // Set to default color using wpColorPicker method
    }
  };

  /**
   * Toggle transparent background color option
   * Used for styling options that can have transparent backgrounds
   * @param {HTMLElement} checkbox The checkbox element
   * @param {string} fieldName The field name for the color input
   */
  window.toggleTransparent = (checkbox, fieldName) => {
    const colorInput = checkbox.closest('td').querySelector('.itf-color-picker');
    const hiddenInput = checkbox.closest('td').querySelector(`input[name="${fieldName}_transparent"]`);

    if (checkbox.checked) {
      $(colorInput).closest('.wp-picker-container').css('opacity', '0.5');
      colorInput.disabled = true;
      if (hiddenInput) {
        hiddenInput.value = '1';
      }
    } else {
      $(colorInput).closest('.wp-picker-container').css('opacity', '1');
      colorInput.disabled = false;
      if (hiddenInput) {
        hiddenInput.value = '0';
      }
    }
  };

  /**
   * Toggle header icon color picker (specialized version for header icons)
   * @param {HTMLElement} checkbox The checkbox element
   * @param {string} fieldName The field name for the color input
   * @param {string} defaultColor The default color to use when unchecked
   */
  window.toggleHeaderIconColor = (checkbox, fieldName, defaultColor) => {
    // Use the same logic as toggleColorPicker
    window.toggleColorPicker(checkbox, fieldName, defaultColor);
  };
});
