/**
 * Form Fields Handlers
 * Handles password toggle, character counter, and range slider functionality
 */
(($) => {
    $(document).ready(() => {
        initPasswordToggle();
        initCharacterCounter();
        initRangeSliders();
    });

    /**
     * Password visibility toggle
     */
    function initPasswordToggle() {
        $(document).on('click', '.intucart-password-toggle', function (e) {
            e.preventDefault();
            var $btn = $(this);
            var targetId = $btn.data('target');
            var $input = $(`#${targetId}`);
            var $icon = $btn.find('.dashicons');

            if ($input.attr('type') === 'password') {
                $input.attr('type', 'text');
                $icon.removeClass('dashicons-visibility').addClass('dashicons-hidden');
                $btn.attr('title', 'Hide password');
            } else {
                $input.attr('type', 'password');
                $icon.removeClass('dashicons-hidden').addClass('dashicons-visibility');
                $btn.attr('title', 'Show password');
            }
        });

        // Prevent input blur when clicking button
        $(document).on('mousedown', '.intucart-password-toggle', (e) => {
            e.preventDefault();
        });
    }

    /**
     * Character counter for textareas with maxlength
     */
    function initCharacterCounter() {
        $('textarea[maxlength]').each(function () {
            var $textarea = $(this);
            var textareaId = $textarea.attr('id');
            var $counter = $(`.intucart-char-count[data-textarea-id="${textareaId}"]`);
            var maxlength = parseInt($textarea.attr('maxlength'), 10);

            if ($counter.length && maxlength) {
                $textarea.on('input', function () {
                    var length = $(this).val().length;
                    $counter.text(length);

                    // Visual feedback when approaching limit
                    var $counterDiv = $counter.parent();
                    if (length > maxlength * 0.9) {
                        $counterDiv.css('color', '#d63638'); // Red
                    } else if (length > maxlength * 0.8) {
                        $counterDiv.css('color', '#dba617'); // Orange
                    } else {
                        $counterDiv.css('color', '#646970'); // Default gray
                    }
                });
            }
        });
    }

    /**
     * Range slider with value display and progress fill
     */
    function initRangeSliders() {
        // Function to update range slider progress
        function updateRangeProgress($input) {
            var min = parseFloat($input.attr('min') || 0);
            var max = parseFloat($input.attr('max') || 100);
            var value = parseFloat($input.val());
            var percentage = ((value - min) / (max - min)) * 100;
            $input.css('--range-progress', `${percentage}%`);
        }

        // Update on input
        $(document).on('input', '.intucart-range', function () {
            var $input = $(this);
            var id = $input.attr('id');
            var $valueDisplay = $(`#${id}_value`);
            var suffix = $valueDisplay
                .text()
                .replace(/[0-9.-]/g, '')
                .trim();
            $valueDisplay.text($input.val() + suffix);
            updateRangeProgress($input);
        });

        // Initialize all range sliders on page load
        $('.intucart-range').each(function () {
            var $input = $(this);
            updateRangeProgress($input);
            // Add initialized class to show the slider with proper fill
            $input.addClass('initialized');
        });
    }
})(jQuery);
