/**
 * Intucart Popovers - Floating UI powered popover system
 * 
 * Usage:
 * <element data-popover="Your popover content here">Trigger text</element>
 * 
 * Features:
 * - Powered by Floating UI for robust positioning
 * - Auto-positioning with collision detection
 * - Viewport boundary detection and smart fallbacks
 * - Click outside to close
 * - Escape key to close
 * - Smooth fade in/out animations
 * - Multiple popover support
 */

(($) => {

    class IntucartPopovers {
        constructor() {
            this.activePopover = null;
            this.activeReference = null;
            this.cleanup = null;
            this.init();
        }

        init() {
            // Wait for both jQuery and Floating UI to be available
            if (typeof $ === 'undefined' || typeof window.FloatingUIDOM === 'undefined') {
                setTimeout(() => this.init(), 100);
                return;
            }

            this.bindEvents();
        }

        bindEvents() {
            // Handle popover triggers
            $(document).on('click', '[data-popover]', (e) => {
                e.preventDefault();
                e.stopPropagation();

                const $trigger = $(e.currentTarget);
                this.showPopover($trigger);
            });

            // Close popover when clicking elsewhere
            $(document).on('click', (e) => {
                if (this.activePopover && !$(e.target).closest('.intucart-popover, [data-popover]').length) {
                    this.hidePopover();
                }
            });

            // Close popover on escape key
            $(document).on('keydown', (e) => {
                if (e.keyCode === 27 && this.activePopover) { // Escape key
                    this.hidePopover();
                }
            });
        }

        async showPopover($trigger) {
            const popoverText = $trigger.attr('data-popover');

            if (!popoverText || !window.FloatingUIDOM) {
                return;
            }

            // Remove existing popover
            this.hidePopover();

            // Create new popover
            const $popover = $(`<div class="intucart-popover">${popoverText}</div>`);
            $('body').append($popover);

            // Store references
            this.activePopover = $popover[0];
            this.activeReference = $trigger[0];

            // Create arrow element
            const arrowElement = document.createElement('div');
            arrowElement.className = 'intucart-popover-arrow';
            this.activePopover.appendChild(arrowElement);

            // Access Floating UI functions from global object
            const { computePosition, flip, shift, offset, arrow, autoUpdate } = window.FloatingUIDOM;

            // Compute position with Floating UI
            const updatePosition = async () => {
                if (!this.activePopover || !this.activeReference) return;

                try {
                    const { x, y, placement, middlewareData } = await computePosition(
                        this.activeReference,
                        this.activePopover,
                        {
                            placement: 'top',
                            middleware: [
                                offset(8), // 8px gap between trigger and popover
                                flip(), // Flip to bottom if no space above
                                shift({ padding: 10 }), // Keep 10px from viewport edges
                                arrow({ element: arrowElement }) // Position arrow
                            ],
                        }
                    );

                    // Apply computed position
                    Object.assign(this.activePopover.style, {
                        left: `${x}px`,
                        top: `${y}px`,
                        position: 'absolute',
                    });

                    // Position arrow (following tutorial pattern)
                    if (middlewareData.arrow) {
                        const { x: arrowX, y: arrowY } = middlewareData.arrow;
                        const side = placement.split('-')[0];

                        const staticSide = {
                            top: 'bottom',
                            right: 'left',
                            bottom: 'top',
                            left: 'right',
                        }[side];

                        Object.assign(arrowElement.style, {
                            left: arrowX != null ? `${arrowX}px` : '',
                            top: arrowY != null ? `${arrowY}px` : '',
                            right: '',
                            bottom: '',
                            [staticSide]: '-4px',
                        });
                    }
                } catch (error) {
                    console.error('Floating UI positioning error:', error);
                    // Fallback to simple positioning
                    const triggerRect = this.activeReference.getBoundingClientRect();
                    const popoverRect = this.activePopover.getBoundingClientRect();

                    Object.assign(this.activePopover.style, {
                        left: `${triggerRect.left + (triggerRect.width / 2) - (popoverRect.width / 2)}px`,
                        top: `${triggerRect.top - popoverRect.height - 8}px`,
                        position: 'absolute',
                    });
                }
            };

            // Initial positioning
            await updatePosition();

            // Auto-update position on scroll/resize
            if (autoUpdate) {
                this.cleanup = autoUpdate(this.activeReference, this.activePopover, updatePosition);
            }

            // Show with animation
            requestAnimationFrame(() => {
                if (this.activePopover) {
                    $(this.activePopover).addClass('show');
                }
            });
        }

        hidePopover() {
            if (this.activePopover) {
                // Cleanup auto-update
                if (this.cleanup) {
                    this.cleanup();
                    this.cleanup = null;
                }

                // Hide with animation
                $(this.activePopover).removeClass('show');

                setTimeout(() => {
                    if (this.activePopover) {
                        $(this.activePopover).remove();
                        this.activePopover = null;
                        this.activeReference = null;
                    }
                }, 200);
            }
        }

        // Public method to manually create popovers
        static create(trigger, content, options = {}) {
            const $trigger = $(trigger);
            $trigger.attr('data-popover', content);

            // Apply any custom options
            if (options.className) {
                $trigger.addClass(options.className);
            }
        }

        // Public method to destroy popovers
        static destroy(trigger) {
            const $trigger = $(trigger);
            $trigger.removeAttr('data-popover');
        }
    }

    // Initialize when DOM is ready
    $(document).ready(() => {
        window.IntucartPopovers = new IntucartPopovers();
    });

    // Expose the class for manual instantiation if needed
    window.IntucartPopoversClass = IntucartPopovers;

})(jQuery);