(($) => {
  const getSelectLib = ($el) => {
    // WooCommerce ships selectWoo; fall back to select2 if present
    if ($.fn.selectWoo) return $.fn.selectWoo.bind($el);
    if ($.fn.select2) return $.fn.select2.bind($el);
    return null;
  };

  const initPromptSelect = ($el) => {
    const selectLib = getSelectLib($el);
    if (!selectLib) {
      // retry once after a short delay to allow CDN enqueues to register
      setTimeout(() => {
        const libRetry = getSelectLib($el);
        if (libRetry) {
          initPromptSelect($el);
        }
      }, 200);
      return;
    }

    const ajaxAction = $el.data('action');
    const nonce = $el.data('nonce');
    if (!ajaxAction) return;

    try {
      selectLib({
        multiple: true,
        width: '100%',
        placeholder: $el.data('placeholder') || 'Start typing to search...',
        ajax: {
          url: typeof window !== 'undefined' && typeof window.ajaxurl !== 'undefined' ? window.ajaxurl : '',
          dataType: 'json',
          delay: 250,
          data: (params) => ({
            action: ajaxAction,
            q: params?.term ?? '',
            nonce: nonce,
            exclude_id: $el.data('current-id') || '',
          }),
          processResults: (data) => {
            // Expecting [{ id, text }]
            if (!Array.isArray(data)) {
              console.warn(`Intucart: Invalid AJAX response for ${ajaxAction}`, data);
              return { results: [] };
            }
            return { results: data };
          },
          error: (xhr, status, error) => {
            console.error(`Intucart: AJAX error for ${ajaxAction}`, { xhr, status, error });
          },
          cache: true,
        },
        minimumInputLength: 1,
        language: {
          noResults: () => 'No results. Create a prompt first, or try a different search.',
          searching: () => 'Searching...',
          inputTooShort: () => 'Type at least 1 character to search',
          errorLoading: () => 'Error loading results. Check console for details.',
        },
        // Preserve existing preselected options rendered server-side
      });
      // Force-hide native select in case styles didn't load yet
      $el.addClass('select2-hidden-accessible');
    } catch (e) {
      console.error('Intucart: Failed to initialize select2 for prompts', e);
    }
  };

  const initAll = () => {
    // Initialize any of our AJAX selects (products, posts, prompts, etc.)
    $('select[data-action^="intucart_search_"]').each((_, el) => {
      initPromptSelect($(el));
    });

    // Media library for icon selection
    const $iconInput = $('#intucart-icon-input');
    const $preview = $('.intucart-icon-preview');
    let frame = null;

    const updatePreview = (url) => {
      $preview.empty();
      if (url && /^https?:\/\//i.test(url)) {
        const $img = $('<img>', {
          src: url,
          alt: 'Icon preview',
          css: {
            maxWidth: '80px',
            height: 'auto',
            display: 'block',
            border: '1px solid #ddd',
            padding: '2px',
            borderRadius: '2px',
          },
        });
        $preview.append($img);
      }
    };

    $('.intucart-icon-select')
      .off('click')
      .on('click', (e) => {
        e.preventDefault();
        if (!wp || !wp.media) {
          return;
        }
        if (frame) {
          frame.open();
          return;
        }
        frame = wp.media({
          title: 'Select Icon',
          button: { text: 'Use this image' },
          multiple: false,
        });
        frame.on('select', () => {
          const attachment = frame.state().get('selection').first().toJSON();
          const url = attachment?.sizes?.thumbnail?.url || attachment?.url || '';
          $iconInput.val(url).trigger('change');
          $('.intucart-icon-remove').show();
          updatePreview(url);
        });
        frame.open();
      });

    $('.intucart-icon-remove')
      .off('click')
      .on('click', (e) => {
        e.preventDefault();
        $iconInput.val('').trigger('change');
        $(e.currentTarget).hide();
        updatePreview('');
      });

    $iconInput.off('change').on('change', () => {
      const val = $iconInput.val();
      updatePreview(typeof val === 'string' ? val : '');
    });
  };

  $(document).ready(() => {
    initAll();
    // If the UI is dynamically re-rendered, consumers can call: window.IntucartPromptsInit && window.IntucartPromptsInit();
    window.IntucartPromptsInit = initAll;

    // Handle dismissible prompt indexing notice
    initPromptIndexingNotice();
  });

  /**
   * Initialize the dismissible prompt indexing notice
   */
  const initPromptIndexingNotice = () => {
    const $notice = $('#intucart-prompt-indexing-notice');
    if (!$notice.length) return;

    // Check if notice was previously dismissed
    if (localStorage.getItem('intucart_prompt_indexing_notice_dismissed') === 'true') {
      $notice.hide();
    }

    // Handle dismiss click
    $notice.find('.intucart-notice-dismiss').on('click', () => {
      $notice.fadeOut(300);
      localStorage.setItem('intucart_prompt_indexing_notice_dismissed', 'true');
    });
  };
})(jQuery);
