/**
 * Intucart Status Dashboard JavaScript
 * Handles sync buttons and notices for the status dashboard
 */
(($) => {
  // Initialize when document is ready
  $(document).ready(() => {
    initSyncButtons();
    initCacheActions();
    initLogActions();
    initHealthActions();
    initUsageActions();
  });

  /**
   * Initialize sync now buttons with enhanced indexing delay handling
   */
  function initSyncButtons() {
    $(".sync-now-button").on("click", function (e) {
      e.preventDefault();

      const $button = $(this);
      const originalText = $button.text();
      const syncType = $button.data("type");
      const postType = $button.data("post-type"); // Get post type for post_types sync
      const taxonomy = $button.data("taxonomy"); // Get taxonomy for individual taxonomy sync
      const taxonomyLabel = $button.data("taxonomy-label"); // Get friendly taxonomy name

      // Disable button and show loading state
      $button.prop("disabled", true).text("Syncing...");

      // Get user-friendly sync type name
      const syncTypeNames = {
        post_types: postType ? `${postType}s` : 'content',
        taxonomies: taxonomy ? (taxonomyLabel || taxonomy) : 'taxonomies',
        users: 'users',
        products: 'products',
        pages: 'pages',
        posts: 'posts'
      };

      const displayName = syncTypeNames[syncType] || syncType;

      // Show initial sync started notification
      const toastId = manageSyncNotification('started', displayName);

      // Prepare AJAX data
      const ajaxData = {
        action: "intucart_sync",
        sync_type: syncType,
        source: "status",
        nonce: intucartStatus.nonce,
      };

      // Add post_type parameter if this is a post_types sync
      if (syncType === "post_types" && postType) {
        ajaxData.post_type = postType;
      }

      // Add taxonomy parameter if this is an individual taxonomy sync
      if (syncType === "taxonomies" && taxonomy) {
        ajaxData.taxonomy = taxonomy;
      }

      // Make AJAX request to trigger sync
      $.ajax({
        url: intucartStatus.ajaxUrl,
        type: "POST",
        data: ajaxData,
        timeout: 30000,
        success: (response) => {
          if (response.success) {
            // Show sync completed notification with server message + indexing info
            const serverMessage = response.data.message || '';
            const enhancedMessage = serverMessage ?
              `${serverMessage}. Intufind is now indexing content for search...` :
              null;
            manageSyncNotification('completed', displayName, enhancedMessage, toastId);

            // Schedule page refresh after indexing delay
            setTimeout(() => {
              manageSyncNotification('refreshing', displayName, null, toastId);
              setTimeout(() => {
                location.reload();
              }, 1000);
            }, 8000); // 8 second delay for OpenSearch indexing

          } else {
            manageSyncNotification('error', displayName, response.data.message || "Failed to sync. Please try again.", toastId);
            // Re-enable button after error
            setTimeout(() => {
              $button.prop("disabled", false).text(originalText);
            }, 2000);
          }
        },
        error: (_xhr, status, _error) => {
          let errorMsg = 'Server error during sync. Please try again.';
          if (status === 'timeout') {
            errorMsg = 'Sync request timed out - sync may still be running in background';
          }
          manageSyncNotification('error', displayName, errorMsg, toastId);
          // Re-enable button after error
          setTimeout(() => {
            $button.prop("disabled", false).text(originalText);
          }, 2000);
        },
      });
    });
  }

  /**
   * Manage sync notification using existing toast system
   * Handles both creating new toasts and updating existing ones
   */
  function manageSyncNotification(stage, syncDescription, customMessage = null, existingToastId = null) {
    const toastId = existingToastId || `sync-${syncDescription.replace(/\s+/g, '-').toLowerCase()}`;
    const capitalizedDesc = syncDescription.charAt(0).toUpperCase() + syncDescription.slice(1);

    let message, type;

    switch (stage) {
      case 'started':
        message = `${capitalizedDesc} sync started. Uploading content to Intufind and preparing search index...`;
        type = 'loading';
        break;
      case 'completed':
        message = customMessage || `${capitalizedDesc} sync complete! Intufind is indexing content. Page will refresh in 8 seconds to show new search results.`;
        type = 'loading';
        break;
      case 'refreshing':
        message = 'Loading updated search results...';
        type = 'loading';
        break;
      case 'error':
        message = customMessage || `Failed to index ${syncDescription.toLowerCase()} to Intufind. Please check connection and try again.`;
        type = 'error';
        break;
    }

    // Use existing toast system - create new or update existing
    if (typeof window.showToast === 'function' && typeof window.updateToast === 'function') {
      if (existingToastId) {
        // Update existing toast
        window.updateToast(toastId, type, message, {
          persistent: (stage === 'completed' || stage === 'refreshing'),
          autoDismiss: (stage === 'error'),
          dismissDelay: (stage === 'error') ? 5000 : undefined
        });
      } else {
        // Create new toast
        window.showToast(type, message, {
          id: toastId,
          persistent: (stage === 'started' || stage === 'completed'),
          autoDismiss: (stage === 'error')
        });
      }
    }

    return toastId;
  }


  /**
   * Initialize cache action buttons
   */
  function initCacheActions() {
    // Handle clear cache group button
    $(".clear-cache-group").on("click", function () {
      const group = $(this).data("group");
      if (
        confirm(
          `Are you sure you want to clear the cache for the "${group}" group?`
        )
      ) {
        $.ajax({
          url: intucartStatus.ajaxUrl,
          type: "POST",
          data: {
            action: "intucart_clear_cache_group",
            group: group,
            nonce: intucartStatus.cacheNonce,
          },
          success: (response) => {
            if (response.success) {
              alert(`Cache cleared successfully for group: ${group}`);
              location.reload();
            } else {
              alert(`Failed to clear cache: ${response.data.message}`);
            }
          },
          error: () => {
            alert("An error occurred while clearing the cache.");
          },
        });
      }
    });

    // Handle clear all cache button
    $(".clear-all-cache").on("click", () => {
      if (confirm("Are you sure you want to clear all cache?")) {
        $.ajax({
          url: intucartStatus.ajaxUrl,
          type: "POST",
          data: {
            action: "intucart_clear_all_cache",
            nonce: intucartStatus.cacheNonce,
          },
          beforeSend: () => {
            $(".clear-all-cache").prop("disabled", true).text("Clearing...");
          },
          success: (response) => {
            if (response.success) {
              alert("All cache cleared successfully.");
              location.reload();
            } else {
              alert(`Failed to clear cache: ${response.data.message}`);
              $(".clear-all-cache")
                .prop("disabled", false)
                .text("Clear All Cache");
            }
          },
          error: () => {
            alert("An error occurred while clearing the cache.");
            $(".clear-all-cache")
              .prop("disabled", false)
              .text("Clear All Cache");
          },
        });
      }
    });

    // Handle reset cache stats button
    $(".reset-cache-stats").on("click", () => {
      if (confirm("Are you sure you want to reset all cache statistics?")) {
        $.ajax({
          url: intucartStatus.ajaxUrl,
          type: "POST",
          data: {
            action: "intucart_reset_cache_stats",
            nonce: intucartStatus.cacheNonce,
          },
          beforeSend: () => {
            $(".reset-cache-stats").prop("disabled", true).text("Resetting...");
          },
          success: (response) => {
            if (response.success) {
              alert("Cache statistics reset successfully.");
              location.reload();
            } else {
              alert(
                `Failed to reset cache statistics: ${response.data.message}`
              );
              $(".reset-cache-stats")
                .prop("disabled", false)
                .text("Reset Statistics");
            }
          },
          error: () => {
            alert("An error occurred while resetting cache statistics.");
            $(".reset-cache-stats")
              .prop("disabled", false)
              .text("Reset Statistics");
          },
        });
      }
    });
  }

  /**
   * Initialize log action buttons
   */
  function initLogActions() {
    // Handle clear logs button
    $(".clear-all-logs").on("click", () => {
      if (confirm("Are you sure you want to clear all error logs?")) {
        $.ajax({
          url: intucartStatus.ajaxUrl,
          type: "POST",
          data: {
            action: "intucart_clear_logs",
            nonce: intucartStatus.nonce,
          },
          beforeSend: () => {
            $(".clear-all-logs").prop("disabled", true).text("Clearing...");
          },
          success: (response) => {
            if (response.success) {
              alert(response.data.message);
              location.reload();
            } else {
              alert(response.data.message || "Failed to clear logs.");
              $(".clear-all-logs")
                .prop("disabled", false)
                .text("Clear All Logs");
            }
          },
          error: () => {
            alert("An error occurred while clearing logs.");
            $(".clear-all-logs").prop("disabled", false).text("Clear All Logs");
          },
        });
      }
    });
  }

  /**
   * Initialize health action buttons
   */
  function initHealthActions() {
    // Handle refresh health status button
    $(".refresh-health-btn").on("click", function () {
      const $button = $(this);
      const $lastCheck = $(".health-last-check");

      $button.prop("disabled", true).text("Refreshing...");
      const toastId = showToast(
        "loading",
        "Refreshing health status...",
        { id: "refresh-health", persistent: true }
      );

      $.ajax({
        url: intucartStatus.ajaxUrl,
        type: "POST",
        data: {
          action: "intucart_refresh_health",
          nonce: intucartStatus.nonce,
        },
        success: (response) => {
          if (response.success) {
            updateToast(
              toastId,
              "success",
              "Health status refreshed successfully.",
              { autoDismiss: true, dismissDelay: 3000 }
            );
            // Update last check time
            $lastCheck.text(`Last checked: ${new Date().toLocaleString()}`);
            // Reload page to show updated data
            setTimeout(() => location.reload(), 1000);
          } else {
            updateToast(
              toastId,
              "error",
              response.data.message || "Failed to refresh health status.",
              { autoDismiss: false }
            );
            $button.prop("disabled", false).text("Refresh Health Status");
          }
        },
        error: () => {
          updateToast(
            toastId,
            "error",
            "Server error while refreshing health status.",
            { autoDismiss: false }
          );
          $button.prop("disabled", false).text("Refresh Health Status");
        },
      });
    });

  }

  /**
   * Initialize usage data refresh actions
   */
  function initUsageActions() {
    // Handle refresh usage data button
    $(".refresh-usage-data").on("click", function () {
      const $button = $(this);

      $button.prop("disabled", true).text("Refreshing...");
      const toastId = showToast(
        "loading",
        "Refreshing usage data...",
        { id: "refresh-usage", persistent: true }
      );

      $.ajax({
        url: intucartStatus.ajaxUrl,
        type: "POST",
        data: {
          action: "intucart_refresh_usage_data",
          nonce: intucartStatus.nonce,
        },
        success: (response) => {
          if (response.success) {
            updateToast(
              toastId,
              "success",
              response.data.message || "Usage data refreshed successfully.",
              { autoDismiss: true, dismissDelay: 3000 }
            );
            // Reload page to show updated data
            if (response.data.reload) {
              setTimeout(() => location.reload(), 1000);
            }
          } else {
            updateToast(
              toastId,
              "error",
              response.data.message || "Failed to refresh usage data.",
              { autoDismiss: false }
            );
            $button.prop("disabled", false).text("Refresh Usage Data");
          }
        },
        error: () => {
          updateToast(
            toastId,
            "error",
            "Server error while refreshing usage data.",
            { autoDismiss: false }
          );
          $button.prop("disabled", false).text("Refresh Usage Data");
        },
      });
    });
  }
})(jQuery);
