/**
 * Intucart Sync Columns Admin JavaScript
 * Handles AJAX functionality for sync control checkboxes in admin list tables
 */

(($) => {
  /**
   * Initialize sync columns functionality
   */
  function initSyncColumns() {
    // Initialize enhanced popovers with proper line break support
    initEnhancedPopovers();

    // Handle individual checkbox toggles
    $(document).on('change', '.intucart-sync-checkbox', handleSyncToggle);
    $(document).on('change', '.intucart-searchable-checkbox', handleSearchableToggle);

    // Handle bulk action form submission
    $('#doaction, #doaction2').on('click', function (e) {
      const action = $(this).siblings('select').val();
      const validActions = [
        'intucart_enable_sync',
        'intucart_disable_sync',
        'intucart_make_searchable',
        'intucart_hide_from_search',
      ];
      if (validActions.includes(action)) {
        return confirmBulkAction(e, action);
      }
    });
  }

  /**
   * Handle individual sync checkbox toggle
   */
  function handleSyncToggle(e) {
    const $checkbox = $(this);
    const $container = $checkbox.closest('.intucart-sync-toggle');
    const postId = $checkbox.data('post-id');
    const nonce = $checkbox.data('nonce');

    if (!postId || !nonce) {
      showError($container, 'Invalid request data');
      e.preventDefault();
      return;
    }

    // Show loading state
    $container.addClass('loading').removeClass('error');
    $checkbox.prop('disabled', true);

    // Hide status icon and store it for later restoration
    const $statusIcon = $container.find('.intucart-sync-status');
    let storedIconHtml = '';
    if ($statusIcon.length) {
      storedIconHtml = $statusIcon.prop('outerHTML');
      $statusIcon.hide();
    }

    // Make AJAX request
    $.ajax({
      url: intucartSync.ajaxUrl,
      type: 'POST',
      data: {
        action: 'intucart_toggle_sync',
        post_id: postId,
        nonce: nonce,
      },
      success: (response) => {
        if (response.success) {
          // Update checkbox state to match server response
          $checkbox.prop('checked', !response.data.excluded);

          // Update the status icon if provided
          if (response.data.status_icon_html) {
            const $existingIcon = $container.find('.intucart-sync-status');
            if ($existingIcon.length) {
              // Replace existing icon (including hidden ones)
              $existingIcon.replaceWith(response.data.status_icon_html);
            } else {
              // Add new icon if none exists
              $container.append(response.data.status_icon_html);
            }
          } else if (storedIconHtml) {
            // Restore original icon if no new icon provided
            $container.append(storedIconHtml);
          }

          // Update the searchable checkbox state based on sync status
          const $row = $checkbox.closest('tr');
          const $searchableContainer = $row.find('.intucart-searchable-toggle');
          const $searchableCheckbox = $searchableContainer.find('.intucart-searchable-checkbox');
          const $searchableStatus = $searchableContainer.find('.intucart-searchable-status');

          if (!response.data.excluded) {
            // Item is now synced - enable searchable checkbox
            $searchableContainer.removeClass('disabled');
            $searchableCheckbox.prop('disabled', false);

            // Update status icon to show it's searchable (default state)
            if ($searchableStatus.hasClass('disabled')) {
              $searchableCheckbox.prop('checked', true);
              $searchableStatus.removeClass('disabled').addClass('searchable');
              $searchableStatus.attr('data-popover', `Visible in ${intucartSync.pluginName} search results`);
              $searchableStatus.html('<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512" width="12" height="12"><path d="M173.898 439.404l-166.4-166.4c-9.997-9.997-9.997-26.206 0-36.204l36.203-36.204c9.997-9.998 26.207-9.998 36.204 0L192 312.69 432.095 72.596c9.997-9.997 26.207-9.997 36.204 0l36.203 36.204c9.997 9.997 9.997 26.206 0 36.204l-294.4 294.401c-9.998 9.997-26.207 9.997-36.204-.001z"/></svg>');
            }
          } else {
            // Item is not synced - disable searchable checkbox
            $searchableContainer.addClass('disabled');
            $searchableCheckbox.prop('disabled', true).prop('checked', false);

            // Update status icon to show it's disabled
            $searchableStatus.removeClass('searchable not-searchable').addClass('disabled');
            $searchableStatus.attr('data-popover', 'Not indexed - enable AI Knowledge first to make searchable');
            $searchableStatus.html('<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512" width="12" height="12"><path d="M256 8C119 8 8 119 8 256s111 248 248 248 248-111 248-248S393 8 256 8zm0 448c-110.5 0-200-89.5-200-200S145.5 56 256 56s200 89.5 200 200-89.5 200-200 200zm101.8-262.2L295.6 256l62.2 62.2c4.7 4.7 4.7 12.3 0 17l-22.6 22.6c-4.7 4.7-12.3 4.7-17 0L256 295.6l-62.2 62.2c-4.7 4.7-12.3 4.7-17 0l-22.6-22.6c-4.7-4.7-4.7-12.3 0-17l62.2-62.2-62.2-62.2c-4.7-4.7-4.7-12.3 0-17l22.6-22.6c4.7-4.7 12.3-4.7 17 0l62.2 62.2 62.2-62.2c4.7-4.7 12.3-4.7 17 0l22.6 22.6c4.7 4.7 4.7 12.3 0 17z"/></svg>');
          }

          // Handle operation status if an operation was attempted
          if (response.data.operation_attempted) {
            const $statusIcon = $container.find('.intucart-sync-status');
            if ($statusIcon.length) {
              if (response.data.operation_success) {
                // Mark icon as operation successful (normal green)
                $statusIcon.removeClass('sync-warning sync-error').addClass('sync-success');
                const operationType = response.data.operation_type;
                const operationName = operationType === 'delete' ? 'removed from cloud' : 'synced successfully';

                // Build enhanced popover content with metadata
                const enhancedPopoverContent = buildEnhancedPopoverContent(
                  `Content ${operationName}`,
                  'Realtime',
                  operationType.charAt(0).toUpperCase() + operationType.slice(1),
                );

                $statusIcon.attr('data-popover', enhancedPopoverContent);
                $statusIcon.removeAttr('data-sync-error');
              } else {
                // Mark icon as operation failed (warning orange)
                const errorMsg = response.data.operation_error || 'Operation failed';
                const operationType = response.data.operation_type;
                const operationName = operationType === 'delete' ? 'deletion' : 'sync';

                // Build enhanced popover content with error metadata
                const enhancedPopoverContent = buildEnhancedPopoverContent(
                  `${operationName.charAt(0).toUpperCase() + operationName.slice(1)} failed: ${errorMsg}`,
                  'Realtime',
                  operationType.charAt(0).toUpperCase() + operationType.slice(1),
                );

                $statusIcon.removeClass('sync-success sync-error').addClass('sync-warning');
                $statusIcon.attr('data-popover', enhancedPopoverContent);
                $statusIcon.attr('data-sync-error', errorMsg);
              }
            }

            // Reinitialize enhanced popovers for the updated element
            initEnhancedPopovers();
          }

          // Visual feedback is handled by the icon color and hover popover
        } else {
          // Revert checkbox state on error
          $checkbox.prop('checked', !$checkbox.prop('checked'));

          // Restore original icon on error
          if (storedIconHtml) {
            const $hiddenIcon = $container.find('.intucart-sync-status:hidden');
            if ($hiddenIcon.length) {
              $hiddenIcon.show();
            } else {
              $container.append(storedIconHtml);
            }
          }

          // Visual feedback is handled by reverting the checkbox and icon state
        }
      },
      error: (_xhr, _status, error) => {
        // Revert checkbox state on error
        $checkbox.prop('checked', !$checkbox.prop('checked'));

        // Restore original icon on error
        if (storedIconHtml) {
          const $hiddenIcon = $container.find('.intucart-sync-status:hidden');
          if ($hiddenIcon.length) {
            $hiddenIcon.show();
          } else {
            $container.append(storedIconHtml);
          }
        }

        // Visual feedback is handled by reverting the checkbox and icon state
        console.error('Intucart sync toggle error:', error);
      },
      complete: () => {
        // Remove loading state
        $container.removeClass('loading');
        $checkbox.prop('disabled', false);
      },
    });
  }

  /**
   * Handle individual searchable checkbox toggle
   */
  function handleSearchableToggle(e) {
    const $checkbox = $(this);
    const $container = $checkbox.closest('.intucart-searchable-toggle');
    const postId = $checkbox.data('post-id');
    const nonce = $checkbox.data('nonce');

    if (!postId || !nonce) {
      showError($container, 'Invalid request data');
      e.preventDefault();
      return;
    }

    // Show loading state
    $container.addClass('loading').removeClass('error');
    $checkbox.prop('disabled', true);

    // Hide status icon and store it for later restoration
    const $statusIcon = $container.find('.intucart-searchable-status');
    let storedIconHtml = '';
    if ($statusIcon.length) {
      storedIconHtml = $statusIcon.prop('outerHTML');
      $statusIcon.hide();
    }

    // Make AJAX request
    $.ajax({
      url: intucartSync.ajaxUrl,
      type: 'POST',
      data: {
        action: 'intucart_toggle_searchable',
        post_id: postId,
        nonce: nonce,
      },
      success: (response) => {
        if (response.success) {
          // Update checkbox state to match server response
          $checkbox.prop('checked', response.data.searchable);

          // Update the status icon if provided
          if (response.data.status_icon_html) {
            const $existingIcon = $container.find('.intucart-searchable-status');
            if ($existingIcon.length) {
              // Replace existing icon (including hidden ones)
              $existingIcon.replaceWith(response.data.status_icon_html);
            } else {
              // Add new icon if none exists
              $container.append(response.data.status_icon_html);
            }
          } else if (storedIconHtml) {
            // Restore original icon if no new icon provided
            $container.append(storedIconHtml);
          }

          // Handle operation status if an operation was attempted
          if (response.data.operation_attempted) {
            const $statusIcon = $container.find('.intucart-searchable-status');
            if ($statusIcon.length) {
              if (response.data.operation_success) {
                // Mark icon as operation successful
                $statusIcon.addClass('sync-success');

                // Update tooltip to show success
                const currentTitle = $statusIcon.attr('title') || '';
                $statusIcon.attr('title', `${currentTitle} (Updated successfully)`);
              } else {
                // Mark icon as operation failed
                $statusIcon.addClass('sync-warning');

                // Update tooltip to show error
                const currentTitle = $statusIcon.attr('title') || '';
                const errorMsg = response.data.operation_error || 'Update failed';
                $statusIcon.attr('title', `${currentTitle} (Update failed: ${errorMsg})`);
              }
            }
          }

          // Reinitialize popovers for the updated element
          initEnhancedPopovers();
        } else {
          // Revert checkbox state on error
          $checkbox.prop('checked', !$checkbox.prop('checked'));

          // Restore original icon on error
          if (storedIconHtml) {
            const $hiddenIcon = $container.find('.intucart-searchable-status:hidden');
            if ($hiddenIcon.length) {
              $hiddenIcon.show();
            } else {
              $container.append(storedIconHtml);
            }
          }
        }
      },
      error: (_xhr, _status, error) => {
        // Revert checkbox state on error
        $checkbox.prop('checked', !$checkbox.prop('checked'));

        // Restore original icon on error
        if (storedIconHtml) {
          const $hiddenIcon = $container.find('.intucart-searchable-status:hidden');
          if ($hiddenIcon.length) {
            $hiddenIcon.show();
          } else {
            $container.append(storedIconHtml);
          }
        }

        console.error('Intucart searchable toggle error:', error);
      },
      complete: () => {
        // Remove loading state
        $container.removeClass('loading');
        $checkbox.prop('disabled', false);
      },
    });
  }

  /**
   * Confirm bulk action
   */
  function confirmBulkAction(e, action) {
    const $form = $(e.target).closest('form');
    const $selected = $form.find('input[name="post[]"]:checked');

    if ($selected.length === 0) {
      alert('Please select at least one item.');
      e.preventDefault();
      return false;
    }

    const count = $selected.length;
    const itemText = count === 1 ? 'item' : 'items';
    let actionText = '';

    switch (action) {
      case 'intucart_enable_sync':
        actionText = 'enable Intucart sync';
        break;
      case 'intucart_disable_sync':
        actionText = 'disable Intucart sync';
        break;
      case 'intucart_make_searchable':
        actionText = 'make searchable';
        break;
      case 'intucart_hide_from_search':
        actionText = 'hide from search';
        break;
      default:
        actionText = 'perform this action';
    }

    const message = `Are you sure you want to ${actionText} for ${count} ${itemText}?`;

    if (!confirm(message)) {
      e.preventDefault();
      return false;
    }

    return true;
  }

  /**
   * Handle bulk action success notices
   */
  function handleBulkActionNotices() {
    // Check for bulk action success parameters in URL
    const urlParams = new URLSearchParams(window.location.search);
    const bulkSync = urlParams.get('intucart_bulk_sync');
    const bulkCount = urlParams.get('intucart_bulk_count');

    if (bulkSync && bulkCount) {
      const count = Number.parseInt(bulkCount, 10);
      const action = bulkSync === 'enabled' ? 'enabled' : 'disabled';
      const itemText = count === 1 ? 'item' : 'items';
      const message = `Intucart sync ${action} for ${count} ${itemText}.`;

      showNotice(message, 'success');

      // Clean up URL parameters
      if (history.replaceState) {
        const newUrl = `${window.location.pathname}?${urlParams
          .toString()
          .replace(/[&?]intucart_bulk_sync=[^&]*/, '')
          .replace(/[&?]intucart_bulk_count=[^&]*/, '')
          .replace(/^&/, '')}`;

        history.replaceState({}, document.title, newUrl);
      }
    }
  }

  /**
   * Initialize enhanced popovers with proper multi-line support
   */
  function initEnhancedPopovers() {
    $('.intucart-sync-status[data-popover], .intucart-searchable-status[data-popover]').each(function () {
      const $element = $(this);
      const popoverText = $element.attr('data-popover');

      if (popoverText?.includes('\n')) {
        // Create a proper tooltip element for multi-line content
        $element.on('mouseenter', () => {
          showEnhancedPopover($element, popoverText);
        });

        $element.on('mouseleave', () => {
          hideEnhancedPopover();
        });

        // Remove the data-popover attribute to prevent CSS popover
        $element.removeAttr('data-popover');
      }
    });
  }

  /**
   * Show enhanced popover with HTML support
   */
  function showEnhancedPopover($element, text) {
    // Remove any existing popover
    hideEnhancedPopover();

    // Convert newlines to HTML breaks
    const htmlText = text.replace(/\n/g, '<br>');

    // Create popover element
    const $popover = $(`<div class="intucart-enhanced-popover">${htmlText}</div>`);

    // Position the popover above the icon with proper spacing
    const offset = $element.offset();
    const elementWidth = $element.outerWidth();

    // Add the popover to body first to get its dimensions
    $('body').append($popover);
    const popoverHeight = $popover.outerHeight();

    $popover.css({
      position: 'absolute',
      top: offset.top - popoverHeight - 8, // Position above with 8px gap
      left: offset.left + elementWidth / 2,
      transform: 'translateX(-50%)',
      zIndex: 1000,
    });

    // Animate in
    setTimeout(() => {
      $popover.addClass('show');
    }, 10);
  }

  /**
   * Hide enhanced popover
   */
  function hideEnhancedPopover() {
    $('.intucart-enhanced-popover').remove();
  }

  /**
   * Build enhanced popover content with metadata formatting
   */
  function buildEnhancedPopoverContent(statusMessage, method, operation) {
    const lines = [statusMessage];

    if (method) {
      lines.push(`• Method: ${method}`);
    }

    if (operation) {
      lines.push(`• Operation: ${operation}`);
    }

    return lines.join('\n');
  }

  /**
   * Initialize when document is ready
   */
  $(document).ready(() => {
    initSyncColumns();
    handleBulkActionNotices();
  });
})(jQuery);
