/**
 * Webhook Management JavaScript
 * Handles the webhook endpoint form and AJAX operations
 */

jQuery(document).ready(($) => {
  // Show form for adding new endpoint
  $('#add-webhook-endpoint').on('click', () => {
    showWebhookForm();
  });

  // Show form for editing endpoint
  $(document).on('click', '.edit-endpoint', function () {
    const index = $(this).data('index');
    const endpointData = $(this).closest('tr').data('endpoint');
    showWebhookForm(endpointData, index);
  });

  // Cancel form - using event delegation since form is initially hidden
  $(document).on('click', '#cancel-webhook-form', () => {
    hideWebhookForm();
  });

  // Test webhook from form - using event delegation since form is initially hidden
  $(document).on('click', '#test-webhook-form', () => {
    testWebhookFromForm();
  });

  // Handle save button click directly for webhook endpoint form ONLY
  $(document).on('click', '#save-webhook-endpoint', (e) => {
    // Double-check we're targeting the right button
    if (e.target.id === 'save-webhook-endpoint' || $(e.target).closest('#save-webhook-endpoint').length > 0) {
      e.preventDefault(); // Prevent the default form submission
      e.stopPropagation(); // Stop event from bubbling
      saveWebhookEndpoint();
    }
  });

  // Delete endpoint
  $(document).on('click', '.delete-endpoint', function () {
    const endpointData = $(this).closest('tr').data('endpoint');

    if (confirm(`Are you sure you want to delete the webhook "${endpointData.name}"?`)) {
      deleteEndpoint(endpointData.id);
    }
  });

  // Test endpoint from table
  $(document).on('click', '.test-endpoint', function () {
    const endpointData = $(this).closest('tr').data('endpoint');
    testEndpoint(endpointData);
  });

  // Reactivate suspended endpoint
  $(document).on('click', '.reactivate-endpoint', function () {
    const endpointData = $(this).closest('tr').data('endpoint');

    if (confirm(`Reactivate webhook "${endpointData.name}"?\n\nMake sure you've fixed the endpoint issue before reactivating.`)) {
      reactivateEndpoint(endpointData);
    }
  });

  /**
   * Show the webhook form for adding or editing
   */
  function showWebhookForm(endpointData = null, editIndex = null) {
    const isEdit = endpointData !== null;
    currentEditingIndex = editIndex;

    // Show form first so DOM elements are accessible
    $('#webhook-form-container').show();
    $('#add-webhook-endpoint').hide();

    // Update form title
    $('#webhook-form-title').text(isEdit ? 'Edit Webhook Endpoint' : 'Add Webhook Endpoint');
    $('#webhook-form-mode').val(isEdit ? 'edit' : 'add');

    // Clear all input fields in the webhook endpoint form div
    $(
      '#webhook-endpoint-form input[type="text"], #webhook-endpoint-form input[type="url"], #webhook-endpoint-form textarea',
    ).val('');
    $('#webhook-endpoint-form input[type="checkbox"]').prop('checked', false);
    $('#webhook-endpoint-form select').prop('selectedIndex', 0);

    if (isEdit) {
      $('#webhook-endpoint-id').val(endpointData.id || '');
      $('#webhook-name').val(endpointData.name || '');
      $('#webhook-description').val(endpointData.description || '');
      $('#webhook-url').val(endpointData.url || '');
      $('#webhook-method').val(endpointData.method || 'POST');
      $('#webhook-secret').val(endpointData.secret === '[CONFIGURED]' ? '' : endpointData.secret || '');
      $('#webhook-payload-template').val(endpointData.payload_template || '');
      $('#webhook-enabled').prop('checked', endpointData.enabled !== false);

      // Check the appropriate event checkboxes
      $('input[name="events[]"]').prop('checked', false);
      if (endpointData.events && Array.isArray(endpointData.events)) {
        endpointData.events.forEach((event) => {
          $(`input[name="events[]"][value="${event}"]`).prop('checked', true);
        });
      }
    } else {
      $('#webhook-endpoint-id').val('');
      $('#webhook-payload-template').val('');
      $('#webhook-enabled').prop('checked', true);
    }
  }

  /**
   * Hide the webhook form
   */
  function hideWebhookForm() {
    $('#webhook-form-container').slideUp();
    $('#add-webhook-endpoint').show();
    currentEditingIndex = null;
  }

  /**
   * Save webhook endpoint via AJAX
   */
  function saveWebhookEndpoint() {
    // Collect checked events
    const events = [];
    $('input[name="events[]"]:checked').each(function () {
      events.push($(this).val());
    });

    if (events.length === 0) {
      alert('Please select at least one event type.');
      return;
    }

    // Show loading state
    const $saveBtn = $('#save-webhook-endpoint');
    const originalText = $saveBtn.html();
    $saveBtn.prop('disabled', true).html('<span class="dashicons dashicons-update spin"></span> Saving...');

    // Prepare AJAX data
    const ajaxData = {
      action: 'intucart_save_webhook_endpoint',
      nonce: intucartAdmin.nonce,
      id: $('#webhook-endpoint-id').val(),
      name: $('#webhook-name').val(),
      description: $('#webhook-description').val(),
      url: $('#webhook-url').val(),
      method: $('#webhook-method').val(),
      secret: $('#webhook-secret').val(),
      payload_template: $('#webhook-payload-template').val(),
      enabled: $('#webhook-enabled').is(':checked') ? 1 : 0,
      events: events,
    };

    // Show initial loading toast
    const webhookName = $('#webhook-name').val() || 'webhook';
    const toastId = `webhook-save-${Date.now()}`;

    if (typeof window.showToast === 'function') {
      window.showToast('loading', `Saving webhook "${webhookName}"...`, {
        id: toastId,
        persistent: true,
      });
    }

    $.post(ajaxurl, ajaxData)
      .done((response) => {
        if (response.success) {
          // Show success and inform about indexing delay
          if (typeof window.updateToast === 'function') {
            window.updateToast(
              toastId,
              'success',
              `Webhook "${webhookName}" saved successfully. Updating webhook index...`,
              {
                persistent: true,
              },
            );
          } else {
            showNotice('success', response.data.message);
          }

          hideWebhookForm();

          if (typeof window.updateToast === 'function') {
            window.updateToast(
              toastId,
              'success',
              `Webhook "${webhookName}" is now active and ready to receive events.`,
              {
                autoDismiss: true,
                dismissDelay: 4000,
              },
            );
          }
          refreshEndpointsTable();
        } else {
          if (typeof window.updateToast === 'function') {
            window.updateToast(toastId, 'error', response.data.message || 'Failed to save webhook endpoint.', {
              autoDismiss: true,
              dismissDelay: 5000,
            });
          } else {
            showNotice('error', response.data.message || 'Failed to save webhook endpoint.');
          }
        }
      })
      .fail((xhr, status, error) => {
        console.error('AJAX Save Failed:', {
          xhr: xhr,
          status: status,
          error: error,
          responseText: xhr.responseText,
        });

        if (typeof window.updateToast === 'function') {
          window.updateToast(toastId, 'error', 'Network error occurred while saving webhook endpoint.', {
            autoDismiss: true,
            dismissDelay: 5000,
          });
        } else {
          showNotice('error', 'Network error occurred while saving webhook endpoint.');
        }
      })
      .always(() => {
        $saveBtn.prop('disabled', false).html(originalText);
      });
  }

  /**
   * Delete webhook endpoint via AJAX
   */
  function deleteEndpoint(endpointId) {
    // Find the webhook name for better UX
    const endpointData = $(`.delete-endpoint[data-index]`)
      .filter(function () {
        return $(this).closest('tr').data('endpoint').id === endpointId;
      })
      .closest('tr')
      .data('endpoint');
    const webhookName = endpointData?.name || 'webhook';

    const toastId = `webhook-delete-${Date.now()}`;

    if (typeof window.showToast === 'function') {
      window.showToast('loading', `Deleting webhook "${webhookName}"...`, {
        id: toastId,
        persistent: true,
      });
    }

    $.post(ajaxurl, {
      action: 'intucart_delete_webhook_endpoint',
      nonce: intucartAdmin.nonce,
      endpoint_id: endpointId,
    })
      .done((response) => {
        if (response.success) {
          if (typeof window.updateToast === 'function') {
            window.updateToast(toastId, 'success', `Webhook "${webhookName}" has been removed.`, {
              autoDismiss: true,
              dismissDelay: 3000,
            });
          } else {
            showNotice('success', response.data.message);
          }
          refreshEndpointsTable();
        } else {
          if (typeof window.updateToast === 'function') {
            window.updateToast(toastId, 'error', response.data.message || 'Failed to delete webhook endpoint.', {
              autoDismiss: true,
              dismissDelay: 5000,
            });
          } else {
            showNotice('error', response.data.message || 'Failed to delete webhook endpoint.');
          }
        }
      })
      .fail(() => {
        if (typeof window.updateToast === 'function') {
          window.updateToast(toastId, 'error', 'Network error occurred while deleting webhook endpoint.', {
            autoDismiss: true,
            dismissDelay: 5000,
          });
        } else {
          showNotice('error', 'Network error occurred while deleting webhook endpoint.');
        }
      });
  }

  /**
   * Test webhook endpoint from table
   */
  function testEndpoint(endpointData) {
    const $testBtn = $('.test-endpoint').filter(function () {
      return $(this).closest('tr').data('endpoint').id === endpointData.id;
    });

    const originalText = $testBtn.html();
    $testBtn.prop('disabled', true).html('<span class="dashicons dashicons-update spin"></span> Testing...');

    $.post(ajaxurl, {
      action: 'intucart_test_webhook',
      nonce: intucartAdmin.nonce,
      webhook_id: endpointData.id,
    })
      .done((response) => {
        if (response.success) {
          showTestResultModal(endpointData.name, response.data);
        } else {
          showNotice('error', `Test webhook failed: ${response.data.message || 'Unknown error'}`);
        }
      })
      .fail(() => {
        showNotice('error', 'Network error occurred while testing webhook.');
      })
      .always(() => {
        $testBtn.prop('disabled', false).html(originalText);
      });
  }

  /**
   * Test webhook from form
   */
  function testWebhookFromForm() {
    const webhookId = $('#webhook-endpoint-id').val();
    const webhookName = $('#webhook-name').val() || 'Webhook';

    if (!webhookId) {
      alert('Please save the webhook first before testing.');
      return;
    }

    const $testBtn = $('#test-webhook-form');
    const originalText = $testBtn.html();
    $testBtn.prop('disabled', true).html('<span class="dashicons dashicons-update spin"></span> Testing...');

    $.post(ajaxurl, {
      action: 'intucart_test_webhook',
      nonce: intucartAdmin.nonce,
      webhook_id: webhookId,
    })
      .done((response) => {
        if (response.success) {
          showTestResultModal(webhookName, response.data);
        } else {
          showNotice('error', `Test webhook failed: ${response.data.message || 'Unknown error'}`);
        }
      })
      .fail(() => {
        showNotice('error', 'Network error occurred while testing webhook.');
      })
      .always(() => {
        $testBtn.prop('disabled', false).html(originalText);
      });
  }

  /**
   * Show test result modal with detailed information
   */
  function showTestResultModal(webhookName, data) {
    // Remove any existing modal
    $('#webhook-test-result-modal').remove();

    const isSuccess = data.success && data.response_code >= 200 && data.response_code < 300;
    const statusClass = isSuccess ? 'success' : 'error';
    const statusIcon = isSuccess ? '✓' : '✗';
    const statusText = isSuccess ? 'Success' : 'Failed';

    // Format response body for display
    let formattedBody = data.response_body || 'No response body';
    try {
      // Try to pretty-print JSON
      const parsed = JSON.parse(formattedBody);
      formattedBody = JSON.stringify(parsed, null, 2);
    } catch {
      // Not JSON, keep as-is
    }

    const modalHtml = `
      <div id="webhook-test-result-modal" class="webhook-test-modal-overlay">
        <div class="webhook-test-modal">
          <div class="webhook-test-modal-header ${statusClass}">
            <span class="status-icon">${statusIcon}</span>
            <h3>Webhook Test ${statusText}</h3>
            <button type="button" class="webhook-test-modal-close" aria-label="Close">&times;</button>
          </div>
          <div class="webhook-test-modal-body">
            <table class="webhook-test-result-table">
              <tr>
                <th>Webhook</th>
                <td>${escapeHtml(webhookName)}</td>
              </tr>
              <tr>
                <th>Status Code</th>
                <td><span class="webhook-test-status-badge ${statusClass}">${data.response_code || 'N/A'}</span></td>
              </tr>
              <tr>
                <th>Response Time</th>
                <td>${data.response_time || 'N/A'}</td>
              </tr>
              ${data.error_message
        ? `<tr>
                <th>Error</th>
                <td class="webhook-test-error-text">${escapeHtml(data.error_message)}</td>
              </tr>`
        : ''
      }
              <tr>
                <th>Response Body</th>
                <td><pre class="webhook-test-response-body">${escapeHtml(formattedBody)}</pre></td>
              </tr>
            </table>
          </div>
          <div class="webhook-test-modal-footer">
            <button type="button" class="button button-primary webhook-test-modal-close-btn">Close</button>
          </div>
        </div>
      </div>
    `;

    $('body').append(modalHtml);

    // Close modal handlers
    $('#webhook-test-result-modal').on('click', function (e) {
      if ($(e.target).is('.webhook-test-modal-overlay, .webhook-test-modal-close, .webhook-test-modal-close-btn')) {
        $(this).fadeOut(200, function () {
          $(this).remove();
        });
        $(document).off('keydown.webhookModal');
      }
    });

    // Close on escape key
    $(document).on('keydown.webhookModal', (e) => {
      if (e.key === 'Escape') {
        $('#webhook-test-result-modal').fadeOut(200, function () {
          $(this).remove();
        });
        $(document).off('keydown.webhookModal');
      }
    });
  }

  /**
   * Escape HTML to prevent XSS
   */
  function escapeHtml(text) {
    if (!text) return '';
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }

  /**
   * Refresh the endpoints table by fetching updated data via AJAX
   */
  function refreshEndpointsTable() {
    // Make AJAX request to get updated webhook list
    $.post(ajaxurl, {
      action: 'intucart_refresh_webhooks_table',
      nonce: intucartAdmin.nonce,
    })
      .done((response) => {
        if (response.success && response.data.html) {
          // Replace the table content with updated HTML
          $('.intucart-webhook-endpoints-table tbody').html(response.data.html);
        } else {
          window.location.reload();
        }
      })
      .fail(() => {
        window.location.reload();
      });
  }

  /**
   * Show notification message
   */
  function showNotice(type, message) {
    // Create notice element
    const noticeClass = type === 'success' ? 'notice-success' : 'notice-error';
    const $notice = $(`<div class="notice ${noticeClass} is-dismissible"><p>${message}</p></div>`);

    // Insert at top of form container or at top of page if form is hidden
    if ($('#webhook-form-container').is(':visible')) {
      $('#webhook-form-container').prepend($notice);
    } else {
      $('.wrap h1').after($notice);
    }

    // Auto-remove after 5 seconds
    setTimeout(() => {
      $notice.fadeOut(function () {
        $(this).remove();
      });
    }, 5000);

    // Make dismissible
    $notice.on('click', '.notice-dismiss', () => {
      $notice.fadeOut(function () {
        $(this).remove();
      });
    });
  }

  /**
   * Reactivate a suspended webhook endpoint
   */
  function reactivateEndpoint(endpointData) {
    const $btn = $(`.reactivate-endpoint[data-index]`).filter(function () {
      return $(this).closest('tr').data('endpoint').id === endpointData.id;
    });

    $btn.prop('disabled', true).text('Reactivating...');

    // Update webhook with status: active
    $.post(ajaxurl, {
      action: 'intucart_save_webhook_endpoint',
      nonce: intucartAdmin.nonce,
      id: endpointData.id,
      name: endpointData.name,
      url: endpointData.url,
      events: endpointData.events,
      secret: endpointData.secret === '[CONFIGURED]' ? '' : endpointData.secret,
      enabled: true, // Reactivate
      method: endpointData.method || 'POST',
      description: endpointData.description || '',
    })
      .done((response) => {
        if (response.success) {
          showNotice('success', 'Webhook reactivated successfully. Monitor its delivery stats to ensure it\'s working.');
          refreshEndpointsTable();
        } else {
          showNotice('error', response.data.message || 'Failed to reactivate webhook.');
          $btn.prop('disabled', false).text('Reactivate');
        }
      })
      .fail(() => {
        showNotice('error', 'Failed to reactivate webhook. Please try again.');
        $btn.prop('disabled', false).text('Reactivate');
      });
  }

  // Add CSS for spinning animation
  if (!$('#webhook-spinner-css').length) {
    $('<style id="webhook-spinner-css">')
      .text(
        '.dashicons.spin { animation: spin 1s linear infinite; } @keyframes spin { 0% { transform: rotate(0deg); } 100% { transform: rotate(360deg); } }',
      )
      .appendTo('head');
  }
});
