jQuery(document).ready(($) => {
  const TaxonomySync = {
    init: () => {
      this.bindEvents();
      this.loadSyncStatus();
    },

    bindEvents: () => {
      // Individual taxonomy sync buttons
      $(document).on("click", ".sync-taxonomy-btn", this.handleTaxonomySync);

      // Bulk sync buttons
      $("#sync-all-taxonomies").on("click", this.handleBulkSync);
      $("#force-full-sync").on("click", this.handleForceFullSync);

      // Refresh status button
      $("#refresh-status").on("click", this.loadSyncStatus);
    },

    handleTaxonomySync: (e) => {
      e.preventDefault();

      const $button = $(this);
      const taxonomy = $button.data("taxonomy");

      if (!confirm(intucartTaxonomySync.strings.confirmSync)) {
        return;
      }

      TaxonomySync.syncTaxonomy(taxonomy, $button);
    },

    syncTaxonomy: (taxonomy, $button) => {
      const originalText = $button.text();

      $button.prop("disabled", true).text(intucartTaxonomySync.strings.syncing);

      $.ajax({
        url: intucartTaxonomySync.ajaxUrl,
        type: "POST",
        data: {
          action: "intucart_sync_taxonomy",
          taxonomy: taxonomy,
          nonce: intucartTaxonomySync.nonce,
        },
        success: (response) => {
          if (response.success) {
            TaxonomySync.showNotice("success", response.data.message);

            // Update last sync time
            $(`.last-sync-${taxonomy}`).text(response.data.last_sync);

            // Update status indicator
            TaxonomySync.updateStatusIndicator(taxonomy, "success");
          } else {
            TaxonomySync.showNotice(
              "error",
              response.data || intucartTaxonomySync.strings.syncError
            );
            TaxonomySync.updateStatusIndicator(taxonomy, "error");
          }
        },
        error: () => {
          TaxonomySync.showNotice(
            "error",
            intucartTaxonomySync.strings.syncError
          );
          TaxonomySync.updateStatusIndicator(taxonomy, "error");
        },
        complete: () => {
          $button.prop("disabled", false).text(originalText);
        },
      });
    },

    handleBulkSync: (e) => {
      e.preventDefault();

      if (
        !confirm(
          "Are you sure you want to sync all taxonomies? This may take several minutes."
        )
      ) {
        return;
      }

      TaxonomySync.performBulkSync(false);
    },

    handleForceFullSync: (e) => {
      e.preventDefault();

      if (
        !confirm(
          "Are you sure you want to force a full re-sync? This will clear all cached data and re-sync everything."
        )
      ) {
        return;
      }

      TaxonomySync.performBulkSync(true);
    },

    performBulkSync: (forceFullSync) => {
      const $progressContainer = $("#bulk-sync-progress");
      const $progressFill = $(".progress-fill");
      const $progressText = $(".progress-text");
      const $bulkButtons = $(".bulk-actions .button");

      // Show progress bar
      $progressContainer.show();
      $bulkButtons.prop("disabled", true);

      // Get all taxonomies to sync
      const taxonomies = [];
      $(".sync-taxonomy-btn").each(() => {
        taxonomies.push($(this).data("taxonomy"));
      });

      let completed = 0;
      const total = taxonomies.length;

      $progressText.text(`Syncing taxonomies... (0/${total})`);
      $progressFill.css("width", "0%");

      // Sync taxonomies one by one
      const syncNext = (index) => {
        if (index >= taxonomies.length) {
          // All done
          $progressText.text(`Sync completed! (${total}/${total})`);
          $progressFill.css("width", "100%");

          setTimeout(() => {
            $progressContainer.hide();
            $bulkButtons.prop("disabled", false);
            TaxonomySync.loadSyncStatus();
          }, 2000);

          TaxonomySync.showNotice(
            "success",
            "All taxonomies synced successfully!"
          );
          return;
        }

        const taxonomy = taxonomies[index];
        $progressText.text(`Syncing ${taxonomy}... (${completed}/${total})`);

        $.ajax({
          url: intucartTaxonomySync.ajaxUrl,
          type: "POST",
          data: {
            action: "intucart_sync_taxonomy",
            taxonomy: taxonomy,
            nonce: intucartTaxonomySync.nonce,
          },
          success: (response) => {
            completed++;
            const progress = (completed / total) * 100;
            $progressFill.css(`width: ${progress}%`);

            if (response.success) {
              TaxonomySync.updateStatusIndicator(taxonomy, "success");
            } else {
              TaxonomySync.updateStatusIndicator(taxonomy, "error");
            }
          },
          error: () => {
            completed++;
            const progress = (completed / total) * 100;
            $progressFill.css(`width: ${progress}%`);
            TaxonomySync.updateStatusIndicator(taxonomy, "error");
          },
          complete: () => {
            // Sync next taxonomy
            setTimeout(() => {
              syncNext(index + 1);
            }, 500); // Small delay between requests
          },
        });
      };

      // Start syncing
      syncNext(0);
    },

    loadSyncStatus: () => {
      $.ajax({
        url: intucartTaxonomySync.ajaxUrl,
        type: "POST",
        data: {
          action: "intucart_get_taxonomy_status",
          nonce: intucartTaxonomySync.nonce,
        },
        success: (response) => {
          if (response.success) {
            TaxonomySync.updateSyncStatusDisplay(response.data);
          }
        },
      });
    },

    updateSyncStatusDisplay: (statusData) => {
      const $statusContainer = $("#taxonomy-sync-status");

      let html = '<div class="sync-status-grid">';

      for (const taxonomy in statusData) {
        const data = statusData[taxonomy];
        const statusClass =
          data.status === "synced"
            ? "success"
            : data.status === "pending"
              ? "pending"
              : "error";

        html += `
                    <div class="sync-status-item">
                        <span class="sync-status-indicator sync-status-${statusClass}"></span>
                        <strong>${taxonomy}</strong>: ${data.term_count} terms
                        ${data.last_sync
            ? `(Last sync: ${data.last_sync})`
            : "(Never synced)"
          }
                    </div>
                `;
      }

      html += "</div>";
      $statusContainer.html(html);
    },

    updateStatusIndicator: (taxonomy, status) => {
      // This would update visual indicators in the tables
      // Implementation depends on your specific UI needs
    },

    showNotice: (type, message) => {
      const $notice = $(
        `<div class="notice notice-${type} is-dismissible"><p>${message}</p></div>`
      );
      $(".wrap h1").after($notice);

      // Auto-dismiss after 5 seconds
      setTimeout(() => {
        $notice.fadeOut();
      }, 5000);
    },
  };

  // Initialize
  TaxonomySync.init();
});
