<?php

declare(strict_types=1);

namespace Intucart\Admin;

use Intucart\Services\Logger;
use Intucart\Services\Constants;

/**
 * Handles admin script and style enqueueing
 */
class AdminAssets
{
    private Logger $logger;

    public function __construct(Logger $logger)
    {
        $this->logger = $logger;
    }

    /**
     * Register the enqueue hook
     */
    public function register(): void
    {
        add_action('admin_enqueue_scripts', [$this, 'enqueue']);
    }

    /**
     * Enqueue admin scripts and styles
     *
     * @param string $hook Current admin page hook
     */
    public function enqueue(string $hook): void
    {
        // Only load on our plugin pages
        if (strpos($hook, 'intucart') === false) {
            return;
        }

        $this->enqueueWordPressAssets();
        $this->enqueueCoreStyles();
        $this->enqueueTabSpecificAssets($hook);
        $this->enqueueCoreScripts();
        $this->enqueueVendorScripts();
        $this->enqueueOptionalAssets($hook);
        $this->localizeScripts();
    }

    /**
     * Enqueue WordPress built-in assets
     */
    private function enqueueWordPressAssets(): void
    {
        wp_enqueue_media();
        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');
    }

    /**
     * Enqueue core plugin styles
     */
    private function enqueueCoreStyles(): void
    {
        wp_enqueue_style(
            'intucart-admin-common',
            INTUCART_PLUGIN_URL . 'assets/css/admin/common.css',
            [],
            $this->getFileVersion('assets/css/admin/common.css')
        );

        wp_enqueue_style(
            'intucart-admin-components',
            INTUCART_PLUGIN_URL . 'assets/css/admin/components.css',
            ['intucart-admin-common'],
            $this->getFileVersion('assets/css/admin/components.css')
        );
    }

    /**
     * Enqueue tab-specific assets based on current tab
     */
    private function enqueueTabSpecificAssets(string $hook): void
    {
        $currentTab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'license';

        if ($currentTab === 'webhooks') {
            $this->enqueueWebhookAssets();
        }
    }

    /**
     * Enqueue webhook-specific assets
     */
    private function enqueueWebhookAssets(): void
    {
        wp_enqueue_style(
            'intucart-admin-webhooks',
            INTUCART_PLUGIN_URL . 'assets/css/admin/webhooks.css',
            ['intucart-admin-common'],
            $this->getFileVersion('assets/css/admin/webhooks.css')
        );

        wp_enqueue_style(
            'intucart-webhook-form',
            INTUCART_PLUGIN_URL . 'assets/css/admin/webhook-form.css',
            ['intucart-admin-webhooks'],
            $this->getFileVersion('assets/css/admin/webhook-form.css')
        );

        wp_enqueue_script(
            'intucart-admin-webhooks',
            INTUCART_PLUGIN_URL . 'assets/js/admin/webhooks.js',
            ['jquery', 'intucart-admin-page'],
            $this->getFileVersion('assets/js/admin/webhooks.js'),
            true
        );
    }

    /**
     * Enqueue core plugin scripts
     */
    private function enqueueCoreScripts(): void
    {
        wp_enqueue_script(
            'intucart-admin-page',
            INTUCART_PLUGIN_URL . 'assets/js/admin/core.js',
            ['jquery', 'wp-color-picker'],
            $this->getFileVersion('assets/js/admin/core.js'),
            true
        );

        wp_enqueue_script(
            'intucart-popovers',
            INTUCART_PLUGIN_URL . 'assets/js/admin/popovers.js',
            ['jquery', 'intucart-admin-page', 'floating-ui-dom'],
            $this->getFileVersion('assets/js/admin/popovers.js'),
            true
        );
    }

    /**
     * Enqueue vendor scripts (Floating UI, Select2)
     */
    private function enqueueVendorScripts(): void
    {
        // Floating UI (bundled locally for WordPress.org compliance)
        wp_enqueue_script(
            'floating-ui-core',
            INTUCART_PLUGIN_URL . 'assets/js/vendor/floating-ui.core.min.js',
            [],
            '1.7.3',
            true
        );

        wp_enqueue_script(
            'floating-ui-dom',
            INTUCART_PLUGIN_URL . 'assets/js/vendor/floating-ui.dom.min.js',
            ['floating-ui-core'],
            '1.7.3',
            true
        );
    }

    /**
     * Enqueue optional assets based on context
     */
    private function enqueueOptionalAssets(string $hook): void
    {
        // Prompts assets (only on chatbot > prompts subtab)
        if ($this->shouldLoadPromptsAssets($hook)) {
            $this->enqueuePromptsAssets();
        }

        // Media script
        $this->enqueueMediaScript();

        // Chatbot styling script
        $this->enqueueChatbotStylingScript();
    }

    /**
     * Check if prompts assets should be loaded
     */
    private function shouldLoadPromptsAssets(string $hook): bool
    {
        if ($hook !== Constants::ADMIN_PAGE_HOOK) {
            return false;
        }

        $tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'license';
        $subtab = isset($_GET['chatbot_subtab']) ? sanitize_text_field($_GET['chatbot_subtab']) : 'settings';

        return ($tab === 'chatbot' && $subtab === 'prompts');
    }

    /**
     * Enqueue prompts-related assets
     */
    private function enqueuePromptsAssets(): void
    {
        // Try to use existing Select2 from WooCommerce or other plugins
        $this->enqueueSelect2();

        // Prompts CSS
        $promptsCssPath = INTUCART_PLUGIN_DIR . 'assets/css/admin/prompts.css';
        if (file_exists($promptsCssPath)) {
            wp_enqueue_style(
                'intucart-admin-prompts',
                INTUCART_PLUGIN_URL . 'assets/css/admin/prompts.css',
                [],
                filemtime($promptsCssPath)
            );
        }

        // Prompts JS
        $promptsJsPath = INTUCART_PLUGIN_DIR . 'assets/js/admin/prompts.js';
        if (file_exists($promptsJsPath)) {
            wp_enqueue_script(
                'intucart-admin-prompts',
                INTUCART_PLUGIN_URL . 'assets/js/admin/prompts.js',
                ['jquery'],
                filemtime($promptsJsPath),
                true
            );
        }
    }

    /**
     * Enqueue Select2 from various sources
     */
    private function enqueueSelect2(): void
    {
        // Try WooCommerce select2 first
        if (class_exists('WooCommerce')) {
            if (!wp_style_is('select2', 'enqueued') && wp_style_is('select2', 'registered')) {
                wp_enqueue_style('select2');
            }
            if (!wp_style_is('woocommerce_admin_styles', 'enqueued') && wp_style_is('woocommerce_admin_styles', 'registered')) {
                wp_enqueue_style('woocommerce_admin_styles');
            }
        }

        // Try from other plugins
        if (wp_style_is('select2', 'registered') && !wp_style_is('select2', 'enqueued')) {
            wp_enqueue_style('select2');
        }

        // Use bundled version as fallback
        $haveSelectJs = wp_script_is('selectWoo', 'registered') ||
                        wp_script_is('select2', 'registered') ||
                        wp_script_is('select2', 'enqueued') ||
                        wp_script_is('selectWoo', 'enqueued');

        $haveSelectCss = wp_style_is('select2', 'registered') || wp_style_is('select2', 'enqueued');

        if (!$haveSelectJs) {
            wp_register_script(
                'intucart-select2',
                INTUCART_PLUGIN_URL . 'assets/js/vendor/select2.full.min.js',
                ['jquery'],
                '4.1.0-rc.0',
                true
            );
            wp_enqueue_script('intucart-select2');
        }

        if (!$haveSelectCss) {
            wp_register_style(
                'intucart-select2',
                INTUCART_PLUGIN_URL . 'assets/css/vendor/select2.min.css',
                [],
                '4.1.0-rc.0'
            );
            wp_enqueue_style('intucart-select2');
        }
    }

    /**
     * Enqueue media script
     */
    private function enqueueMediaScript(): void
    {
        $mediaJsPath = INTUCART_PLUGIN_DIR . 'assets/js/admin/media.js';

        if (file_exists($mediaJsPath)) {
            wp_enqueue_script(
                'intucart-admin-media',
                INTUCART_PLUGIN_URL . 'assets/js/admin/media.js',
                ['jquery'],
                filemtime($mediaJsPath),
                true
            );
        } else {
            $this->logger->warning('Admin media script file not found (expected assets/js/admin/media.js)');
        }
    }

    /**
     * Enqueue chatbot styling script
     */
    private function enqueueChatbotStylingScript(): void
    {
        $stylingJsPath = INTUCART_PLUGIN_DIR . 'assets/js/admin/chatbot/styling.js';

        if (file_exists($stylingJsPath)) {
            wp_enqueue_script(
                'intucart-admin-chatbot-styling',
                INTUCART_PLUGIN_URL . 'assets/js/admin/chatbot/styling.js',
                ['jquery', 'wp-color-picker'],
                filemtime($stylingJsPath),
                true
            );
        }
    }

    /**
     * Localize scripts with data
     */
    private function localizeScripts(): void
    {
        wp_localize_script(
            'intucart-admin-page',
            'intucartAdmin',
            [
                'ajaxUrl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('intucart_admin_nonce')
            ]
        );
    }

    /**
     * Get file version based on modification time
     */
    private function getFileVersion(string $relativePath): string
    {
        $fullPath = INTUCART_PLUGIN_DIR . $relativePath;

        if (file_exists($fullPath)) {
            return (string) filemtime($fullPath);
        }

        return INTUCART_VERSION;
    }
}

