<?php

declare(strict_types=1);

namespace Intucart\Admin;

use Intucart\Admin\Ajax\SyncAjaxHandler;
use Intucart\Admin\Ajax\ContentTypesAjaxHandler;
use Intucart\Admin\Ajax\SystemAjaxHandler;
use Intucart\Admin\Tabs\TabRendererInterface;
use Intucart\Admin\Tabs\LicenseTabRenderer;
use Intucart\Admin\Tabs\SearchTabRenderer;
use Intucart\Admin\Tabs\WooCommerceTabRenderer;
use Intucart\Admin\Tabs\ChatbotTabRenderer;
use Intucart\Admin\Tabs\WebhooksTabRenderer;
use Intucart\Admin\Tabs\IndexingTabRenderer;
use Intucart\Components\Tabs;
use Intucart\Pages\Settings\EcommerceSettings;
use Intucart\Pages\Settings\SyncSettings;
use Intucart\Pages\Settings\SearchSettings;
use Intucart\Pages\Settings\WebhookSettings;
use Intucart\Pages\Settings\ChatSettings;
use Intucart\Pages\Status;
use Intucart\Pages\Analytics;
use Intucart\Services\Constants;
use Intucart\Services\Logger;
use Intucart\Base\Enqueue;

/**
 * Main admin controller - orchestrates admin page functionality
 */
class AdminController
{
    private Logger $logger;
    private AdminAssets $assets;
    private AdminNotices $notices;

    // AJAX handlers
    private SyncAjaxHandler $syncHandler;
    private ContentTypesAjaxHandler $contentTypesHandler;
    private SystemAjaxHandler $systemHandler;

    // Tab renderers
    /** @var array<string, TabRendererInterface> */
    private array $tabs = [];

    // Settings pages (for registerSettings)
    private EcommerceSettings $ecommerceSettings;
    private SyncSettings $syncSettings;
    private SearchSettings $searchSettings;
    private WebhookSettings $webhookSettings;
    private ChatSettings $chatSettings;

    // Sub-pages
    private Status $statusDashboard;
    private Analytics $analytics;

    public function __construct(
        Logger $logger,
        AdminAssets $assets,
        AdminNotices $notices,
        SyncAjaxHandler $syncHandler,
        ContentTypesAjaxHandler $contentTypesHandler,
        SystemAjaxHandler $systemHandler,
        LicenseTabRenderer $licenseTab,
        SearchTabRenderer $searchTab,
        WooCommerceTabRenderer $wooCommerceTab,
        ChatbotTabRenderer $chatbotTab,
        WebhooksTabRenderer $webhooksTab,
        IndexingTabRenderer $indexingTab,
        EcommerceSettings $ecommerceSettings,
        SyncSettings $syncSettings,
        SearchSettings $searchSettings,
        WebhookSettings $webhookSettings,
        ChatSettings $chatSettings,
        Status $statusDashboard,
        Analytics $analytics
    ) {
        $this->logger = $logger;
        $this->assets = $assets;
        $this->notices = $notices;
        $this->syncHandler = $syncHandler;
        $this->contentTypesHandler = $contentTypesHandler;
        $this->systemHandler = $systemHandler;

        // Register tab renderers
        $this->tabs = [
            'license' => $licenseTab,
            'search' => $searchTab,
            'woocommerce' => $wooCommerceTab,
            'chatbot' => $chatbotTab,
            'webhooks' => $webhooksTab,
            'indexing' => $indexingTab,
        ];

        $this->ecommerceSettings = $ecommerceSettings;
        $this->syncSettings = $syncSettings;
        $this->searchSettings = $searchSettings;
        $this->webhookSettings = $webhookSettings;
        $this->chatSettings = $chatSettings;
        $this->statusDashboard = $statusDashboard;
        $this->analytics = $analytics;
    }

    /**
     * Initialize admin functionality
     */
    public function initialize(): void
    {
        // Register hooks
        add_action('admin_init', [$this, 'registerSettings']);
        add_action('admin_menu', [$this, 'addAdminPage']);

        // Register asset loader
        $this->assets->register();

        // Register notices
        $this->notices->register();
        
        // Register cache invalidation hooks for widget settings
        Enqueue::registerCacheInvalidationHooks();

        // Register AJAX handlers
        $this->syncHandler->register();
        $this->contentTypesHandler->register();
        $this->systemHandler->register();

        // Initialize sub-pages with their own AJAX handlers
        $this->analytics->initialize();
        $this->webhookSettings->initialize();

        // Register Status page AJAX handler
        add_action('wp_ajax_intucart_clear_logs', [$this->statusDashboard, 'handleAjaxClearLogs']);

        // Register ChatSettings AJAX handlers
        add_action('wp_ajax_intucart_reset_styling_settings', [$this->chatSettings, 'resetStylingSettings']);
        add_action('wp_ajax_intucart_generate_color_scheme', [$this->chatSettings, 'generateColorScheme']);
    }

    /**
     * Register all settings
     */
    public function registerSettings(): void
    {
        $this->ecommerceSettings->registerSettings();
        $this->syncSettings->registerSettings();
        $this->searchSettings->registerSettings();
        $this->webhookSettings->registerSettings();
        $this->chatSettings->registerSettings();
    }

    /**
     * Add admin menu pages
     */
    public function addAdminPage(): void
    {
        $icon = $this->getMenuIcon();
        $bridgePluginActive = is_plugin_active('intucart-bridge/intucart-bridge.php');

        // Add separator before
        $this->addMenuSeparator('intucart_sep_before', 53.999999);

        // Main menu page
        add_menu_page(
            Constants::PLUGIN_SHORT_NAME,
            Constants::PLUGIN_NAME,
            'manage_options',
            'intucart',
            [$this, 'renderPage'],
            $icon,
            54
        );

        // Settings submenu (same as main)
        add_submenu_page(
            'intucart',
            'Settings',
            'Settings',
            'manage_options',
            'intucart',
            [$this, 'renderPage']
        );

        // Analytics submenu
        add_submenu_page(
            'intucart',
            'Analytics',
            'Analytics',
            'manage_options',
            'intucart-analytics',
            [$this->analytics, 'renderPage']
        );

        // Status submenu
        add_submenu_page(
            'intucart',
            'Status',
            'Status',
            'manage_options',
            'intucart-status',
            [$this, 'renderStatusPage']
        );

        // Add separator after (if bridge plugin is not active)
        if (!$bridgePluginActive) {
            $this->addMenuSeparator('intucart_sep_after', 54.000001);
        }
    }

    /**
     * Get base64-encoded SVG icon for menu
     */
    private function getMenuIcon(): string
    {
        $svg = file_get_contents(INTUCART_PLUGIN_DIR . 'assets/images/icon.svg');
        return 'data:image/svg+xml;base64,' . base64_encode($svg);
    }

    /**
     * Add a menu separator
     */
    private function addMenuSeparator(string $slug, float $position): void
    {
        add_menu_page('', '', 'manage_options', $slug, '', '', $position);

        global $menu;
        foreach ($menu as $key => $item) {
            if ($item[2] === $slug) {
                $menu[$key][4] = 'wp-menu-separator';
            }
        }
    }

    /**
     * Render the main admin page
     */
    public function renderPage(): void
    {
        if (!current_user_can('manage_options')) {
            return;
        }

        $this->enqueueInlineStyles();

        echo '<div class="wrap">';
        $this->renderPageHeader();
        $this->renderTabbedInterface();
        echo '</div>';
    }

    /**
     * Enqueue inline styles needed for rendering
     */
    private function enqueueInlineStyles(): void
    {
        if (!wp_style_is('intucart-admin-common', 'enqueued')) {
            wp_enqueue_style(
                'intucart-admin-common',
                INTUCART_PLUGIN_URL . 'assets/css/admin/common.css',
                [],
                filemtime(INTUCART_PLUGIN_DIR . 'assets/css/admin/common.css')
            );
        }

        if (!wp_style_is('intucart-admin-components', 'enqueued')) {
            wp_enqueue_style(
                'intucart-admin-components',
                INTUCART_PLUGIN_URL . 'assets/css/admin/components.css',
                ['intucart-admin-common'],
                filemtime(INTUCART_PLUGIN_DIR . 'assets/css/admin/components.css')
            );
        }
    }

    /**
     * Render the page header
     */
    private function renderPageHeader(): void
    {
        echo '<h1>';
        echo '<span class="intucart-page-header-icon">';
        echo '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 640 512"><path opacity=".4" d="M638.41 387a12.34 12.34 0 0 0-12.2-10.3h-16.5a86.33 86.33 0 0 0-15.9-27.4L602 335a12.42 12.42 0 0 0-2.8-15.7 110.5 110.5 0 0 0-32.1-18.6 12.36 12.36 0 0 0-15.1 5.4l-8.2 14.3a88.86 88.86 0 0 0-31.7 0l-8.2-14.3a12.36 12.36 0 0 0-15.1-5.4 111.83 111.83 0 0 0-32.1 18.6 12.3 12.3 0 0 0-2.8 15.7l8.2 14.3a86.33 86.33 0 0 0-15.9 27.4h-16.5a12.43 12.43 0 0 0-12.2 10.4 112.66 112.66 0 0 0 0 37.1 12.34 12.34 0 0 0 12.2 10.3h16.5a86.33 86.33 0 0 0 15.9 27.4l-8.2 14.3a12.42 12.42 0 0 0 2.8 15.7 110.5 110.5 0 0 0 32.1 18.6 12.36 12.36 0 0 0 15.1-5.4l8.2-14.3a88.86 88.86 0 0 0 31.7 0l8.2 14.3a12.36 12.36 0 0 0 15.1 5.4 111.83 111.83 0 0 0 32.1-18.6 12.3 12.3 0 0 0 2.8-15.7l-8.2-14.3a86.33 86.33 0 0 0 15.9-27.4h16.5a12.43 12.43 0 0 0 12.2-10.4 112.66 112.66 0 0 0 .01-37.1zm-136.8 44.9c-29.6-38.5 14.3-82.4 52.8-52.8 29.59 38.49-14.3 82.39-52.8 52.79zm136.8-343.8a12.34 12.34 0 0 0-12.2-10.3h-16.5a86.33 86.33 0 0 0-15.9-27.4l8.2-14.3a12.42 12.42 0 0 0-2.8-15.7 110.5 110.5 0 0 0-32.1-18.6A12.36 12.36 0 0 0 552 7.19l-8.2 14.3a88.86 88.86 0 0 0-31.7 0l-8.2-14.3a12.36 12.36 0 0 0-15.1-5.4 111.83 111.83 0 0 0-32.1 18.6 12.3 12.3 0 0 0-2.8 15.7l8.2 14.3a86.33 86.33 0 0 0-15.9 27.4h-16.5a12.43 12.43 0 0 0-12.2 10.4 112.66 112.66 0 0 0 0 37.1 12.34 12.34 0 0 0 12.2 10.3h16.5a86.33 86.33 0 0 0 15.9 27.4l-8.2 14.3a12.42 12.42 0 0 0 2.8 15.7 110.5 110.5 0 0 0 32.1 18.6 12.36 12.36 0 0 0 15.1-5.4l8.2-14.3a88.86 88.86 0 0 0 31.7 0l8.2 14.3a12.36 12.36 0 0 0 15.1 5.4 111.83 111.83 0 0 0 32.1-18.6 12.3 12.3 0 0 0 2.8-15.7l-8.2-14.3a86.33 86.33 0 0 0 15.9-27.4h16.5a12.43 12.43 0 0 0 12.2-10.4 112.66 112.66 0 0 0 .01-37.1zm-136.8 45c-29.6-38.5 14.3-82.5 52.8-52.8 29.59 38.49-14.3 82.39-52.8 52.79z"/><path d="M420 303.79L386.31 287a173.78 173.78 0 0 0 0-63.5l33.7-16.8c10.1-5.9 14-18.2 10-29.1-8.9-24.2-25.9-46.4-42.1-65.8a23.93 23.93 0 0 0-30.3-5.3l-29.1 16.8a173.66 173.66 0 0 0-54.9-31.7V58a24 24 0 0 0-20-23.6 228.06 228.06 0 0 0-76 .1A23.82 23.82 0 0 0 158 58v33.7a171.78 171.78 0 0 0-54.9 31.7L74 106.59a23.91 23.91 0 0 0-30.3 5.3c-16.2 19.4-33.3 41.6-42.2 65.8a23.84 23.84 0 0 0 10.5 29l33.3 16.9a173.24 173.24 0 0 0 0 63.4L12 303.79a24.13 24.13 0 0 0-10.5 29.1c8.9 24.1 26 46.3 42.2 65.7a23.93 23.93 0 0 0 30.3 5.3l29.1-16.7a173.66 173.66 0 0 0 54.9 31.7v33.6a24 24 0 0 0 20 23.6 224.88 224.88 0 0 0 75.9 0 23.93 23.93 0 0 0 19.7-23.6v-33.6a171.78 171.78 0 0 0 54.9-31.7l29.1 16.8a23.91 23.91 0 0 0 30.3-5.3c16.2-19.4 33.7-41.6 42.6-65.8a24 24 0 0 0-10.5-29.1zm-151.3 4.3c-77 59.2-164.9-28.7-105.7-105.7 77-59.2 164.91 28.7 105.71 105.7z"/></svg>';
        echo '</span>';
        echo esc_html(get_admin_page_title());
        echo '</h1>';
        echo '<p class="intucart-page-description">' . esc_html__('Manage all aspects of your AI-powered site including search, chat, recommendations, license activation, content indexing, and advanced configurations.', 'intufind') . '</p>';
    }

    /**
     * Render the tabbed interface
     */
    private function renderTabbedInterface(): void
    {
        $currentTab = $this->getCurrentTab();

        // Build tab configurations
        $tabConfigs = [];
        foreach ($this->tabs as $id => $renderer) {
            $config = $renderer->getTabConfig();

            // Skip tabs that have a condition set to false
            if (isset($config['condition']) && !$config['condition']) {
                continue;
            }

            $tabConfigs[$id] = $config;
        }

        // Render tabs
        Tabs::renderOuter($tabConfigs, $currentTab);
        Tabs::openContent();

        // Render current tab content
        if (isset($this->tabs[$currentTab])) {
            $this->tabs[$currentTab]->render();
        } else {
            $this->tabs['license']->render();
        }

        Tabs::closeContent();
        Tabs::closeOuter();
    }

    /**
     * Get the current tab with validation
     */
    private function getCurrentTab(): string
    {
        $currentTab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'license';

        // Validate WooCommerce tab availability
        if ($currentTab === 'woocommerce') {
            $wooTab = $this->tabs['woocommerce'];
            if ($wooTab instanceof WooCommerceTabRenderer && !$wooTab->isAvailable()) {
                return 'license';
            }
        }

        // Validate tab exists
        if (!isset($this->tabs[$currentTab])) {
            return 'license';
        }

        return $currentTab;
    }

    /**
     * Render the status page
     */
    public function renderStatusPage(): void
    {
        $this->statusDashboard->render();
    }
}

