<?php

declare(strict_types=1);

namespace Intucart\Admin;

use Intucart\Services\Constants;
use Intucart\Components\Button;

/**
 * Handles admin notice display
 */
class AdminNotices
{
    /**
     * Register notice hooks
     */
    public function register(): void
    {
        add_action('admin_notices', [$this, 'displayTransientNotices']);
        add_action('admin_notices', [$this, 'showLicenseKeyNotice']);
    }

    /**
     * Display transient admin notices
     */
    public function displayTransientNotices(): void
    {
        $screen = get_current_screen();
        if (!$screen || strpos($screen->id, 'intucart') === false) {
            return;
        }

        $notices = get_transient('intucart_admin_notices');
        if (!$notices || !is_array($notices)) {
            return;
        }

        foreach ($notices as $notice) {
            $type = $notice['type'] ?? 'info';
            $message = $notice['message'] ?? '';

            if (empty($message)) {
                continue;
            }

            $class = 'notice notice-' . esc_attr($type) . ' is-dismissible';
            echo '<div class="' . $class . '">';
            echo '<p>' . wp_kses_post($message) . '</p>';
            echo '</div>';
        }

        delete_transient('intucart_admin_notices');
    }

    /**
     * Show API key notice on Intucart admin pages
     */
    public function showLicenseKeyNotice(): void
    {
        $screen = get_current_screen();
        if (!$screen || !$this->isIntucartPage($screen->id)) {
            return;
        }

        // Don't show if API key is already set
        $api_key = get_option(Constants::API_KEY_OPTION, '');
        if (!empty($api_key)) {
            return;
        }

        // Don't show if dismissed
        if (get_option(Constants::LICENSE_NOTICE_DISMISSED_OPTION, false)) {
            return;
        }

        $this->renderLicenseNotice();
    }

    /**
     * Check if current screen is an Intucart page
     */
    private function isIntucartPage(string $screenId): bool
    {
        return in_array($screenId, [
            Constants::ADMIN_PAGE_HOOK,
            Constants::ADMIN_PAGE_ANALYTICS_HOOK,
            Constants::ADMIN_PAGE_STATUS_HOOK
        ], true);
    }

    /**
     * Render the license notice HTML
     */
    private function renderLicenseNotice(): void
    {
        ?>
        <div class="notice notice-info is-dismissible" id="intucart-license-notice" style="border-left-color: #2271b1;">
            <p>
                <strong><?php esc_html_e('🚀 Get Started Free:', 'intufind'); ?></strong>
                <?php echo sprintf(esc_html__('%s requires a Secret Key. Start your 14-day free trial – no credit card required!', 'intufind'), Constants::PLUGIN_NAME); ?>
                <span style="margin-left: 10px;">
                    <?php
                    Button::primary(__('Start Free Trial', 'intufind'), [
                        'href' => Constants::WEBSITE_URL . 'register',
                        'attributes' => ['target' => '_blank']
                    ]);
                    ?>
                </span>
            </p>
        </div>
        <script>
        jQuery(document).ready(function($) {
            $('#intucart-license-notice').on('click', '.notice-dismiss', function() {
                $.post(ajaxurl, {
                    action: 'intucart_dismiss_license_notice',
                    nonce: '<?php echo wp_create_nonce('intucart_dismiss_license_notice'); ?>'
                });
            });
        });
        </script>
        <?php
    }
}

