<?php

declare(strict_types=1);

namespace Intucart\Admin\Ajax;

use Intucart\Services\Logger;

/**
 * Abstract base class for AJAX handlers
 */
abstract class AbstractAjaxHandler implements AjaxHandlerInterface
{
    protected Logger $logger;

    public function __construct(Logger $logger)
    {
        $this->logger = $logger;
    }

    /**
     * Verify the current user has admin capabilities
     *
     * @return bool
     */
    protected function verifyCapability(): bool
    {
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'intufind')]);
            return false;
        }
        return true;
    }

    /**
     * Verify nonce and capability together
     *
     * @param string $action Nonce action name
     * @param string $queryArg Query argument name for nonce
     * @return bool
     */
    protected function verifyRequest(string $action, string $queryArg = 'nonce'): bool
    {
        if (!$this->verifyCapability()) {
            return false;
        }

        if (!check_ajax_referer($action, $queryArg, false)) {
            wp_send_json_error(['message' => __('Security check failed.', 'intufind')]);
            return false;
        }

        return true;
    }

    /**
     * Get sanitized POST value
     *
     * @param string $key
     * @param string $default
     * @return string
     */
    protected function getPostValue(string $key, string $default = ''): string
    {
        return isset($_POST[$key]) ? sanitize_text_field(wp_unslash($_POST[$key])) : $default;
    }

    /**
     * Get sanitized POST array
     *
     * @param string $key
     * @return array
     */
    protected function getPostArray(string $key): array
    {
        if (!isset($_POST[$key]) || !is_array($_POST[$key])) {
            return [];
        }
        return array_map('sanitize_text_field', $_POST[$key]);
    }

    /**
     * Extend execution time for long-running operations
     *
     * @return void
     */
    protected function extendExecutionTime(): void
    {
        if (function_exists('set_time_limit')) {
            @set_time_limit(0);
        }
        if (function_exists('ini_set')) {
            @ini_set('max_execution_time', '0');
        }
    }
}

