<?php

declare(strict_types=1);

namespace Intucart\Admin\Ajax;

use Intucart\Services\Logger;
use Intucart\Services\Managers\PostTypeManager;
use Intucart\Services\Managers\TaxonomyManager;
use Intucart\Services\Managers\CronManager;

/**
 * Handles AJAX requests for content type management
 */
class ContentTypesAjaxHandler extends AbstractAjaxHandler
{
    private PostTypeManager $postTypeManager;
    private TaxonomyManager $taxonomyManager;
    private CronManager $cronManager;

    public function __construct(
        Logger $logger,
        PostTypeManager $postTypeManager,
        TaxonomyManager $taxonomyManager,
        CronManager $cronManager
    ) {
        parent::__construct($logger);
        $this->postTypeManager = $postTypeManager;
        $this->taxonomyManager = $taxonomyManager;
        $this->cronManager = $cronManager;
    }

    /**
     * Register AJAX actions
     */
    public function register(): void
    {
        add_action('wp_ajax_intucart_update_content_types', [$this, 'handleUpdateContentTypes']);
    }

    /**
     * Handle request to update content type preferences
     */
    public function handleUpdateContentTypes(): void
    {
        if (!$this->verifyRequest('intucart_admin_nonce')) {
            return;
        }

        try {
            $result = $this->processContentTypeUpdate();
            wp_send_json_success($result);
        } catch (\Exception $e) {
            $this->logger->error('Failed to update content type preferences', [
                'error' => $e->getMessage()
            ]);
            wp_send_json_error(['message' => sprintf(__('Failed to save preferences: %s', 'intufind'), $e->getMessage())]);
        }
    }

    /**
     * Process the content type update and return result
     */
    private function processContentTypeUpdate(): array
    {
        // Get all available post types and taxonomies
        $allPostTypes = $this->postTypeManager->getAvailablePostTypes();
        $syncablePostTypes = $this->postTypeManager->getSyncablePostTypes();
        $allTaxonomies = $this->taxonomyManager->getAllAvailableTaxonomyNames($syncablePostTypes);

        // Get current preferences
        $oldPostTypePreferences = get_option('intucart_syncable_post_types', []);
        $oldTaxonomyPreferences = get_option('intucart_syncable_taxonomies', []);

        // Get selected items from form
        $selectedPostTypes = $this->getPostArray('post_types');
        $selectedTaxonomies = $this->getPostArray('taxonomies');

        // Build new preferences
        $preferences = $this->buildPostTypePreferences($allPostTypes, $selectedPostTypes);
        $taxonomyPreferences = $this->buildTaxonomyPreferences($allTaxonomies, $selectedTaxonomies);

        // Auto-align taxonomies with post type changes
        $autoDeselectedTaxonomies = $this->alignTaxonomiesWithPostTypes(
            $taxonomyPreferences,
            $selectedPostTypes
        );

        // Identify changes for sync scheduling
        $changes = $this->identifyChanges(
            $allPostTypes,
            $allTaxonomies,
            $oldPostTypePreferences,
            $oldTaxonomyPreferences,
            $preferences,
            $taxonomyPreferences,
            $autoDeselectedTaxonomies
        );

        // Schedule sync if needed
        $syncScheduled = $this->scheduleSyncIfNeeded($changes);

        // Save preferences
        update_option('intucart_syncable_post_types', $preferences);
        update_option('intucart_syncable_taxonomies', $taxonomyPreferences);

        $this->logPreferencesUpdate($changes, $syncScheduled);

        return [
            'message' => $this->buildSuccessMessage($autoDeselectedTaxonomies, $syncScheduled, $changes),
            'sync_scheduled' => $syncScheduled
        ];
    }

    /**
     * Build post type preferences from selected values
     */
    private function buildPostTypePreferences(array $allPostTypes, array $selectedPostTypes): array
    {
        $preferences = [];
        foreach ($allPostTypes as $postType) {
            $preferences[$postType] = in_array($postType, $selectedPostTypes, true);
        }
        return $preferences;
    }

    /**
     * Build taxonomy preferences from selected values
     */
    private function buildTaxonomyPreferences(array $allTaxonomies, array $selectedTaxonomies): array
    {
        $preferences = [];
        foreach ($allTaxonomies as $taxonomy) {
            $preferences[$taxonomy] = in_array($taxonomy, $selectedTaxonomies, true);
        }
        return $preferences;
    }

    /**
     * Align taxonomies with post types and return auto-deselected ones
     */
    private function alignTaxonomiesWithPostTypes(
        array &$taxonomyPreferences,
        array $selectedPostTypes
    ): array {
        $autoDeselected = [];
        $availableTaxonomies = $this->taxonomyManager->getAllAvailableTaxonomyNames($selectedPostTypes);

        foreach ($taxonomyPreferences as $taxonomy => $isEnabled) {
            if ($isEnabled && !in_array($taxonomy, $availableTaxonomies, true)) {
                $taxonomyPreferences[$taxonomy] = false;
                $autoDeselected[] = $taxonomy;
            }
        }

        return $autoDeselected;
    }

    /**
     * Identify what changed between old and new preferences
     */
    private function identifyChanges(
        array $allPostTypes,
        array $allTaxonomies,
        array $oldPostTypePrefs,
        array $oldTaxonomyPrefs,
        array $newPostTypePrefs,
        array $newTaxonomyPrefs,
        array $autoDeselectedTaxonomies
    ): array {
        $changes = [
            'deselected_post_types' => [],
            'selected_post_types' => [],
            'deselected_taxonomies' => [],
            'selected_taxonomies' => []
        ];

        // Find post type changes
        foreach ($allPostTypes as $postType) {
            $wasEnabled = $oldPostTypePrefs[$postType] ?? false;
            $isEnabled = $newPostTypePrefs[$postType] ?? false;

            if ($wasEnabled && !$isEnabled) {
                $changes['deselected_post_types'][] = $postType;
            } elseif (!$wasEnabled && $isEnabled) {
                $changes['selected_post_types'][] = $postType;
            }
        }

        // Find taxonomy changes
        foreach ($allTaxonomies as $taxonomy) {
            $wasEnabled = $oldTaxonomyPrefs[$taxonomy] ?? false;
            $isEnabled = $newTaxonomyPrefs[$taxonomy] ?? false;

            if ($wasEnabled && !$isEnabled) {
                $changes['deselected_taxonomies'][] = $taxonomy;
            } elseif (!$wasEnabled && $isEnabled) {
                $changes['selected_taxonomies'][] = $taxonomy;
            }
        }

        // Add auto-deselected taxonomies
        foreach ($autoDeselectedTaxonomies as $taxonomy) {
            if (!in_array($taxonomy, $changes['deselected_taxonomies'], true)) {
                $changes['deselected_taxonomies'][] = $taxonomy;
            }
        }

        return $changes;
    }

    /**
     * Schedule sync if there are changes to process
     */
    private function scheduleSyncIfNeeded(array $changes): bool
    {
        $hasChanges = !empty($changes['deselected_post_types']) ||
                      !empty($changes['deselected_taxonomies']) ||
                      !empty($changes['selected_post_types']) ||
                      !empty($changes['selected_taxonomies']);

        if (!$hasChanges) {
            $this->logger->info('No content changes detected, skipping synchronization');
            return false;
        }

        $this->logger->info('Content preferences changed, scheduling full sync', $changes);
        $this->cronManager->triggerFullSync('intucart-content-preferences', 1);

        return true;
    }

    /**
     * Log the preferences update
     */
    private function logPreferencesUpdate(array $changes, bool $syncScheduled): void
    {
        $this->logger->info('Content type preferences updated', [
            'newly_selected_post_types' => $changes['selected_post_types'],
            'newly_selected_taxonomies' => $changes['selected_taxonomies'],
            'deselected_post_types' => $changes['deselected_post_types'],
            'deselected_taxonomies' => $changes['deselected_taxonomies'],
            'sync_scheduled' => $syncScheduled
        ]);
    }

    /**
     * Build user-friendly success message
     */
    private function buildSuccessMessage(
        array $autoDeselectedTaxonomies,
        bool $syncScheduled,
        array $changes
    ): string {
        $message = __('Content type and taxonomy preferences saved successfully.', 'intufind');

        if (!empty($autoDeselectedTaxonomies)) {
            $taxonomyObjects = $this->taxonomyManager->getAllAvailableTaxonomies();
            $labels = [];
            foreach ($autoDeselectedTaxonomies as $taxonomy) {
                $labels[] = $taxonomyObjects[$taxonomy]->labels->name ?? $taxonomy;
            }

            $taxonomyList = implode(', ', $labels);
            $isPlural = count($labels) !== 1;
            $message .= ' ' . sprintf(
                $isPlural
                    ? __('Note: %s were automatically deselected because they are not tied to your selected post types.', 'intufind')
                    : __('Note: %s was automatically deselected because it is not tied to your selected post types.', 'intufind'),
                $taxonomyList
            );
        }

        if ($syncScheduled) {
            $message .= ' ' . __('Full synchronization scheduled to update the AI index with your changes.', 'intufind');
        }

        return $message;
    }
}

