<?php

declare(strict_types=1);

namespace Intucart\Admin\Ajax;

use Intucart\Services\Logger;
use Intucart\Services\Managers\PostTypeManager;
use Intucart\Services\Managers\TaxonomyManager;
use Intucart\Services\UsageService;

/**
 * Handles AJAX requests for content synchronization
 */
class SyncAjaxHandler extends AbstractAjaxHandler
{
    private PostTypeManager $postTypeManager;
    private TaxonomyManager $taxonomyManager;
    private UsageService $usageService;

    public function __construct(
        Logger $logger,
        PostTypeManager $postTypeManager,
        TaxonomyManager $taxonomyManager,
        UsageService $usageService
    ) {
        parent::__construct($logger);
        $this->postTypeManager = $postTypeManager;
        $this->taxonomyManager = $taxonomyManager;
        $this->usageService = $usageService;
    }

    /**
     * Register AJAX actions
     */
    public function register(): void
    {
        add_action('wp_ajax_intucart_sync', [$this, 'handleSync']);
        add_action('wp_ajax_intucart_sync_taxonomy', [$this, 'handleTaxonomySync']);
        add_action('wp_ajax_intucart_get_post_types', [$this, 'handleGetPostTypes']);
    }

    /**
     * Handle unified sync operations (post types, taxonomies)
     */
    public function handleSync(): void
    {
        if (!$this->verifyRequest('intucart_status_nonce')) {
            return;
        }

        $this->extendExecutionTime();

        $syncType = $this->getPostValue('sync_type');
        $postType = $this->getPostValue('post_type');

        if (!in_array($syncType, ['post_types', 'users', 'taxonomies'], true)) {
            wp_send_json_error(['message' => __('Invalid sync type. Must be: post_types, users, or taxonomies', 'intufind')]);
            return;
        }

        // Validate post type if syncing post types
        if ($syncType === 'post_types') {
            if (empty($postType)) {
                wp_send_json_error(['message' => __('post_type parameter is required when sync_type is post_types', 'intufind')]);
                return;
            }

            $validPostTypes = get_post_types(['public' => true], 'names');
            if (class_exists('WooCommerce')) {
                $validPostTypes[] = 'product';
            }

            if (!in_array($postType, $validPostTypes, true)) {
                wp_send_json_error(['message' => sprintf(__('Invalid post type: %s', 'intufind'), $postType)]);
                return;
            }
        }

        try {
            $message = $this->executeSyncOperation($syncType, $postType);

            // Invalidate usage cache to show updated data immediately
            $this->usageService->invalidateUsageCache();

            wp_send_json_success([
                'message' => $message,
                'dismissible' => true
            ]);
        } catch (\Exception $e) {
            $this->logSyncError($syncType, $postType, $e);
            wp_send_json_error(['message' => sprintf(__('Sync failed: %s', 'intufind'), $e->getMessage())]);
        }
    }

    /**
     * Execute the appropriate sync operation
     */
    private function executeSyncOperation(string $syncType, string $postType): string
    {
        if ($syncType === 'post_types') {
            $this->logger->info('Manual post type sync triggered via AJAX', [
                'post_type' => $postType,
                'source' => $syncType
            ]);

            $_POST['post_type'] = $postType;
            $this->postTypeManager->handleManualSync();
            update_option("intucart_last_{$postType}_sync", time());

            return sprintf(__('%s sync completed successfully.', 'intufind'), ucfirst($postType));
        }

        if ($syncType === 'taxonomies') {
            $taxonomy = $this->getPostValue('taxonomy');

            $this->logger->info('Manual taxonomy sync triggered via AJAX', [
                'taxonomy' => $taxonomy,
                'source' => $syncType
            ]);

            if (!empty($taxonomy)) {
                $_POST['taxonomy'] = $taxonomy;
                $this->taxonomyManager->handleManualTaxonomySync();
                update_option("intucart_last_{$taxonomy}_sync", time());
                return sprintf(__('Taxonomy "%s" sync completed successfully.', 'intufind'), $taxonomy);
            }

            unset($_POST['taxonomy']);
            $this->taxonomyManager->handleManualTaxonomySync();
            update_option('intucart_last_taxonomy_sync', time());
            return __('All taxonomies sync completed successfully.', 'intufind');
        }

        return __('Sync completed.', 'intufind');
    }

    /**
     * Log sync error with context
     */
    private function logSyncError(string $syncType, string $postType, \Exception $e): void
    {
        $errorContext = [
            'sync_type' => $syncType,
            'error' => $e->getMessage()
        ];

        if ($syncType === 'post_types') {
            $errorContext['post_type'] = $postType;
        } elseif ($syncType === 'taxonomies') {
            $taxonomy = $this->getPostValue('taxonomy');
            if (!empty($taxonomy)) {
                $errorContext['taxonomy'] = $taxonomy;
            }
        }

        $this->logger->error('Manual sync failed', $errorContext);
    }

    /**
     * Handle taxonomy sync request
     */
    public function handleTaxonomySync(): void
    {
        if (!$this->verifyRequest('intucart_admin_nonce')) {
            return;
        }

        $taxonomy = $this->getPostValue('taxonomy');

        if (empty($taxonomy)) {
            wp_send_json_error(__('Taxonomy parameter required', 'intufind'));
            return;
        }

        try {
            $this->taxonomyManager->handleManualTaxonomySync();
            update_option("intucart_last_sync_{$taxonomy}", current_time('mysql'));

            wp_send_json_success([
                'message' => sprintf(__('Taxonomy "%s" synced successfully', 'intufind'), $taxonomy),
                'last_sync' => current_time('mysql')
            ]);
        } catch (\Exception $e) {
            wp_send_json_error(__('Taxonomy sync failed. Check logs for details.', 'intufind'));
        }
    }

    /**
     * Handle request to get available post types
     */
    public function handleGetPostTypes(): void
    {
        if (!$this->verifyRequest('intucart_admin_nonce')) {
            return;
        }

        try {
            $availablePostTypeNames = $this->postTypeManager->getAvailablePostTypes();
            $formattedTypes = [];

            foreach ($availablePostTypeNames as $slug) {
                $postType = get_post_type_object($slug);
                if (!$postType) {
                    continue;
                }

                $formattedTypes[] = [
                    'slug' => $slug,
                    'label' => $postType->labels->name ?? $postType->label,
                    'singular_label' => $postType->labels->singular_name ?? $postType->label,
                    'count' => wp_count_posts($slug)->publish ?? 0
                ];
            }

            usort($formattedTypes, fn($a, $b) => strcmp($a['label'], $b['label']));

            wp_send_json_success(['post_types' => $formattedTypes]);
        } catch (\Exception $e) {
            $this->logger->error('Failed to get post types', ['error' => $e->getMessage()]);
            wp_send_json_error(['message' => sprintf(__('Failed to get post types: %s', 'intufind'), $e->getMessage())]);
        }
    }
}

