<?php

declare(strict_types=1);

namespace Intucart\Admin\Ajax;

use Intucart\Services\Logger;
use Intucart\Services\Cache\CacheService;
use Intucart\Services\UsageService;
use Intucart\Services\Constants;

/**
 * Handles AJAX requests for system operations (usage, health, notices)
 */
class SystemAjaxHandler extends AbstractAjaxHandler
{
    private CacheService $cache;
    private UsageService $usageService;

    public function __construct(
        Logger $logger,
        CacheService $cache,
        UsageService $usageService
    ) {
        parent::__construct($logger);
        $this->cache = $cache;
        $this->usageService = $usageService;
    }

    /**
     * Register AJAX actions
     */
    public function register(): void
    {
        add_action('wp_ajax_intucart_refresh_usage_data', [$this, 'handleRefreshUsage']);
        add_action('wp_ajax_intucart_refresh_health', [$this, 'handleRefreshHealth']);
        add_action('wp_ajax_intucart_dismiss_license_notice', [$this, 'handleDismissLicenseNotice']);
        add_action('wp_ajax_intucart_reset_cache_stats', [$this, 'handleResetCacheStats']);
    }

    /**
     * Handle request to refresh usage data
     */
    public function handleRefreshUsage(): void
    {
        if (!$this->verifyRequest('intucart_status_nonce')) {
            return;
        }

        try {
            $this->usageService->invalidateUsageCache();

            wp_send_json_success([
                'message' => __('Usage data refreshed successfully. The page will reload to show updated data.', 'intufind'),
                'reload' => true
            ]);
        } catch (\Exception $e) {
            wp_send_json_error([
                'message' => sprintf(__('Failed to refresh usage data: %s', 'intufind'), $e->getMessage())
            ]);
        }
    }

    /**
     * Handle request to refresh health data
     */
    public function handleRefreshHealth(): void
    {
        if (!$this->verifyRequest('intucart_status_nonce')) {
            return;
        }

        try {
            $this->cache->delete('system_health_check', 'intucart_system');
            $this->logger->info('Health cache cleared via AJAX');

            wp_send_json_success(['message' => __('Health data refreshed successfully.', 'intufind')]);
        } catch (\Exception $e) {
            $this->logger->error('Failed to refresh health data', ['error' => $e->getMessage()]);
            wp_send_json_error(['message' => sprintf(__('Failed to refresh health data: %s', 'intufind'), $e->getMessage())]);
        }
    }

    /**
     * Handle request to dismiss license notice
     */
    public function handleDismissLicenseNotice(): void
    {
        if (!wp_verify_nonce($_POST['nonce'] ?? '', 'intucart_dismiss_license_notice')) {
            wp_die('Security check failed');
        }

        if (!current_user_can('manage_options')) {
            wp_die('Permission denied');
        }

        update_option(Constants::LICENSE_NOTICE_DISMISSED_OPTION, true);
        wp_die('success');
    }

    /**
     * Handle request to reset cache stats
     */
    public function handleResetCacheStats(): void
    {
        if (!$this->verifyRequest('intucart_status_nonce')) {
            return;
        }

        try {
            $this->cache->clearCacheStats();
            $this->logger->info('Cache stats reset via AJAX');

            wp_send_json_success(['message' => __('Cache statistics reset successfully.', 'intufind')]);
        } catch (\Exception $e) {
            $this->logger->error('Failed to reset cache stats', ['error' => $e->getMessage()]);
            wp_send_json_error(['message' => sprintf(__('Failed to reset cache stats: %s', 'intufind'), $e->getMessage())]);
        }
    }
}

