<?php

namespace Intucart\Admin;

use Intucart\Services\Logger;
use Intucart\Services\Managers\PostTypeManager;
use Intucart\Services\AIClientManager;
use Intucart\Services\Constants;

/**
 * SyncAdmin Component
 *
 * Adds sync control columns to WordPress admin list tables for posts, pages, and products
 * Provides bulk management of sync inclusion/exclusion settings
 */
class SyncAdmin
{
    private Logger $logger;
    private PostTypeManager $postTypeManager;
    private AIClientManager $aiClientManager;

    /**
     * Cache for supported post types to avoid repeated calls
     *
     * @var array<string>|null
     */
    private ?array $supportedPostTypes = null;

    /**
     * Constructor
     *
     * @param Logger $logger Logger service
     * @param PostTypeManager $postTypeManager PostTypeManager service
     * @param AIClientManager $aiClientManager AI Client Manager service
     */
    public function __construct(Logger $logger, PostTypeManager $postTypeManager, AIClientManager $aiClientManager)
    {
        $this->logger = $logger;
        $this->postTypeManager = $postTypeManager;
        $this->aiClientManager = $aiClientManager;
    }

    /**
     * Get supported post types for sync columns
     *
     * @return array<string>
     */
    private function getSupportedPostTypes(): array
    {
        if ($this->supportedPostTypes === null) {
            // Get syncable post types from PostTypeManager
            $syncablePostTypes = $this->postTypeManager->getSyncablePostTypes();

            // Filter out post types that don't have admin list tables or aren't suitable for sync columns
            $this->supportedPostTypes = array_filter($syncablePostTypes, function ($postType) {
                $postTypeObject = get_post_type_object($postType);

                // Must have admin UI and be shown in admin menu
                if (!$postTypeObject || !$postTypeObject->show_ui || !$postTypeObject->show_in_menu) {
                    return false;
                }

                // Skip system post types that shouldn't have sync columns even if somehow enabled
                $systemTypes = ['attachment', 'revision', 'nav_menu_item', 'customize_changeset'];
                if (in_array($postType, $systemTypes, true)) {
                    return false;
                }

                return true;
            });
        }

        return $this->supportedPostTypes;
    }

    /**
     * Initialize the sync columns admin
     *
     * @return void
     */
    public function initialize(): void
    {
        // Delay initialization until after all post types are registered
        // ACF custom post types are registered at 'init' priority 5
        add_action('init', [$this, 'initializeSyncColumns'], 20);

        // Add AJAX handlers for individual checkbox toggles (can be registered early)
        add_action('wp_ajax_intucart_toggle_sync', [$this, 'handleAjaxToggleSync']);
        add_action('wp_ajax_intucart_toggle_searchable', [$this, 'handleAjaxToggleSearchable']);

        // Add admin scripts and styles (can be registered early)
        add_action('admin_enqueue_scripts', [$this, 'enqueueAdminAssets']);
    }

    /**
     * Initialize sync columns after all post types are registered
     *
     * @return void
     */
    public function initializeSyncColumns(): void
    {
        // Clear cached post types since we're now running after post types are registered
        $this->supportedPostTypes = null;

        // Add columns to supported post types
        foreach ($this->getSupportedPostTypes() as $postType) {
            // Add column headers
            add_filter("manage_{$postType}_posts_columns", [$this, 'addSyncColumn']);
            add_filter("manage_{$postType}_posts_columns", [$this, 'addSearchableColumn']);

            // Add column content
            add_action("manage_{$postType}_posts_custom_column", [$this, 'renderSyncColumn'], 10, 2);
            add_action("manage_{$postType}_posts_custom_column", [$this, 'renderSearchableColumn'], 10, 2);

            // Make columns sortable
            add_filter("manage_edit-{$postType}_sortable_columns", [$this, 'makeSyncColumnSortable']);
            add_filter("manage_edit-{$postType}_sortable_columns", [$this, 'makeSearchableColumnSortable']);

            // Add bulk actions
            add_filter("bulk_actions-edit-{$postType}", [$this, 'addBulkActions']);

            // Handle bulk actions
            add_filter("handle_bulk_actions-edit-{$postType}", [$this, 'handleBulkActions'], 10, 3);
        }

        // Handle sorting by sync status and searchable status
        add_action('pre_get_posts', [$this, 'handleSyncColumnSorting']);
    }

    /**
     * Add sync column to post list tables
     *
     * @param array $columns Existing columns
     * @return array Modified columns
     */
    public function addSyncColumn(array $columns): array
    {
        $pluginName = Constants::PLUGIN_SHORT_NAME;
        
        // Get the same SVG icon used in admin menu - with tooltip
        $iconSvg = '<span class="intucart-column-header-icon" title="' . esc_attr($pluginName) . ' AI Knowledge - Controls what content the AI assistant learns from"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="16" height="16" fill="currentColor" style="vertical-align: middle;"><path d="M19 9l1.25-2.75L23 5l-2.75-1.25L19 1l-1.25 2.75L15 5l2.75 1.25L19 9zm-7.5.5L9 4 6.5 9.5 1 12l5.5 2.5L9 20l2.5-5.5L17 12l-5.5-2.5zM19 15l-1.25 2.75L15 19l2.75 1.25L19 23l1.25-2.75L23 19l-2.75-1.25L19 15z"/></svg></span>';

        // Insert sync column before the date column
        $newColumns = [];
        foreach ($columns as $key => $title) {
            if ($key === 'date') {
                $newColumns['intucart_sync'] = $iconSvg;
            }
            $newColumns[$key] = $title;
        }

        // If date column doesn't exist, add sync column at the end
        if (!isset($newColumns['intucart_sync'])) {
            $newColumns['intucart_sync'] = $iconSvg;
        }

        return $newColumns;
    }

    /**
     * Render sync column content
     *
     * @param string $column Column name
     * @param int $postId Post ID
     * @return void
     */
    public function renderSyncColumn(string $column, int $postId): void
    {
        if ($column !== 'intucart_sync') {
            return;
        }

        $post = get_post($postId);
        if (!$post) {
            return;
        }

        // Get current manual setting
        $excludeFromSync = get_post_meta($postId, '_intucart_exclude_from_sync', true);

        // Get auto-exclusion status
        $isAutoExcluded = $this->postTypeManager->isExcludedPost($postId, $post, $post->post_type);

        // Check if this is a private post
        $isPrivatePost = ($post->post_status === 'private');
        
        // Check if post status is syncable (only publish and private are allowed)
        $isSyncableStatus = in_array($post->post_status, ['publish', 'private'], true);

        // Determine current sync status
        // Private posts are excluded by default unless explicitly enabled
        if ($isPrivatePost && $excludeFromSync !== 'no') {
            $isExcluded = true;
        } else {
            $isExcluded = ($excludeFromSync === 'yes') || ($isAutoExcluded && $excludeFromSync !== 'no');
        }
        
        $isManuallyOverridden = ($excludeFromSync === 'no' && $isAutoExcluded) || ($excludeFromSync === 'yes' && !$isAutoExcluded);

        // Create checkbox
        echo '<div class="intucart-sync-toggle">';
        
        // For non-syncable statuses (drafts, pending, etc.), show disabled state
        if (!$isSyncableStatus) {
            echo '<label>';
            echo '<input type="checkbox" ';
            echo 'class="intucart-sync-checkbox" ';
            echo 'disabled ';
            echo '> ';
            echo '</label>';
            
            // Show appropriate status icon for non-syncable posts
            $statusLabel = ucfirst($post->post_status);
            echo '<span class="intucart-sync-status disabled" data-popover="' . esc_attr($statusLabel) . ' posts cannot be synced - publish first">';
            echo '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512" width="12" height="12"><path fill="#999" d="M256 8C119 8 8 119 8 256s111 248 248 248 248-111 248-248S393 8 256 8zm0 448c-110.5 0-200-89.5-200-200S145.5 56 256 56s200 89.5 200 200-89.5 200-200 200zm101.8-262.2L295.6 256l62.2 62.2c4.7 4.7 4.7 12.3 0 17l-22.6 22.6c-4.7 4.7-12.3 4.7-17 0L256 295.6l-62.2 62.2c-4.7 4.7-12.3 4.7-17 0l-22.6-22.6c-4.7-4.7-4.7-12.3 0-17l62.2-62.2-62.2-62.2c-4.7-4.7-4.7-12.3 0-17l22.6-22.6c4.7-4.7 12.3-4.7 17 0l62.2 62.2 62.2-62.2c4.7-4.7 12.3-4.7 17 0l22.6 22.6c4.7 4.7 4.7 12.3 0 17z"/></svg>';
            echo '</span>';
            echo '</div>';
            return; // Exit early for non-syncable statuses
        }
        
        // For syncable statuses, show normal controls
        echo '<label>';
        echo '<input type="checkbox" ';
        echo 'class="intucart-sync-checkbox" ';
        echo 'data-post-id="' . esc_attr($postId) . '" ';
        echo 'data-nonce="' . wp_create_nonce('intucart_toggle_sync_' . $postId) . '" ';
        echo checked(!$isExcluded, true, false);
        echo '> ';
        echo '</label>';

        // Add status indicator with icons based on sync status and exclusion
        $pluginName = Constants::PLUGIN_SHORT_NAME;

        if ($isExcluded) {
            // Show "not syncing" icon (circle with slash)
            echo '<span class="intucart-sync-status not-syncing" data-popover="' . esc_attr(sprintf(__('Not syncing to %s', 'intufind'), $pluginName)) . '">';
            echo '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512" width="12" height="12"><path opacity=".4" d="M406.78 361.53a186.53 186.53 0 0 1-45.25 45.25L105.22 150.47a186.53 186.53 0 0 1 45.25-45.25z"/><path d="M256 8C119 8 8 119 8 256s111 248 248 248 248-111 248-248S393 8 256 8zm130.11 378.11A184 184 0 1 1 440 256a182.82 182.82 0 0 1-53.89 130.11z"/></svg>';
            echo '</span>';
        } else {
            // Get persisted sync status
            $syncStatus = $this->postTypeManager->getSyncStatus($postId);
            $statusClass = 'syncing';
            $popoverText = sprintf(__('Syncing to %s', 'intufind'), $pluginName);
            $dataAttributes = '';

            // Build enhanced popover text with metadata
            $popoverText = $this->buildPopoverText($syncStatus, $pluginName, $postId);

            // Apply status-specific styling and attributes
            if ($syncStatus['status'] === 'error' && !empty($syncStatus['error'])) {
                $statusClass .= ' sync-warning';
                $dataAttributes = ' data-sync-error="' . esc_attr($syncStatus['error']) . '"';
            } elseif ($syncStatus['status'] === 'success') {
                $statusClass .= ' sync-success';
            } elseif ($syncStatus['status'] === 'pending') {
                // Keep default styling for pending
            }

            // Show "syncing" icon (refresh arrows) with appropriate status
            echo '<span class="intucart-sync-status ' . $statusClass . '" data-popover="' . esc_attr($popoverText) . '"' . $dataAttributes . '>';
            echo '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512" width="12" height="12"><path opacity=".4" d="M0 500V299.67a12 12 0 0 1 12-12h200.33a12 12 0 0 1 12 12v47.41a12 12 0 0 1-12.57 12l-101.87-4.88a176.07 176.07 0 0 0 317.25-56.94 12 12 0 0 1 11.67-9.26h49.09a12 12 0 0 1 11.8 14.18C478.07 417.08 377.19 504 256 504a247.43 247.43 0 0 1-188.76-87.17l4.13 82.57a12 12 0 0 1-12 12.6H12a12 12 0 0 1-12-12z"/><path d="M12.3 209.82C33.93 94.92 134.81 8 256 8a247.4 247.4 0 0 1 188.9 87.34l-4-82.77A12 12 0 0 1 452.92 0h47.41a12 12 0 0 1 12 12v200.33a12 12 0 0 1-12 12H300a12 12 0 0 1-12-12v-47.41a12 12 0 0 1 12.57-12l101.53 4.88a176.07 176.07 0 0 0-317.24 56.94A12 12 0 0 1 73.19 224H24.1a12 12 0 0 1-11.8-14.18z"/></svg>';
            echo '</span>';
        }

        echo '</div>';
    }

    /**
     * Make sync column sortable
     *
     * @param array $columns Sortable columns
     * @return array Modified sortable columns
     */
    public function makeSyncColumnSortable(array $columns): array
    {
        $columns['intucart_sync'] = 'intucart_sync';
        return $columns;
    }

    /**
     * Handle sorting by sync status
     *
     * @param \WP_Query $query WordPress query object
     * @return void
     */
    public function handleSyncColumnSorting(\WP_Query $query): void
    {
        if (!is_admin() || !$query->is_main_query()) {
            return;
        }

        $orderby = $query->get('orderby');
        if ($orderby !== 'intucart_sync') {
            return;
        }

        $query->set('meta_key', '_intucart_exclude_from_sync');
        $query->set('orderby', 'meta_value');

        // Handle posts without the meta key (treat as included by default)
        $query->set('meta_query', [
            'relation' => 'OR',
            [
                'key' => '_intucart_exclude_from_sync',
                'compare' => 'EXISTS'
            ],
            [
                'key' => '_intucart_exclude_from_sync',
                'compare' => 'NOT EXISTS'
            ]
        ]);
    }

    /**
     * Add searchable column to post list tables
     *
     * @param array $columns Existing columns
     * @return array Modified columns
     */
    public function addSearchableColumn(array $columns): array
    {
        $pluginName = Constants::PLUGIN_SHORT_NAME;
        
        // Use magnifying glass icon with tooltip
        $iconSvg = '<span class="intucart-column-header-icon" title="' . esc_attr($pluginName) . ' AI Search - Controls whether content appears in search results"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512" width="16" height="16" fill="currentColor" style="vertical-align: middle;"><path d="M505 442.7L405.3 343c-4.5-4.5-10.6-7-17-7H372c27.6-35.3 44-79.7 44-128C416 93.1 322.9 0 208 0S0 93.1 0 208s93.1 208 208 208c48.3 0 92.7-16.4 128-44v16.3c0 6.4 2.5 12.5 7 17l99.7 99.7c9.4 9.4 24.6 9.4 33.9 0l28.3-28.3c9.4-9.4 9.4-24.6.1-34zM208 336c-70.7 0-128-57.2-128-128 0-70.7 57.2-128 128-128 70.7 0 128 57.2 128 128 0 70.7-57.2 128-128 128z"/></svg></span>';

        // Insert searchable column before the date column
        $newColumns = [];
        foreach ($columns as $key => $title) {
            if ($key === 'date') {
                $newColumns['intucart_searchable'] = $iconSvg;
            }
            $newColumns[$key] = $title;
        }

        // If date column doesn't exist, add searchable column at the end
        if (!isset($newColumns['intucart_searchable'])) {
            $newColumns['intucart_searchable'] = $iconSvg;
        }

        return $newColumns;
    }

    /**
     * Render searchable column content
     *
     * @param string $column Column name
     * @param int $postId Post ID
     * @return void
     */
    public function renderSearchableColumn(string $column, int $postId): void
    {
        if ($column !== 'intucart_searchable') {
            return;
        }

        $post = get_post($postId);
        if (!$post) {
            return;
        }

        // Check if this is a private post
        $isPrivatePost = ($post->post_status === 'private');
        
        // Check if post status is syncable (only publish and private are allowed)
        $isSyncableStatus = in_array($post->post_status, ['publish', 'private'], true);

        // Check if item is synced to cloud first
        $excludeFromSync = get_post_meta($postId, '_intucart_exclude_from_sync', true);
        $isAutoExcluded = $this->postTypeManager->isExcludedPost($postId, $post, $post->post_type);
        
        // Private posts require explicit opt-in
        if ($isPrivatePost && $excludeFromSync !== 'no') {
            $isSynced = false;
        } else {
            $isSynced = !(($excludeFromSync === 'yes') || ($isAutoExcluded && $excludeFromSync !== 'no'));
        }

        // Get searchable setting (default to true for public posts, false for private)
        $searchableMeta = get_post_meta($postId, '_intucart_searchable', true);
        $isSearchable = ($searchableMeta !== 'no'); // Default true unless explicitly set to 'no'
        
        // Private posts are always non-searchable
        if ($isPrivatePost) {
            $isSearchable = false;
        }

        $pluginName = Constants::PLUGIN_SHORT_NAME;

        echo '<div class="intucart-searchable-toggle' . (!$isSynced ? ' disabled' : '') . '">';
        
        // For non-syncable statuses, show disabled state
        if (!$isSyncableStatus) {
            echo '<label>';
            echo '<input type="checkbox" ';
            echo 'class="intucart-searchable-checkbox" ';
            echo 'disabled ';
            echo '> ';
            echo '</label>';
            
            $statusLabel = ucfirst($post->post_status);
            echo '<span class="intucart-searchable-status disabled" data-popover="' . esc_attr($statusLabel) . ' posts cannot be searched - publish first">';
            echo '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512" width="12" height="12"><path fill="#999" d="M256 8C119 8 8 119 8 256s111 248 248 248 248-111 248-248S393 8 256 8zm0 448c-110.5 0-200-89.5-200-200S145.5 56 256 56s200 89.5 200 200-89.5 200-200 200zm101.8-262.2L295.6 256l62.2 62.2c4.7 4.7 4.7 12.3 0 17l-22.6 22.6c-4.7 4.7-12.3 4.7-17 0L256 295.6l-62.2 62.2c-4.7 4.7-12.3 4.7-17 0l-22.6-22.6c-4.7-4.7-4.7-12.3 0-17l62.2-62.2-62.2-62.2c-4.7-4.7-4.7-12.3 0-17l22.6-22.6c4.7-4.7 12.3-4.7 17 0l62.2 62.2 62.2-62.2c4.7-4.7 12.3-4.7 17 0l22.6 22.6c4.7 4.7 4.7 12.3 0 17z"/></svg>';
            echo '</span>';
        } elseif (!$isSynced) {
            // Not synced - show disabled state
            echo '<label>';
            echo '<input type="checkbox" ';
            echo 'class="intucart-searchable-checkbox" ';
            echo 'disabled ';
            echo '> ';
            echo '</label>';
            
            $disabledMessage = $isPrivatePost 
                ? '🔒 Private content - enable AI Knowledge to sync (never searchable, chatbot-only)' 
                : 'Not indexed - enable AI Knowledge first to make searchable';
            
            echo '<span class="intucart-searchable-status disabled" data-popover="' . esc_attr($disabledMessage) . '">';
            echo '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512" width="12" height="12"><path d="M256 8C119 8 8 119 8 256s111 248 248 248 248-111 248-248S393 8 256 8zm0 448c-110.5 0-200-89.5-200-200S145.5 56 256 56s200 89.5 200 200-89.5 200-200 200zm101.8-262.2L295.6 256l62.2 62.2c4.7 4.7 4.7 12.3 0 17l-22.6 22.6c-4.7 4.7-12.3 4.7-17 0L256 295.6l-62.2 62.2c-4.7 4.7-12.3 4.7-17 0l-22.6-22.6c-4.7-4.7-4.7-12.3 0-17l62.2-62.2-62.2-62.2c-4.7-4.7-4.7-12.3 0-17l22.6-22.6c4.7-4.7 12.3-4.7 17 0l62.2 62.2 62.2-62.2c4.7-4.7 12.3-4.7 17 0l22.6 22.6c4.7 4.7 4.7 12.3 0 17z"/></svg>';
            echo '</span>';
        } elseif ($isPrivatePost) {
            // Private post synced - always non-searchable, show locked state
            echo '<label>';
            echo '<input type="checkbox" ';
            echo 'class="intucart-searchable-checkbox" ';
            echo 'checked disabled ';
            echo '> ';
            echo '</label>';
            echo '<span class="intucart-searchable-status not-searchable" data-popover="🔒 Private content is chatbot-only (never searchable)">';
            echo '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 640 512" width="12" height="12"><path opacity=".4" d="M634 471L36 3.51A16 16 0 0 0 13.51 6l-10 12.49A16 16 0 0 0 6 41l598 467.49a16 16 0 0 0 22.49-2.49l10-12.49A16 16 0 0 0 634 471z"/><path d="M320 400c-75.85 0-137.25-58.71-142.9-133.11L72.2 185.82c-13.79 17.3-26.48 35.59-36.72 55.59a32.35 32.35 0 0 0 0 29.19C89.71 376.41 197.07 448 320 448c26.91 0 52.87-4 77.89-10.46L346 397.39c-8.59 1.79-17.42 2.61-26 2.61zm313.82 58.1l-110.55-85.44a331.25 331.25 0 0 0 81.25-102.07 32.35 32.35 0 0 0 0-29.19C550.29 135.59 442.93 64 320 64a308.15 308.15 0 0 0-147.32 37.7L45.46 3.37A16 16 0 0 0 23 6.18L3.37 31.45A16 16 0 0 0 6.18 53.9l588.36 454.73a16 16 0 0 0 22.46-2.81l19.64-25.27a16 16 0 0 0-2.82-22.45zm-183.72-142l-39.3-30.38A94.75 94.75 0 0 0 416 256a94.76 94.76 0 0 0-121.31-92.21A47.65 47.65 0 0 1 304 192a46.64 46.64 0 0 1-1.54 10l-73.61-56.89A142.31 142.31 0 0 1 320 112a143.92 143.92 0 0 1 144 144c0 21.63-5.29 41.79-13.9 60.11z"/></svg>';
            echo '</span>';
        } else {
            // Synced public post - show normal searchable controls
            echo '<label>';
            echo '<input type="checkbox" ';
            echo 'class="intucart-searchable-checkbox" ';
            echo 'data-post-id="' . esc_attr($postId) . '" ';
            echo 'data-nonce="' . wp_create_nonce('intucart_toggle_searchable_' . $postId) . '" ';
            echo checked($isSearchable, true, false);
            echo '> ';
            echo '</label>';

            // Add status indicator
            if ($isSearchable) {
                echo '<span class="intucart-searchable-status searchable" data-popover="' . esc_attr(sprintf(__('Visible in %s search results', 'intufind'), $pluginName)) . '">';
                echo '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512" width="12" height="12"><path d="M173.898 439.404l-166.4-166.4c-9.997-9.997-9.997-26.206 0-36.204l36.203-36.204c9.997-9.998 26.207-9.998 36.204 0L192 312.69 432.095 72.596c9.997-9.997 26.207-9.997 36.204 0l36.203 36.204c9.997 9.997 9.997 26.206 0 36.204l-294.4 294.401c-9.998 9.997-26.207 9.997-36.204-.001z"/></svg>';
                echo '</span>';
            } else {
                echo '<span class="intucart-searchable-status not-searchable" data-popover="' . esc_attr(sprintf(__('Hidden from %s search results (chatbot can still reference)', 'intufind'), $pluginName)) . '">';
                echo '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 640 512" width="12" height="12"><path opacity=".4" d="M634 471L36 3.51A16 16 0 0 0 13.51 6l-10 12.49A16 16 0 0 0 6 41l598 467.49a16 16 0 0 0 22.49-2.49l10-12.49A16 16 0 0 0 634 471z"/><path d="M320 400c-75.85 0-137.25-58.71-142.9-133.11L72.2 185.82c-13.79 17.3-26.48 35.59-36.72 55.59a32.35 32.35 0 0 0 0 29.19C89.71 376.41 197.07 448 320 448c26.91 0 52.87-4 77.89-10.46L346 397.39c-8.59 1.79-17.42 2.61-26 2.61zm313.82 58.1l-110.55-85.44a331.25 331.25 0 0 0 81.25-102.07 32.35 32.35 0 0 0 0-29.19C550.29 135.59 442.93 64 320 64a308.15 308.15 0 0 0-147.32 37.7L45.46 3.37A16 16 0 0 0 23 6.18L3.37 31.45A16 16 0 0 0 6.18 53.9l588.36 454.73a16 16 0 0 0 22.46-2.81l19.64-25.27a16 16 0 0 0-2.82-22.45zm-183.72-142l-39.3-30.38A94.75 94.75 0 0 0 416 256a94.76 94.76 0 0 0-121.31-92.21A47.65 47.65 0 0 1 304 192a46.64 46.64 0 0 1-1.54 10l-73.61-56.89A142.31 142.31 0 0 1 320 112a143.92 143.92 0 0 1 144 144c0 21.63-5.29 41.79-13.9 60.11z"/></svg>';
                echo '</span>';
            }
        }

        echo '</div>';
    }

    /**
     * Make searchable column sortable
     *
     * @param array $columns Sortable columns
     * @return array Modified sortable columns
     */
    public function makeSearchableColumnSortable(array $columns): array
    {
        $columns['intucart_searchable'] = 'intucart_searchable';
        return $columns;
    }

    /**
     * Add bulk actions for sync management
     *
     * @param array $actions Existing bulk actions
     * @return array Modified bulk actions
     */
    public function addBulkActions(array $actions): array
    {
        $actions['intucart_enable_sync'] = __('Enable Intucart Sync', 'intufind');
        $actions['intucart_disable_sync'] = __('Disable Intucart Sync', 'intufind');
        $actions['intucart_make_searchable'] = __('Make Searchable', 'intufind');
        $actions['intucart_hide_from_search'] = __('Hide from Search', 'intufind');
        return $actions;
    }

    /**
     * Handle bulk actions for sync and searchable management
     *
     * @param string $redirectTo Redirect URL
     * @param string $action Bulk action name
     * @param array $postIds Array of post IDs
     * @return string Modified redirect URL
     */
    public function handleBulkActions(string $redirectTo, string $action, array $postIds): string
    {
        $validActions = ['intucart_enable_sync', 'intucart_disable_sync', 'intucart_make_searchable', 'intucart_hide_from_search'];
        if (!in_array($action, $validActions, true)) {
            return $redirectTo;
        }

        if (empty($postIds)) {
            return $redirectTo;
        }

        $updated = 0;

        foreach ($postIds as $postId) {
            $post = get_post($postId);
            if (!$post) {
                continue;
            }

            // Handle sync actions
            if ($action === 'intucart_enable_sync') {
                update_post_meta($postId, '_intucart_exclude_from_sync', 'no');
                $this->logger->info('Bulk sync enabled', [
                    'post_id' => $postId,
                    'post_type' => $post->post_type,
                    'title' => $post->post_title
                ]);
                $updated++;
            } elseif ($action === 'intucart_disable_sync') {
                update_post_meta($postId, '_intucart_exclude_from_sync', 'yes');
                $this->logger->info('Bulk sync disabled', [
                    'post_id' => $postId,
                    'post_type' => $post->post_type,
                    'title' => $post->post_title
                ]);
                $updated++;
            }
            // Handle searchable actions
            elseif ($action === 'intucart_make_searchable') {
                update_post_meta($postId, '_intucart_searchable', 'yes');
                $this->logger->info('Bulk made searchable', [
                    'post_id' => $postId,
                    'post_type' => $post->post_type,
                    'title' => $post->post_title
                ]);
                $updated++;
            } elseif ($action === 'intucart_hide_from_search') {
                update_post_meta($postId, '_intucart_searchable', 'no');
                $this->logger->info('Bulk hidden from search', [
                    'post_id' => $postId,
                    'post_type' => $post->post_type,
                    'title' => $post->post_title
                ]);
                $updated++;
            }
        }

        return $redirectTo;
    }

    /**
     * Handle AJAX toggle sync request
     *
     * @return void
     */
    public function handleAjaxToggleSync(): void
    {
        // Check user capabilities
        if (!current_user_can('edit_posts')) {
            wp_send_json_error(['message' => __('Permission denied.', 'intufind')]);
            return;
        }

        $postId = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        $nonce = isset($_POST['nonce']) ? sanitize_text_field($_POST['nonce']) : '';

        if (!$postId || !wp_verify_nonce($nonce, 'intucart_toggle_sync_' . $postId)) {
            wp_send_json_error(['message' => __('Invalid request.', 'intufind')]);
            return;
        }

        $post = get_post($postId);
        if (!$post) {
            wp_send_json_error(['message' => __('Post not found.', 'intufind')]);
            return;
        }

        // Check if user can edit this specific post
        if (!current_user_can('edit_post', $postId)) {
            wp_send_json_error(['message' => __('Permission denied for this post.', 'intufind')]);
            return;
        }

        try {
            // Get current state
            $excludeFromSync = get_post_meta($postId, '_intucart_exclude_from_sync', true);
            $isAutoExcluded = $this->postTypeManager->isExcludedPost($postId, $post, $post->post_type);
            $currentlyExcluded = ($excludeFromSync === 'yes') || ($isAutoExcluded && $excludeFromSync !== 'no');

            // Toggle the state
            $newMetaValue = $currentlyExcluded ? 'no' : 'yes';
            update_post_meta($postId, '_intucart_exclude_from_sync', $newMetaValue);

            // Calculate new state for response
            $newlyExcluded = ($newMetaValue === 'yes') || ($isAutoExcluded && $newMetaValue !== 'no');

                        // Attempt real-time operations based on the state change
            $operationAttempted = false;
            $operationSuccess = false;
            $operationError = null;
            $operationType = null;

            if (!$newlyExcluded && $currentlyExcluded) {
                // Post was just enabled for sync (was excluded, now included) - attempt immediate sync
                $operationAttempted = true;
                $operationType = 'sync';
                try {
                    // Check if content has changed since last sync to avoid unnecessary operations
                    $hasContentChanged = $this->postTypeManager->hasContentChanged($postId);
                    $existingSyncStatus = get_post_meta($postId, '_intucart_sync_status', true);
                    
                    $this->logger->info('Attempting real-time sync for newly enabled post', [
                        'post_id' => $postId,
                        'post_type' => $post->post_type,
                        'title' => $post->post_title,
                        'content_changed' => $hasContentChanged,
                        'existing_status' => $existingSyncStatus
                    ]);

                    // Skip sync only if content hasn't changed AND previous sync was successful
                    // This ensures failed syncs get retried even with unchanged content
                    if (!$hasContentChanged && $existingSyncStatus === 'success') {
                        $this->logger->info('Skipping real-time sync - content unchanged since last successful sync', [
                            'post_id' => $postId,
                            'post_type' => $post->post_type
                        ]);
                        
                        // Just update the sync status to show it's enabled but unchanged
                        $contentHash = $this->postTypeManager->generateContentHash($postId);
                        $this->postTypeManager->setSyncStatus($postId, 'success', null, [
                            'method' => 'realtime',
                            'operation' => 'skipped_unchanged',
                            'content_hash' => $contentHash
                        ]);
                        
                        $operationSuccess = true;
                        $operationType = 'skipped_unchanged';
                    } else {
                        // Get post data using PostTypeManager
                        $postData = $this->postTypeManager->getCloudPayload($postId, $post->post_type);

                    if (!empty($postData)) {
                        // Generate content hash and determine operation type
                        $contentHash = $this->postTypeManager->generateContentHash($postId);
                        $existingSyncStatus = get_post_meta($postId, '_intucart_sync_status', true);
                        $operationType = ($existingSyncStatus === 'success') ? 'update' : 'create';

                        // Set pending status before sync attempt
                        $this->postTypeManager->setSyncStatus($postId, 'pending', null, [
                            'method' => 'realtime',
                            'operation' => $operationType,
                            'content_hash' => $contentHash
                        ]);

                        // Add permalink to the payload
                        $postData['url'] = get_permalink($postId);

                        // Attempt to sync to cloud using AI SDK
                        $aiClient = $this->aiClientManager->getClient();
                        if ($post->post_type === 'product') {
                            $result = $aiClient->products()->upsert($postData);
                        } else {
                            $result = $aiClient->posts()->upsert($postData);
                        }
                        $operationSuccess = $result['success'] ?? false;

                        if ($operationSuccess) {
                            $this->postTypeManager->setSyncStatus($postId, 'success', null, [
                                'method' => 'realtime',
                                'operation' => $operationType,
                                'content_hash' => $contentHash
                            ]);
                            $this->logger->info('Real-time sync successful', [
                                'post_id' => $postId,
                                'post_type' => $post->post_type
                            ]);
                        } else {
                            $operationError = __('Cloud sync failed', 'intufind');
                            $this->postTypeManager->setSyncStatus($postId, 'error', $operationError, [
                                'method' => 'realtime',
                                'operation' => $operationType,
                                'content_hash' => $contentHash
                            ]);
                            $this->logger->warning('Real-time sync failed - cloud upsert returned false', [
                                'post_id' => $postId,
                                'post_type' => $post->post_type
                            ]);
                        }
                    } else {
                        $operationError = __('No post data available', 'intufind');
                        $contentHash = $this->postTypeManager->generateContentHash($postId);
                        $this->postTypeManager->setSyncStatus($postId, 'error', $operationError, [
                            'method' => 'realtime',
                            'operation' => 'create',
                            'content_hash' => $contentHash
                        ]);
                        $this->logger->warning('Real-time sync skipped - no post data available', [
                            'post_id' => $postId,
                            'post_type' => $post->post_type
                        ]);
                        }
                    }
                } catch (\Exception $e) {
                    $operationError = $e->getMessage();
                    $operationSuccess = false;
                    $contentHash = $this->postTypeManager->generateContentHash($postId);
                    $existingSyncStatus = get_post_meta($postId, '_intucart_sync_status', true);
                    $operationType = ($existingSyncStatus === 'success') ? 'update' : 'create';
                    $this->postTypeManager->setSyncStatus($postId, 'error', $operationError, [
                        'method' => 'realtime',
                        'operation' => $operationType,
                        'content_hash' => $contentHash
                    ]);
                    $this->logger->error('Real-time sync failed with exception', [
                        'post_id' => $postId,
                        'post_type' => $post->post_type,
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                }
            } elseif ($newlyExcluded && !$currentlyExcluded) {
                // Post was just disabled for sync (was included, now excluded) - attempt immediate deletion
                $operationAttempted = true;
                $operationType = 'delete';
                try {
                                        $this->logger->info('Attempting real-time deletion for newly excluded post', [
                        'post_id' => $postId,
                        'post_type' => $post->post_type,
                        'title' => $post->post_title
                                        ]);

                    // Set pending status before delete attempt
                    $this->postTypeManager->setSyncStatus($postId, 'pending', null, [
                        'method' => 'realtime',
                        'operation' => 'delete'
                    ]);

                    // Attempt to delete from cloud using AI SDK
                    $aiClient = $this->aiClientManager->getClient();
                    if ($post->post_type === 'product') {
                        $result = $aiClient->products()->delete((string)$postId);
                    } else {
                        $result = $aiClient->posts()->delete((string)$postId);
                    }
                    $operationSuccess = $result['success'] ?? false;

                    if ($operationSuccess) {
                        // Clear sync status after successful deletion (post is no longer in cloud)
                        $this->postTypeManager->clearSyncStatus($postId);
                        $this->logger->info('Real-time deletion successful', [
                            'post_id' => $postId,
                            'post_type' => $post->post_type
                        ]);
                    } else {
                        $operationError = __('Cloud deletion failed', 'intufind');
                        $this->postTypeManager->setSyncStatus($postId, 'error', $operationError, [
                            'method' => 'realtime',
                            'operation' => 'delete'
                        ]);
                        $this->logger->warning('Real-time deletion failed - cloud delete returned false', [
                            'post_id' => $postId,
                            'post_type' => $post->post_type
                        ]);
                    }
                } catch (\Exception $e) {
                    $operationError = $e->getMessage();
                    $operationSuccess = false;
                    $this->postTypeManager->setSyncStatus($postId, 'error', $operationError, [
                        'method' => 'realtime',
                        'operation' => 'delete'
                    ]);
                    $this->logger->error('Real-time deletion failed with exception', [
                        'post_id' => $postId,
                        'post_type' => $post->post_type,
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                }
            }

            // Generate the appropriate status icon HTML
            $statusIconHtml = '';
            if ($newlyExcluded) {
                // Not syncing icon (circle with slash) - red
                $statusIconHtml = '<span class="intucart-sync-status not-syncing" title="' . esc_attr__('Not syncing to Intucart', 'intufind') . '">';
                $statusIconHtml .= '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512" width="12" height="12"><path opacity=".4" d="M406.78 361.53a186.53 186.53 0 0 1-45.25 45.25L105.22 150.47a186.53 186.53 0 0 1 45.25-45.25z"/><path d="M256 8C119 8 8 119 8 256s111 248 248 248 248-111 248-248S393 8 256 8zm130.11 378.11A184 184 0 1 1 440 256a182.82 182.82 0 0 1-53.89 130.11z"/></svg>';
                $statusIconHtml .= '</span>';
            } else {
                // Determine sync status class and attributes
                $statusClass = 'syncing';
                $title = __('Syncing to Intucart', 'intufind');
                $dataAttributes = '';

                // If operation was attempted and failed, mark with warning status
                if ($operationAttempted && !$operationSuccess) {
                    $statusClass .= ' sync-warning';
                    $operationName = $operationType === 'delete' ? __('deletion', 'intufind') : __('sync', 'intufind');
                    $title = sprintf(__('Syncing to Intucart - %s failed: %s', 'intufind'), ucfirst($operationName), ($operationError ?: __('Unknown error', 'intufind')));
                    $dataAttributes = ' data-sync-error="' . esc_attr($operationError ?: __('Unknown error', 'intufind')) . '"';
                } elseif ($operationAttempted && $operationSuccess) {
                    $statusClass .= ' sync-success';
                    $operationName = $operationType === 'delete' ? __('removed from cloud', 'intufind') : __('synced successfully', 'intufind');
                    $title = sprintf(__('Syncing to Intucart - Content %s', 'intufind'), $operationName);
                }

                // Syncing icon (refresh arrows) with appropriate status
                $statusIconHtml = '<span class="intucart-sync-status ' . $statusClass . '" title="' . esc_attr($title) . '"' . $dataAttributes . '>';
                $statusIconHtml .= '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512" width="12" height="12"><path opacity=".4" d="M0 500V299.67a12 12 0 0 1 12-12h200.33a12 12 0 0 1 12 12v47.41a12 12 0 0 1-12.57 12l-101.87-4.88a176.07 176.07 0 0 0 317.25-56.94 12 12 0 0 1 11.67-9.26h49.09a12 12 0 0 1 11.8 14.18C478.07 417.08 377.19 504 256 504a247.43 247.43 0 0 1-188.76-87.17l4.13 82.57a12 12 0 0 1-12 12.6H12a12 12 0 0 1-12-12z"/><path d="M12.3 209.82C33.93 94.92 134.81 8 256 8a247.4 247.4 0 0 1 188.9 87.34l-4-82.77A12 12 0 0 1 452.92 0h47.41a12 12 0 0 1 12 12v200.33a12 12 0 0 1-12 12H300a12 12 0 0 1-12-12v-47.41a12 12 0 0 1 12.57-12l101.53 4.88a176.07 176.07 0 0 0-317.24 56.94A12 12 0 0 1 73.19 224H24.1a12 12 0 0 1-11.8-14.18z"/></svg>';
                $statusIconHtml .= '</span>';
            }

            // Log the change
            $this->logger->info('Sync setting toggled via AJAX', [
                'post_id' => $postId,
                'post_type' => $post->post_type,
                'title' => $post->post_title,
                'exclude_from_sync' => $newMetaValue,
                'is_excluded' => $newlyExcluded
            ]);

            $response = [
                'excluded' => $newlyExcluded,
                'meta_value' => $newMetaValue,
                'status_icon_html' => $statusIconHtml
            ];

            // Add operation information if an operation was attempted
            if ($operationAttempted) {
                $response['operation_attempted'] = true;
                $response['operation_success'] = $operationSuccess;
                $response['operation_type'] = $operationType;
                if ($operationError) {
                    $response['operation_error'] = $operationError;
                }
            }

            wp_send_json_success($response);
        } catch (\Exception $e) {
            $this->logger->error('Failed to toggle sync setting', [
                'post_id' => $postId,
                'error' => $e->getMessage()
            ]);
            wp_send_json_error(['message' => __('Failed to update sync setting.', 'intufind')]);
        }
    }

    /**
     * Handle AJAX toggle searchable request
     *
     * @return void
     */
    public function handleAjaxToggleSearchable(): void
    {
        // Check user capabilities
        if (!current_user_can('edit_posts')) {
            wp_send_json_error(['message' => __('Permission denied.', 'intufind')]);
            return;
        }

        $postId = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        $nonce = isset($_POST['nonce']) ? sanitize_text_field($_POST['nonce']) : '';

        if (!$postId || !wp_verify_nonce($nonce, 'intucart_toggle_searchable_' . $postId)) {
            wp_send_json_error(['message' => __('Invalid request.', 'intufind')]);
            return;
        }

        $post = get_post($postId);
        if (!$post) {
            wp_send_json_error(['message' => __('Post not found.', 'intufind')]);
            return;
        }

        // Check if user can edit this specific post
        if (!current_user_can('edit_post', $postId)) {
            wp_send_json_error(['message' => __('Permission denied for this post.', 'intufind')]);
            return;
        }

        try {
            // Check if item is synced to cloud first
            $excludeFromSync = get_post_meta($postId, '_intucart_exclude_from_sync', true);
            $isAutoExcluded = $this->postTypeManager->isExcludedPost($postId, $post, $post->post_type);
            $isSynced = !(($excludeFromSync === 'yes') || ($isAutoExcluded && $excludeFromSync !== 'no'));

            if (!$isSynced) {
                wp_send_json_error(['message' => __('Item must be synced to cloud before changing searchable status.', 'intufind')]);
                return;
            }

            // Get current state (default to true/searchable)
            $searchableMeta = get_post_meta($postId, '_intucart_searchable', true);
            $currentlySearchable = ($searchableMeta !== 'no'); // Default true unless explicitly set to 'no'

            // Toggle the state
            $newMetaValue = $currentlySearchable ? 'no' : 'yes';
            update_post_meta($postId, '_intucart_searchable', $newMetaValue);

            // Calculate new state for response
            $newlySearchable = ($newMetaValue !== 'no');

            // Attempt real-time cloud update
            $operationAttempted = false;
            $operationSuccess = false;
            $operationError = null;

            try {
                $this->logger->info('Attempting real-time searchable update', [
                    'post_id' => $postId,
                    'post_type' => $post->post_type,
                    'searchable' => $newlySearchable
                ]);

                // Get post data for cloud update
                $postData = $this->postTypeManager->getCloudPayload($postId, $post->post_type);
                
                if (!empty($postData)) {
                    $operationAttempted = true;
                    
                    // Update the searchable field in the payload
                    $postData['searchable'] = $newlySearchable;
                    $postData['url'] = get_permalink($postId);

                    // Update in cloud using AI SDK
                    $aiClient = $this->aiClientManager->getClient();
                    if ($post->post_type === 'product') {
                        $result = $aiClient->products()->upsert($postData);
                    } else {
                        $result = $aiClient->posts()->upsert($postData);
                    }
                    $operationSuccess = $result['success'] ?? false;

                    if ($operationSuccess) {
                        $this->logger->info('Real-time searchable update successful', [
                            'post_id' => $postId,
                            'searchable' => $newlySearchable
                        ]);
                    } else {
                        $operationError = __('Cloud update failed', 'intufind');
                        $this->logger->warning('Real-time searchable update failed', [
                            'post_id' => $postId,
                            'searchable' => $newlySearchable
                        ]);
                    }
                }
            } catch (\Exception $e) {
                $operationAttempted = true;
                $operationError = $e->getMessage();
                $operationSuccess = false;
                $this->logger->error('Real-time searchable update failed with exception', [
                    'post_id' => $postId,
                    'error' => $e->getMessage()
                ]);
            }

            $pluginName = Constants::PLUGIN_SHORT_NAME;

            // Generate the appropriate status icon HTML
            $statusIconHtml = '';
            if ($newlySearchable) {
                // Searchable icon (checkmark) - green
                $statusIconHtml = '<span class="intucart-searchable-status searchable" title="' . esc_attr(sprintf(__('Visible in %s search results', 'intufind'), $pluginName)) . '">';
                $statusIconHtml .= '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512" width="12" height="12"><path d="M173.898 439.404l-166.4-166.4c-9.997-9.997-9.997-26.206 0-36.204l36.203-36.204c9.997-9.998 26.207-9.998 36.204 0L192 312.69 432.095 72.596c9.997-9.997 26.207-9.997 36.204 0l36.203 36.204c9.997 9.997 9.997 26.206 0 36.204l-294.4 294.401c-9.998 9.997-26.207 9.997-36.204-.001z"/></svg>';
                $statusIconHtml .= '</span>';
            } else {
                // Not searchable icon (eye with slash) - orange
                $statusIconHtml = '<span class="intucart-searchable-status not-searchable" title="' . esc_attr(sprintf(__('Hidden from %s search results (chatbot can still reference)', 'intufind'), $pluginName)) . '">';
                $statusIconHtml .= '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 640 512" width="12" height="12"><path opacity=".4" d="M634 471L36 3.51A16 16 0 0 0 13.51 6l-10 12.49A16 16 0 0 0 6 41l598 467.49a16 16 0 0 0 22.49-2.49l10-12.49A16 16 0 0 0 634 471z"/><path d="M320 400c-75.85 0-137.25-58.71-142.9-133.11L72.2 185.82c-13.79 17.3-26.48 35.59-36.72 55.59a32.35 32.35 0 0 0 0 29.19C89.71 376.41 197.07 448 320 448c26.91 0 52.87-4 77.89-10.46L346 397.39c-8.59 1.79-17.42 2.61-26 2.61zm313.82 58.1l-110.55-85.44a331.25 331.25 0 0 0 81.25-102.07 32.35 32.35 0 0 0 0-29.19C550.29 135.59 442.93 64 320 64a308.15 308.15 0 0 0-147.32 37.7L45.46 3.37A16 16 0 0 0 23 6.18L3.37 31.45A16 16 0 0 0 6.18 53.9l588.36 454.73a16 16 0 0 0 22.46-2.81l19.64-25.27a16 16 0 0 0-2.82-22.45zm-183.72-142l-39.3-30.38A94.75 94.75 0 0 0 416 256a94.76 94.76 0 0 0-121.31-92.21A47.65 47.65 0 0 1 304 192a46.64 46.64 0 0 1-1.54 10l-73.61-56.89A142.31 142.31 0 0 1 320 112a143.92 143.92 0 0 1 144 144c0 21.63-5.29 41.79-13.9 60.11z"/></svg>';
                $statusIconHtml .= '</span>';
            }

            // Log the change
            $this->logger->info('Searchable setting toggled via AJAX', [
                'post_id' => $postId,
                'post_type' => $post->post_type,
                'title' => $post->post_title,
                'searchable' => $newMetaValue,
                'is_searchable' => $newlySearchable
            ]);

            $response = [
                'searchable' => $newlySearchable,
                'meta_value' => $newMetaValue,
                'status_icon_html' => $statusIconHtml
            ];

            // Add operation information if an operation was attempted
            if ($operationAttempted) {
                $response['operation_attempted'] = true;
                $response['operation_success'] = $operationSuccess;
                $response['operation_type'] = 'searchable_update';
                if ($operationError) {
                    $response['operation_error'] = $operationError;
                }
            }

            wp_send_json_success($response);
        } catch (\Exception $e) {
            $this->logger->error('Failed to toggle searchable setting', [
                'post_id' => $postId,
                'error' => $e->getMessage()
            ]);
            wp_send_json_error(['message' => __('Failed to update searchable setting.', 'intufind')]);
        }
    }

    /**
     * Enqueue admin assets for sync columns
     *
     * @param string $hook Current admin page hook
     * @return void
     */
    public function enqueueAdminAssets(string $hook): void
    {
        // Only load on edit.php pages (post list tables)
        if ($hook !== 'edit.php') {
            return;
        }

        // Get current post type
        $postType = get_current_screen()->post_type ?? '';
        if (!in_array($postType, $this->getSupportedPostTypes(), true)) {
            return;
        }

        // Enqueue CSS
        wp_enqueue_style(
            'intucart-sync',
            INTUCART_PLUGIN_URL . 'assets/css/admin/sync.css',
            [],
            filemtime(INTUCART_PLUGIN_DIR . 'assets/css/admin/sync.css')
        );

        // Enqueue JavaScript
        wp_enqueue_script(
            'intucart-sync',
            INTUCART_PLUGIN_URL . 'assets/js/admin/sync.js',
            ['jquery'],
            filemtime(INTUCART_PLUGIN_DIR . 'assets/js/admin/sync.js'),
            true
        );

        // Localize script
        wp_localize_script('intucart-sync', 'intucartSync', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'pluginName' => Constants::PLUGIN_SHORT_NAME
        ]);
    }

    /**
     * Build enhanced popover text with all metadata
     *
     * @param array $syncStatus Sync status array from PostTypeManager
     * @param string $pluginName Plugin name for display
     * @param int $postId Post ID for additional checks
     * @return string Enhanced popover text
     */
    private function buildPopoverText(array $syncStatus, string $pluginName, int $postId): string
    {
        $lines = [];

        // Main status line
        if ($syncStatus['status'] === 'error' && !empty($syncStatus['error'])) {
            $lines[] = sprintf(__('Sync failed: %s', 'intufind'), $syncStatus['error']);
        } elseif ($syncStatus['status'] === 'success') {
            $lines[] = __('Content synced successfully', 'intufind');
            if ($syncStatus['timestamp']) {
                $lines[0] .= ' (' . sprintf(__('%s ago', 'intufind'), human_time_diff($syncStatus['timestamp'])) . ')';
            }
        } elseif ($syncStatus['status'] === 'pending') {
            $lines[] = __('Sync in progress...', 'intufind');
        } else {
            $lines[] = sprintf(__('Syncing to %s', 'intufind'), $pluginName);
        }

                // Add metadata if available
        $metadataLines = [];

                // Method and operation
        if (!empty($syncStatus['method']) || !empty($syncStatus['operation'])) {
            $methodText = !empty($syncStatus['method']) ? ucfirst($syncStatus['method']) : __('Unknown', 'intufind');
            $operationText = !empty($syncStatus['operation']) ? ucfirst($syncStatus['operation']) : '';

            if ($operationText) {
                $metadataLines[] = sprintf(__('• Method: %s', 'intufind'), $methodText);
                $metadataLines[] = sprintf(__('• Operation: %s', 'intufind'), $operationText);
            } else {
                $metadataLines[] = sprintf(__('• Method: %s', 'intufind'), $methodText);
            }
        }

        // Content change detection
        if (!empty($syncStatus['content_hash']) && $syncStatus['status'] === 'success') {
            if ($this->postTypeManager->hasContentChanged($postId)) {
                $metadataLines[] = __('⚠️ Content modified since last sync', 'intufind');
            }
        }

        // Combine main status with metadata, each on its own line
        if (!empty($metadataLines)) {
            return $lines[0] . "\n" . implode("\n", $metadataLines);
        }

        return $lines[0];
    }
}
