<?php

namespace Intucart\Admin;

use Intucart\Services\Logger;
use Intucart\Services\Managers\PostTypeManager;

/**
 * SyncMetaBox Admin Component
 *
 * Handles the admin UI for sync control meta boxes in the WordPress editor
 * Provides per-post control over sync inclusion/exclusion
 */
class SyncMetaBox
{
    private Logger $logger;
    private PostTypeManager $postTypeManager;

    /**
     * Constructor
     *
     * @param Logger $logger Logger service
     * @param PostTypeManager $postTypeManager PostTypeManager service
     */
    public function __construct(Logger $logger, PostTypeManager $postTypeManager)
    {
        $this->logger = $logger;
        $this->postTypeManager = $postTypeManager;
    }

    /**
     * Initialize the sync meta box
     *
     * @return void
     */
    public function initialize(): void
    {
        // Admin UI hooks for page exclusion control
        add_action('add_meta_boxes', [$this, 'addSyncMetaBox']);
        add_action('save_post', [$this, 'saveSyncMetaBox'], 10, 2);
    }

    /**
     * Add meta box for sync control
     *
     * @return void
     */
    public function addSyncMetaBox(): void
    {
        // Add meta box to pages and posts
        $screens = ['page', 'post'];

        // Add to other public post types as well
        $postTypes = get_post_types(['public' => true], 'names');
        foreach ($postTypes as $postType) {
            if (!in_array($postType, ['attachment', 'revision', 'nav_menu_item'])) {
                $screens[] = $postType;
            }
        }

        // Add to product if WooCommerce is active
        if (class_exists('WooCommerce')) {
            $screens[] = 'product';
        }

        foreach ($screens as $screen) {
            add_meta_box(
                'intucart_sync_control',
                'Intufind',
                [$this, 'renderSyncMetaBox'],
                $screen,
                'side',
                'default'
            );
        }
    }

    /**
     * Render sync control meta box
     *
     * @param \WP_Post $post The post being edited
     * @return void
     */
    public function renderSyncMetaBox(\WP_Post $post): void
    {
        // Add nonce for security
        wp_nonce_field('intucart_sync_meta_box', 'intucart_sync_meta_box_nonce');

        // Get current manual settings
        $excludeFromSync = get_post_meta($post->ID, '_intucart_exclude_from_sync', true);
        $searchableMeta = get_post_meta($post->ID, '_intucart_searchable', true);

        // Get auto-exclusion status for information using the unified post exclusion
        $isAutoExcluded = $this->postTypeManager->isExcludedPost($post->ID, $post, $post->post_type);

        // Check if this is a private post
        $isPrivatePost = ($post->post_status === 'private');
        
        // Check if post status is syncable (only publish and private are allowed)
        $isSyncableStatus = in_array($post->post_status, ['publish', 'private'], true);

        // Determine checkbox state: checked if manually excluded OR auto-excluded with no manual override
        $isSyncExcluded = ($excludeFromSync === 'yes') || ($isAutoExcluded && $excludeFromSync !== 'no');
        
        // For private posts, default is excluded unless explicitly enabled
        if ($isPrivatePost && $excludeFromSync !== 'no') {
            $isSyncExcluded = true;
        }
        
        $isSynced = !$isSyncExcluded;

        // Searchable state (default true unless explicitly set to 'no', but private posts are always non-searchable)
        $isSearchable = ($searchableMeta !== 'no');
        if ($isPrivatePost) {
            $isSearchable = false; // Private posts are never searchable
        }

        echo '<div style="padding: 10px 0;">';
        
        // Non-syncable status notice (drafts, pending, etc.)
        if (!$isSyncableStatus) {
            $statusLabel = ucfirst($post->post_status);
            echo '<div class="notice notice-warning inline" style="margin: 0 0 16px 0; padding: 8px;">';
            echo '<p style="margin: 0;"><strong>' . esc_html($statusLabel) . ' Post:</strong> Only published content can be synced to AI. ';
            echo 'Publish this post to enable AI indexing.</p></div>';
        }
        
        // Private post notice (if applicable)
        if ($isPrivatePost) {
            echo '<div class="notice notice-info inline" style="margin: 0 0 16px 0; padding: 8px;">';
            echo '<p style="margin: 0;"><strong>Private Post:</strong> Private content is excluded from AI by default for security. ';
            if ($isSynced) {
                echo 'You have explicitly enabled this content for AI assistance.';
            } else {
                echo 'Enable below to make available to AI chatbot (non-searchable).';
            }
            echo '</p></div>';
        }
        
        // AI Knowledge control
        echo '<div style="margin-bottom: 16px;">';
        echo '<label style="display: flex; align-items: center; gap: 8px; font-weight: 600;';
        if (!$isSyncableStatus) {
            echo ' opacity: 0.5;';
        }
        echo '">';
        echo '<input type="checkbox" ';
        echo 'id="intucart_exclude_from_sync" ';
        echo 'name="intucart_exclude_from_sync" ';
        echo 'value="yes" ';
        echo checked($isSyncExcluded, true, false);
        if (!$isSyncableStatus) {
            echo 'disabled ';
        }
        echo '> ';
        echo '<span>Don\'t index this content</span>';
        echo '</label>';

        if (!$isSyncableStatus) {
            echo '<p class="description" style="margin: 8px 0 0 0;">';
            echo '⚠ Publish this post to enable AI indexing.';
            echo '</p>';
        } elseif ($isAutoExcluded && !$isPrivatePost) {
            echo '<div class="notice notice-warning inline" style="margin: 10px 0; padding: 8px;">';
            echo '<p style="margin: 0;"><strong>Auto-excluded:</strong> This page is automatically excluded based on its type/title/slug ';
            echo '(e.g., Cart, Checkout, My Account, Sample Page). ';

            if ($excludeFromSync === 'no') {
                echo 'Manual override is <strong>enabled</strong> - this will be indexed despite auto-exclusion.';
            } else {
                echo 'Uncheck the box above to override and force indexing.';
            }
            echo '</p></div>';
        } else {
            echo '<p class="description" style="margin: 8px 0 0 0;">';
            if ($isSyncExcluded) {
                echo '✗ This content is excluded from AI Knowledge.';
            } else {
                echo '✓ This content is indexed for AI assistance.';
            }
            echo '</p>';
        }
        echo '</div>';

        // AI Search control
        echo '<div style="margin-bottom: 8px; padding-top: 16px; border-top: 1px solid #dcdcde;">';
        echo '<label style="display: flex; align-items: center; gap: 8px; font-weight: 600;';
        if (!$isSynced || $isPrivatePost || !$isSyncableStatus) {
            echo ' opacity: 0.5;';
        }
        echo '">';
        echo '<input type="checkbox" ';
        echo 'id="intucart_hide_from_search" ';
        echo 'name="intucart_hide_from_search" ';
        echo 'value="yes" ';
        echo checked(!$isSearchable, true, false);
        if (!$isSynced || $isPrivatePost || !$isSyncableStatus) {
            echo 'disabled ';
        }
        echo '> ';
        echo '<span>Hide from search results</span>';
        echo '</label>';

        echo '<p class="description" style="margin: 8px 0 0 0;">';
        if (!$isSyncableStatus) {
            echo '⚠ Publish to enable search control.';
        } elseif ($isPrivatePost) {
            echo '🔒 Private posts are always hidden from search (chatbot-only access).';
        } elseif (!$isSynced) {
            echo '⚠ Content must be indexed first to control search visibility.';
        } elseif (!$isSearchable) {
            echo '✗ Hidden from search results (AI can still reference).';
        } else {
            echo '✓ Visible in search results.';
        }
        echo '</p>';
        echo '</div>';

        echo '</div>';

        // Enqueue sync metabox script with localized data
        wp_enqueue_script(
            'intucart-sync-metabox',
            INTUCART_PLUGIN_URL . 'assets/js/admin/sync-metabox.js',
            ['jquery'],
            INTUCART_VERSION,
            true
        );
        wp_localize_script('intucart-sync-metabox', 'intucartSyncMetabox', [
            'isPrivatePost' => $isPrivatePost,
            'isSyncableStatus' => $isSyncableStatus
        ]);
    }

    /**
     * Save sync control meta box
     *
     * @param int      $post_id The post ID being saved
     * @param \WP_Post $post    The post being saved
     * @return void
     */
    public function saveSyncMetaBox(int $post_id, \WP_Post $post): void
    {
        // Verify nonce
        if (
            !isset($_POST['intucart_sync_meta_box_nonce']) ||
            !wp_verify_nonce($_POST['intucart_sync_meta_box_nonce'], 'intucart_sync_meta_box')
        ) {
            return;
        }

        // Don't save on autosave
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        // Check permissions
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        // Save the sync exclusion meta value
        $excludeFromSync = isset($_POST['intucart_exclude_from_sync']) ? 'yes' : 'no';
        update_post_meta($post_id, '_intucart_exclude_from_sync', $excludeFromSync);

        // Save the searchable meta value (inverted logic: checkbox is "hide from search")
        $hideFromSearch = isset($_POST['intucart_hide_from_search']) ? 'yes' : 'no';
        $searchable = ($hideFromSearch === 'yes') ? 'no' : 'yes';
        update_post_meta($post_id, '_intucart_searchable', $searchable);

        // Log the change
        $this->logger->info('Sync settings updated', [
            'post_id' => $post_id,
            'post_type' => $post->post_type,
            'title' => $post->post_title,
            'exclude_from_sync' => $excludeFromSync,
            'searchable' => $searchable
        ]);
    }
}
