<?php

declare(strict_types=1);

namespace Intucart\Admin\Tabs;

use Intucart\Services\Logger;

/**
 * Abstract base class for admin tab renderers
 */
abstract class AbstractTabRenderer implements TabRendererInterface
{
    protected Logger $logger;

    public function __construct(Logger $logger)
    {
        $this->logger = $logger;
    }

    /**
     * Render a header section with title and description
     *
     * @param string $title
     * @param string $description
     * @return void
     */
    protected function renderHeader(string $title, string $description): void
    {
        echo '<div class="intucart-header-section">';
        echo '<h2 class="intucart-main-header">' . esc_html($title) . '</h2>';
        echo '<p class="intucart-main-description">' . esc_html($description) . '</p>';
        echo '</div>';
    }

    /**
     * Render a content panel with optional header
     *
     * @param string $title Panel title
     * @param string $description Panel description
     * @param callable $contentCallback Callback to render panel content
     * @param string $icon Optional dashicon class
     * @return void
     */
    protected function renderPanel(
        string $title,
        string $description,
        callable $contentCallback,
        string $icon = ''
    ): void {
        echo '<div class="intucart-content-panel">';
        echo '<h3 class="intucart-panel-header">';
        if (!empty($icon)) {
            echo '<span class="dashicons ' . esc_attr($icon) . '"></span>';
        }
        echo esc_html($title);
        echo '</h3>';
        echo '<p class="intucart-panel-description">' . esc_html($description) . '</p>';

        $contentCallback();

        echo '</div>';
    }

    /**
     * Render a settings form with standard WordPress options
     *
     * @param string $optionGroup Settings group name
     * @param string $section Section name
     * @param string $submitLabel Submit button label
     * @return void
     */
    protected function renderSettingsForm(
        string $optionGroup,
        string $section,
        string $submitLabel
    ): void {
        echo '<form action="options.php" method="post" novalidate>';
        settings_fields($optionGroup);

        global $wp_settings_sections;
        if (isset($wp_settings_sections[$optionGroup][$section])) {
            echo '<table class="form-table">';
            do_settings_fields($optionGroup, $section);
            echo '</table>';
        }

        \Intucart\Components\Button::primary($submitLabel, ['type' => 'submit']);
        echo '</form>';
    }
}

