<?php

declare(strict_types=1);

namespace Intucart\Admin\Tabs;

use Intucart\Services\Logger;
use Intucart\Services\Constants;
use Intucart\Services\AIClientManager;
use Intucart\Pages\Settings\ChatSettings;
use Intucart\Pages\Prompts;
use Intucart\Components\Tabs;
use Intucart\Components\Button;

/**
 * Renders the Chatbot tab in admin settings with subtabs
 */
class ChatbotTabRenderer extends AbstractTabRenderer
{
    private AIClientManager $aiClientManager;
    private ChatSettings $chatSettings;
    private Prompts $promptsPage;

    public function __construct(
        Logger $logger,
        AIClientManager $aiClientManager,
        ChatSettings $chatSettings,
        Prompts $promptsPage
    ) {
        parent::__construct($logger);
        $this->aiClientManager = $aiClientManager;
        $this->chatSettings = $chatSettings;
        $this->promptsPage = $promptsPage;
    }

    public function getId(): string
    {
        return 'chatbot';
    }

    public function getTabConfig(): array
    {
        return [
            'url' => '?page=intucart&tab=chatbot',
            'label' => __('Chatbot', 'intufind'),
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path opacity=".4" d="M256 32C114.6 32 0 125.1 0 240c0 49.6 21.4 95 57 130.7C44.5 421.1 2.7 466 2.2 466.5a8 8 0 0 0-1.5 8.7A7.83 7.83 0 0 0 8 480c66.3 0 116-31.8 140.6-51.4A305 305 0 0 0 256 448c141.4 0 256-93.1 256-208S397.4 32 256 32zM128 272a32 32 0 1 1 32-32 32 32 0 0 1-32 32zm128 0a32 32 0 1 1 32-32 32 32 0 0 1-32 32zm128 0a32 32 0 1 1 32-32 32 32 0 0 1-32 32z"/><path d="M128 208a32 32 0 1 0 32 32 32 32 0 0 0-32-32zm128 0a32 32 0 1 0 32 32 32 32 0 0 0-32-32zm128 0a32 32 0 1 0 32 32 32 32 0 0 0-32-32z"/></svg>'
        ];
    }

    public function render(): void
    {
        $this->renderHeader(
            __('AI Chat Assistant', 'intufind'),
            __('Configure your AI-powered chat widget and manage suggested prompts to provide intelligent customer support.', 'intufind')
        );

        $apiKeyValid = $this->isApiKeyValid();
        $subtab = $this->getCurrentSubtab($apiKeyValid);

        $this->renderSubtabs($subtab, $apiKeyValid);
        $this->renderSubtabContent($subtab, $apiKeyValid);
    }

    /**
     * Check if API key is valid
     */
    private function isApiKeyValid(): bool
    {
        $api_key = get_option(Constants::API_KEY_OPTION, '');

        if (empty($api_key)) {
            return false;
        }

        return $this->aiClientManager->hasApiKey();
    }

    /**
     * Get current subtab with validation
     */
    private function getCurrentSubtab(bool $apiKeyValid): string
    {
        $subtab = isset($_GET['chatbot_subtab']) ? sanitize_text_field($_GET['chatbot_subtab']) : 'settings';

        $availableSubtabs = ['settings', 'styling', 'woocommerce', 'liveagent'];
        if ($apiKeyValid) {
            $availableSubtabs[] = 'prompts';
        }

        if (!in_array($subtab, $availableSubtabs, true)) {
            return 'settings';
        }

        // Redirect prompts tab if not available
        if (!$apiKeyValid && $subtab === 'prompts') {
            return 'settings';
        }

        return $subtab;
    }

    /**
     * Render subtab navigation
     */
    private function renderSubtabs(string $currentSubtab, bool $apiKeyValid): void
    {
        $base = admin_url('admin.php?page=intucart&tab=chatbot');

        $innerTabs = [
            'settings' => [
                'url' => $base . '&chatbot_subtab=settings',
                'label' => __('Settings', 'intufind')
            ],
            'styling' => [
                'url' => $base . '&chatbot_subtab=styling',
                'label' => __('Styling', 'intufind')
            ],
            'woocommerce' => [
                'url' => $base . '&chatbot_subtab=woocommerce',
                'label' => __('WooCommerce', 'intufind')
            ],
            'liveagent' => [
                'url' => $base . '&chatbot_subtab=liveagent',
                'label' => __('Live Agent', 'intufind')
            ],
            'prompts' => [
                'url' => $base . '&chatbot_subtab=prompts',
                'label' => __('Prompts', 'intufind'),
                'condition' => $apiKeyValid
            ]
        ];

        Tabs::renderInner($innerTabs, $currentSubtab);

        // Show success message if settings were saved
        if (isset($_GET['settings-updated']) && $_GET['settings-updated']) {
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Chat settings saved successfully!', 'intufind') . '</p></div>';
        }
    }

    /**
     * Render subtab content
     */
    private function renderSubtabContent(string $subtab, bool $apiKeyValid): void
    {
        if (in_array($subtab, ['settings', 'styling', 'woocommerce'], true)) {
            $this->renderChatbotSubtabsForm($subtab);
        } elseif ($subtab === 'liveagent') {
            $this->chatSettings->renderLiveAgentSettings();
        } else {
            $this->renderPromptsContent($apiKeyValid);
        }
    }

    /**
     * Render the combined settings form for settings/styling/woocommerce subtabs
     */
    private function renderChatbotSubtabsForm(string $activeSubtab): void
    {
        echo '<form action="options.php" method="post" novalidate>';
        settings_fields('intucart_chat_settings');

        // Settings Tab Content
        echo '<div id="chatbot-settings-tab" class="intucart-subtab-content" style="' . ($activeSubtab !== 'settings' ? 'display:none;' : '') . '">';
        $this->chatSettings->renderChatSettings();
        echo '</div>';

        // Styling Tab Content
        echo '<div id="chatbot-styling-tab" class="intucart-subtab-content" style="' . ($activeSubtab !== 'styling' ? 'display:none;' : '') . '">';
        $this->chatSettings->renderChatStylingSettings();
        echo '</div>';

        // WooCommerce Tab Content
        echo '<div id="chatbot-woocommerce-tab" class="intucart-subtab-content" style="' . ($activeSubtab !== 'woocommerce' ? 'display:none;' : '') . '">';
        $this->chatSettings->renderWooCommerceSettings();
        echo '</div>';

        Button::primary(__('Save Chat Settings', 'intufind'), ['type' => 'submit']);
        echo '</form>';

        $this->renderTabSwitchingScript();
    }

    /**
     * Render prompts content (or API key required message)
     */
    private function renderPromptsContent(bool $apiKeyValid): void
    {
        if ($apiKeyValid) {
            $this->promptsPage->renderEmbedded();
            return;
        }

        $api_key = get_option(Constants::API_KEY_OPTION, '');

        echo '<div class="intucart-content-panel">';
        echo '<div class="intucart-notice intucart-notice-warning" style="margin: 0;">';
        echo '<p><strong>' . esc_html__('⚠️ API Key Required', 'intufind') . '</strong></p>';
        echo '<p>' . esc_html__('Prompt management requires a valid API key. Prompts are stored and managed through our cloud service to provide intelligent suggestions across all your chat interactions.', 'intufind') . '</p>';

        echo '<p style="margin-top: 12px;">';
        if (empty($api_key)) {
            Button::primary(__('Configure API Key →', 'intufind'), [
                'href' => admin_url('admin.php?page=intucart&tab=license')
            ]);
        } else {
            echo esc_html__('Your API key appears to be invalid. ', 'intufind');
            Button::primary(__('Check API Key →', 'intufind'), [
                'href' => admin_url('admin.php?page=intucart&tab=license')
            ]);
        }
        echo '</p>';

        echo '</div>';
        echo '</div>';
    }

    /**
     * Render JavaScript for tab switching without page reload
     */
    private function renderTabSwitchingScript(): void
    {
        wp_enqueue_script(
            'intucart-chatbot-tabs',
            INTUCART_PLUGIN_URL . 'assets/js/admin/chatbot-tabs.js',
            ['jquery'],
            INTUCART_VERSION,
            true
        );
    }
}

