<?php

declare(strict_types=1);

namespace Intucart\Admin\Tabs;

use Intucart\Services\Logger;
use Intucart\Services\Constants;
use Intucart\Services\Managers\PostTypeManager;
use Intucart\Services\Managers\TaxonomyManager;
use Intucart\Components\Button;

/**
 * Renders the Indexing tab in admin settings
 */
class IndexingTabRenderer extends AbstractTabRenderer
{
    private PostTypeManager $postTypeManager;
    private TaxonomyManager $taxonomyManager;

    public function __construct(
        Logger $logger,
        PostTypeManager $postTypeManager,
        TaxonomyManager $taxonomyManager
    ) {
        parent::__construct($logger);
        $this->postTypeManager = $postTypeManager;
        $this->taxonomyManager = $taxonomyManager;
    }

    public function getId(): string
    {
        return 'indexing';
    }

    public function getTabConfig(): array
    {
        return [
            'url' => '?page=intucart&tab=indexing',
            'label' => __('Indexing', 'intufind'),
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 576 512"><path opacity=".4" d="M272 439.93a71.81 71.81 0 0 1-140 22.78 70.91 70.91 0 0 1-84-70.77 72.51 72.51 0 0 1 1.42-14.1A80 80 0 0 1 0 304c0-29.66 16.34-55.26 40.34-69.07A79.05 79.05 0 0 1 32 200a80 80 0 0 1 49.66-74A63.6 63.6 0 0 1 144 48c.75 0 1.45.2 2.2.22A63.87 63.87 0 0 1 272 64zM576 304c0-29.66-16.34-55.26-40.34-69.07A79.05 79.05 0 0 0 544 200a80 80 0 0 0-49.66-74A63.6 63.6 0 0 0 432 48c-.75 0-1.45.2-2.2.22A63.87 63.87 0 0 0 304 64v375.9a71.81 71.81 0 0 0 140 22.78 71.73 71.73 0 0 0 82.61-84.87A80 80 0 0 0 576 304z"/><path d="M304 439.93a8 8 0 0 1-8 8h-16a8 8 0 0 1-8-8V72a8 8 0 0 1 8-8h16a8 8 0 0 1 8 8z"/></svg>'
        ];
    }

    public function render(): void
    {
        $this->renderHeader(
            __('Content Indexing', 'intufind'),
            sprintf(
                __('Control how often your website data is indexed with %s\'s AI system to power search results, chat responses, and product recommendations. Keeping your content index up to date ensures optimal performance across all AI features.', 'intufind'),
                Constants::PLUGIN_SHORT_NAME
            )
        );

        $this->renderTimingSettings();
        $this->renderContentTypesSection();
    }

    /**
     * Render the index timing settings form
     */
    private function renderTimingSettings(): void
    {
        echo '<form action="options.php" method="post" novalidate>';
        settings_fields(Constants::SETTINGS_GROUP_SYNC);

        $this->renderPanel(
            __('Index Timing Settings', 'intufind'),
            __('Configure how frequently your content is indexed with the AI system. The index happens automatically in the background.', 'intufind'),
            function () {
                global $wp_settings_sections;
                if (isset($wp_settings_sections[Constants::SETTINGS_GROUP_SYNC]['intucart_sync_section'])) {
                    echo '<table class="form-table">';
                    do_settings_fields(Constants::SETTINGS_GROUP_SYNC, 'intucart_sync_section');
                    echo '</table>';
                }
            },
            'dashicons-clock'
        );

        Button::primary(__('Save Indexing Settings', 'intufind'), ['type' => 'submit']);
        echo '</form>';
    }

    /**
     * Render the content types management section
     */
    private function renderContentTypesSection(): void
    {
        // Get available post types
        $availablePostTypeNames = $this->postTypeManager->getAvailablePostTypes();
        $allPostTypes = [];
        foreach ($availablePostTypeNames as $postTypeName) {
            $postTypeObject = get_post_type_object($postTypeName);
            if ($postTypeObject) {
                $allPostTypes[$postTypeName] = $postTypeObject;
            }
        }

        // Get available taxonomies
        $syncablePostTypes = $this->postTypeManager->getSyncablePostTypes();
        $allTaxonomies = $this->taxonomyManager->getAllAvailableTaxonomies($syncablePostTypes);

        // Get current preferences
        $userPreferences = $this->getOrInitializePostTypePreferences($allPostTypes);
        $taxonomyPreferences = $this->getOrInitializeTaxonomyPreferences($syncablePostTypes);

        $features = class_exists('WooCommerce') 
            ? __('search, chatbot responses, and product recommendations', 'intufind')
            : __('search and chatbot responses', 'intufind');
        $this->renderHeader(
            __('Content Types', 'intufind'),
            sprintf(
                __('Choose which content types and taxonomies to include in your %s index. Selected content will be available for AI-powered %s. Only indexed content types and taxonomies will appear in search results and chat interactions.', 'intufind'),
                Constants::PLUGIN_NAME,
                $features
            )
        );

        echo '<div class="intucart-content-panel">';
        echo '<h3 class="intucart-panel-header">';
        echo '<span class="dashicons dashicons-list-view"></span>';
        echo esc_html__('Content Selection', 'intufind');
        echo '</h3>';
        $features = class_exists('WooCommerce') 
            ? __('search results, chatbot responses, and product recommendations', 'intufind')
            : __('search results and chatbot responses', 'intufind');
        echo '<p class="intucart-panel-description">' . sprintf(
            esc_html__('Select which content types and taxonomies should be included in the AI index. These will power %s. Changes will trigger automatic synchronization in the background.', 'intufind'),
            $features
        ) . '</p>';

        echo '<form id="intucart-content-types-form">';

        $this->renderContentTypesTable($allPostTypes, $userPreferences, $allTaxonomies, $taxonomyPreferences);

        echo '<p class="submit">';
        Button::primary(__('Save Content Types', 'intufind'), ['type' => 'submit']);
        echo '</p>';

        echo '</form>';
        echo '</div>';

        $this->renderContentTypesScript();
    }

    /**
     * Get or initialize post type preferences
     */
    private function getOrInitializePostTypePreferences(array $allPostTypes): array
    {
        $preferences = get_option('intucart_syncable_post_types', []);

        if (empty($preferences)) {
            $preferences = array_fill_keys(array_keys($allPostTypes), true);
            update_option('intucart_syncable_post_types', $preferences);
        }

        return $preferences;
    }

    /**
     * Get or initialize taxonomy preferences
     */
    private function getOrInitializeTaxonomyPreferences(array $syncablePostTypes): array
    {
        $preferences = get_option('intucart_syncable_taxonomies', []);

        if (empty($preferences)) {
            $allTaxonomyNames = $this->taxonomyManager->getAllAvailableTaxonomyNames($syncablePostTypes);
            $preferences = $this->taxonomyManager->getDefaultTaxonomyPreferences($allTaxonomyNames);
            update_option('intucart_syncable_taxonomies', $preferences);
        }

        return $preferences;
    }

    /**
     * Render the content types table
     */
    private function renderContentTypesTable(
        array $allPostTypes,
        array $userPreferences,
        array $allTaxonomies,
        array $taxonomyPreferences
    ): void {
        echo '<table class="widefat">';
        echo '<thead><tr>';
        echo '<th>' . esc_html__('Content Type', 'intufind') . '</th>';
        echo '<th>' . esc_html__('Available', 'intufind') . '</th>';
        echo '<th>' . esc_html__('Include in Index', 'intufind') . '</th>';
        echo '</tr></thead>';
        echo '<tbody>';

        // Render post types grouped
        $groupedPostTypes = $this->groupPostTypes($allPostTypes);
        foreach ($groupedPostTypes as $groupName => $postTypes) {
            if (!empty($postTypes)) {
                $this->renderGroupHeader($groupName);
                foreach ($postTypes as $postType) {
                    $this->renderPostTypeRow($postType, $userPreferences);
                }
            }
        }

        // Render taxonomies grouped
        if (!empty($allTaxonomies)) {
            $groupedTaxonomies = $this->groupTaxonomies($allTaxonomies);
            foreach ($groupedTaxonomies as $groupName => $taxonomies) {
                if (!empty($taxonomies)) {
                    $this->renderGroupHeader($groupName);
                    foreach ($taxonomies as $taxonomy) {
                        $this->renderTaxonomyRow($taxonomy, $taxonomyPreferences);
                    }
                }
            }
        }

        echo '</tbody>';
        echo '</table>';
    }

    /**
     * Render a group header row
     */
    private function renderGroupHeader(string $groupName): void
    {
        echo '<tr class="post-type-group-header">';
        echo '<td colspan="3">' . esc_html($groupName) . '</td>';
        echo '</tr>';
    }

    /**
     * Render a post type row
     */
    private function renderPostTypeRow(\WP_Post_Type $postType, array $userPreferences): void
    {
        $isChecked = isset($userPreferences[$postType->name]) && $userPreferences[$postType->name] === true;
        $postCount = $this->getSyncablePostCount($postType->name);

        echo '<tr>';
        echo '<td><strong>' . esc_html($postType->labels->name ?? $postType->name) . '</strong></td>';
        echo '<td class="available-items-column">';
        if ($postCount > 0) {
            echo '<span class="status-text-ok">' . number_format($postCount) . ' ' . esc_html__('items', 'intufind') . '</span>';
        } else {
            echo '<span class="status-text-neutral">0 ' . esc_html__('items', 'intufind') . '</span>';
        }
        echo '</td>';
        echo '<td>';
        echo '<label>';
        echo '<input type="checkbox" name="post_types[]" value="' . esc_attr($postType->name) . '"';
        if ($isChecked) {
            echo ' checked';
        }
        echo '> ' . esc_html__('Include', 'intufind');
        echo '</label>';
        echo '</td>';
        echo '</tr>';
    }

    /**
     * Render a taxonomy row
     */
    private function renderTaxonomyRow(\WP_Taxonomy $taxonomy, array $taxonomyPreferences): void
    {
        $isChecked = isset($taxonomyPreferences[$taxonomy->name]) && $taxonomyPreferences[$taxonomy->name] === true;
        $termCount = wp_count_terms(['taxonomy' => $taxonomy->name, 'hide_empty' => false]);
        $termCount = is_wp_error($termCount) ? 0 : (int) $termCount;

        echo '<tr>';
        echo '<td><strong>' . esc_html($taxonomy->labels->name ?? $taxonomy->name) . '</strong></td>';
        echo '<td class="available-items-column">';
        if ($termCount > 0) {
            echo '<span class="status-text-ok">' . number_format($termCount) . ' ' . esc_html__('terms', 'intufind') . '</span>';
        } else {
            echo '<span class="status-text-neutral">0 ' . esc_html__('terms', 'intufind') . '</span>';
        }
        echo '</td>';
        echo '<td>';
        echo '<label>';
        echo '<input type="checkbox" name="taxonomies[]" value="' . esc_attr($taxonomy->name) . '"';
        if ($isChecked) {
            echo ' checked';
        }
        echo '> ' . esc_html__('Include', 'intufind');
        echo '</label>';
        echo '</td>';
        echo '</tr>';
    }

    /**
     * Get count of syncable posts for a post type
     */
    private function getSyncablePostCount(string $postTypeName): int
    {
        $posts = get_posts([
            'post_type' => $postTypeName,
            'post_status' => 'publish',
            'numberposts' => -1,
            'fields' => 'ids'
        ]);

        if (empty($posts)) {
            return 0;
        }

        $count = 0;
        foreach ($posts as $postId) {
            if ($this->postTypeManager->shouldSyncPost($postId, $postTypeName)) {
                $count++;
            }
        }

        return $count;
    }

    /**
     * Group post types for display
     */
    private function groupPostTypes(array $allPostTypes): array
    {
        $groups = [
            __('Core Content', 'intufind') => [],
            __('E-commerce', 'intufind') => [],
            __('Custom Content', 'intufind') => [],
        ];

        $coreLabel = __('Core Content', 'intufind');
        $ecommerceLabel = __('E-commerce', 'intufind');
        $customLabel = __('Custom Content', 'intufind');
        foreach ($allPostTypes as $postType) {
            if (in_array($postType->name, ['post', 'page'], true)) {
                $groups[$coreLabel][] = $postType;
            } elseif ($postType->name === 'product') {
                $groups[$ecommerceLabel][] = $postType;
            } else {
                $groups[$customLabel][] = $postType;
            }
        }

        return array_filter($groups, fn($group) => !empty($group));
    }

    /**
     * Group taxonomies for display
     */
    private function groupTaxonomies(array $allTaxonomies): array
    {
        $groups = [
            __('Core Taxonomies', 'intufind') => [],
            __('E-commerce Taxonomies', 'intufind') => [],
            __('Custom Taxonomies', 'intufind') => [],
        ];

        $coreTaxLabel = __('Core Taxonomies', 'intufind');
        $ecommerceTaxLabel = __('E-commerce Taxonomies', 'intufind');
        $customTaxLabel = __('Custom Taxonomies', 'intufind');
        foreach ($allTaxonomies as $taxonomy) {
            if (in_array($taxonomy->name, ['category', 'post_tag'], true)) {
                $groups[$coreTaxLabel][] = $taxonomy;
            } elseif (strpos($taxonomy->name, 'product_') === 0 || strpos($taxonomy->name, 'pa_') === 0) {
                $groups[$ecommerceTaxLabel][] = $taxonomy;
            } else {
                $groups[$customTaxLabel][] = $taxonomy;
            }
        }

        return array_filter($groups, fn($group) => !empty($group));
    }

    /**
     * Render JavaScript for content types form
     */
    private function renderContentTypesScript(): void
    {
        wp_enqueue_script(
            'intucart-content-types',
            INTUCART_PLUGIN_URL . 'assets/js/admin/content-types.js',
            ['jquery'],
            INTUCART_VERSION,
            true
        );
    }
}

