<?php

declare(strict_types=1);

namespace Intucart\Admin\Tabs;

use Intucart\Services\Logger;
use Intucart\Services\Constants;
use Intucart\Services\Licensing\License;
use Intucart\Services\AIClientManager;
use Intucart\Services\UsageService;
use Intucart\Components\Button;
use Intucart\Components\Field;

/**
 * Renders the License/API Key tab in admin settings
 */
class LicenseTabRenderer extends AbstractTabRenderer
{
    private License $license;
    private AIClientManager $aiClientManager;
    private UsageService $usageService;

    public function __construct(
        Logger $logger,
        License $license,
        AIClientManager $aiClientManager,
        UsageService $usageService
    ) {
        parent::__construct($logger);
        $this->license = $license;
        $this->aiClientManager = $aiClientManager;
        $this->usageService = $usageService;
    }

    public function getId(): string
    {
        return 'license';
    }

    public function getTabConfig(): array
    {
        return [
            'url' => '?page=intucart&tab=license',
            'label' => __('Secret Key', 'intufind'),
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path opacity=".4" d="M336 0c-97.2 0-176 78.8-176 176c0 18.7 2.9 36.8 8.3 53.7L7 391c-4.5 4.5-7 10.6-7 17v80c0 13.3 10.7 24 24 24h80c13.3 0 24-10.7 24-24v-40h40c13.3 0 24-10.7 24-24v-40h40c6.4 0 12.5-2.5 17-7l11.3-11.3c16.9 5.4 35 8.3 53.7 8.3c97.2 0 176-78.8 176-176S433.2 0 336 0z"/><path d="M376 144a40 40 0 1 0 0-80 40 40 0 1 0 0 80z"/></svg>'
        ];
    }

    public function render(): void
    {
        $api_key = get_option(Constants::API_KEY_OPTION, '');
        $usageSummary = $this->usageService->getUsageSummary();

        $this->renderHeaderSection($api_key);
        $this->renderApiKeyForm($api_key, $usageSummary);
        $this->renderHelpSection($api_key);
    }

    /**
     * Render the header section with description and privacy notice
     */
    private function renderHeaderSection(string $api_key): void
    {
        echo '<div class="intucart-header-section">';
        echo '<h2 class="intucart-main-header">' . esc_html__('Secret Key', 'intufind') . '</h2>';

        if (empty($api_key)) {
            $features = class_exists('WooCommerce') 
                ? __('search, chatbot, and product recommendations', 'intufind')
                : __('search and chatbot', 'intufind');
            $message = sprintf(
                __('A Secret Key is required to use %s.', 'intufind'),
                Constants::PLUGIN_NAME
            );
            $description = sprintf(
                __('Enter your Secret Key below to activate all AI-powered features including %s.', 'intufind'),
                $features
            );
            $linkText = __('Learn how to get your Secret Key →', 'intufind');
            echo '<p class="intucart-main-description"><strong>' . esc_html($message) . '</strong> ' . esc_html($description) . ' ';
            echo sprintf(
                __('Need help? %s', 'intufind'),
                '<a href="' . esc_url(Constants::WEBSITE_URL . 'docs/wordpress/installation#getting-your-api-key') . '" target="_blank">' . esc_html($linkText) . '</a>'
            ) . '</p>';
        } else {
            $features = class_exists('WooCommerce') 
                ? __('search, chatbot, and product recommendations', 'intufind')
                : __('search and chatbot', 'intufind');
            echo '<p class="intucart-main-description">' . sprintf(
                esc_html__('Your Secret Key connects this site to %s cloud services for AI-powered %s.', 'intufind'),
                Constants::PLUGIN_NAME,
                $features
            ) . '</p>';
        }

        // Privacy notice
        echo '<div class="intucart-privacy-notice intucart-notice intucart-notice-info" style="margin-top: 15px;">';
        echo '<p><strong>' . esc_html__('🔒 Privacy-First AI Indexing', 'intufind') . '</strong> ';
        echo '<span style="cursor: pointer; text-decoration: underline;" data-popover="' . esc_attr($this->getPrivacyPopoverContent()) . '" title="' . esc_attr__('Click for privacy details', 'intufind') . '">' . esc_html__('What content is indexed?', 'intufind') . '</span>';
        echo '</p>';
        echo '</div>';

        echo '</div>';
    }

    /**
     * Get privacy popover content
     */
    private function getPrivacyPopoverContent(): string
    {
        $wooText = class_exists('WooCommerce') 
            ? ', ' . __('and product listings', 'intufind')
            : '';
        $wooDefaultText = class_exists('WooCommerce') 
            ? ', ' . __('Products, and Product Categories', 'intufind')
            : '';

        $approachTitle = '<strong>' . esc_html__('Privacy-First Approach:', 'intufind') . '</strong>';
        $approachText = sprintf(
            __('We take your privacy seriously. %s only indexes publicly available content from your website - such as published posts, pages%s. We never index private content, draft posts, user data, personal information, or any content marked as private.', 'intufind'),
            Constants::PLUGIN_NAME,
            $wooText
        );

        $selectionTitle = '<strong>' . esc_html__('Default Content Selection:', 'intufind') . '</strong>';
        $selectionText = sprintf(
            __('For your convenience, we\'ve pre-selected common content types (Posts, Pages, Categories, Tags%s) that most websites want indexed. You can customize these selections anytime in the Content Types section.', 'intufind'),
            $wooDefaultText
        );

        $controlTitle = '<strong>' . esc_html__('Complete Administrative Control:', 'intufind') . '</strong>';
        $controlText = __('You have full control over what is indexed via Content Selection settings. Enable or disable any content type, and completely remove indexed content when necessary. Your content, your control.', 'intufind');

        return $approachTitle . '<br/>' . esc_html($approachText) . '<br/><br/>' .
               $selectionTitle . '<br/>' . esc_html($selectionText) . '<br/><br/>' .
               $controlTitle . '<br/>' . esc_html($controlText);
    }

    /**
     * Render the API key form
     */
    private function renderApiKeyForm(string $api_key, array $usageSummary): void
    {
        echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" novalidate>';
        echo '<input type="hidden" name="action" value="intucart_api_key_form">';
        wp_nonce_field('intucart_api_key_form', 'intucart_api_key_nonce');

        $api_key_is_valid = $this->validateApiKey($api_key, $usageSummary);

        echo '<table class="form-table">';
        echo '<tr>';
        echo '<th scope="row">' . esc_html__('Secret Key', 'intufind') . '</th>';
        echo '<td>';

        $this->renderApiKeyInput($api_key, $api_key_is_valid);
        $this->renderApiKeyStatus($api_key, $api_key_is_valid);

        echo '</td>';
        echo '</tr>';
        echo '</table>';

        $this->renderActionButtons($api_key_is_valid);

        echo '</form>';
    }

    /**
     * Validate API key and return validation status
     */
    private function validateApiKey(string $api_key, array &$usageSummary): bool
    {
        if (empty($api_key)) {
            return false;
        }

        $hasApiKey = $this->aiClientManager->hasApiKey();

        if (!$hasApiKey) {
            return false;
        }

        $validation = $this->license->validateLicense();
        $api_key_is_valid = $validation['valid'] ?? false;
        $this->license->updateLicenseCache($validation);

        if (!$api_key_is_valid) {
            $this->usageService->invalidateUsageCache();
            $this->license->invalidateTierCache();
            $usageSummary = $this->usageService->getUsageSummary();
            if ($usageSummary['success'] && isset($usageSummary['license_valid'])) {
                $api_key_is_valid = $usageSummary['license_valid'];
            }
        }

        return $api_key_is_valid;
    }

    /**
     * Render the API key input field
     */
    private function renderApiKeyInput(string $api_key, bool $api_key_is_valid): void
    {
        $apiKeyDescription = '';
        if (!empty($api_key) && $api_key_is_valid) {
            $apiKeyDescription = '<em>' . esc_html__('To change your Secret Key, clear it first using the button below.', 'intufind') . '</em>';
        }

        Field::password('api_key', '', $api_key, [
            'id' => 'api_key_input',
            'description' => $apiKeyDescription,
            'wrapper_class' => 'no-margin',
            'attributes' => [
                'placeholder' => __('Enter your Secret Key (starts with if_sk_)...', 'intufind')
            ] + ($api_key_is_valid ? ['disabled' => 'disabled'] : [])
        ]);

        echo '<div style="margin-top: 10px;">';

        // Show status and auto-sync option
        if (!$api_key_is_valid) {
            Field::checkbox('auto_sync', __('Automatically index content after connecting', 'intufind'), 1, [
                'hint' => [[
                    'title' => __('Why Content Indexing?', 'intufind'),
                    'content' => sprintf(
                        __('To provide AI-powered search, chat, and recommendations, %s needs to understand your content. Indexing creates a searchable knowledge base of your public content, enabling features like:<br/><br/>• <strong>Intelligent Search:</strong> Find content by meaning, not just keywords<br/>• <strong>AI Chat:</strong> Answer visitor questions about your content<br/>• <strong>Smart Recommendations:</strong> Suggest relevant products or posts<br/><br/>Content is automatically synced to keep the AI index current when you publish, update, or delete content. Everything stays secure and under your control.', 'intufind'),
                        Constants::PLUGIN_NAME
                    )
                ]]
            ]);
        }

        echo '</div>';
    }

    /**
     * Render the API key status
     */
    private function renderApiKeyStatus(string $api_key, bool $api_key_is_valid): void
    {
        if (empty($api_key)) {
            return;
        }

        echo '<div style="display: flex; align-items: center; gap: 10px; margin-bottom: 8px;">';

        if ($api_key_is_valid) {
            $tierName = $this->license->getTierName();
            echo '<span style="color: #46b450; font-weight: bold;">✓ ' . esc_html__('Connected', 'intufind') . '</span>';
            if ($tierName) {
                echo '<span style="color: #666;">(' . esc_html($tierName) . ' ' . esc_html__('plan', 'intufind') . ')</span>';
            }
        } else {
            echo '<span style="color: #dc3232; font-weight: bold;">✗ ' . esc_html__('Invalid Secret Key', 'intufind') . '</span>';
        }

        echo '</div>';
    }

    /**
     * Render the action buttons
     */
    private function renderActionButtons(bool $api_key_is_valid): void
    {
        echo '<div class="submit" style="display: flex; gap: 10px; align-items: center;">';

        if ($api_key_is_valid) {
            Button::danger(__('Clear Secret Key', 'intufind'), [
                'type' => 'submit',
                'attributes' => [
                    'name' => 'api_key_action',
                    'value' => 'clear',
                    'onclick' => 'return confirm("' . esc_js(__('Are you sure you want to clear your Secret Key? This will disable all plugin features until you enter a new key.', 'intufind')) . '");'
                ]
            ]);
        } else {
            Button::primary(__('Save Secret Key', 'intufind'), [
                'type' => 'submit',
                'attributes' => [
                    'name' => 'api_key_action',
                    'value' => 'save'
                ]
            ]);
            
            // Show refresh button for users with invalid/expired key who might have just subscribed
            $api_key = get_option(Constants::API_KEY_OPTION, '');
            if (!empty($api_key)) {
                Button::secondary(__('Refresh Key Status', 'intufind'), [
                    'type' => 'submit',
                    'attributes' => [
                        'name' => 'api_key_action',
                        'value' => 'refresh',
                        'title' => __('Check if your subscription has been activated', 'intufind')
                    ]
                ]);
            }
        }

        echo '</div>';
    }

    /**
     * Render the help section for users without API keys
     */
    private function renderHelpSection(string $api_key): void
    {
        if (!empty($api_key)) {
            return;
        }

        echo '<br>';
        echo '<div class="intucart-content-panel" style="background: linear-gradient(135deg, #f0f7ff 0%, #e8f4f8 100%); padding: 24px; border: 1px solid #c3e0f0; border-radius: 8px;">';
        echo '<div style="display: flex; align-items: center; gap: 8px; margin-bottom: 12px;">';
        echo '<span style="font-size: 24px;">🚀</span>';
        echo '<h3 style="margin: 0; color: #1e3a5f;">' . esc_html__('Get Started Free', 'intufind') . '</h3>';
        echo '</div>';
        echo '<p style="margin: 0 0 16px 0; color: #2c5282; font-weight: 500;">' . esc_html__('14-day free trial • No credit card required', 'intufind') . '</p>';
        echo '<ol style="margin-bottom: 16px; padding-left: 20px;">';
        echo '<li style="margin-bottom: 8px;">' . sprintf(
            __('%s on %s', 'intufind'),
            '<a href="' . esc_url(Constants::WEBSITE_URL . 'register') . '" target="_blank" style="color: #2b6cb0; font-weight: 500;">' . esc_html__('Create a free account', 'intufind') . '</a>',
            Constants::PLUGIN_SHORT_NAME
        ) . '</li>';
        echo '<li style="margin-bottom: 8px;">' . esc_html__('Your Secret Key is shown immediately after signup', 'intufind') . '</li>';
        echo '<li>' . esc_html__('Paste your Secret Key above to activate AI features', 'intufind') . '</li>';
        echo '</ol>';
        echo '<p style="margin: 0; font-size: 13px; color: #4a5568;">' . esc_html__('Full access to all features during your trial. Upgrade anytime to continue after 14 days.', 'intufind') . '</p>';
        echo '</div>';
    }
}

