<?php

declare(strict_types=1);

namespace Intucart\Admin\Tabs;

use Intucart\Services\Logger;
use Intucart\Services\Constants;
use Intucart\Components\Button;
use Intucart\Components\Tabs;
use Intucart\Pages\Settings\SearchSettings;
use Intucart\Pages\Settings\SearchWidgetSettings;

/**
 * Renders the Search tab in admin settings with subtabs for Settings and Widget
 */
class SearchTabRenderer extends AbstractTabRenderer
{
    private SearchSettings $searchSettings;
    private SearchWidgetSettings $searchWidgetSettings;

    public function __construct(
        Logger $logger,
        SearchSettings $searchSettings,
        SearchWidgetSettings $searchWidgetSettings
    ) {
        parent::__construct($logger);
        $this->searchSettings = $searchSettings;
        $this->searchWidgetSettings = $searchWidgetSettings;
    }

    public function getId(): string
    {
        return 'search';
    }

    public function getTabConfig(): array
    {
        return [
            'url' => '?page=intucart&tab=search',
            'label' => __('Search', 'intufind'),
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path opacity=".4" d="M208 80a128 128 0 1 1-90.51 37.49A127.15 127.15 0 0 1 208 80m0-80C93.12 0 0 93.12 0 208s93.12 208 208 208 208-93.12 208-208S322.88 0 208 0z"/><path d="M504.9 476.7L476.6 505a23.9 23.9 0 0 1-33.9 0L343 405.3a24 24 0 0 1-7-17V372l36-36h16.3a24 24 0 0 1 17 7l99.7 99.7a24.11 24.11 0 0 1-.1 34z"/></svg>'
        ];
    }

    public function render(): void
    {
        $this->renderHeader(
            __('AI Search', 'intufind'),
            __('Configure AI-powered search behavior and the search widget for your website visitors.', 'intufind')
        );

        // Get current subtab
        $subtab = $this->getCurrentSubtab();

        // Render subtab navigation
        $this->renderSubtabs($subtab);

        // Render subtab content
        $this->renderSubtabContent($subtab);
    }

    /**
     * Get current subtab with validation.
     */
    private function getCurrentSubtab(): string
    {
        $subtab = isset($_GET['search_subtab']) ? sanitize_text_field($_GET['search_subtab']) : 'settings';

        $availableSubtabs = ['settings', 'widget'];

        if (!in_array($subtab, $availableSubtabs, true)) {
            return 'settings';
        }

        return $subtab;
    }

    /**
     * Render subtab navigation.
     */
    private function renderSubtabs(string $currentSubtab): void
    {
        $base = admin_url('admin.php?page=intucart&tab=search');

        $innerTabs = [
            'settings' => [
                'url' => $base . '&search_subtab=settings',
                'label' => __('Settings', 'intufind')
            ],
            'widget' => [
                'url' => $base . '&search_subtab=widget',
                'label' => __('Widget', 'intufind')
            ]
        ];

        Tabs::renderInner($innerTabs, $currentSubtab);

        // Show success message if settings were saved
        if (isset($_GET['settings-updated']) && $_GET['settings-updated']) {
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Search settings saved successfully!', 'intufind') . '</p></div>';
        }
    }

    /**
     * Render subtab content.
     */
    private function renderSubtabContent(string $subtab): void
    {
        if ($subtab === 'settings') {
            $this->renderSearchSettings();
        } else {
            $this->renderWidgetSettings();
        }
    }

    /**
     * Render search settings subtab (server-side search configuration).
     */
    private function renderSearchSettings(): void
    {
        echo '<form action="options.php" method="post" novalidate>';
        settings_fields(Constants::SETTINGS_GROUP_SEARCH);

        $this->renderPanel(
            __('Search Configuration', 'intufind'),
            __('Customize how AI search behaves on your site, including result ranking, filtering options, and performance optimizations.', 'intufind'),
            function () {
                global $wp_settings_sections;
                if (isset($wp_settings_sections[Constants::SETTINGS_GROUP_SEARCH]['intucart_search_section'])) {
                    echo '<table class="form-table">';
                    do_settings_fields(Constants::SETTINGS_GROUP_SEARCH, 'intucart_search_section');
                    echo '</table>';
                }
            }
        );

        Button::primary(__('Save Search Settings', 'intufind'), ['type' => 'submit']);
        echo '</form>';
    }

    /**
     * Render widget settings subtab (search widget configuration).
     */
    private function renderWidgetSettings(): void
    {
        // Check if publishable key is configured
        $publishable_key = get_option(Constants::PUBLISHABLE_KEY_OPTION, '');
        
        if (empty($publishable_key)) {
            $this->renderApiKeyRequired();
            return;
        }

        echo '<form action="options.php" method="post" novalidate>';
        settings_fields(Constants::SETTINGS_GROUP_SEARCH_WIDGET);

        $this->renderPanel(
            __('Search Widget Configuration', 'intufind'),
            __('Configure the instant search overlay widget that visitors can access via keyboard shortcut or trigger button.', 'intufind'),
            function () {
                $this->searchWidgetSettings->renderWidgetSettings();
            }
        );

        Button::primary(__('Save Widget Settings', 'intufind'), ['type' => 'submit']);
        echo '</form>';
    }

    /**
     * Render API key required message.
     */
    private function renderApiKeyRequired(): void
    {
        echo '<div class="intucart-content-panel">';
        echo '<div class="intucart-notice intucart-notice-warning" style="margin: 0;">';
        echo '<p><strong>' . esc_html__('⚠️ API Key Required', 'intufind') . '</strong></p>';
        echo '<p>' . esc_html__('The search widget requires a valid API key and publishable key to communicate with the Intufind cloud service.', 'intufind') . '</p>';

        echo '<p style="margin-top: 12px;">';
        Button::primary(__('Configure API Key →', 'intufind'), [
            'href' => admin_url('admin.php?page=intucart&tab=license')
        ]);
        echo '</p>';

        echo '</div>';
        echo '</div>';
    }
}
