<?php

declare(strict_types=1);

namespace Intucart\Admin\Tabs;

use Intucart\Services\Logger;
use Intucart\Services\Constants;
use Intucart\Components\Button;

/**
 * Renders the WooCommerce tab in admin settings
 */
class WooCommerceTabRenderer extends AbstractTabRenderer
{
    private bool $isAvailable;

    public function __construct(Logger $logger, bool $recommendationsAvailable)
    {
        parent::__construct($logger);
        $this->isAvailable = class_exists('WooCommerce') && $recommendationsAvailable;
    }

    public function getId(): string
    {
        return 'woocommerce';
    }

    public function getTabConfig(): array
    {
        return [
            'url' => '?page=intucart&tab=woocommerce',
            'label' => __('WooCommerce', 'intufind'),
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 576 512"><path opacity=".4" d="M552 64H159.21l52.36 256h293.15a24 24 0 0 0 23.4-18.68l47.27-208a24 24 0 0 0-18.08-28.72A23.69 23.69 0 0 0 552 64z"/><path d="M218.12 352h268.42a24 24 0 0 1 23.4 29.32l-5.52 24.28a56 56 0 1 1-63.6 10.4H231.18a56 56 0 1 1-67.05-8.57L93.88 64H24A24 24 0 0 1 0 40V24A24 24 0 0 1 24 0h102.53A24 24 0 0 1 150 19.19z"/></svg>',
            'condition' => $this->isAvailable
        ];
    }

    public function isAvailable(): bool
    {
        return $this->isAvailable;
    }

    public function render(): void
    {
        $this->renderHeader(
            __('Product Recommendations', 'intufind'),
            __('Configure AI-powered product recommendations to enhance your WooCommerce store with intelligent product suggestions and personalized shopping experiences.', 'intufind')
        );

        echo '<form action="options.php" method="post" novalidate>';
        settings_fields(Constants::SETTINGS_GROUP_WOOCOMMERCE);

        if (class_exists('WooCommerce')) {
            $this->renderPanel(
                __('Recommendation Settings', 'intufind'),
                __('Customize how AI-powered product recommendations appear and behave in your WooCommerce store.', 'intufind'),
                function () {
                    global $wp_settings_sections;
                    if (isset($wp_settings_sections[Constants::SETTINGS_GROUP_WOOCOMMERCE]['intucart_woocommerce_section'])) {
                        echo '<table class="form-table">';
                        do_settings_fields(Constants::SETTINGS_GROUP_WOOCOMMERCE, 'intucart_woocommerce_section');
                        echo '</table>';
                    }
                }
            );

            Button::primary(__('Save Recommendation Settings', 'intufind'), ['type' => 'submit']);
        } else {
            echo '<div class="intucart-content-panel" style="background: #fff3cd; border: 1px solid #ffeaa7;">';
            echo '<h3 class="intucart-panel-header" style="color: #856404;">' . esc_html__('WooCommerce Required', 'intufind') . '</h3>';
            echo '<p class="intucart-panel-description" style="color: #856404;">' . esc_html__('WooCommerce is not currently active on your site. Please install and activate WooCommerce to access AI-powered product recommendation features.', 'intufind') . '</p>';
            echo '</div>';
        }

        echo '</form>';
    }
}

