<?php

namespace Intucart\Base;

use Intucart\Services\Constants;
use Intucart\Services\AIClientManager;
use Intucart\Services\AssetOptimizationService;
use Intucart\Services\Cache\CacheService;
use Intucart\Services\Licensing\License;

/**
 * Enqueue class
 *
 * @category Base
 * @package  Intucart
 * @author   Intucart <support@intufind.com>
 * @license  GPL-2.0+ https://www.gnu.org/licenses/gpl-2.0.html
 * @link     https://intufind.com
 */
class Enqueue
{
    private AIClientManager $aiClientManager;
    private CacheService $cache;
    private License $license;
    private AssetOptimizationService $assetOptimization;
    
    /** @var array|null Cached widget settings to avoid repeated get_option calls */
    private ?array $cachedSettings = null;
    
    private const SETTINGS_CACHE_KEY = 'widget_settings';
    private const SETTINGS_CACHE_GROUP = 'intucart_enqueue';
    private const SETTINGS_CACHE_TTL = 300; // 5 minutes

    public function __construct(
        AIClientManager $aiClientManager, 
        CacheService $cache, 
        License $license, 
        ?AssetOptimizationService $assetOptimization = null
    )
    {
        $this->aiClientManager = $aiClientManager;
        $this->cache = $cache;
        $this->license = $license;
        $this->assetOptimization = $assetOptimization ?: new AssetOptimizationService();
    }
    
    /**
     * Get all widget settings from cache or database
     * Reduces ~80 get_option() calls to a single cached lookup
     *
     * @return array All widget settings
     */
    private function getWidgetSettings(): array
    {
        // Check request-level cache first
        if ($this->cachedSettings !== null) {
            return $this->cachedSettings;
        }
        
        // Check persistent cache
        $cached = $this->cache->get(self::SETTINGS_CACHE_KEY, self::SETTINGS_CACHE_GROUP);
        if ($cached !== false && is_array($cached)) {
            $this->cachedSettings = $cached;
            return $cached;
        }
        
        // Build settings from individual options
        $settings = $this->buildAllSettings();
        
        // Cache for subsequent requests
        $this->cache->set(self::SETTINGS_CACHE_KEY, $settings, self::SETTINGS_CACHE_GROUP, self::SETTINGS_CACHE_TTL);
        $this->cachedSettings = $settings;
        
        return $settings;
    }
    
    /**
     * Build all widget settings from individual options
     * This is only called on cache miss
     *
     * @return array All settings keyed by option name
     */
    private function buildAllSettings(): array
    {
        // Define all options with their defaults in a single array
        // This makes it easy to add/remove options and see all settings in one place
        $optionDefaults = [
            // Core settings
            Constants::OPTION_ENABLE_CHAT_ASSISTANT => '1',
            Constants::PUBLISHABLE_KEY_OPTION => '',
            
            // Trigger button
            Constants::OPTION_TOGGLE_ICON_ID => 0,
            Constants::OPTION_CHAT_TRIGGER_BUTTON_SHOW_BACKGROUND => true,
            Constants::OPTION_CHAT_TRIGGER_ICON_SHOW_BORDER => true,
            
            // Header
            Constants::OPTION_CHAT_HEADER_ICON_ID => 0,
            Constants::OPTION_CHAT_HEADER_TEXT => Constants::ASSISTANT_DEFAULT_CHAT_HEADER_TEXT,
            Constants::OPTION_CHAT_HEADER_SUBTITLE => '',
            Constants::OPTION_CHAT_HEADER_AVATAR_SHOW_BACKGROUND => true,
            Constants::OPTION_CHAT_HEADER_AVATAR_SHOW_BORDER => true,
            
            // Greeting/Empty state
            Constants::OPTION_CHAT_GREETING_ICON_ID => 0,
            Constants::OPTION_ASSISTANT_WELCOME_MESSAGE => Constants::ASSISTANT_DEFAULT_WELCOME_MESSAGE,
            Constants::OPTION_CHAT_GREETING_SUBTEXT => '',
            
            // Widget behavior
            Constants::OPTION_CHAT_WIDGET_POSITION => 'bottom-right',
            Constants::OPTION_CHAT_CARD_LAYOUT => 'carousel',
            Constants::OPTION_CHAT_PROMPT_STYLE => 'footer',
            Constants::OPTION_CHAT_CUSTOM_CSS => '',
            
            // Message actions
            Constants::OPTION_CHAT_SHOW_MESSAGE_ACTIONS => false,
            Constants::OPTION_CHAT_SHOW_USER_COPY_BUTTON => false,
            Constants::OPTION_CHAT_SHOW_MESSAGE_TIMESTAMPS => false,
            
            // Post card display
            Constants::OPTION_CHAT_SHOW_POST_IMAGE => false,
            Constants::OPTION_CHAT_SHOW_POST_AUTHOR => true,
            Constants::OPTION_CHAT_SHOW_POST_DATE => true,
            Constants::OPTION_CHAT_SHOW_POST_CATEGORY => true,
            
            // Live agent
            Constants::OPTION_LIVE_AGENT_ENABLED => false,
            Constants::OPTION_LIVE_AGENT_CONFIGURED => false,
            Constants::OPTION_LIVE_AGENT_MANUAL_HANDOFF => true,
            Constants::OPTION_LIVE_AGENT_AUTO_HANDOFF => true,
            Constants::OPTION_LIVE_AGENT_PROVIDER => Constants::LIVE_AGENT_PROVIDER_NONE,
            Constants::OPTION_LIVE_AGENT_BUSINESS_HOURS => '',
            Constants::OPTION_LIVE_AGENT_INACTIVITY_TIMEOUT => Constants::DEFAULT_LIVE_AGENT_INACTIVITY_TIMEOUT,
            Constants::OPTION_LIVE_AGENT_ESCALATION_PROMPT => '',
            Constants::OPTION_CONTACT_FORM_ENABLED => false,
            
            // Theme - Core
            Constants::OPTION_CHAT_COLOR_SCHEME => 'auto',
            Constants::OPTION_CHAT_PRIMARY_COLOR => '#0084ff',
            Constants::OPTION_CHAT_BORDER_RADIUS => 16,
            Constants::OPTION_CHAT_WIDGET_MAX_WIDTH => 800,
            Constants::OPTION_CHAT_WIDGET_MAX_HEIGHT => 900,
            
            // Theme - Colors
            Constants::OPTION_CHAT_LINK_COLOR => '#0084ff',
            Constants::OPTION_CHAT_WIDGET_BACKGROUND_CUSTOM => '#ffffff',
            Constants::OPTION_CHAT_INPUT_BACKGROUND => '#f8f9fa',
            Constants::OPTION_CHAT_ASSISTANT_BUBBLE_BG => '#f1f3f4',
            Constants::OPTION_CHAT_ACCENT_COLOR => '#dadce0',
            
            // Theme - Header
            Constants::OPTION_CHAT_HEADER_BACKGROUND => '#ffffff',
            Constants::OPTION_CHAT_HEADER_TITLE_COLOR => '#0f172a',
            Constants::OPTION_CHAT_HEADER_SUBTITLE_COLOR => '#64748b',
            Constants::OPTION_CHAT_HEADER_ICON_COLOR => '#64748b',
            
            // Theme - Trigger button
            Constants::OPTION_CHAT_TRIGGER_BUTTON_COLOR => '#0084ff',
            Constants::OPTION_CHAT_TRIGGER_BUTTON_HOVER_COLOR => '#0073e6',
            Constants::OPTION_CHAT_TRIGGER_BUTTON_BORDER_COLOR => '#dadce0',
            
            // Theme - Reference cards
            Constants::OPTION_CHAT_CARD_BACKGROUND => '#ffffff',
            Constants::OPTION_CHAT_CARD_BORDER => '#e1e5e9',
            Constants::OPTION_CHAT_CARD_HOVER_BG => '#f8f9fa',
            Constants::OPTION_CHAT_CARD_BORDER_RADIUS => 8,
            Constants::OPTION_CHAT_CARD_TITLE_COLOR => '#1a1a1a',
            Constants::OPTION_CHAT_CARD_TITLE_SIZE => 16,
            Constants::OPTION_CHAT_CARD_TITLE_WEIGHT => '600',
            Constants::OPTION_CHAT_CARD_META_COLOR => '#6c757d',
            Constants::OPTION_CHAT_CARD_DESC_COLOR => '#495057',
            Constants::OPTION_CHAT_CARD_DESC_LENGTH => 150,
            
            // Theme - Prompt chips
            Constants::OPTION_CHAT_PROMPT_CHIP_BACKGROUND => '#f8f9fa',
            Constants::OPTION_CHAT_PROMPT_CHIP_TEXT_COLOR => '#202124',
            Constants::OPTION_CHAT_PROMPT_CHIP_BORDER => '#d1d5db',
            Constants::OPTION_CHAT_PROMPT_CHIP_HOVER_BG => '#f1f3f4',
            
            // Theme - Dark mode
            Constants::OPTION_CHAT_DARK_PRIMARY_COLOR => '#0084ff',
            Constants::OPTION_CHAT_DARK_LINK_COLOR => '#6495ed',
            Constants::OPTION_CHAT_DARK_WIDGET_BACKGROUND_CUSTOM => '#1f1f1f',
            Constants::OPTION_CHAT_DARK_INPUT_BACKGROUND => '#2d2d2d',
            Constants::OPTION_CHAT_DARK_ASSISTANT_BUBBLE_BG => '#2d2d2d',
            Constants::OPTION_CHAT_DARK_ACCENT_COLOR => '#404040',
            Constants::OPTION_CHAT_DARK_HEADER_BACKGROUND => '#1f1f1f',
            Constants::OPTION_CHAT_DARK_HEADER_TITLE_COLOR => '#e8eaed',
            Constants::OPTION_CHAT_DARK_HEADER_SUBTITLE_COLOR => '#9aa0a6',
            Constants::OPTION_CHAT_DARK_HEADER_ICON_COLOR => '#9aa0a6',
            Constants::OPTION_CHAT_DARK_TRIGGER_BUTTON_COLOR => '#0084ff',
            Constants::OPTION_CHAT_DARK_TRIGGER_BUTTON_HOVER_COLOR => '#0073e6',
            Constants::OPTION_CHAT_DARK_TRIGGER_BUTTON_BORDER_COLOR => '#404040',
            Constants::OPTION_CHAT_DARK_CARD_BACKGROUND => '#2d2d2d',
            Constants::OPTION_CHAT_DARK_CARD_BORDER => '#404040',
            Constants::OPTION_CHAT_DARK_CARD_HOVER_BG => '#383838',
            Constants::OPTION_CHAT_DARK_CARD_TITLE_COLOR => '#f5f5f5',
            Constants::OPTION_CHAT_DARK_CARD_META_COLOR => '#a0a0a0',
            Constants::OPTION_CHAT_DARK_CARD_DESC_COLOR => '#d0d0d0',
            Constants::OPTION_CHAT_DARK_PROMPT_CHIP_BACKGROUND => '#2d2d2d',
            Constants::OPTION_CHAT_DARK_PROMPT_CHIP_TEXT_COLOR => '#ffffff',
            Constants::OPTION_CHAT_DARK_PROMPT_CHIP_BORDER => '#404040',
            Constants::OPTION_CHAT_DARK_PROMPT_CHIP_HOVER_BG => '#383838',
            
            // Search widget
            Constants::OPTION_SEARCH_WIDGET_ENABLED => '0',
            Constants::OPTION_SEARCH_WIDGET_PRODUCTS => '1',
            Constants::OPTION_SEARCH_WIDGET_POSTS => '1',
            Constants::OPTION_SEARCH_WIDGET_KEYBOARD_SHORTCUT => Constants::DEFAULT_SEARCH_WIDGET_KEYBOARD_SHORTCUT,
            Constants::OPTION_SEARCH_WIDGET_SHOW_TRIGGER => '1',
            Constants::OPTION_SEARCH_WIDGET_TRIGGER_POSITION => Constants::DEFAULT_SEARCH_WIDGET_TRIGGER_POSITION,
            Constants::OPTION_SEARCH_WIDGET_PRIMARY_COLOR => '#0084ff',
            Constants::OPTION_SEARCH_WIDGET_DARK_PRIMARY_COLOR => '#0084ff',
            Constants::OPTION_SEARCH_WIDGET_COLOR_SCHEME => 'auto',
            Constants::OPTION_SEARCH_WIDGET_TRIGGER_BG => Constants::DEFAULT_SEARCH_WIDGET_TRIGGER_BG,
            Constants::OPTION_SEARCH_WIDGET_TRIGGER_TEXT => Constants::DEFAULT_SEARCH_WIDGET_TRIGGER_TEXT,
            Constants::OPTION_SEARCH_WIDGET_TRIGGER_BORDER => Constants::DEFAULT_SEARCH_WIDGET_TRIGGER_BORDER,
            Constants::OPTION_SEARCH_WIDGET_DARK_TRIGGER_BG => Constants::DEFAULT_SEARCH_WIDGET_DARK_TRIGGER_BG,
            Constants::OPTION_SEARCH_WIDGET_DARK_TRIGGER_TEXT => Constants::DEFAULT_SEARCH_WIDGET_DARK_TRIGGER_TEXT,
            Constants::OPTION_SEARCH_WIDGET_DARK_TRIGGER_BORDER => Constants::DEFAULT_SEARCH_WIDGET_DARK_TRIGGER_BORDER,
            Constants::OPTION_SEARCH_WIDGET_TRIGGER_ICON_ID => 0,
        ];
        
        // Fetch all options in one go
        $settings = [];
        foreach ($optionDefaults as $option => $default) {
            $settings[$option] = get_option($option, $default);
        }
        
        return $settings;
    }
    
    /**
     * Get a single setting value from the cached settings
     *
     * @param string $key Option constant
     * @param mixed $default Default value (only used if not in cache)
     * @return mixed Setting value
     */
    private function getSetting(string $key, $default = null)
    {
        $settings = $this->getWidgetSettings();
        return $settings[$key] ?? $default;
    }
    
    /**
     * Clear the widget settings cache
     * Should be called when settings are updated in admin
     *
     * @return void
     */
    public static function clearSettingsCache(): void
    {
        // Delete transient using the correct key format: {group}_{key}
        delete_transient(self::SETTINGS_CACHE_GROUP . '_' . self::SETTINGS_CACHE_KEY);
        
        // Also clear object cache if using external cache
        if (wp_using_ext_object_cache()) {
            wp_cache_delete(self::SETTINGS_CACHE_KEY, self::SETTINGS_CACHE_GROUP);
        }
    }
    
    /**
     * Get the list of all option keys that affect the widget settings cache
     * Used for registering cache invalidation hooks
     *
     * @return array List of option names
     */
    public static function getCacheableOptionKeys(): array
    {
        return [
            // Core settings
            Constants::OPTION_ENABLE_CHAT_ASSISTANT,
            Constants::PUBLISHABLE_KEY_OPTION,
            
            // Trigger button
            Constants::OPTION_TOGGLE_ICON_ID,
            Constants::OPTION_CHAT_TRIGGER_BUTTON_SHOW_BACKGROUND,
            Constants::OPTION_CHAT_TRIGGER_ICON_SHOW_BORDER,
            
            // Header
            Constants::OPTION_CHAT_HEADER_ICON_ID,
            Constants::OPTION_CHAT_HEADER_TEXT,
            Constants::OPTION_CHAT_HEADER_SUBTITLE,
            Constants::OPTION_CHAT_HEADER_AVATAR_SHOW_BACKGROUND,
            Constants::OPTION_CHAT_HEADER_AVATAR_SHOW_BORDER,
            
            // Greeting/Empty state
            Constants::OPTION_CHAT_GREETING_ICON_ID,
            Constants::OPTION_ASSISTANT_WELCOME_MESSAGE,
            Constants::OPTION_CHAT_GREETING_SUBTEXT,
            
            // Widget behavior
            Constants::OPTION_CHAT_WIDGET_POSITION,
            Constants::OPTION_CHAT_CARD_LAYOUT,
            Constants::OPTION_CHAT_PROMPT_STYLE,
            Constants::OPTION_CHAT_CUSTOM_CSS,
            
            // Message actions
            Constants::OPTION_CHAT_SHOW_MESSAGE_ACTIONS,
            Constants::OPTION_CHAT_SHOW_USER_COPY_BUTTON,
            Constants::OPTION_CHAT_SHOW_MESSAGE_TIMESTAMPS,
            
            // Post card display
            Constants::OPTION_CHAT_SHOW_POST_IMAGE,
            Constants::OPTION_CHAT_SHOW_POST_AUTHOR,
            Constants::OPTION_CHAT_SHOW_POST_DATE,
            Constants::OPTION_CHAT_SHOW_POST_CATEGORY,
            
            // Live agent
            Constants::OPTION_LIVE_AGENT_ENABLED,
            Constants::OPTION_LIVE_AGENT_CONFIGURED,
            Constants::OPTION_LIVE_AGENT_MANUAL_HANDOFF,
            Constants::OPTION_LIVE_AGENT_AUTO_HANDOFF,
            Constants::OPTION_LIVE_AGENT_PROVIDER,
            Constants::OPTION_LIVE_AGENT_BUSINESS_HOURS,
            Constants::OPTION_LIVE_AGENT_INACTIVITY_TIMEOUT,
            Constants::OPTION_LIVE_AGENT_ESCALATION_PROMPT,
            Constants::OPTION_CONTACT_FORM_ENABLED,
            
            // Theme - Core
            Constants::OPTION_CHAT_COLOR_SCHEME,
            Constants::OPTION_CHAT_PRIMARY_COLOR,
            Constants::OPTION_CHAT_BORDER_RADIUS,
            Constants::OPTION_CHAT_WIDGET_MAX_WIDTH,
            Constants::OPTION_CHAT_WIDGET_MAX_HEIGHT,
            
            // Theme - Colors
            Constants::OPTION_CHAT_LINK_COLOR,
            Constants::OPTION_CHAT_WIDGET_BACKGROUND_CUSTOM,
            Constants::OPTION_CHAT_INPUT_BACKGROUND,
            Constants::OPTION_CHAT_ASSISTANT_BUBBLE_BG,
            Constants::OPTION_CHAT_ACCENT_COLOR,
            
            // Theme - Header
            Constants::OPTION_CHAT_HEADER_BACKGROUND,
            Constants::OPTION_CHAT_HEADER_TITLE_COLOR,
            Constants::OPTION_CHAT_HEADER_SUBTITLE_COLOR,
            Constants::OPTION_CHAT_HEADER_ICON_COLOR,
            
            // Theme - Trigger button
            Constants::OPTION_CHAT_TRIGGER_BUTTON_COLOR,
            Constants::OPTION_CHAT_TRIGGER_BUTTON_HOVER_COLOR,
            Constants::OPTION_CHAT_TRIGGER_BUTTON_BORDER_COLOR,
            
            // Theme - Reference cards
            Constants::OPTION_CHAT_CARD_BACKGROUND,
            Constants::OPTION_CHAT_CARD_BORDER,
            Constants::OPTION_CHAT_CARD_HOVER_BG,
            Constants::OPTION_CHAT_CARD_BORDER_RADIUS,
            Constants::OPTION_CHAT_CARD_TITLE_COLOR,
            Constants::OPTION_CHAT_CARD_TITLE_SIZE,
            Constants::OPTION_CHAT_CARD_TITLE_WEIGHT,
            Constants::OPTION_CHAT_CARD_META_COLOR,
            Constants::OPTION_CHAT_CARD_DESC_COLOR,
            Constants::OPTION_CHAT_CARD_DESC_LENGTH,
            
            // Theme - Prompt chips
            Constants::OPTION_CHAT_PROMPT_CHIP_BACKGROUND,
            Constants::OPTION_CHAT_PROMPT_CHIP_TEXT_COLOR,
            Constants::OPTION_CHAT_PROMPT_CHIP_BORDER,
            Constants::OPTION_CHAT_PROMPT_CHIP_HOVER_BG,
            
            // Theme - Dark mode
            Constants::OPTION_CHAT_DARK_PRIMARY_COLOR,
            Constants::OPTION_CHAT_DARK_LINK_COLOR,
            Constants::OPTION_CHAT_DARK_WIDGET_BACKGROUND_CUSTOM,
            Constants::OPTION_CHAT_DARK_INPUT_BACKGROUND,
            Constants::OPTION_CHAT_DARK_ASSISTANT_BUBBLE_BG,
            Constants::OPTION_CHAT_DARK_ACCENT_COLOR,
            Constants::OPTION_CHAT_DARK_HEADER_BACKGROUND,
            Constants::OPTION_CHAT_DARK_HEADER_TITLE_COLOR,
            Constants::OPTION_CHAT_DARK_HEADER_SUBTITLE_COLOR,
            Constants::OPTION_CHAT_DARK_HEADER_ICON_COLOR,
            Constants::OPTION_CHAT_DARK_TRIGGER_BUTTON_COLOR,
            Constants::OPTION_CHAT_DARK_TRIGGER_BUTTON_HOVER_COLOR,
            Constants::OPTION_CHAT_DARK_TRIGGER_BUTTON_BORDER_COLOR,
            Constants::OPTION_CHAT_DARK_CARD_BACKGROUND,
            Constants::OPTION_CHAT_DARK_CARD_BORDER,
            Constants::OPTION_CHAT_DARK_CARD_HOVER_BG,
            Constants::OPTION_CHAT_DARK_CARD_TITLE_COLOR,
            Constants::OPTION_CHAT_DARK_CARD_META_COLOR,
            Constants::OPTION_CHAT_DARK_CARD_DESC_COLOR,
            Constants::OPTION_CHAT_DARK_PROMPT_CHIP_BACKGROUND,
            Constants::OPTION_CHAT_DARK_PROMPT_CHIP_TEXT_COLOR,
            Constants::OPTION_CHAT_DARK_PROMPT_CHIP_BORDER,
            Constants::OPTION_CHAT_DARK_PROMPT_CHIP_HOVER_BG,
            
            // Search widget
            Constants::OPTION_SEARCH_WIDGET_ENABLED,
            Constants::OPTION_SEARCH_WIDGET_PRODUCTS,
            Constants::OPTION_SEARCH_WIDGET_POSTS,
            Constants::OPTION_SEARCH_WIDGET_KEYBOARD_SHORTCUT,
            Constants::OPTION_SEARCH_WIDGET_SHOW_TRIGGER,
            Constants::OPTION_SEARCH_WIDGET_TRIGGER_POSITION,
            Constants::OPTION_SEARCH_WIDGET_PRIMARY_COLOR,
            Constants::OPTION_SEARCH_WIDGET_DARK_PRIMARY_COLOR,
            Constants::OPTION_SEARCH_WIDGET_COLOR_SCHEME,
            Constants::OPTION_SEARCH_WIDGET_TRIGGER_BG,
            Constants::OPTION_SEARCH_WIDGET_TRIGGER_TEXT,
            Constants::OPTION_SEARCH_WIDGET_TRIGGER_BORDER,
            Constants::OPTION_SEARCH_WIDGET_DARK_TRIGGER_BG,
            Constants::OPTION_SEARCH_WIDGET_DARK_TRIGGER_TEXT,
            Constants::OPTION_SEARCH_WIDGET_DARK_TRIGGER_BORDER,
            Constants::OPTION_SEARCH_WIDGET_TRIGGER_ICON_ID,
        ];
    }
    
    /**
     * Register hooks to invalidate settings cache when options are updated
     * Should be called during admin init
     *
     * @return void
     */
    public static function registerCacheInvalidationHooks(): void
    {
        foreach (self::getCacheableOptionKeys() as $optionKey) {
            add_action("update_option_{$optionKey}", [self::class, 'clearSettingsCache']);
            add_action("add_option_{$optionKey}", [self::class, 'clearSettingsCache']);
        }
    }
    /**
     * Register scripts and styles
     *
     * @return void
     */
    public function register(): void
    {
        // Priority 999 to load after Elementor and other page builders
        add_action('wp_enqueue_scripts', array($this, 'enqueueFrontend'), 999);
        
        // Register asset optimization exclusions (prevents WP Rocket, Autoptimize, etc. from breaking chatbot)
        $this->assetOptimization->register();
        
        // Log active optimization plugins in debug mode
        if (defined('WP_DEBUG') && WP_DEBUG) {
            $this->assetOptimization->logActivePlugins();
        }
    }

    /**
     * Check if chatbot is available (publishable key configured AND license valid)
     *
     * @return bool True if publishable key is configured and license is valid
     */
    private function isChatbotAvailable(): bool
    {
        $publishable_key = $this->getSetting(Constants::PUBLISHABLE_KEY_OPTION, '');
        if (empty($publishable_key)) {
            return false;
        }

        // Check if license is valid (uses cached result for performance)
        // This prevents widgets from loading when trial has expired
        return $this->license->isValid();
    }

    /**
     * Enqueue frontend scripts and styles
     *
     * @return void
     */
    public function enqueueFrontend(): void
    {
        $frontend_css = \INTUCART_PLUGIN_DIR . 'assets/css/frontend.css';
        $frontend_js = \INTUCART_PLUGIN_DIR . 'assets/js/frontend.js';

        if (file_exists($frontend_css)) {
            wp_enqueue_style(
                'intucart-frontend',
                \INTUCART_PLUGIN_URL . 'assets/css/frontend.css',
                array(),
                \INTUCART_VERSION
            );
        }

        if (file_exists($frontend_js)) {
            wp_enqueue_script(
                'intucart-frontend',
                \INTUCART_PLUGIN_URL . 'assets/js/frontend.js',
                array('jquery'),
                \INTUCART_VERSION,
                true
            );
        }

        // Check if the chat assistant is enabled and license is valid before loading its assets
        // Also ensure we're not in the WordPress admin area
        $enable_chat = $this->getSetting(Constants::OPTION_ENABLE_CHAT_ASSISTANT, '1');
        $is_chatbot_available = $this->isChatbotAvailable();
        $is_not_admin = !is_admin();

        if ('1' === $enable_chat && $is_chatbot_available && $is_not_admin) {
            // Chat widget base URL - override via INTUFIND_CHAT_WIDGET_URL in wp-config.php
            // For local dev: define('INTUFIND_CHAT_WIDGET_URL', 'http://localhost:4001');
            $chat_base_url = defined('INTUFIND_CHAT_WIDGET_URL')
                ? rtrim(\INTUFIND_CHAT_WIDGET_URL, '/')
                : 'https://cdn.intufind.com';

            $loader_url = $chat_base_url . '/chat-loader.js';

            wp_enqueue_script(
                'intucart-chatbot-loader',
                $loader_url,
                [],
                null,
                true
            );

            $chat_script_handle = 'intucart-chatbot-loader';

            // --- Prepare Page Context --- (Moved from Shortcode)
            $page_context = [
                'currentPageUrl' => is_singular() ? get_permalink() : home_url(add_query_arg(null, null)),
                'currentPageTitle' => is_singular() ? get_the_title() : wp_get_document_title(),
                'currentProductId' => null,
            ];

            // Check if we are on a single product page (ensure WC functions exist)
            if (function_exists('is_product') && \is_product()) {
                global $product;
                if ($product instanceof \WC_Product) {
                    $page_context['currentProductId'] = $product->get_id();
                } else {
                    $post_id = get_the_ID();
                    if ($post_id && function_exists('wc_get_product')) {
                        $_product = \wc_get_product($post_id);
                        if ($_product) {
                            $page_context['currentProductId'] = $_product->get_id();
                        }
                    }
                }
            }

            // Get trigger button avatar URL
            $trigger_icon_id = $this->getSetting(Constants::OPTION_TOGGLE_ICON_ID, 0);
            $trigger_icon_url = $trigger_icon_id ? wp_get_attachment_image_url($trigger_icon_id, 'thumbnail') : null;

            // Get header avatar URL
            $header_icon_id = $this->getSetting(Constants::OPTION_CHAT_HEADER_ICON_ID, 0);
            $header_icon_url = $header_icon_id ? wp_get_attachment_image_url($header_icon_id, 'thumbnail') : null;

            // Get greeting icon URL (for empty state)
            $greeting_icon_id = $this->getSetting(Constants::OPTION_CHAT_GREETING_ICON_ID, 0);
            $greeting_icon_url = $greeting_icon_id ? wp_get_attachment_image_url($greeting_icon_id, 'thumbnail') : false;

            // Get custom CSS (from minimal ChatSettings.php)
            $custom_css = $this->getSetting(Constants::OPTION_CHAT_CUSTOM_CSS, '');

            // Get publishable key for direct cloud communication
            $publishableKey = $this->license->getPublishableKey();
            $workspaceId = $this->aiClientManager->getWorkspaceId();

            // Cloud API base URL (defined in intucart.php, can be overridden via wp-config.php)
            $cloudApiUrl = INTUFIND_API_URL;

            // Cloud Streaming API URL
            $streamingApiUrl = INTUFIND_STREAMING_API_URL;

            // Widget URL (iframe loads this HTML page)
            $widgetUrl = $chat_base_url . '/chat-widget.html';

            // Prepare Widget Configuration
            $chatbotConfig = [
                'apiUrl' => $cloudApiUrl,
                'streamingApiUrl' => $streamingApiUrl,
                'publishableKey' => $publishableKey ?: null,
                'workspaceId' => $workspaceId ?: null,
                // Widget URL (loader creates iframe pointing to this)
                'widgetUrl' => $widgetUrl,
                
                // Page Context
                'pageContext' => $page_context,
                
                // Debug Mode
                'debugMode' => defined('WP_DEBUG') && WP_DEBUG,
                
                // Trigger Button Configuration
                'triggerAvatar' => $trigger_icon_url,
                'triggerShowBackground' => (bool) $this->getSetting(
                    Constants::OPTION_CHAT_TRIGGER_BUTTON_SHOW_BACKGROUND,
                    true
                ),
                'triggerIconShowBorder' => (bool) $this->getSetting(
                    Constants::OPTION_CHAT_TRIGGER_ICON_SHOW_BORDER,
                    true
                ),

                // Theme Configuration - Full color scheme from AI Color Generator
                'theme' => $this->buildThemeConfig(),
                
                // Custom CSS
                'customCSS' => $custom_css,
            ];

            // Content & Behavior Configuration (aligns with React ChatbotConfig interface)
            $contentConfig = [
                'enableFloatingChat' => true,

                // Header Configuration
                'title' => $this->getSetting(Constants::OPTION_CHAT_HEADER_TEXT),
                'subtitle' => $this->getSetting(Constants::OPTION_CHAT_HEADER_SUBTITLE),
                'avatar' => $header_icon_url,
                'headerAvatarShowBackground' => (bool) $this->getSetting(Constants::OPTION_CHAT_HEADER_AVATAR_SHOW_BACKGROUND),
                'headerAvatarShowBorder' => (bool) $this->getSetting(Constants::OPTION_CHAT_HEADER_AVATAR_SHOW_BORDER),

                // Greeting Configuration
                'greeting' => $this->getSetting(Constants::OPTION_ASSISTANT_WELCOME_MESSAGE),
                'greetingSubtext' => $this->getSetting(Constants::OPTION_CHAT_GREETING_SUBTEXT),
                'greetingIcon' => $greeting_icon_url,

                // AI Custom Instructions & Context
                'customInstructions' => $this->getSetting(Constants::OPTION_CHAT_CUSTOM_INSTRUCTIONS),
                'workspaceDescription' => $this->getSetting(Constants::OPTION_CHAT_SITE_DESCRIPTION),

                // Widget Behavior
                'widgetPosition' => $this->getSetting(Constants::OPTION_CHAT_WIDGET_POSITION),
                'cardLayout' => $this->getSetting(Constants::OPTION_CHAT_CARD_LAYOUT),
                'promptStyle' => $this->getSetting(Constants::OPTION_CHAT_PROMPT_STYLE),

                // Message Actions
                'showMessageActions' => (bool) $this->getSetting(Constants::OPTION_CHAT_SHOW_MESSAGE_ACTIONS),
                'showUserCopyButton' => (bool) $this->getSetting(Constants::OPTION_CHAT_SHOW_USER_COPY_BUTTON),
                'showMessageTimestamps' => (bool) $this->getSetting(Constants::OPTION_CHAT_SHOW_MESSAGE_TIMESTAMPS),

                // Post Card Display
                'showPostImage' => (bool) $this->getSetting(Constants::OPTION_CHAT_SHOW_POST_IMAGE),
                'showPostAuthor' => (bool) $this->getSetting(Constants::OPTION_CHAT_SHOW_POST_AUTHOR),
                'showPostDate' => (bool) $this->getSetting(Constants::OPTION_CHAT_SHOW_POST_DATE),
                'showPostCategory' => (bool) $this->getSetting(Constants::OPTION_CHAT_SHOW_POST_CATEGORY),

                // Live Agent Handoff
                'liveAgentEnabled' => (bool) $this->getSetting(Constants::OPTION_LIVE_AGENT_ENABLED),
                'liveAgentConfigured' => (bool) $this->getSetting(Constants::OPTION_LIVE_AGENT_CONFIGURED),
                'liveAgentManualHandoff' => (bool) $this->getSetting(Constants::OPTION_LIVE_AGENT_MANUAL_HANDOFF),
                'liveAgentAutoHandoff' => (bool) $this->getSetting(Constants::OPTION_LIVE_AGENT_AUTO_HANDOFF),
                'liveAgentProvider' => $this->getSetting(Constants::OPTION_LIVE_AGENT_PROVIDER),
                'liveAgentBusinessHours' => $this->getSetting(Constants::OPTION_LIVE_AGENT_BUSINESS_HOURS),
                'liveAgentInactivityTimeout' => (int) $this->getSetting(Constants::OPTION_LIVE_AGENT_INACTIVITY_TIMEOUT),
                'liveAgentEscalationPrompt' => $this->getSetting(Constants::OPTION_LIVE_AGENT_ESCALATION_PROMPT),
                'contactFormEnabled' => (bool) $this->getSetting(Constants::OPTION_CONTACT_FORM_ENABLED),

                // User Capabilities (computed per-request)
                'userCapabilities' => [
                    'canManageOptions' => current_user_can('manage_options'),
                ],

                // Privacy & Analytics
                'analyticsEnabled' => (bool) $this->getSetting(Constants::OPTION_CHAT_ANALYTICS_ENABLED, true),
                'analyticsConsentRequired' => (bool) $this->getSetting(Constants::OPTION_CHAT_ANALYTICS_CONSENT_REQUIRED, false),
            ];

            // Use wp_add_inline_script with JSON to preserve proper data types (booleans, etc.)
            // wp_localize_script converts boolean true to '1' string, which breaks our type checking
            $mergedConfig = array_merge($chatbotConfig, $contentConfig);
            $configJson = wp_json_encode($mergedConfig, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
            
            $inlineScript = "window.intufindConfig = {$configJson};";
            wp_add_inline_script($chat_script_handle, $inlineScript, 'before');
        }

        // Enqueue Search Widget if enabled
        $this->enqueueSearchWidget($is_chatbot_available, $is_not_admin, $enable_chat === '1');
    }

    /**
     * Enqueue the search widget scripts and configuration.
     *
     * @param bool $is_available Whether publishable key is configured
     * @param bool $is_not_admin Whether we're not in admin area
     * @param bool $chatbot_enabled Whether chatbot is enabled
     * @return void
     */
    private function enqueueSearchWidget(bool $is_available, bool $is_not_admin, bool $chatbot_enabled): void
    {
        // Check if search widget is enabled
        $search_widget_enabled = $this->getSetting(Constants::OPTION_SEARCH_WIDGET_ENABLED, '0') === '1';

        if (!$search_widget_enabled || !$is_available || !$is_not_admin) {
            return;
        }

        // Search widget base URL - override via INTUFIND_SEARCH_WIDGET_URL in wp-config.php
        $search_base_url = defined('INTUFIND_SEARCH_WIDGET_URL')
            ? rtrim(\INTUFIND_SEARCH_WIDGET_URL, '/')
            : 'https://cdn.intufind.com';

        $search_loader_url = $search_base_url . '/search-loader.js';

        wp_enqueue_script(
            'intucart-search-loader',
            $search_loader_url,
            [],
            null,
            true
        );

        // Get publishable key and API URL
        $publishableKey = $this->license->getPublishableKey();
        $cloudApiUrl = INTUFIND_API_URL;

        // Get search widget settings from cache
        $search_products = $this->getSetting(Constants::OPTION_SEARCH_WIDGET_PRODUCTS) === '1';
        $search_posts = $this->getSetting(Constants::OPTION_SEARCH_WIDGET_POSTS) === '1';
        $keyboard_shortcut = $this->getSetting(Constants::OPTION_SEARCH_WIDGET_KEYBOARD_SHORTCUT);
        $show_trigger = $this->getSetting(Constants::OPTION_SEARCH_WIDGET_SHOW_TRIGGER) === '1';
        $trigger_position = $this->getSetting(Constants::OPTION_SEARCH_WIDGET_TRIGGER_POSITION);
        $primary_color = $this->getSetting(Constants::OPTION_SEARCH_WIDGET_PRIMARY_COLOR);
        $dark_primary_color = $this->getSetting(Constants::OPTION_SEARCH_WIDGET_DARK_PRIMARY_COLOR);
        $color_scheme = $this->getSetting(Constants::OPTION_SEARCH_WIDGET_COLOR_SCHEME);
        
        // Trigger button styling
        $trigger_bg = $this->getSetting(Constants::OPTION_SEARCH_WIDGET_TRIGGER_BG);
        $trigger_text = $this->getSetting(Constants::OPTION_SEARCH_WIDGET_TRIGGER_TEXT);
        $trigger_border = $this->getSetting(Constants::OPTION_SEARCH_WIDGET_TRIGGER_BORDER);
        $dark_trigger_bg = $this->getSetting(Constants::OPTION_SEARCH_WIDGET_DARK_TRIGGER_BG);
        $dark_trigger_text = $this->getSetting(Constants::OPTION_SEARCH_WIDGET_DARK_TRIGGER_TEXT);
        $dark_trigger_border = $this->getSetting(Constants::OPTION_SEARCH_WIDGET_DARK_TRIGGER_BORDER);
        
        // Trigger button custom icon
        $trigger_icon_id = $this->getSetting(Constants::OPTION_SEARCH_WIDGET_TRIGGER_ICON_ID, 0);
        $trigger_icon_url = $trigger_icon_id ? wp_get_attachment_image_url($trigger_icon_id, 'thumbnail') : null;

        // Determine if floating trigger should show
        // Hide if chatbot is enabled to avoid button clutter
        $should_show_trigger = $show_trigger && !$chatbot_enabled;

        // Search widget URL (iframe loads this HTML page)
        $searchWidgetUrl = $search_base_url . '/search-widget.html';

        // Prepare search widget configuration
        $searchConfig = [
            'publishableKey' => $publishableKey ?: null,
            'apiUrl' => $cloudApiUrl,
            'widgetUrl' => $searchWidgetUrl,
            
            // Search scope
            'searchProducts' => $search_products,
            'searchPosts' => $search_posts,
            
            // Trigger settings
            'keyboardShortcut' => $keyboard_shortcut,
            'showTriggerButton' => $should_show_trigger,
            'triggerPosition' => $trigger_position,
            
            // Theme
            'theme' => [
                'primaryColor' => $primary_color,
                'darkPrimaryColor' => $dark_primary_color,
                'colorScheme' => $color_scheme,
                'trigger' => [
                    'background' => $trigger_bg,
                    'textColor' => $trigger_text,
                    'borderColor' => $trigger_border,
                    'darkBackground' => $dark_trigger_bg,
                    'darkTextColor' => $dark_trigger_text,
                    'darkBorderColor' => $dark_trigger_border,
                    'iconUrl' => $trigger_icon_url,
                ],
            ],
            
            // Debug mode
            'debugMode' => defined('WP_DEBUG') && WP_DEBUG,
        ];

        $configJson = wp_json_encode($searchConfig, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        $inlineScript = "window.intufindSearchConfig = {$configJson};";
        wp_add_inline_script('intucart-search-loader', $inlineScript, 'before');
    }

    /**
     * Build complete theme configuration for chat widget
     * Maps WordPress options to widget's expected theme structure
     *
     * @return array Theme configuration matching React Theme interface
     */
    private function buildThemeConfig(): array
    {
        // Get primary colors from cache
        $colorScheme = $this->getSetting(Constants::OPTION_CHAT_COLOR_SCHEME);
        $primaryColor = $this->getSetting(Constants::OPTION_CHAT_PRIMARY_COLOR);
        $primaryHover = $this->darkenColor($primaryColor, 10);
        
        return [
            'colorScheme' => $colorScheme,
            
            // Spacing/layout
            'spacing' => [
                'borderRadius' => $this->getSetting(Constants::OPTION_CHAT_BORDER_RADIUS) . 'px',
                'widgetMaxWidth' => $this->getSetting(Constants::OPTION_CHAT_WIDGET_MAX_WIDTH) . 'px',
                'widgetMaxHeight' => $this->getSetting(Constants::OPTION_CHAT_WIDGET_MAX_HEIGHT) . 'px',
            ],
            
            // Core colors
            'colors' => [
                'primaryColor' => $primaryColor,
                'primaryHover' => $primaryHover,
                'linkColor' => $this->getSetting(Constants::OPTION_CHAT_LINK_COLOR),
                'widgetBackground' => $this->getSetting(Constants::OPTION_CHAT_WIDGET_BACKGROUND_CUSTOM),
                'backgroundSecondary' => $this->getSetting(Constants::OPTION_CHAT_INPUT_BACKGROUND),
                'assistantBubbleBackground' => $this->getSetting(Constants::OPTION_CHAT_ASSISTANT_BUBBLE_BG),
                'borderColor' => $this->getSetting(Constants::OPTION_CHAT_ACCENT_COLOR),
                'userBubbleBackground' => $primaryColor,
            ],
            
            // Header
            'header' => [
                'background' => $this->getSetting(Constants::OPTION_CHAT_HEADER_BACKGROUND),
                'title' => ['color' => $this->getSetting(Constants::OPTION_CHAT_HEADER_TITLE_COLOR)],
                'subtitle' => ['color' => $this->getSetting(Constants::OPTION_CHAT_HEADER_SUBTITLE_COLOR)],
                'closeButton' => ['color' => $this->getSetting(Constants::OPTION_CHAT_HEADER_ICON_COLOR)],
            ],
            
            // Trigger button
            'triggerButton' => [
                'background' => $this->getSetting(Constants::OPTION_CHAT_TRIGGER_BUTTON_COLOR),
                'hoverBackground' => $this->getSetting(Constants::OPTION_CHAT_TRIGGER_BUTTON_HOVER_COLOR),
                'borderColor' => $this->getSetting(Constants::OPTION_CHAT_TRIGGER_BUTTON_BORDER_COLOR),
                'hoverBorderColor' => $this->getSetting(Constants::OPTION_CHAT_TRIGGER_BUTTON_BORDER_COLOR),
            ],
            
            // Chat bubbles
            'chatBubble' => [
                'user' => ['background' => $primaryColor, 'textColor' => '#ffffff'],
                'assistant' => ['background' => $this->getSetting(Constants::OPTION_CHAT_ASSISTANT_BUBBLE_BG)],
            ],
            
            // Reference cards
            'referenceCard' => [
                'background' => $this->getSetting(Constants::OPTION_CHAT_CARD_BACKGROUND),
                'borderColor' => $this->getSetting(Constants::OPTION_CHAT_CARD_BORDER),
                'hoverBackground' => $this->getSetting(Constants::OPTION_CHAT_CARD_HOVER_BG),
                'borderRadius' => $this->getSetting(Constants::OPTION_CHAT_CARD_BORDER_RADIUS) . 'px',
                'title' => [
                    'color' => $this->getSetting(Constants::OPTION_CHAT_CARD_TITLE_COLOR),
                    'fontSize' => $this->getSetting(Constants::OPTION_CHAT_CARD_TITLE_SIZE) . 'px',
                    'fontWeight' => $this->getSetting(Constants::OPTION_CHAT_CARD_TITLE_WEIGHT),
                ],
                'meta' => ['color' => $this->getSetting(Constants::OPTION_CHAT_CARD_META_COLOR)],
                'description' => [
                    'color' => $this->getSetting(Constants::OPTION_CHAT_CARD_DESC_COLOR),
                    'maxLength' => (int) $this->getSetting(Constants::OPTION_CHAT_CARD_DESC_LENGTH),
                ],
            ],
            
            // Prompt chips
            'promptChip' => [
                'background' => $this->getSetting(Constants::OPTION_CHAT_PROMPT_CHIP_BACKGROUND),
                'textColor' => $this->getSetting(Constants::OPTION_CHAT_PROMPT_CHIP_TEXT_COLOR),
                'borderColor' => $this->getSetting(Constants::OPTION_CHAT_PROMPT_CHIP_BORDER),
                'hover' => ['background' => $this->getSetting(Constants::OPTION_CHAT_PROMPT_CHIP_HOVER_BG)],
            ],
            
            // Dark mode overrides
            'darkTheme' => [
                'colors' => [
                    'primaryColor' => $this->getSetting(Constants::OPTION_CHAT_DARK_PRIMARY_COLOR),
                    'linkColor' => $this->getSetting(Constants::OPTION_CHAT_DARK_LINK_COLOR),
                    'widgetBackground' => $this->getSetting(Constants::OPTION_CHAT_DARK_WIDGET_BACKGROUND_CUSTOM),
                    'backgroundSecondary' => $this->getSetting(Constants::OPTION_CHAT_DARK_INPUT_BACKGROUND),
                    'assistantBubbleBackground' => $this->getSetting(Constants::OPTION_CHAT_DARK_ASSISTANT_BUBBLE_BG),
                    'borderColor' => $this->getSetting(Constants::OPTION_CHAT_DARK_ACCENT_COLOR),
                ],
                'header' => [
                    'background' => $this->getSetting(Constants::OPTION_CHAT_DARK_HEADER_BACKGROUND),
                    'title' => ['color' => $this->getSetting(Constants::OPTION_CHAT_DARK_HEADER_TITLE_COLOR)],
                    'subtitle' => ['color' => $this->getSetting(Constants::OPTION_CHAT_DARK_HEADER_SUBTITLE_COLOR)],
                    'closeButton' => ['color' => $this->getSetting(Constants::OPTION_CHAT_DARK_HEADER_ICON_COLOR)],
                ],
                'triggerButton' => [
                    'background' => $this->getSetting(Constants::OPTION_CHAT_DARK_TRIGGER_BUTTON_COLOR),
                    'hoverBackground' => $this->getSetting(Constants::OPTION_CHAT_DARK_TRIGGER_BUTTON_HOVER_COLOR),
                    'borderColor' => $this->getSetting(Constants::OPTION_CHAT_DARK_TRIGGER_BUTTON_BORDER_COLOR),
                ],
                'chatBubble' => [
                    'assistant' => ['background' => $this->getSetting(Constants::OPTION_CHAT_DARK_ASSISTANT_BUBBLE_BG)],
                ],
                'referenceCard' => [
                    'background' => $this->getSetting(Constants::OPTION_CHAT_DARK_CARD_BACKGROUND),
                    'borderColor' => $this->getSetting(Constants::OPTION_CHAT_DARK_CARD_BORDER),
                    'hoverBackground' => $this->getSetting(Constants::OPTION_CHAT_DARK_CARD_HOVER_BG),
                    'title' => ['color' => $this->getSetting(Constants::OPTION_CHAT_DARK_CARD_TITLE_COLOR)],
                    'meta' => ['color' => $this->getSetting(Constants::OPTION_CHAT_DARK_CARD_META_COLOR)],
                    'description' => ['color' => $this->getSetting(Constants::OPTION_CHAT_DARK_CARD_DESC_COLOR)],
                ],
                'promptChip' => [
                    'background' => $this->getSetting(Constants::OPTION_CHAT_DARK_PROMPT_CHIP_BACKGROUND),
                    'textColor' => $this->getSetting(Constants::OPTION_CHAT_DARK_PROMPT_CHIP_TEXT_COLOR),
                    'borderColor' => $this->getSetting(Constants::OPTION_CHAT_DARK_PROMPT_CHIP_BORDER),
                    'hover' => ['background' => $this->getSetting(Constants::OPTION_CHAT_DARK_PROMPT_CHIP_HOVER_BG)],
                ],
            ],
        ];
    }

    /**
     * Darken a hex color by a percentage
     *
     * @param string $hex Hex color code (e.g., '#0084ff')
     * @param int $percent Percentage to darken (0-100)
     * @return string Darkened hex color
     */
    private function darkenColor(string $hex, int $percent): string
    {
        // Remove # if present
        $hex = ltrim($hex, '#');
        
        // Parse RGB values
        if (strlen($hex) === 3) {
            $hex = $hex[0] . $hex[0] . $hex[1] . $hex[1] . $hex[2] . $hex[2];
        }
        
        if (strlen($hex) !== 6) {
            return '#' . $hex; // Return original if invalid
        }
        
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));
        
        // Darken by reducing each component
        $factor = (100 - $percent) / 100;
        $r = max(0, min(255, (int) ($r * $factor)));
        $g = max(0, min(255, (int) ($g * $factor)));
        $b = max(0, min(255, (int) ($b * $factor)));
        
        return sprintf('#%02x%02x%02x', $r, $g, $b);
    }
}
