<?php
/**
 * Button Component
 * 
 * Reusable button component for admin pages with support for icons, variants, and sizes
 * 
 * @package Intucart
 * @subpackage Components
 */

namespace Intucart\Components;

class Button
{
    /**
     * Render a button
     * 
     * @param array $config Button configuration:
     *   - text: string (required) - Button text
     *   - variant: string - 'primary'|'secondary'|'danger'|'success'|'warning' (default: 'primary')
     *   - size: string - 'small'|'medium'|'large' (default: 'medium')
     *   - icon: string - SVG icon HTML (optional)
     *   - icon_position: string - 'before'|'after' (default: 'before')
     *   - href: string - URL for link button (optional)
     *   - type: string - Button type for form buttons (default: 'button')
     *   - id: string - HTML id attribute (optional)
     *   - class: string - Additional CSS classes (optional)
     *   - disabled: bool - Whether button is disabled (default: false)
     *   - attributes: array - Additional HTML attributes as key => value pairs (optional)
     * @return void
     */
    public static function render(array $config): void
    {
        // Required parameter
        if (!isset($config['text'])) {
            return;
        }

        // Default values
        $text = esc_html($config['text']);
        $variant = $config['variant'] ?? 'primary';
        $size = $config['size'] ?? 'medium';
        $icon = $config['icon'] ?? '';
        $iconPosition = $config['icon_position'] ?? 'before';
        $href = $config['href'] ?? '';
        $type = $config['type'] ?? 'button';
        $id = $config['id'] ?? '';
        $additionalClasses = $config['class'] ?? '';
        $disabled = $config['disabled'] ?? false;
        $attributes = $config['attributes'] ?? [];

        // Build CSS classes
        $classes = [
            'itf-btn',
            "itf-btn-{$variant}",
            "itf-btn-{$size}"
        ];

        if ($disabled) {
            $classes[] = 'itf-btn-disabled';
        }

        if ($additionalClasses) {
            $classes[] = $additionalClasses;
        }

        // Merge class from attributes if provided
        if (isset($attributes['class'])) {
            $classes[] = $attributes['class'];
            unset($attributes['class']); // Remove from attributes to avoid duplicate
        }

        $classString = implode(' ', $classes);

        // Build button content
        $content = '';
        
        if ($icon && $iconPosition === 'before') {
            $content .= '<span class="itf-btn-icon itf-btn-icon-before">' . $icon . '</span>';
        }
        
        $content .= '<span class="itf-btn-text">' . $text . '</span>';
        
        if ($icon && $iconPosition === 'after') {
            $content .= '<span class="itf-btn-icon itf-btn-icon-after">' . $icon . '</span>';
        }

        // Build additional attributes string
        $attributesString = '';
        foreach ($attributes as $key => $value) {
            $attributesString .= sprintf(' %s="%s"', esc_attr($key), esc_attr($value));
        }

        // Render as link or button
        if ($href && !$disabled) {
            printf(
                '<a href="%s" class="%s"%s%s>%s</a>',
                esc_url($href),
                esc_attr($classString),
                $id ? ' id="' . esc_attr($id) . '"' : '',
                $attributesString,
                $content
            );
        } else {
            printf(
                '<button type="%s" class="%s"%s%s%s>%s</button>',
                esc_attr($type),
                esc_attr($classString),
                $id ? ' id="' . esc_attr($id) . '"' : '',
                $disabled ? ' disabled' : '',
                $attributesString,
                $content
            );
        }
    }

    /**
     * Helper method to render a primary button
     * 
     * @param string $text Button text
     * @param array $config Additional configuration
     * @return void
     */
    public static function primary(string $text, array $config = []): void
    {
        $config['text'] = $text;
        $config['variant'] = 'primary';
        self::render($config);
    }

    /**
     * Helper method to render a secondary button
     * 
     * @param string $text Button text
     * @param array $config Additional configuration
     * @return void
     */
    public static function secondary(string $text, array $config = []): void
    {
        $config['text'] = $text;
        $config['variant'] = 'secondary';
        self::render($config);
    }

    /**
     * Helper method to render a danger button
     * 
     * @param string $text Button text
     * @param array $config Additional configuration
     * @return void
     */
    public static function danger(string $text, array $config = []): void
    {
        $config['text'] = $text;
        $config['variant'] = 'danger';
        self::render($config);
    }

    /**
     * Helper method to render a success button
     * 
     * @param string $text Button text
     * @param array $config Additional configuration
     * @return void
     */
    public static function success(string $text, array $config = []): void
    {
        $config['text'] = $text;
        $config['variant'] = 'success';
        self::render($config);
    }
}

