<?php
/**
 * Color Picker Component
 * 
 * Reusable color picker component with WordPress color picker integration
 * Supports single and dual (light/dark) mode pickers
 * 
 * @package Intucart
 * @subpackage Components
 */

namespace Intucart\Components;

class ColorPicker
{
    /**
     * Render a color picker input
     * 
     * @param array $config Configuration:
     *   - name: string - Field name (required)
     *   - value: string - Current color value (required)
     *   - label: string - Label text (optional)
     *   - description: string - Help text below picker (optional)
     *   - disabled: bool - Whether disabled (optional, default: false)
     *   - wrapper_class: string - Additional wrapper classes (optional)
     * @return void
     */
    public static function render(array $config): void
    {
        // Required parameters
        if (!isset($config['name']) || !isset($config['value'])) {
            return;
        }

        $name = $config['name'];
        $value = $config['value'];
        $label = $config['label'] ?? '';
        $description = $config['description'] ?? '';
        $disabled = $config['disabled'] ?? false;
        $wrapper_class = $config['wrapper_class'] ?? '';

        $disabled_attr = $disabled ? 'disabled' : '';
        $disabled_style = $disabled ? 'opacity: 0.5;' : '';

        // Wrapper
        echo '<div class="intucart-color-picker-group ' . esc_attr($wrapper_class) . '">';
        
        // Label
        if ($label) {
            echo '<label class="intucart-color-picker-label">';
            echo esc_html($label);
            echo '</label>';
        }

        // Color input (WordPress color picker)
        $inline_style = 'border-color: ' . esc_attr($value) . ' !important;';
        if ($disabled_style) {
            $inline_style .= ' ' . $disabled_style;
        }
        
        echo '<input type="text" ';
        echo 'name="' . esc_attr($name) . '" ';
        echo 'value="' . esc_attr($value) . '" ';
        echo 'class="itf-color-picker" ';
        echo 'style="' . esc_attr($inline_style) . '" ';
        if ($disabled_attr) {
            echo $disabled_attr . ' ';
        }
        echo '/>';

        // Description
        if ($description) {
            echo '<p class="intucart-field-description">' . wp_kses_post($description) . '</p>';
        }

        echo '</div>';
    }

    /**
     * Render dual color pickers (light and dark mode side by side)
     * 
     * @param array $config Configuration:
     *   - light_name: string - Light mode field name (required)
     *   - light_value: string - Light mode color value (required)
     *   - dark_name: string - Dark mode field name (required)
     *   - dark_value: string - Dark mode color value (required)
     *   - description: string - Help text below pickers (optional)
     *   - wrapper_class: string - Additional wrapper classes (optional)
     * @return void
     */
    public static function dual(array $config): void
    {
        // Required parameters
        if (!isset($config['light_name'], $config['light_value'], $config['dark_name'], $config['dark_value'])) {
            return;
        }

        $light_name = $config['light_name'];
        $light_value = $config['light_value'];
        $dark_name = $config['dark_name'];
        $dark_value = $config['dark_value'];
        $description = $config['description'] ?? '';
        $wrapper_class = $config['wrapper_class'] ?? '';

        // Wrapper
        echo '<div class="intucart-dual-color-picker ' . esc_attr($wrapper_class) . '">';
        
        // Light mode picker
        echo '<div class="intucart-color-picker-item">';
        echo '<label class="intucart-color-picker-label">' . esc_html__('Light Mode', 'intufind') . '</label>';
        echo '<input type="text" ';
        echo 'name="' . esc_attr($light_name) . '" ';
        echo 'value="' . esc_attr($light_value) . '" ';
        echo 'class="itf-color-picker" ';
        echo 'style="border-color: ' . esc_attr($light_value) . ' !important;" />';
        echo '</div>';
        
        // Dark mode picker
        echo '<div class="intucart-color-picker-item">';
        echo '<label class="intucart-color-picker-label">' . esc_html__('Dark Mode', 'intufind') . '</label>';
        echo '<input type="text" ';
        echo 'name="' . esc_attr($dark_name) . '" ';
        echo 'value="' . esc_attr($dark_value) . '" ';
        echo 'class="itf-color-picker" ';
        echo 'style="border-color: ' . esc_attr($dark_value) . ' !important;" />';
        echo '</div>';

        echo '</div>';

        // Description (below both pickers)
        if ($description) {
            echo '<p class="intucart-field-description">' . wp_kses_post($description) . '</p>';
        }
    }

    /**
     * Helper method for single color picker
     * 
     * @param string $name Field name
     * @param string $value Current color value
     * @param array $config Additional configuration
     * @return void
     */
    public static function single(string $name, string $value, array $config = []): void
    {
        $config['name'] = $name;
        $config['value'] = $value;
        self::render($config);
    }
}

