<?php
/**
 * Field Component
 * 
 * Reusable form field component for admin pages with support for various input types,
 * labels, descriptions, and hint sections
 * 
 * @package Intucart
 * @subpackage Components
 */

namespace Intucart\Components;

class Field
{
    /**
     * Render a form field with label and description
     * 
     * @param array $config Field configuration:
     *   - type: string - 'checkbox'|'text'|'number'|'select'|'textarea'|'radio'|'range' (required)
     *   - name: string - Field name attribute (required)
     *   - id: string - Field id attribute (optional, defaults to name)
     *   - value: mixed - Current value (optional)
     *   - label: string - Field label text (optional, can be empty for table rows)
     *   - description: string - Field description/help text (optional)
     *   - hint: array - Hint sections [['title' => '...', 'content' => '...'], ...] (optional)
     *   - options: array - Options for select/radio [value => label] (required for select/radio)
     *   - attributes: array - Additional HTML attributes (optional)
     *   - class: string - Additional CSS classes (optional)
     *   - wrapper_class: string - Additional wrapper classes (optional)
     *   - show_value: bool - Show current value next to range slider (optional, default true)
     * @return void
     */
    public static function render(array $config): void
    {
        // Required parameters (label is optional for table rows)
        if (!isset($config['type']) || !isset($config['name'])) {
            return;
        }

        $type = $config['type'];
        $name = $config['name'];
        $id = $config['id'] ?? $name;
        $value = $config['value'] ?? '';
        $label = $config['label'] ?? '';
        $description = $config['description'] ?? '';
        $hint = $config['hint'] ?? [];
        $options = $config['options'] ?? [];
        $attributes = $config['attributes'] ?? [];
        $class = $config['class'] ?? '';
        $wrapperClass = $config['wrapper_class'] ?? '';
        $toggleable = $config['toggleable'] ?? true;
        $suffix = $config['suffix'] ?? '';
        $showValue = $config['show_value'] ?? true;

        // Build attributes string
        $attributesString = '';
        foreach ($attributes as $key => $val) {
            $attributesString .= sprintf(' %s="%s"', esc_attr($key), esc_attr($val));
        }

        // Render based on type
        echo '<div class="intucart-field-group ' . esc_attr($wrapperClass) . '">';

        switch ($type) {
            case 'checkbox':
                self::renderCheckbox($id, $name, $value, $label, $description, $hint, $class, $attributesString);
                break;
            case 'text':
            case 'number':
            case 'email':
            case 'url':
                self::renderInput($type, $id, $name, $value, $label, $description, $hint, $class, $attributesString, $suffix);
                break;
            case 'password':
                self::renderPassword($id, $name, $value, $label, $description, $hint, $class, $attributesString, $toggleable);
                break;
            case 'select':
                self::renderSelect($id, $name, $value, $label, $description, $hint, $options, $class, $attributesString);
                break;
            case 'textarea':
                self::renderTextarea($id, $name, $value, $label, $description, $hint, $class, $attributesString);
                break;
            case 'radio':
                self::renderRadio($id, $name, $value, $label, $description, $hint, $options, $class, $attributesString);
                break;
            case 'range':
                self::renderRange($id, $name, $value, $label, $description, $hint, $class, $attributesString, $suffix, $showValue);
                break;
        }

        echo '</div>';
    }

    /**
     * Render checkbox field
     */
    private static function renderCheckbox(string $id, string $name, $value, string $label, string $description, array $hint, string $class, string $attributes): void
    {
        echo '<label class="itf-checkbox-label">';
        printf(
            '<input type="checkbox" id="%s" name="%s" value="1" class="itf-checkbox %s"%s%s>',
            esc_attr($id),
            esc_attr($name),
            esc_attr($class),
            checked($value, 1, false),
            $attributes
        );
        echo '<span class="itf-checkbox-text">';
        echo '<strong>' . esc_html($label) . '</strong>';
        if ($description) {
            echo '<span class="intucart-field-description">' . wp_kses_post($description) . '</span>';
        }
        self::renderHint($hint);
        echo '</span>';
        echo '</label>';
    }

    /**
     * Render text/number/email/url/password input field
     */
    private static function renderInput(string $type, string $id, string $name, $value, string $label, string $description, array $hint, string $class, string $attributes, string $suffix = ''): void
    {
        if ($label) {
            echo '<label for="' . esc_attr($id) . '"><strong>' . esc_html($label) . '</strong></label>';
        }
        
        $inputClass = $type === 'number' ? 'small-text' : 'regular-text';
        printf(
            '<input type="%s" id="%s" name="%s" value="%s" class="%s %s"%s>',
            esc_attr($type),
            esc_attr($id),
            esc_attr($name),
            esc_attr($value),
            esc_attr($inputClass),
            esc_attr($class),
            $attributes
        );
        
        if ($suffix) {
            echo ' <span class="intucart-field-suffix">' . esc_html($suffix) . '</span>';
        }
        
        if ($description) {
            echo '<p class="intucart-field-description">' . wp_kses_post($description) . '</p>';
        }
        self::renderHint($hint);
    }

    /**
     * Render password field with optional visibility toggle
     */
    private static function renderPassword(string $id, string $name, $value, string $label, string $description, array $hint, string $class, string $attributes, bool $toggleable = true): void
    {
        if ($label) {
            echo '<label for="' . esc_attr($id) . '"><strong>' . esc_html($label) . '</strong></label>';
        }
        
        if ($toggleable) {
            echo '<div class="intucart-password-wrapper">';
        }
        
        printf(
            '<input type="password" id="%s" name="%s" value="%s" class="regular-text %s"%s>',
            esc_attr($id),
            esc_attr($name),
            esc_attr($value),
            esc_attr($class),
            $attributes
        );
        
        if ($toggleable) {
            echo '<button type="button" class="intucart-password-toggle" data-target="' . esc_attr($id) . '" title="' . esc_attr__('Show password', 'intufind') . '">';
            echo '<span class="dashicons dashicons-visibility"></span>';
            echo '</button>';
            echo '</div>';
            
            // Enqueue fields script (handles password toggle, char counter, range slider)
            self::enqueueFieldsScript();
        }
        
        if ($description) {
            echo '<p class="intucart-field-description">' . wp_kses_post($description) . '</p>';
        }
        self::renderHint($hint);
    }

    /**
     * Render select field
     */
    private static function renderSelect(string $id, string $name, $value, string $label, string $description, array $hint, array $options, string $class, string $attributes): void
    {
        if ($label) {
            echo '<label for="' . esc_attr($id) . '"><strong>' . esc_html($label) . '</strong></label>';
        }
        
        printf(
            '<select id="%s" name="%s" class="%s"%s>',
            esc_attr($id),
            esc_attr($name),
            esc_attr($class),
            $attributes
        );
        
        foreach ($options as $optValue => $optLabel) {
            printf(
                '<option value="%s"%s>%s</option>',
                esc_attr($optValue),
                selected($value, $optValue, false),
                esc_html($optLabel)
            );
        }
        
        echo '</select>';
        
        if ($description) {
            echo '<p class="intucart-field-description">' . wp_kses_post($description) . '</p>';
        }
        self::renderHint($hint);
    }

    /**
     * Render textarea field
     */
    private static function renderTextarea(string $id, string $name, $value, string $label, string $description, array $hint, string $class, string $attributes): void
    {
        if ($label) {
            echo '<label for="' . esc_attr($id) . '"><strong>' . esc_html($label) . '</strong></label>';
        }
        
        // Extract maxlength for character counter
        $maxlength = null;
        if (preg_match('/maxlength=["\']?(\d+)["\']?/', $attributes, $matches)) {
            $maxlength = (int) $matches[1];
        }
        
        printf(
            '<textarea id="%s" name="%s" class="large-text %s" rows="5"%s>%s</textarea>',
            esc_attr($id),
            esc_attr($name),
            esc_attr($class),
            $attributes,
            esc_textarea($value)
        );
        
        // Character counter if maxlength is set
        if ($maxlength) {
            $currentLength = strlen($value);
            echo '<div class="intucart-textarea-counter" style="text-align: right; margin-top: 4px; font-size: 12px; color: #646970;">';
            echo '<span class="intucart-char-count" data-textarea-id="' . esc_attr($id) . '">' . esc_html($currentLength) . '</span>';
            echo ' / ' . esc_html(number_format($maxlength)) . ' ' . esc_html__('characters', 'intufind');
            echo '</div>';
            
            // Enqueue fields script (handles password toggle, char counter, range slider)
            self::enqueueFieldsScript();
        }
        
        if ($description) {
            echo '<p class="intucart-field-description">' . wp_kses_post($description) . '</p>';
        }
        self::renderHint($hint);
    }

    /**
     * Render radio field group
     */
    private static function renderRadio(string $id, string $name, $value, string $label, string $description, array $hint, array $options, string $class, string $attributes): void
    {
        if ($label) {
            echo '<strong>' . esc_html($label) . '</strong>';
        }
        
        if ($description) {
            echo '<p class="intucart-field-description">' . wp_kses_post($description) . '</p>';
        }
        
        $counter = 0;
        foreach ($options as $optValue => $optLabel) {
            $radioId = $id . '_' . $counter;
            echo '<label class="intucart-radio-label">';
            printf(
                '<input type="radio" id="%s" name="%s" value="%s" class="%s"%s%s> %s',
                esc_attr($radioId),
                esc_attr($name),
                esc_attr($optValue),
                esc_attr($class),
                checked($value, $optValue, false),
                $attributes,
                esc_html($optLabel)
            );
            echo '</label><br>';
            $counter++;
        }
        
        self::renderHint($hint);
    }

    /**
     * Render range slider field
     */
    private static function renderRange(string $id, string $name, $value, string $label, string $description, array $hint, string $class, string $attributes, string $suffix = '', bool $showValue = true): void
    {
        if ($label) {
            echo '<label for="' . esc_attr($id) . '"><strong>' . esc_html($label) . '</strong></label>';
        }
        
        echo '<div class="intucart-range-wrapper">';
        
        printf(
            '<input type="range" id="%s" name="%s" value="%s" class="intucart-range %s"%s>',
            esc_attr($id),
            esc_attr($name),
            esc_attr($value),
            esc_attr($class),
            $attributes
        );
        
        if ($showValue) {
            $valueId = $id . '_value';
            echo ' <span class="intucart-range-value" id="' . esc_attr($valueId) . '">' . esc_html($value) . esc_html($suffix) . '</span>';
            
            // Enqueue fields script (handles password toggle, char counter, range slider)
            self::enqueueFieldsScript();
        }
        
        echo '</div>';
        
        if ($description) {
            echo '<p class="intucart-field-description">' . wp_kses_post($description) . '</p>';
        }
        self::renderHint($hint);
    }

    /**
     * Render hint sections
     */
    private static function renderHint(array $hint): void
    {
        if (empty($hint)) {
            return;
        }

        foreach ($hint as $hintSection) {
            echo '<div class="intucart-hint">';
            if (isset($hintSection['title'])) {
                echo '<div class="intucart-hint-section"><strong>' . esc_html($hintSection['title']) . '</strong></div>';
            }
            if (isset($hintSection['content'])) {
                echo '<div class="intucart-hint-section">' . wp_kses_post($hintSection['content']) . '</div>';
            }
            echo '</div>';
        }
    }

    /**
     * Helper method to render a checkbox field
     * 
     * @param string $name Field name
     * @param string $label Field label
     * @param mixed $value Current value
     * @param array $config Additional configuration
     * @return void
     */
    public static function checkbox(string $name, string $label, $value = 0, array $config = []): void
    {
        $config['type'] = 'checkbox';
        $config['name'] = $name;
        $config['label'] = $label;
        $config['value'] = $value;
        self::render($config);
    }

    /**
     * Helper method to render a text field
     * 
     * @param string $name Field name
     * @param string $label Field label
     * @param string $value Current value
     * @param array $config Additional configuration
     * @return void
     */
    public static function text(string $name, string $label, string $value = '', array $config = []): void
    {
        $config['type'] = 'text';
        $config['name'] = $name;
        $config['label'] = $label;
        $config['value'] = $value;
        self::render($config);
    }

    /**
     * Helper method to render a number field
     * 
     * @param string $name Field name
     * @param string $label Field label
     * @param mixed $value Current value
     * @param array $config Additional configuration
     * @return void
     */
    public static function number(string $name, string $label, $value = '', array $config = []): void
    {
        $config['type'] = 'number';
        $config['name'] = $name;
        $config['label'] = $label;
        $config['value'] = $value;
        self::render($config);
    }

    /**
     * Helper method to render a select field
     * 
     * @param string $name Field name
     * @param string $label Field label
     * @param array $options Options array [value => label]
     * @param mixed $value Current value
     * @param array $config Additional configuration
     * @return void
     */
    public static function select(string $name, string $label, array $options, $value = '', array $config = []): void
    {
        $config['type'] = 'select';
        $config['name'] = $name;
        $config['label'] = $label;
        $config['options'] = $options;
        $config['value'] = $value;
        self::render($config);
    }

    /**
     * Helper method to render a password field
     * 
     * @param string $name Field name
     * @param string $label Field label
     * @param string $value Current value
     * @param array $config Additional configuration (can include 'toggleable' => false)
     * @return void
     */
    public static function password(string $name, string $label, string $value = '', array $config = []): void
    {
        $config['type'] = 'password';
        $config['name'] = $name;
        $config['label'] = $label;
        $config['value'] = $value;
        self::render($config);
    }

    /**
     * Helper method to render a range slider field
     * 
     * @param string $name Field name
     * @param string $label Field label
     * @param mixed $value Current value
     * @param array $config Additional configuration (min, max, step, suffix, show_value)
     * @return void
     */
    public static function range(string $name, string $label, $value = '', array $config = []): void
    {
        $config['type'] = 'range';
        $config['name'] = $name;
        $config['label'] = $label;
        $config['value'] = $value;
        self::render($config);
    }

    /**
     * Enqueue the fields JavaScript file
     * Handles password toggle, character counter, and range slider functionality
     * 
     * @return void
     */
    private static function enqueueFieldsScript(): void
    {
        static $enqueued = false;
        if (!$enqueued) {
            wp_enqueue_script(
                'intucart-fields',
                INTUCART_PLUGIN_URL . 'assets/js/admin/fields.js',
                ['jquery'],
                INTUCART_VERSION,
                true
            );
            $enqueued = true;
        }
    }
}

