<?php
/**
 * Image Uploader Component
 * 
 * Reusable image uploader component with WordPress media.js integration
 * 
 * @package Intucart
 * @subpackage Components
 */

namespace Intucart\Components;

class ImageUploader
{
    /**
     * Render an image uploader with preview and action buttons
     * 
     * @param array $config Configuration:
     *   - name: string - Field name (required)
     *   - value: int - Attachment ID (required)
     *   - button_text: string - Text for upload button (optional, default: 'Image')
     *   - hint: string - Help text below uploader (optional)
     *   - preview_size: array - [width, height] for preview (optional, default: [60, 60])
     *   - preview_style: string - Additional preview styles (optional)
     *   - show_placeholder: bool - Show placeholder when no image (optional, default: true)
     *   - wrapper_class: string - Additional wrapper classes (optional)
     * @return void
     */
    public static function render(array $config): void
    {
        // Required parameters
        if (!isset($config['name'])) {
            return;
        }

        $name = $config['name'];
        $value = $config['value'] ?? 0;
        $button_text = $config['button_text'] ?? __('Image', 'intufind');
        $hint = $config['hint'] ?? '';
        $preview_size = $config['preview_size'] ?? [60, 60];
        $preview_style = $config['preview_style'] ?? '';
        $show_placeholder = $config['show_placeholder'] ?? true;
        $wrapper_class = $config['wrapper_class'] ?? '';

        $icon_url = $value ? wp_get_attachment_image_url($value, 'thumbnail') : '';

        // Wrapper
        echo '<div class="intucart-image-uploader ' . esc_attr($wrapper_class) . '" style="display: flex; align-items: center; gap: 12px;">';

        // Hidden input (required by media.js)
        echo '<input type="hidden" name="' . esc_attr($name) . '" value="' . esc_attr($value) . '">';

        // Preview wrapper
        $preview_width = is_array($preview_size) ? $preview_size[0] : 60;
        $preview_height = is_array($preview_size) ? $preview_size[1] : 60;
        
        $default_preview_style = "width: {$preview_width}px; height: {$preview_height}px; border-radius: 8px; overflow: hidden;";
        
        if (!$icon_url && $show_placeholder) {
            $default_preview_style .= ' background: var(--itf-avatar-background, #f5f5f5); border: 2px dashed var(--itf-border-light, #ccc); display: flex; align-items: center; justify-content: center;';
        }
        
        echo '<div class="intucart-image-preview-wrapper" style="' . esc_attr($default_preview_style . ' ' . $preview_style) . '">';

        if ($icon_url) {
            echo '<img src="' . esc_url($icon_url) . '" style="width: 100%; height: 100%; object-fit: contain; border: var(--itf-avatar-border-width, 2px) solid var(--itf-avatar-border-color, #ddd); border-radius: inherit; background: var(--itf-avatar-background, #fff); box-shadow: var(--itf-avatar-shadow, 0 1px 3px rgba(0,0,0,0.1));">';
        } elseif ($show_placeholder) {
            echo '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512" style="width: 32px; height: 32px; fill: #999;"><path opacity=".4" d="M448 384H64v-48l71.51-71.52a12 12 0 0 1 17 0L208 320l135.51-135.52a12 12 0 0 1 17 0L448 272z"/><path d="M464 64H48a48 48 0 0 0-48 48v288a48 48 0 0 0 48 48h416a48 48 0 0 0 48-48V112a48 48 0 0 0-48-48zm-352 56a56 56 0 1 1-56 56 56 56 0 0 1 56-56zm336 264H64v-48l71.51-71.52a12 12 0 0 1 17 0L208 320l135.51-135.52a12 12 0 0 1 17 0L448 272z"/></svg>';
        }

        echo '</div>';

        // Action buttons
        echo '<div style="display: flex; gap: 8px;">';
        
        Button::secondary($icon_url ? __('Change', 'intufind') : __('Upload', 'intufind'), [
            'type' => 'button',
            'size' => 'small',
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path opacity=".4" d="M488 351.92H352v8a56 56 0 0 1-56 56h-80a56 56 0 0 1-56-56v-8H24a23.94 23.94 0 0 0-24 24v112a23.94 23.94 0 0 0 24 24h464a23.94 23.94 0 0 0 24-24v-112a23.94 23.94 0 0 0-24-24zm-120 132a20 20 0 1 1 20-20 20.06 20.06 0 0 1-20 20zm64 0a20 20 0 1 1 20-20 20.06 20.06 0 0 1-20 20z"/><path d="M192 359.93v-168h-87.7c-17.8 0-26.7-21.5-14.1-34.11L242.3 5.62a19.37 19.37 0 0 1 27.3 0l152.2 152.2c12.6 12.61 3.7 34.11-14.1 34.11H320v168a23.94 23.94 0 0 1-24 24h-80a23.94 23.94 0 0 1-24-24z"/></svg>',
            'icon_position' => 'before',
            'attributes' => [
                'class' => 'upload-button'
            ]
        ]);

        Button::danger(__('Remove', 'intufind'), [
            'type' => 'button',
            'size' => 'small',
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path opacity=".4" d="M53.2 467L32 96h384l-21.2 371a48 48 0 0 1-47.9 45H101.1a48 48 0 0 1-47.9-45z"/><path d="M0 80V48a16 16 0 0 1 16-16h120l9.4-18.7A23.72 23.72 0 0 1 166.8 0h114.3a24 24 0 0 1 21.5 13.3L312 32h120a16 16 0 0 1 16 16v32a16 16 0 0 1-16 16H16A16 16 0 0 1 0 80z"/></svg>',
            'icon_position' => 'before',
            'attributes' => [
                'class' => 'remove-button',
                'style' => $icon_url ? '' : 'display: none;'
            ]
        ]);

        echo '</div>';
        echo '</div>';

        // Hint text
        if ($hint) {
            echo '<div class="intucart-hint">';
            echo '<div class="intucart-hint-section">' . wp_kses_post($hint) . '</div>';
            echo '</div>';
        }
    }

    /**
     * Helper method for standard image uploader
     * 
     * @param string $name Field name
     * @param int $value Attachment ID
     * @param array $config Additional configuration
     * @return void
     */
    public static function standard(string $name, int $value, array $config = []): void
    {
        $config['name'] = $name;
        $config['value'] = $value;
        self::render($config);
    }

    /**
     * Helper method for avatar uploader
     * 
     * @param string $name Field name
     * @param int $value Attachment ID
     * @param array $config Additional configuration
     * @return void
     */
    public static function avatar(string $name, int $value, array $config = []): void
    {
        $config['name'] = $name;
        $config['value'] = $value;
        $config['button_text'] = $config['button_text'] ?? __('Avatar', 'intufind');
        $config['preview_size'] = $config['preview_size'] ?? [60, 60];
        self::render($config);
    }

    /**
     * Helper method for icon uploader
     * 
     * @param string $name Field name
     * @param int $value Attachment ID
     * @param array $config Additional configuration
     * @return void
     */
    public static function icon(string $name, int $value, array $config = []): void
    {
        $config['name'] = $name;
        $config['value'] = $value;
        $config['button_text'] = $config['button_text'] ?? __('Icon', 'intufind');
        $config['preview_size'] = $config['preview_size'] ?? [48, 48];
        self::render($config);
    }
}

