<?php
/**
 * Tabs Component
 * 
 * Reusable tab navigation component for admin pages
 * 
 * @package Intucart
 * @subpackage Components
 */

namespace Intucart\Components;

class Tabs
{
    /**
     * Render outer tabs container with navigation
     * 
     * @param array $tabs Array of tab configurations [key => ['url' => string, 'label' => string, 'icon' => string, 'condition' => bool]]
     * @param string $currentTab Currently active tab key
     * @return void
     */
    public static function renderOuter(array $tabs, string $currentTab): void
    {
        echo '<div class="itf-tabs">';
        echo '<h2 class="itf-tab-wrapper">';
        
        foreach ($tabs as $key => $config) {
            // Skip tab if condition is false (for license-gated features)
            if (isset($config['condition']) && !$config['condition']) {
                continue;
            }
            
            $active = $currentTab === $key ? 'itf-tab-active' : '';
            $url = esc_url($config['url']);
            $label = esc_html($config['label']);
            $icon = $config['icon'] ?? '';
            
            // Build tab content with optional icon
            $tabContent = '';
            if (!empty($icon)) {
                $tabContent .= '<span class="itf-tab-icon">' . $icon . '</span>';
            }
            $tabContent .= '<span class="itf-tab-label">' . $label . '</span>';
            
            echo sprintf(
                '<a href="%s" class="itf-tab %s">%s</a>',
                $url,
                $active,
                $tabContent
            );
        }
        
        echo '</h2>';
    }
    
    /**
     * Close outer tabs container
     * 
     * @return void
     */
    public static function closeOuter(): void
    {
        echo '</div>'; // .itf-tabs
    }
    
    /**
     * Render inner tabs (secondary navigation)
     * 
     * @param array $tabs Array of tab configurations [key => ['url' => string, 'label' => string, 'icon' => string, 'condition' => bool]]
     * @param string $currentTab Currently active tab key
     * @return void
     */
    public static function renderInner(array $tabs, string $currentTab): void
    {
        echo '<h2 class="itf-inner-tab-wrapper">';
        
        foreach ($tabs as $key => $config) {
            // Skip tab if condition is false (for license-gated features)
            if (isset($config['condition']) && !$config['condition']) {
                continue;
            }
            
            $active = $currentTab === $key ? 'itf-inner-tab-active' : '';
            $url = esc_url($config['url']);
            $label = esc_html($config['label']);
            $icon = $config['icon'] ?? '';
            
            // Build tab content with optional icon
            $tabContent = '';
            if (!empty($icon)) {
                $tabContent .= '<span class="itf-tab-icon">' . $icon . '</span>';
            }
            $tabContent .= '<span class="itf-tab-label">' . $label . '</span>';
            
            echo sprintf(
                '<a href="%s" class="itf-inner-tab %s">%s</a>',
                $url,
                $active,
                $tabContent
            );
        }
        
        echo '</h2>';
    }
    
    /**
     * Open tab content wrapper
     * 
     * @return void
     */
    public static function openContent(): void
    {
        echo '<div class="intucart-tab-content">';
    }
    
    /**
     * Close tab content wrapper
     * 
     * @return void
     */
    public static function closeContent(): void
    {
        echo '</div>'; // .intucart-tab-content
    }
}

