<?php

declare(strict_types=1);

namespace Intucart\DI;

use Intucart\Admin\AdminController;
use Intucart\Admin\AdminAssets;
use Intucart\Admin\AdminNotices;
use Intucart\Admin\Ajax\SyncAjaxHandler;
use Intucart\Admin\Ajax\ContentTypesAjaxHandler;
use Intucart\Admin\Ajax\SystemAjaxHandler;
use Intucart\Admin\Tabs\LicenseTabRenderer;
use Intucart\Admin\Tabs\SearchTabRenderer;
use Intucart\Admin\Tabs\WooCommerceTabRenderer;
use Intucart\Admin\Tabs\ChatbotTabRenderer;
use Intucart\Admin\Tabs\WebhooksTabRenderer;
use Intucart\Admin\Tabs\IndexingTabRenderer;
use Intucart\Pages\Settings\ChatSettings;
use Intucart\Pages\Settings\EcommerceSettings;
use Intucart\Pages\Settings\SearchSettings;
use Intucart\Pages\Settings\SearchWidgetSettings;
use Intucart\Pages\Settings\SyncSettings;
use Intucart\Pages\Settings\WebhookSettings;
use Intucart\Pages\Prompts;
use Intucart\Pages\Status;
use Intucart\Pages\Analytics;
use Intucart\Services\AIClientManager;
use Intucart\Services\Cache\CacheService;
use Intucart\Services\Licensing\License;
use Intucart\Services\Logger;
use Intucart\Services\Managers\CronManager;
use Intucart\Services\Managers\PostTypeManager;
use Intucart\Services\Managers\TaxonomyManager;
use Intucart\Services\UsageService;
use Psr\Container\ContainerInterface;

/**
 * Service Provider for Admin layer services.
 * Includes AdminController, AJAX handlers, and Tab renderers.
 */
class AdminServicesProvider
{
    /**
     * Returns the definitions for the container.
     *
     * @return array Definitions for the container.
     */
    public function getDefinitions(): array
    {
        return [
            // === ADMIN CORE ===

            AdminAssets::class => \DI\create()
                ->constructor(\DI\get(Logger::class)),

            AdminNotices::class => \DI\create(),

            // === AJAX HANDLERS ===

            SyncAjaxHandler::class => \DI\create()
                ->constructor(
                    \DI\get(Logger::class),
                    \DI\get(PostTypeManager::class),
                    \DI\get(TaxonomyManager::class),
                    \DI\get(UsageService::class)
                ),

            ContentTypesAjaxHandler::class => \DI\create()
                ->constructor(
                    \DI\get(Logger::class),
                    \DI\get(PostTypeManager::class),
                    \DI\get(TaxonomyManager::class),
                    \DI\get(CronManager::class)
                ),

            SystemAjaxHandler::class => \DI\create()
                ->constructor(
                    \DI\get(Logger::class),
                    \DI\get(CacheService::class),
                    \DI\get(UsageService::class)
                ),

            // === TAB RENDERERS ===

            LicenseTabRenderer::class => \DI\create()
                ->constructor(
                    \DI\get(Logger::class),
                    \DI\get(License::class),
                    \DI\get(AIClientManager::class),
                    \DI\get(UsageService::class)
                ),

            SearchTabRenderer::class => \DI\create()
                ->constructor(
                    \DI\get(Logger::class),
                    \DI\get(SearchSettings::class),
                    \DI\get(SearchWidgetSettings::class)
                ),

            WooCommerceTabRenderer::class => \DI\factory(function (ContainerInterface $c) {
                $license = $c->get(License::class);
                $recommendationsAvailable = $license->hasRecommendations();

                return new WooCommerceTabRenderer(
                    $c->get(Logger::class),
                    $recommendationsAvailable
                );
            }),

            ChatbotTabRenderer::class => \DI\create()
                ->constructor(
                    \DI\get(Logger::class),
                    \DI\get(AIClientManager::class),
                    \DI\get(ChatSettings::class),
                    \DI\get(Prompts::class)
                ),

            WebhooksTabRenderer::class => \DI\create()
                ->constructor(
                    \DI\get(Logger::class),
                    \DI\get(WebhookSettings::class)
                ),

            IndexingTabRenderer::class => \DI\create()
                ->constructor(
                    \DI\get(Logger::class),
                    \DI\get(PostTypeManager::class),
                    \DI\get(TaxonomyManager::class)
                ),

            // === SETTINGS PAGES ===

            EcommerceSettings::class => \DI\factory(function (ContainerInterface $c) {
                $license = $c->get(License::class);
                $recommendationsAvailable = $license->hasRecommendations();

                return new EcommerceSettings(
                    $c->get(Logger::class),
                    $license,
                    $recommendationsAvailable
                );
            }),

            SyncSettings::class => \DI\create()
                ->constructor(\DI\get(Logger::class)),

            SearchSettings::class => \DI\create()
                ->constructor(\DI\get(Logger::class)),

            SearchWidgetSettings::class => \DI\create()
                ->constructor(\DI\get(Logger::class)),

            WebhookSettings::class => \DI\create()
                ->constructor(
                    \DI\get(Logger::class),
                    \DI\get(AIClientManager::class)
                ),

            ChatSettings::class => \DI\create()
                ->constructor(
                    \DI\get(Logger::class),
                    \DI\get(AIClientManager::class)
                ),

            Prompts::class => \DI\create()
                ->constructor(
                    \DI\get(Logger::class),
                    \DI\get(AIClientManager::class),
                    \DI\get(CacheService::class)
                ),

            // === ADMIN CONTROLLER ===

            AdminController::class => \DI\create()
                ->constructor(
                    \DI\get(Logger::class),
                    \DI\get(AdminAssets::class),
                    \DI\get(AdminNotices::class),
                    \DI\get(SyncAjaxHandler::class),
                    \DI\get(ContentTypesAjaxHandler::class),
                    \DI\get(SystemAjaxHandler::class),
                    \DI\get(LicenseTabRenderer::class),
                    \DI\get(SearchTabRenderer::class),
                    \DI\get(WooCommerceTabRenderer::class),
                    \DI\get(ChatbotTabRenderer::class),
                    \DI\get(WebhooksTabRenderer::class),
                    \DI\get(IndexingTabRenderer::class),
                    \DI\get(EcommerceSettings::class),
                    \DI\get(SyncSettings::class),
                    \DI\get(SearchSettings::class),
                    \DI\get(WebhookSettings::class),
                    \DI\get(ChatSettings::class),
                    \DI\get(Status::class),
                    \DI\get(Analytics::class)
                ),
        ];
    }
}

