<?php

namespace Intucart\DI;

use DI\Container;
use Intucart\Services\Logger;
use Intucart\Services\Cache\CacheService;
use Intucart\Services\Licensing\License;
use Intucart\Services\Licensing\PluginUpdater;
use Intucart\Services\AIClientManager;
use Intucart\Services\AcfAutoMapper;
use Psr\Container\ContainerInterface;
use Intucart\Services\Providers\PostTypes\PostTypeProviderInterface;
use Intucart\Services\Providers\PostTypes\WooCommerceProductProvider;
use Intucart\Services\Providers\PostTypes\PostTypeProvider;
use Intucart\Services\Managers\PostTypeManager;
use Intucart\Services\UsageService;
use Intucart\Services\PageBuilderService;
use Intucart\Services\ThreadService;
use Intucart\Pages\Analytics;

/**
 * Service Provider for Core application services and infrastructure.
 * Includes foundational services, factories, and interface mappings.
 */
class CoreServicesProvider
{
    /**
     * Returns the definitions for the container.
     *
     * @return array Definitions for the container.
     */
    public function getDefinitions(): array
    {
        return [
            // === FOUNDATIONAL SERVICES ===

            // Logger service (usually simple creation)
            Logger::class => \DI\create(),

            // ACF Auto Mapper service
            AcfAutoMapper::class => \DI\create()
                ->constructor(\DI\get(Logger::class)),

            // Page Builder Service for cache busting and integrations
            PageBuilderService::class => \DI\create()
                ->constructor(\DI\get(Logger::class)),

            // Make the container itself available (handled by Plugin class)
            // Container::class is defined in Plugin::getContainerDefinitions()
            ContainerInterface::class => \DI\get(Container::class), // Alias

            // Cache service
            CacheService::class => \DI\create(),

            // Core business services
            License::class => \DI\create()
                ->constructor(
                    \DI\get(CacheService::class),
                    \DI\get(Logger::class),
                    \DI\get(AIClientManager::class)
                ),

            // Plugin updater for self-hosted updates
            PluginUpdater::class => \DI\create()
                ->constructor(\DI\get(Logger::class)),

            Analytics::class => \DI\create()
                ->constructor(\DI\get(Logger::class)),

            // AI Client Manager for centralized SDK management
            AIClientManager::class => \DI\create()
                ->constructor(\DI\get(CacheService::class)),

            // Usage service for tracking limits and feature availability
            UsageService::class => \DI\create()
                ->constructor(
                    \DI\get(Logger::class),
                    \DI\get(CacheService::class),
                    \DI\get(AIClientManager::class)
                ),

            // === PROVIDER FACTORIES & INTERFACES ===

            // Product providers
            WooCommerceProductProvider::class => \DI\create()
                ->constructor(
                    \DI\get(Logger::class),
                    \DI\get(CacheService::class),
                    \DI\get(AcfAutoMapper::class)
                ),

            PostTypeProvider::class => \DI\create()
                ->constructor(
                    \DI\get(Logger::class),
                    \DI\get(CacheService::class),
                    \DI\get(AcfAutoMapper::class)
                ),

            // Interface mapping for PostTypeProviderInterface
            PostTypeProviderInterface::class => \DI\get(WooCommerceProductProvider::class),

            // PostTypeManager with unified functionality (replaces ProductManager)
            PostTypeManager::class => \DI\create()
                ->constructor(
                    \DI\get(Logger::class),
                    \DI\get(AIClientManager::class),
                    \DI\get(CacheService::class),
                    \DI\get(WooCommerceProductProvider::class),
                    \DI\get(PostTypeProvider::class),
                    \DI\get(License::class),
                    \DI\get(PageBuilderService::class)
                ),

            // ThreadService for conversation management
            ThreadService::class => \DI\create()
                ->constructor(
                    \DI\get(Logger::class),
                    \DI\get(AIClientManager::class),
                    \DI\get(CacheService::class),
                    \DI\get(License::class)
                ),

            // Analytics page with ThreadService
            Analytics::class => \DI\create()
                ->constructor(
                    \DI\get(Logger::class),
                    \DI\get(License::class),
                    \DI\get(UsageService::class),
                    \DI\get(ThreadService::class)
                ),
        ];
    }
}
