<?php

namespace Intucart\Integration;

use Intucart\Services\Recommendations as RecommendationsService;
use Intucart\Services\Logger;
use Intucart\Services\Analytics;
use Intucart\Services\Licensing\License;
use Intucart\Services\Constants;

/**
 * WooCommerce Recommendations Override
 *
 * This class overrides WooCommerce's native recommendation system
 * by replacing the default product query logic with AI-driven recommendations.
 */
class RecommendationsOverride
{
    private RecommendationsService $recommendations;
    private Logger $logger;
    private Analytics $analytics;
    private License $license;

    /**
     * Constructor
     *
     * @param RecommendationsService $recommendations Recommendations service
     * @param Logger                 $logger          Logger service
     * @param Analytics              $analytics       Analytics service
     * @param License                $license         License service
     */
    public function __construct(
        RecommendationsService $recommendations,
        Logger $logger,
        Analytics $analytics,
        License $license
    ) {
        $this->recommendations = $recommendations;
        $this->logger = $logger;
        $this->analytics = $analytics;
        $this->license = $license;
    }

    /**
     * Initialize the service
     *
     * @return void
     */
    public function initialize(): void
    {
        // Check if recommendations feature is available
        if ($this->license->hasRecommendations()) {
            $this->registerFilters();
            $this->logger->debug('Recommendations override initialized successfully', [
                'tier' => $this->license->getTierName(),
                'has_recommendations' => true
            ]);
        } else {
            $this->logger->info('Recommendations feature not available for current tier', [
                'tier' => $this->license->getTierName(),
                'has_recommendations' => false
            ]);
        }
    }

    /**
     * Register WooCommerce filters
     *
     * @return void
     */
    private function registerFilters(): void
    {
        // Related products
        if (get_option(Constants::ENABLE_RELATED_PRODUCTS_OPTION, true)) {
            add_filter('woocommerce_related_products', [$this, 'getRelatedProducts'], 10, 3);
            add_filter('woocommerce_get_related_products_args', [$this, 'modifyRelatedProductsArgs'], 10, 1);
        }

        // Upsells
        if (get_option(Constants::ENABLE_UPSELLS_OPTION, true)) {
            add_filter('woocommerce_product_get_upsell_ids', [$this, 'getUpsellProducts'], 10, 2);
        }

        // Cross-sells
        if (get_option(Constants::ENABLE_CROSS_SELLS_OPTION, true)) {
            add_filter('woocommerce_product_get_cross_sell_ids', [$this, 'getCrossSellProducts'], 10, 2);
        }
    }

    /**
     * Check if results indicate that fallback is needed
     *
     * @param array $results Recommendations results
     * @return bool True if fallback is needed
     */
    private function needsFallback(array $results): bool
    {
        return isset($results['fallback_needed']) && $results['fallback_needed'];
    }

    /**
     * Get detailed fallback message based on the reason
     *
     * @param string $reason Fallback reason
     * @param array  $results Full results array with potential additional data
     * @return string Detailed message for logging
     */
    private function getFallbackMessage(string $reason, array $results = []): string
    {
        switch ($reason) {
            case 'usage_limit_exceeded':
                return 'AI recommendations unavailable: monthly usage limit exceeded. Using native WooCommerce recommendations.';

            case 'recommendations_disabled':
                return 'AI recommendations not available for current tier. Using native WooCommerce recommendations.';

            case 'invalid_license':
                return 'AI recommendations unavailable: invalid license key. Using native WooCommerce recommendations.';

            case 'network_error':
                return 'AI recommendations unavailable: network connectivity issue. Using native WooCommerce recommendations.';

            case 'api_error':
                return 'AI recommendations unavailable: cloud service error. Using native WooCommerce recommendations.';

            case 'technical_error':
                return 'AI recommendations unavailable: technical error. Using native WooCommerce recommendations.';

            case 'multiple_products_not_supported':
                return 'AI recommendations unavailable: multiple product recommendations not yet supported. Using native WooCommerce recommendations.';

            default:
                return 'AI recommendations unavailable. Using native WooCommerce recommendations.';
        }
    }

    /**
     * Log fallback with appropriate level based on reason
     *
     * @param string $context Context (e.g., 'related', 'upsell', 'cross-sell')
     * @param int    $productId Product ID
     * @param array  $results Recommendation results
     * @return void
     */
    private function logFallback(string $context, int $productId, array $results): void
    {
        $reason = $results['reason'] ?? 'unknown';
        $message = $this->getFallbackMessage($reason, $results);

        // Log at different levels based on severity
        switch ($reason) {
            case 'usage_limit_exceeded':
            case 'recommendations_disabled':
            case 'invalid_license':
                // These are important business/licensing issues
                $this->logger->warning(ucfirst($context) . ' products fallback: ' . $message, [
                    'product_id' => $productId,
                    'reason' => $reason,
                    'tier_data' => $results['tier_data'] ?? null,
                    'usage_data' => $results['usage_data'] ?? null,
                    'error_message' => $results['error_message'] ?? null
                ]);
                break;

            case 'network_error':
            case 'api_error':
                // These are technical issues that should be monitored
                $this->logger->error(ucfirst($context) . ' products fallback: ' . $message, [
                    'product_id' => $productId,
                    'reason' => $reason,
                    'error_message' => $results['error_message'] ?? null
                ]);
                break;

            case 'technical_error':
            case 'multiple_products_not_supported':
            default:
                // These are expected or less critical issues
                $this->logger->info(ucfirst($context) . ' products fallback: ' . $message, [
                    'product_id' => $productId,
                    'reason' => $reason,
                    'error_message' => $results['error_message'] ?? null
                ]);
                break;
        }
    }

    /**
     * Get related products using AI recommendations with contextual enhancements
     *
     * @param array $related_product_ids Default related product IDs
     * @param int   $product_id          Current product ID
     * @param array $args                Query arguments
     *
     * @return array
     */
    public function getRelatedProducts($related_product_ids, $product_id, $args): array
    {
        if (is_admin()) {
            return $related_product_ids;
        }

        // Quick runtime check for license status (in case it changed since initialization)
        if (!$this->license->hasRecommendations()) {
            $this->logger->debug('Skipping AI recommendations: feature not available', [
                'product_id' => $product_id,
                'context' => 'related_products'
            ]);
            return $related_product_ids;
        }

        $product = wc_get_product($product_id);
        if (!$product) {
            $this->logger->info('Skipping related products: product not found', ['product_id' => $product_id]);
            return $related_product_ids; // Return original IDs if product not found
        }

        $max_items = $args['posts_per_page'] ?? 4;

        try {
            $recommendations = $this->recommendations->getRecommendations(
                $product_id,
                get_current_user_id(),
                Constants::DEFAULT_SIMILAR_PRODUCTS_WEIGHT,
                Constants::DEFAULT_BASKET_AFFINITY_WEIGHT,
                Constants::DEFAULT_USER_SIMILARITY_WEIGHT,
                $max_items
            );

            // Check if fallback is needed - let WooCommerce handle natively
            if ($this->needsFallback($recommendations)) {
                $this->logFallback('related', $product_id, $recommendations);
                return $related_product_ids;
            }

            // Track impressions with enhanced context
            foreach ($recommendations as $recommendation) {
                $this->analytics->trackRecommendationImpression(
                    $product_id,
                    (int) $recommendation['id'],
                    'related',
                    $recommendation['score'],
                    $this->getPageContext()
                );
            }

            return array_map(function ($item) {
                return $item['id'];
            }, $recommendations);
        } catch (\Exception $e) {
            $this->logger->error('Failed to get AI related product recommendations', [
                'product_id' => $product_id,
                'error' => $e->getMessage()
            ]);
            // Return original related products if AI recommendations fail
            return $related_product_ids;
        }
    }

    /**
     * Get current page context for analytics
     *
     * @return array Page context data
     */
    private function getPageContext(): array
    {
        return [
            'page_type' => $this->getCurrentPageType(),
            'referrer' => wp_get_referer(),
            'user_agent' => isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : '',
            'timestamp' => current_time('mysql'),
            'session_id' => WC()->session->get_customer_id(),
        ];
    }

    /**
     * Get current page type
     *
     * @return string Page type
     */
    private function getCurrentPageType(): string
    {
        if (is_product()) {
            return 'product';
        } elseif (is_shop() || is_product_category() || is_product_tag()) {
            return 'catalog';
        } elseif (is_cart()) {
            return 'cart';
        } elseif (is_checkout()) {
            return 'checkout';
        } elseif (is_account_page()) {
            return 'account';
        }
        return 'other';
    }

    /**
     * Modify related products query arguments
     *
     * @param array $args Query arguments
     *
     * @return array
     */
    public function modifyRelatedProductsArgs($args): array
    {
        // Remove category and tag constraints since we're using AI
        unset($args['tax_query']);
        return $args;
    }

    /**
     * Get upsell products using AI recommendations
     *
     * @param array       $upsell_ids Default upsell IDs
     * @param \WC_Product $product    Current product
     *
     * @return array
     */
    public function getUpsellProducts($upsell_ids, $product): array
    {
        if (is_admin()) {
            return $upsell_ids;
        }

        $product_obj = is_object($product) ? $product : wc_get_product($product);
        if (!$product_obj) {
            $this->logger->info('Skipping upsell products: product not found', ['product_id' => is_object($product) ? $product->get_id() : $product]);
            return $upsell_ids; // Return original IDs if product not found
        }

        $max_items = apply_filters('woocommerce_upsells_total', 4);
        $final_upsells = array_slice($upsell_ids, 0, $max_items);
        $remaining_slots = $max_items - count($final_upsells);

        if ($remaining_slots > 0) {
            try {
                $recommendations = $this->recommendations->getRecommendations(
                    $product_obj->get_id(),
                    get_current_user_id(),
                    Constants::DEFAULT_SIMILAR_PRODUCTS_WEIGHT,
                    Constants::DEFAULT_BASKET_AFFINITY_WEIGHT,
                    Constants::DEFAULT_USER_SIMILARITY_WEIGHT,
                    $remaining_slots
                );

                // Check if fallback is needed - let WooCommerce handle natively
                if ($this->needsFallback($recommendations)) {
                    $this->logFallback('upsell', $product_obj->get_id(), $recommendations);
                    return $upsell_ids;
                }

                // Track impressions with enhanced context
                foreach ($recommendations as $recommendation) {
                    $this->analytics->trackRecommendationImpression(
                        $product_obj->get_id(),
                        (int) $recommendation['id'], // Cast to int to ensure type safety
                        'upsell',
                        $recommendation['score'],
                        $this->getPageContext()
                    );
                }

                $ai_upsells = array_map(function ($item) {
                    return $item['id'];
                }, $recommendations);

                $ai_upsells = array_diff($ai_upsells, $final_upsells);
                $final_upsells = array_merge($final_upsells, array_slice($ai_upsells, 0, $remaining_slots));
            } catch (\Exception $e) {
                $this->logger->error('Failed to get AI upsell recommendations', [
                    'product_id' => $product_obj->get_id(),
                    'error' => $e->getMessage()
                ]);
                // Return original upsells if AI recommendations fail
                return $upsell_ids;
            }
        }

        return $final_upsells;
    }

    /**
     * Get cross-sell products using AI recommendations
     *
     * @param array       $cross_sell_ids Default cross-sell IDs
     * @param \WC_Product $product       Current product
     *
     * @return array
     */
    public function getCrossSellProducts($cross_sell_ids, $product): array
    {
        if (is_admin()) {
            return $cross_sell_ids;
        }

        $product_obj = is_object($product) ? $product : wc_get_product($product);
        if (!$product_obj) {
            $this->logger->info(
                'Skipping cross-sell products: product not found',
                ['product_id' => is_object($product) ? $product->get_id() : $product]
            );
            return $cross_sell_ids; // Return original IDs if product not found
        }

        $max_items = apply_filters('woocommerce_cross_sells_total', 4);

        // Preserve existing cross-sells first, similar to upsells
        $final_cross_sells = array_slice($cross_sell_ids, 0, $max_items);
        $remaining_slots = $max_items - count($final_cross_sells);

        // Only get AI recommendations if we have remaining slots
        if ($remaining_slots > 0) {
            try {
                $recommendations = $this->recommendations->getRecommendations(
                    $product_obj->get_id(),
                    get_current_user_id(),
                    Constants::DEFAULT_SIMILAR_PRODUCTS_WEIGHT,
                    Constants::DEFAULT_BASKET_AFFINITY_WEIGHT,
                    Constants::DEFAULT_USER_SIMILARITY_WEIGHT,
                    $remaining_slots
                );

                // Check if fallback is needed - let WooCommerce handle natively
                if ($this->needsFallback($recommendations)) {
                    $this->logFallback('cross-sell', $product_obj->get_id(), $recommendations);
                    return $cross_sell_ids;
                }

                // Track impressions with enhanced context
                foreach ($recommendations as $recommendation) {
                    $this->analytics->trackRecommendationImpression(
                        $product_obj->get_id(),
                        $recommendation['id'],
                        'cross-sell',
                        $recommendation['score'],
                        $this->getPageContext()
                    );
                }

                $ai_cross_sells = array_map(function ($item) {
                    return $item['id'];
                }, $recommendations);

                // Avoid duplicates and merge with existing cross-sells
                $ai_cross_sells = array_diff($ai_cross_sells, $final_cross_sells);
                $final_cross_sells = array_merge($final_cross_sells, array_slice($ai_cross_sells, 0, $remaining_slots));
            } catch (\Exception $e) {
                $this->logger->error('Failed to get AI cross-sell recommendations', [
                    'product_id' => $product_obj->get_id(),
                    'error' => $e->getMessage()
                ]);
                // Return original cross-sells if AI recommendations fail
                return $cross_sell_ids;
            }
        }

        return $final_cross_sells;
    }
}
