<?php

namespace Intucart\Integration;

use Intucart\Services\Search;
use Intucart\Services\Logger;
use Intucart\Services\Constants;

/**
 * WordPress and WooCommerce search override class
 * Provides semantic search capabilities for all content types:
 * - WordPress posts and pages (via /posts/search endpoint)
 * - WooCommerce products (via /products/search endpoint)
 * - Mixed content searches with intelligent result ranking
 *
 * php version 8.1
 *
 * @category Integration
 * @package  Intucart
 * @author   Intucart <support@intufind.com>
 * @license  GPL-2.0+ https://www.gnu.org/licenses/gpl-2.0.html
 * @link     https://intufind.com
 */
class SearchOverride
{
    private Search $search;
    private Logger $logger;

    /**
     * Constructor
     *
     * @param Search $search Search service
     * @param Logger $logger Logger service
     */
    public function __construct(Search $search, Logger $logger)
    {
        $this->search = $search;
        $this->logger = $logger;
    }

    /**
     * Initialize hooks
     *
     * @return void
     */
    public function initialize(): void
    {
        $semantic_search_enabled = get_option(Constants::ENABLE_SEMANTIC_SEARCH_OPTION, true);
        $woocommerce_available = class_exists('WooCommerce');

        // Debug logging
        $this->logger->info('SearchOverride initialization', [
            'semantic_search_enabled' => $semantic_search_enabled,
            'woocommerce_available' => $woocommerce_available
        ]);

        if ($semantic_search_enabled) {
            // Hook into general WordPress search for all content types
            add_filter('pre_get_posts', [$this, 'modifyGeneralSearch'], 999, 1);

            // If WooCommerce is available, also hook into WooCommerce-specific product queries
            if ($woocommerce_available) {
                add_filter('woocommerce_product_query', [$this, 'modifyProductQuery'], 999, 1);
            }
        }
    }

    /**
     * Extract common data from search result (DRY helper)
     *
     * @param object $entity Search result entity from AI SDK (Product or Post object)
     * @return array Extracted data with id, score, and title
     */
    private function extractResultData($entity): array
    {
        $id = (int)$entity->id;
        $score = $entity->score ?? 0;

        // Products have 'name', posts have 'title'
        $title = $entity->name ?? $entity->title ?? "Item #$id";

        return [
            'id' => $id,
            'score' => (float)$score,
            'title' => $title
        ];
    }

    /**
     * Check if results indicate that fallback is needed
     *
     * @param array $results Search results
     * @return bool True if fallback is needed
     */
    private function needsFallback(array $results): bool
    {
        return isset($results['fallback_needed']) && $results['fallback_needed'];
    }

    /**
     * Modify the WooCommerce product query directly
     *
     * @param \WC_Query $query WooCommerce query object
     *
     * @return \WC_Query
     */
    public function modifyProductQuery($query)
    {
        // Debug logging
        $this->logger->info('modifyProductQuery called', [
            'has_search_term' => isset($query->query_vars['s']) && !empty($query->query_vars['s']),
            'search_term' => $query->query_vars['s'] ?? 'none',
            'query_vars' => array_keys($query->query_vars)
        ]);

        // Only process if this is a search query
        if (!isset($query->query_vars['s']) || empty($query->query_vars['s'])) {
            return $query;
        }

        $search_term = $query->query_vars['s'];

        try {
            // Use injected search service to perform semantic search
            $this->logger->info('Calling searchPostTypes for product search', [
                'search_term' => $search_term,
                'limit' => 10,
                'type' => 'product'
            ]);

            // Check if enhanced AI search is enabled
            $useAgent = get_option(Constants::ENABLE_ENHANCED_AI_SEARCH_OPTION, false);
            
            $results = $this->search->searchPostTypes($search_term, 10, [], false, 'product', $useAgent);

            $this->logger->info('Product search results received', [
                'result_count' => count($results),
                'search_term' => $search_term,
                'fallback_needed' => $this->needsFallback($results)
            ]);

            // Check if fallback is needed
            if ($this->needsFallback($results)) {
                $this->logger->info('Product search needs fallback, letting native WooCommerce search handle it', [
                    'search_term' => $search_term,
                    'fallback_reason' => $results['fallback_reason'] ?? 'unknown'
                ]);
                return $query; // Don't modify the query, let native search handle it
            }

            if (empty($results)) {
                $this->logger->info('No product search results found, returning original query');
                return $query;
            }

            // Process results into a single structure
            $search_data = [];
            $result_log = [];

            foreach ($results as $result) {
                $data = $this->extractResultData($result);

                if (!isset($search_data['product_ids'])) {
                    $search_data['product_ids'] = [];
                }
                $search_data['product_ids'][] = $data['id'];
                $search_data['scores'][$data['id']] = $data['score'];

                // Add to result log without querying the database
                $result_log[] = $data;
            }

            // Sort results by score (highest first)
            usort($result_log, function ($a, $b) {
                return $b['score'] <=> $a['score'];
            });

            // Log the search results with scores (only when in debug mode)
            $this->logger->info('Semantic Search Results for: "' . $search_term . '"', [
                'results' => $result_log
            ]);

            // If we have product IDs, use them to filter the query
            if (!empty($search_data['product_ids'])) {
                // Store the original search term for display purposes
                $query->query_vars['original_search_term'] = $search_term;

                // Set post__in to limit results to these products
                $query->query_vars['post__in'] = $search_data['product_ids'];

                // We need to clear the search term to prevent default search,
                // but we'll add a filter to restore it for display purposes
                add_filter('the_search_query', [$this, 'restoreSearchTerm']);
                add_filter('get_search_query', [$this, 'restoreSearchTerm']);

                // Clear the search term to prevent default search
                $query->query_vars['s'] = '';

                // Set orderby to preserve our custom ordering
                $query->query_vars['orderby'] = 'post__in';
            }

            // Store search data for potential use by other functions
            $query->query_vars['intucart_search_data'] = $search_data;
        } catch (\Exception $e) {
            // Log error with full details
            $this->logger->error('Semantic Search Error: ' . $e->getMessage());
        }

        return $query;
    }

    /**
     * Modify general WordPress search queries to include semantic search
     * This handles site-wide searches that aren't specifically WooCommerce product searches
     *
     * @param \WP_Query $query WordPress query object
     *
     * @return void
     */
    public function modifyGeneralSearch($query)
    {
        // Only modify main query search on frontend, not admin or product-specific searches
        if (is_admin() || !$query->is_main_query() || !$query->is_search()) {
            return;
        }

        $search_term = $query->get('s');
        if (empty($search_term)) {
            return;
        }

        // Handle product searches here instead of relying on woocommerce_product_query
        // which may not trigger for search queries
        if (isset($query->query_vars['post_type']) && $query->query_vars['post_type'] === 'product') {
            // Let modifyProductQuery handle product-only searches
            return;
        }

        try {
            // Determine what to search based on post_type
            $post_types = $query->get('post_type');

            // Check if mixed content search is enabled
            $mixed_content_enabled = get_option(Constants::ENABLE_MIXED_CONTENT_SEARCH_OPTION, true);

            // If no specific post type set, search both products and posts (if enabled)
            if ((empty($post_types) || $post_types === 'any') && $mixed_content_enabled) {
                $this->handleMixedContentSearch($query, $search_term);
            } elseif (is_array($post_types) && in_array('product', $post_types) && $mixed_content_enabled) {
                // Multiple post types including products
                $this->handleMixedContentSearch($query, $search_term);
            } elseif ($post_types === 'product') {
                // Product-only search - let modifyProductQuery handle this
                return;
            } else {
                // Non-product post types (posts, pages, custom post types)
                $this->handlePostSearch($query, $search_term);
            }
        } catch (\Exception $e) {
            $this->logger->error('General semantic search error: ' . $e->getMessage());
        }
    }

    /**
     * Handle mixed content search (products + posts)
     *
     * @param \WP_Query $query
     * @param string $search_term
     * @return void
     */
    private function handleMixedContentSearch($query, $search_term)
    {
        // Check if enhanced AI search is enabled
        $useAgent = get_option(Constants::ENABLE_ENHANCED_AI_SEARCH_OPTION, false);
        
        // Search both products and posts
        $productResults = [];
        if (class_exists('WooCommerce')) {
            $productResults = $this->search->searchPostTypes($search_term, 5, [], false, 'product', $useAgent);
        }
        $postResults = $this->search->searchPostTypes($search_term, 5, [], false, 'post', $useAgent);

        // Check if either search needs fallback
        $productFallback = $this->needsFallback($productResults);
        $postFallback = $this->needsFallback($postResults);

        if ($productFallback || $postFallback) {
            $this->logger->info('Mixed content search needs fallback, letting native search handle it', [
                'search_term' => $search_term,
                'product_fallback' => $productFallback,
                'post_fallback' => $postFallback,
                'product_reason' => $productResults['fallback_reason'] ?? 'unknown',
                'post_reason' => $postResults['fallback_reason'] ?? 'unknown'
            ]);
            return; // Don't modify the query, let native search handle it
        }

        $allResults = [];
        $allIds = [];

        // Process product results
        foreach ($productResults as $result) {
            $data = $this->extractResultData($result);
            $allResults[] = [
                'id' => $data['id'],
                'type' => 'product',
                'score' => $data['score']
            ];
            $allIds[] = $data['id'];
        }

        // Process post results
        foreach ($postResults as $result) {
            $data = $this->extractResultData($result);
            $allResults[] = [
                'id' => $data['id'],
                'type' => 'post',
                'score' => $data['score']
            ];
            $allIds[] = $data['id'];
        }

        if (!empty($allIds)) {
            // Sort by score
            usort($allResults, function ($a, $b) {
                return $b['score'] <=> $a['score'];
            });

            $this->logger->info('Mixed content semantic search results for: "' . $search_term . '"', [
                'results' => $allResults
            ]);

            // Set post__in to include all results
            $query->set('post__in', $allIds);

            // Set post types based on what we found
            $post_types = ['post', 'page'];
            if (class_exists('WooCommerce') && !empty($productResults)) {
                $post_types[] = 'product';
            }
            $query->set('post_type', $post_types);

            $query->set('orderby', 'post__in'); // Preserve semantic search order
            $query->set('s', ''); // Clear search term to prevent default search

            // Store original search term for display
            $query->set('original_search_term', $search_term);
            add_filter('the_search_query', [$this, 'restoreSearchTerm']);
            add_filter('get_search_query', [$this, 'restoreSearchTerm']);
        }
    }

    /**
     * Handle post-only search (non-products)
     *
     * @param \WP_Query $query
     * @param string $search_term
     * @return void
     */
    private function handlePostSearch($query, $search_term)
    {
        // Check if enhanced AI search is enabled
        $useAgent = get_option(Constants::ENABLE_ENHANCED_AI_SEARCH_OPTION, false);
        
        $results = $this->search->searchPostTypes($search_term, 10, [], false, 'post', $useAgent);

        // Check if fallback is needed
        if ($this->needsFallback($results)) {
            $this->logger->info('Post search needs fallback, letting native search handle it', [
                'search_term' => $search_term,
                'fallback_reason' => $results['fallback_reason'] ?? 'unknown'
            ]);
            return; // Don't modify the query, let native search handle it
        }

        if (empty($results)) {
            $this->logger->info('No semantic search results found for post search', [
                'search_term' => $search_term
            ]);
            return;
        }

        $postIds = [];
        $resultLog = [];

        foreach ($results as $result) {
            $data = $this->extractResultData($result);

            $postIds[] = $data['id'];
            $resultLog[] = $data;
        }

        if (!empty($postIds)) {
            $this->logger->info('Post semantic search results for: "' . $search_term . '"', [
                'results' => $resultLog
            ]);

            $query->set('post__in', $postIds);
            $query->set('orderby', 'post__in');
            $query->set('s', '');

            // Store original search term for display
            $query->set('original_search_term', $search_term);
            add_filter('the_search_query', [$this, 'restoreSearchTerm']);
            add_filter('get_search_query', [$this, 'restoreSearchTerm']);
        }
    }

    /**
     * Restore the original search term for display purposes
     *
     * @param string $search_term The current search term
     *
     * @return string The original search term
     */
    public function restoreSearchTerm($search_term): string
    {
        global $wp_query;

        if (isset($wp_query->query_vars['original_search_term'])) {
            return $wp_query->query_vars['original_search_term'];
        }

        return $search_term;
    }
}
