<?php

namespace Intucart\Pages;

use Intucart\Services\Logger;
use Intucart\Services\Licensing\License;
use Intucart\Services\UsageService;
use Intucart\Services\ThreadService;
use Intucart\Components\Tabs;

/**
 * Analytics Page
 */
class Analytics
{
    private Logger $logger;
    private License $license;
    private UsageService $usageService;
    private ThreadService $threadService;

    /**
     * Constructor
     *
     * @param Logger  $logger  Logger instance
     * @param License $license License service
     * @param UsageService $usageService Usage service for modern feature checking
     * @param ThreadService $threadService Thread service for conversation management
     */
    public function __construct(Logger $logger, License $license, UsageService $usageService, ThreadService $threadService)
    {
        $this->logger = $logger;
        $this->license = $license;
        $this->usageService = $usageService;
        $this->threadService = $threadService;
    }

    /**
     * Initialize admin hooks and AJAX endpoints
     *
     * @return void
     */
    public function initialize(): void
    {
        // AJAX endpoints for conversation management
        add_action('wp_ajax_intucart_search_conversations', [$this, 'handleAjaxSearchConversations']);
        add_action('wp_ajax_intucart_get_conversation_analytics', [$this, 'handleAjaxGetConversationAnalytics']);
        add_action('wp_ajax_intucart_delete_conversation', [$this, 'handleAjaxDeleteConversation']);
        add_action('wp_ajax_intucart_get_conversation_details', [$this, 'handleAjaxGetConversationDetails']);
        add_action('wp_ajax_intucart_validate_export', [$this, 'handleAjaxValidateExport']);
        add_action('wp_ajax_intucart_export_conversations', [$this, 'handleAjaxExportConversations']);
    }

    /**
     * Render the full analytics page with permissions, page structure, and assets
     *
     * @return void
     */
    public function renderPage(): void
    {
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'intufind'));
        }

        // Enqueue Chart.js (bundled locally for WordPress.org compliance)
        wp_enqueue_script(
            'chart.js',
            INTUCART_PLUGIN_URL . 'assets/js/vendor/chart.min.js',
            [],
            '4.4.1',
            true
        );

        echo '<div class="wrap">';
        
        // Page title and description before tabs
        echo '<h1>';
        echo '<span class="intucart-page-header-icon">';
        echo '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path opacity=".4" d="M512 400v32a16 16 0 0 1-16 16H32a32 32 0 0 1-32-32V80a16 16 0 0 1 16-16h32a16 16 0 0 1 16 16v304h432a16 16 0 0 1 16 16z"/><path d="M480 112v118.05c0 21.38-25.85 32.09-41 17l-32.4-32.4-96 96a32 32 0 0 1-45.25 0L192 237.25l-46.06 46.07a16 16 0 0 1-22.63 0l-22.62-22.62a16 16 0 0 1 0-22.63l68.69-68.69a32 32 0 0 1 45.25 0L288 242.75l73.37-73.38L329 137c-15.12-15.12-4.41-41 17-41h118a16 16 0 0 1 16 16z"/></svg>';
        echo '</span>';
        echo esc_html(get_admin_page_title());
        echo '</h1>';
        echo '<p class="intucart-page-description">' . esc_html__('Monitor and analyze your website\'s AI-powered features including search performance, chat interactions, and user engagement patterns.', 'intufind') . '</p>';
        
        $this->render();
        
        echo '</div>';
    }

    /**
     * Render the analytics dashboard content (used internally)
     *
     * @return void
     */
    public function render(): void
    {
        // Enqueue our custom script that depends on Chart.js
        wp_enqueue_script(
            'intucart-analytics',
            INTUCART_PLUGIN_URL . 'assets/js/admin/analytics.js',
            ['jquery', 'chart.js'],
            filemtime(INTUCART_PLUGIN_DIR . 'assets/js/admin/analytics.js'),
            true
        );

        // Enqueue common admin CSS first (includes tab styles)
        if (!wp_style_is('intucart-admin-common', 'enqueued')) {
            wp_enqueue_style(
                'intucart-admin-common',
                INTUCART_PLUGIN_URL . 'assets/css/admin/common.css',
                [],
                filemtime(INTUCART_PLUGIN_DIR . 'assets/css/admin/common.css')
            );
        }

        // Enqueue component styles
        if (!wp_style_is('intucart-admin-components', 'enqueued')) {
            wp_enqueue_style(
                'intucart-admin-components',
                INTUCART_PLUGIN_URL . 'assets/css/admin/components.css',
                ['intucart-admin-common'],
                filemtime(INTUCART_PLUGIN_DIR . 'assets/css/admin/components.css')
            );
        }

        // Enqueue analytics CSS file
        wp_enqueue_style(
            'intucart-analytics',
            INTUCART_PLUGIN_URL . 'assets/css/admin/analytics.css',
            ['intucart-admin-common'],
            filemtime(INTUCART_PLUGIN_DIR . 'assets/css/admin/analytics.css')
        );

        // Localize script with necessary data for conversations tab
        wp_localize_script(
            'intucart-analytics',
            'intucartAnalytics',
            [
                'ajaxUrl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('intucart_analytics_nonce')
            ]
        );

        // Render tabbed interface
        $this->renderTabbedInterface();
    }

    /**
     * Render the tabbed interface
     *
     * @return void
     */
    private function renderTabbedInterface(): void
    {
        $current_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'dashboard';

        // Check if license is valid to determine if Conversations tab should be shown
        $license_validation = $this->license->validateLicense();
        $license_is_valid = $license_validation['valid'] ?? false;

        // Configure tabs
        $tabs = [
            'dashboard' => [
                'url' => '?page=intucart-analytics&tab=dashboard',
                'label' => __('Dashboard', 'intufind'),
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path opacity=".4" d="M288 160v256h160V160zM64 416h160V160H64z"/><path d="M464 32H48A48 48 0 0 0 0 80v352a48 48 0 0 0 48 48h416a48 48 0 0 0 48-48V80a48 48 0 0 0-48-48zM224 416H64V160h160zm224 0H288V160h160z"/></svg>'
            ],
            'conversations' => [
                'url' => '?page=intucart-analytics&tab=conversations',
                'label' => __('Conversations', 'intufind'),
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 576 512"><path opacity=".4" d="M208 352c-41 0-79.1-9.3-111.3-25-21.8 12.7-52.1 25-88.7 25a7.83 7.83 0 0 1-7.3-4.8 8 8 0 0 1 1.5-8.7c.3-.3 22.4-24.3 35.8-54.5-23.9-26.1-38-57.7-38-92C0 103.6 93.1 32 208 32s208 71.6 208 160-93.1 160-208 160z"/><path d="M576 320c0 34.3-14.1 66-38 92 13.4 30.3 35.5 54.2 35.8 54.5a8 8 0 0 1 1.5 8.7 7.88 7.88 0 0 1-7.3 4.8c-36.6 0-66.9-12.3-88.7-25-32.2 15.8-70.3 25-111.3 25-86.2 0-160.2-40.4-191.7-97.9A299.82 299.82 0 0 0 208 384c132.3 0 240-86.1 240-192a148.61 148.61 0 0 0-1.3-20.1C522.5 195.8 576 253.1 576 320z"/></svg>',
                'condition' => $license_is_valid
            ]
        ];

        // Render tabs
        Tabs::renderOuter($tabs, $current_tab);
        Tabs::openContent();

        switch ($current_tab) {
            case 'dashboard':
                $this->renderDashboardTab();
                break;
            case 'conversations':
                // Only render conversations tab if license is valid
                if ($license_is_valid) {
                    $this->renderConversationsTab();
                } else {
                    $this->renderDashboardTab();
                }
                break;
            default:
                $this->renderDashboardTab();
                break;
        }

        Tabs::closeContent();
        Tabs::closeOuter();
    }

    /**
     * Render dashboard tab (original analytics content)
     *
     * @return void
     */
    private function renderDashboardTab(): void
    {
        // Verify nonce if period is being changed
        if (isset($_GET['period'])) {
            $nonce = isset($_GET['_wpnonce']) ? sanitize_text_field(wp_unslash($_GET['_wpnonce'])) : '';
            if (!wp_verify_nonce($nonce, 'intucart_analytics_period')) {
                wp_die(esc_html__('Invalid nonce', 'intufind'));
            }
        }

        $period = isset($_GET['period'])
            ? sanitize_text_field(wp_unslash($_GET['period']))
            : '30days';

        echo '<div class="intucart-header-section">';
        echo '<h2 class="intucart-main-header">' . esc_html__('Analytics Dashboard', 'intufind') . '</h2>';
        echo '<p class="intucart-main-description">' . esc_html__('View search trends, chat interactions, and overall engagement metrics for your AI-powered features.', 'intufind') . '</p>';
        echo '</div>';

        // Period selector
        $this->renderPeriodSelector($period);

        // Summary cards
        $this->renderSummaryCards($period);

        // Charts container
        echo '<div class="intucart-analytics-charts">';
        $this->renderSearchChart($period);
        echo '</div>';

        // Detailed stats
        echo '<div class="intucart-analytics-details">';
        $this->renderTopSearches($period);
        echo '</div>';

        // Localize script with chart data
        $this->localizeChartData($period);
    }

    /**
     * Render conversations tab
     *
     * @return void
     */
    private function renderConversationsTab(): void
    {
        // Add wrapper with class that JavaScript uses to detect the tab
        echo '<div class="intucart-conversations-tab">';
        
        echo '<div class="intucart-header-section">';
        echo '<h2 class="intucart-main-header">' . esc_html__('Conversation Management', 'intufind') . '</h2>';
        echo '<p class="intucart-main-description">' . esc_html__('Search, view, and manage chat conversations with detailed analytics and export capabilities.', 'intufind') . '</p>';
        echo '</div>';

        // Conversations search and filters
        $this->renderConversationFilters();
        
        // Conversations analytics summary
        $this->renderConversationAnalytics();
        
        // Conversations table
        $this->renderConversationsTable();
        
        echo '</div>'; // .intucart-conversations-tab
    }

    /**
     * Render period selector
     *
     * @param string $current Current period
     *
     * @return void
     */
    private function renderPeriodSelector(string $current): void
    {
        ?>
        <div class="intucart-period-selector">
            <form method="get">
                <input type="hidden" name="page" value="intucart-analytics">
                <?php wp_nonce_field('intucart_analytics_period'); ?>
                <select name="period" onchange="this.form.submit()">
                    <option value="7days" <?php selected($current, '7days'); ?>><?php echo esc_html__('Last 7 Days', 'intufind'); ?></option>
                    <option value="30days" <?php selected($current, '30days'); ?>><?php echo esc_html__('Last 30 Days', 'intufind'); ?></option>
                    <option value="90days" <?php selected($current, '90days'); ?>><?php echo esc_html__('Last 90 Days', 'intufind'); ?></option>
                    <option value="year" <?php selected($current, 'year'); ?>><?php echo esc_html__('Last Year', 'intufind'); ?></option>
                    <option value="all" <?php selected($current, 'all'); ?>><?php echo esc_html__('All Time', 'intufind'); ?></option>
                </select>
            </form>
        </div>
        <?php
    }

    /**
     * Render summary cards
     *
     * @param string $period Time period
     *
     * @return void
     */
    private function renderSummaryCards(string $period): void
    {
        $stats = $this->getStats($period);
        // Access gating handled elsewhere if needed; analytics page renders regardless

        ?>
        <div class="intucart-summary-cards">
            <div class="intucart-stat-card">
                <div class="stat-count"><?php echo esc_html(number_format($stats['total_searches'])); ?></div>
                <div class="stat-label"><span class="dashicons dashicons-search"></span> <?php echo esc_html__('Total Searches', 'intufind'); ?></div>
            </div>
            
            <div class="intucart-stat-card">
                <div class="stat-count"><?php echo esc_html(number_format($stats['avg_results'], 1)); ?></div>
                <div class="stat-label"><span class="dashicons dashicons-list-view"></span> <?php echo esc_html__('Avg Results', 'intufind'); ?></div>
            </div>
            
            <div class="intucart-stat-card">
                <div class="stat-count"><?php echo esc_html(number_format($stats['total_conversations'])); ?></div>
                <div class="stat-label"><span class="dashicons dashicons-format-chat"></span> <?php echo esc_html__('Total Conversations', 'intufind'); ?></div>
            </div>
            
            <div class="intucart-stat-card">
                <div class="stat-count"><?php echo esc_html(number_format($stats['avg_messages'], 1)); ?></div>
                <div class="stat-label">
                    <span class="dashicons dashicons-admin-comments"></span> <?php echo esc_html__('Avg Messages/Chat', 'intufind'); ?>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Get analytics stats
     *
     * @param string $period Time period
     *
     * @return array
     */
    private function getStats(string $period): array
    {
        global $wpdb;

        $date_condition = $this->getDateCondition($period);

        // Get search stats from local database
        $search_table = $wpdb->prefix . 'intucart_search_analytics';
        $total_searches = (int) $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM $search_table WHERE created_at >= %s",
                $date_condition
            )
        );

        $avg_results = (float) $wpdb->get_var(
            $wpdb->prepare(
                "SELECT AVG(results_count) FROM $search_table WHERE created_at >= %s",
                $date_condition
            )
        );

        // Calculate search trends (compare to previous period)
        $prev_date = $this->getPreviousPeriodDate($period, $date_condition);

        $prev_searches = (int) $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM $search_table WHERE created_at >= %s AND created_at < %s",
                $prev_date,
                $date_condition
            )
        );

        if ($prev_searches > 0) {
            $search_trend = round(($total_searches - $prev_searches) / $prev_searches * 100, 1);
        } elseif ($total_searches > 0) {
            $search_trend = 100;
        } else {
            $search_trend = 0;
        }

        // Get conversation stats from cloud API (via ThreadService)
        $conversationStats = $this->getConversationStats();

        return [
            'total_searches' => $total_searches,
            'avg_results' => $avg_results,
            'search_trend' => $search_trend,
            // Conversation stats from cloud
            'total_conversations' => $conversationStats['total_conversations'],
            'avg_messages' => $conversationStats['avg_messages'],
            'conversations_today' => $conversationStats['conversations_today'],
        ];
    }

    /**
     * Get conversation stats from cloud API
     *
     * @return array Conversation statistics
     */
    private function getConversationStats(): array
    {
        try {
            $stats = $this->threadService->getThreadStats();
            return [
                'total_conversations' => $stats['total_conversations'] ?? 0,
                'avg_messages' => $stats['average_messages_per_conversation'] ?? 0,
                'conversations_today' => $stats['conversations_today'] ?? 0,
            ];
        } catch (\Exception $e) {
            // Fail gracefully - return zeros if cloud is unavailable
            return [
                'total_conversations' => 0,
                'avg_messages' => 0,
                'conversations_today' => 0,
            ];
        }
    }

    /**
     * Get date condition based on period
     *
     * @param string $period Time period
     *
     * @return string
     */
    private function getDateCondition(string $period): string
    {
        $now = current_time('mysql');

        switch ($period) {
            case '7days':
                return date('Y-m-d H:i:s', strtotime('-7 days', strtotime($now)));
            case '30days':
                return date('Y-m-d H:i:s', strtotime('-30 days', strtotime($now)));
            case '90days':
                return date('Y-m-d H:i:s', strtotime('-90 days', strtotime($now)));
            case 'year':
                return date('Y-m-d H:i:s', strtotime('-1 year', strtotime($now)));
            case 'all':
                return '1970-01-01 00:00:00';
            default:
                return date('Y-m-d H:i:s', strtotime('-30 days', strtotime($now)));
        }
    }

    /**
     * Get previous period date
     *
     * @param string $period Current period
     * @param string $current_date Current period start date
     *
     * @return string
     */
    private function getPreviousPeriodDate(string $period, string $current_date): string
    {
        $current_timestamp = strtotime($current_date);

        switch ($period) {
            case '7days':
                return date('Y-m-d H:i:s', strtotime('-7 days', $current_timestamp));
            case '30days':
                return date('Y-m-d H:i:s', strtotime('-30 days', $current_timestamp));
            case '90days':
                return date('Y-m-d H:i:s', strtotime('-90 days', $current_timestamp));
            case 'year':
                return date('Y-m-d H:i:s', strtotime('-1 year', $current_timestamp));
            default:
                return date('Y-m-d H:i:s', strtotime('-30 days', $current_timestamp));
        }
    }

    /**
     * Render search trends chart container
     *
     * @param string $period Time period
     *
     * @return void
     */
    private function renderSearchChart(string $period): void
    {
        ?>
        <div class="chart-container">
            <h3>
                <span class="dashicons dashicons-search"></span>
                <?php echo esc_html__('Search Trends', 'intufind'); ?>
            </h3>
            <p class="chart-description"><?php echo esc_html__('Track search volume over time to understand how visitors discover your content.', 'intufind'); ?></p>
            <canvas id="searchTrendsChart"></canvas>
        </div>
        <?php
    }

    /**
     * Render top searches table
     *
     * @param string $period Time period
     *
     * @return void
     */
    private function renderTopSearches(string $period): void
    {
        global $wpdb;

        $date_condition = $this->getDateCondition($period);
        $search_table = $wpdb->prefix . 'intucart_search_analytics';

        $results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT search_term, 
                    COUNT(*) as count,
                    AVG(results_count) as avg_results
                FROM $search_table 
                WHERE created_at >= %s 
                GROUP BY search_term 
                ORDER BY count DESC 
                LIMIT 10",
                $date_condition
            )
        );

        ?>
        <div class="intucart-content-panel">
            <h3 class="intucart-panel-header">
                <span class="dashicons dashicons-search"></span>
                <?php echo esc_html__('Top Searches', 'intufind'); ?>
            </h3>
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th><?php echo esc_html__('Search Term', 'intufind'); ?></th>
                        <th><?php echo esc_html__('Searches', 'intufind'); ?></th>
                        <th><?php echo esc_html__('Avg Results', 'intufind'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (!empty($results)) : ?>
                        <?php foreach ($results as $row) : ?>
                        <tr>
                            <?php $full = (string) $row->search_term; ?>
                            <td>
                                <span title="<?php echo esc_attr($full); ?>"><?php echo esc_html($full); ?></span>
                            </td>
                            <td><?php echo esc_html(number_format((int)$row->count)); ?></td>
                            <td><?php echo esc_html(number_format((float)$row->avg_results, 1)); ?></td>
                        </tr>
                        <?php endforeach; ?>
                    <?php else : ?>
                        <tr>
                            <td colspan="3" style="text-align: center; padding: 40px 20px; color: #64748b; font-style: italic;">
                                <?php echo esc_html__('No search data available for this period', 'intufind'); ?>
                            </td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
        <?php
    }

    /**
     * Render top recommended products table
     *
     * @param string $period Time period
     *
     * @return void
     */
    private function renderTopRecommendations(string $period): void
    {
        global $wpdb;

        $date_condition = $this->getDateCondition($period);
        $rec_table = $wpdb->prefix . 'intucart_recommendation_analytics';

        $results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT 
                    product_id,
                    SUM(impression_count) as total_impressions,
                    AVG(total_score / impression_count) as avg_score
                FROM $rec_table 
                WHERE last_updated >= %s 
                GROUP BY product_id 
                ORDER BY total_impressions DESC 
                LIMIT 10",
                $date_condition
            )
        );

        ?>
        <div class="intucart-content-panel">
            <h3 class="intucart-panel-header">
                <span class="dashicons dashicons-products"></span>
                <?php echo esc_html__('Top Recommended Products', 'intufind'); ?>
            </h3>
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th><?php echo esc_html__('Product', 'intufind'); ?></th>
                        <th><?php echo esc_html__('Impressions', 'intufind'); ?></th>
                        <th><?php echo esc_html__('Avg Score', 'intufind'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (!empty($results)) : ?>
                        <?php $hasValidProducts = false; ?>
                        <?php foreach ($results as $row) : ?>
                            <?php $product = wc_get_product($row->product_id); ?>
                            <?php if ($product) : ?>
                                <?php $hasValidProducts = true; ?>
                        <tr>
                            <td>
                                <a href="<?php echo esc_url(get_edit_post_link($row->product_id)); ?>">
                                    <?php echo esc_html($product->get_name()); ?>
                                </a>
                            </td>
                            <td><?php echo esc_html(number_format($row->total_impressions)); ?></td>
                            <td><?php echo esc_html(number_format($row->avg_score * 100, 1)) . '%'; ?></td>
                        </tr>
                            <?php endif; ?>
                        <?php endforeach; ?>
                        <?php if (!$hasValidProducts) : ?>
                        <tr>
                            <td colspan="3" style="text-align: center; padding: 40px 20px; color: #64748b; font-style: italic;">
                                <?php echo esc_html__('No product data available for this period', 'intufind'); ?>
                            </td>
                        </tr>
                        <?php endif; ?>
                    <?php else : ?>
                        <tr>
                            <td colspan="3" style="text-align: center; padding: 40px 20px; color: #64748b; font-style: italic;">
                                <?php echo esc_html__('No product data available for this period', 'intufind'); ?>
                            </td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
        <?php
    }

    /**
     * Localize chart data for JavaScript
     *
     * @param string $period Time period
     *
     * @return void
     */
    private function localizeChartData(string $period): void
    {
        global $wpdb;

        // Get search data
        $date_condition = $this->getDateCondition($period);
        $search_table = $wpdb->prefix . 'intucart_search_analytics';

        $search_results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT DATE(created_at) as date, COUNT(*) as count 
                FROM $search_table 
                WHERE created_at >= %s 
                GROUP BY DATE(created_at) 
                ORDER BY date ASC",
                $date_condition
            )
        );

        // Prepare the data for localization
        $chart_data = [
            'search' => [
                'labels' => wp_list_pluck($search_results, 'date'),
                'data' => array_map('intval', wp_list_pluck($search_results, 'count'))
            ]
        ];

        wp_localize_script('intucart-analytics', 'intucartChartData', $chart_data);
    }

    /**
     * Render feature requirement notice
     *
     * @param string $featureName The feature display name
     * @param string $featureKey The feature key for cloud service checks
     * @param bool $showUpgradeButton Whether to show an upgrade button
     *
     * @return void
     */
    private function renderFeatureRequirementNotice(
        string $featureName,
        string $featureKey,
        bool $showUpgradeButton = true
    ): void {
        // Use typed license helper for availability
        if ($this->license->hasAnalytics()) {
            return;
        }

        echo '<div class="license-requirement-notice">';
        echo '<div class="license-icon dashicons dashicons-lock"></div>';
        echo '<div class="license-message">';
        echo '<strong>' . esc_html($featureName) . ' requires upgrade</strong>';
        echo '<p>Upgrade your license to access this feature and more advanced capabilities.</p>';
        echo '</div>';

        if ($showUpgradeButton) {
            echo '<div class="license-action">';
            echo '<a href="' . esc_url(admin_url('admin.php?page=intucart&tab=license')) . '" ';
            echo 'class="button button-primary">Upgrade License</a>';
            echo '</div>';
        }

        echo '</div>';
    }

    /**
     * Get the number of days in the selected period
     *
     * @param string $period Time period
     *
     * @return int
     */
    private function getDaysInPeriod(string $period): int
    {
        switch ($period) {
            case '7days':
                return 7;
            case '30days':
                return 30;
            case '90days':
                return 90;
            case 'year':
                return 365;
            case 'all':
                // For "all time", calculate the actual days between first record and now
                global $wpdb;
                $rec_table = $wpdb->prefix . 'intucart_recommendation_analytics';
                $first_date = $wpdb->get_var("SELECT MIN(DATE(last_updated)) FROM $rec_table");

                if (!$first_date) {
                    return 0;
                }

                $now = current_time('mysql');
                $days = (strtotime($now) - strtotime($first_date)) / (60 * 60 * 24);
                return max(1, round($days)); // At least 1 day
            default:
                return 30;
        }
    }

    /**
     * Get the length of the previous period in days
     *
     * @param string $period Current period
     *
     * @return int
     */
    private function getPreviousPeriodLength(string $period): int
    {
        // For most standard periods, the previous period has the same length
        switch ($period) {
            case '7days':
                return 7;
            case '30days':
                return 30;
            case '90days':
                return 90;
            case 'year':
                return 365;
            case 'all':
                // For "all time", we need to determine the length of the previous equivalent period
                global $wpdb;
                $rec_table = $wpdb->prefix . 'intucart_recommendation_analytics';
                $first_date = $wpdb->get_var("SELECT MIN(DATE(last_updated)) FROM $rec_table");

                if (!$first_date) {
                    return 0;
                }

                $now = current_time('mysql');
                $total_days = (strtotime($now) - strtotime($first_date)) / (60 * 60 * 24);

                // If we have less than 2 days of data, return 1 to avoid division by zero
                if ($total_days < 2) {
                    return 1;
                }

                // Previous period is the same length as the current period
                return max(1, round($total_days / 2));
            default:
                return 30;
        }
    }

    /**
     * Render conversation filters and search
     *
     * @return void
     */
    private function renderConversationFilters(): void
    {
        echo '<div class="intucart-content-panel" style="margin-bottom: 24px;">';
        echo '<h3 class="intucart-panel-header">';
        echo '<span class="dashicons dashicons-filter"></span>';
        echo esc_html__('Search & Filter Conversations', 'intufind');
        echo '</h3>';

        echo '<div class="conversation-filters">';
        echo '<form id="conversation-search-form" class="conversation-search-form">';
        
        echo '<div class="filter-row filter-row-main">';
        
        // Search query - takes more space
        echo '<div class="analytics-filter-group analytics-filter-group-wide">';
        echo '<label for="conversation-query">' . esc_html__('Search:', 'intufind') . '</label>';
        echo '<input type="text" id="conversation-query" name="query" placeholder="' . esc_attr__('Search conversation content...', 'intufind') . '" />';
        echo '</div>';
        
        // User ID filter
        echo '<div class="analytics-filter-group">';
        echo '<label for="conversation-user">' . esc_html__('User ID:', 'intufind') . '</label>';
        echo '<input type="text" id="conversation-user" name="userId" placeholder="' . esc_attr__('Filter by user ID...', 'intufind') . '" />';
        echo '</div>';
        
        // Message count range - smaller width for numbers
        echo '<div class="analytics-filter-group analytics-filter-group-narrow">';
        echo '<label for="conversation-min-messages">' . esc_html__('Min Messages:', 'intufind') . '</label>';
        echo '<input type="number" id="conversation-min-messages" name="minMessages" min="0" />';
        echo '</div>';
        
        echo '<div class="analytics-filter-group analytics-filter-group-narrow">';
        echo '<label for="conversation-max-messages">' . esc_html__('Max Messages:', 'intufind') . '</label>';
        echo '<input type="number" id="conversation-max-messages" name="maxMessages" min="0" />';
        echo '</div>';
        
        echo '</div>'; // .filter-row
        
        echo '<div class="filter-row filter-row-secondary">';
        
        // Date range - needs more width
        echo '<div class="analytics-filter-group analytics-filter-group-date">';
        echo '<label for="conversation-date-from">' . esc_html__('From Date:', 'intufind') . '</label>';
        echo '<input type="date" id="conversation-date-from" name="dateFrom" />';
        echo '</div>';
        
        echo '<div class="analytics-filter-group analytics-filter-group-date">';
        echo '<label for="conversation-date-to">' . esc_html__('To Date:', 'intufind') . '</label>';
        echo '<input type="date" id="conversation-date-to" name="dateTo" />';
        echo '</div>';
        
        // Sort options
        echo '<div class="analytics-filter-group">';
        echo '<label for="conversation-sort">' . esc_html__('Sort By:', 'intufind') . '</label>';
        echo '<select id="conversation-sort" name="sortBy">';
        echo '<option value="timestamp">' . esc_html__('Date (Newest First)', 'intufind') . '</option>';
        echo '<option value="message_count">' . esc_html__('Message Count', 'intufind') . '</option>';
        echo '<option value="token_count">' . esc_html__('Token Count', 'intufind') . '</option>';
        echo '</select>';
        echo '</div>';
        
        echo '</div>'; // .filter-row
        
        // Actions row at bottom
        echo '<div class="filter-actions-row">';
        echo '<div class="filter-actions-left">';
        echo '<button type="submit" class="itf-btn itf-btn-primary itf-btn-medium">';
        echo '<span class="itf-btn-icon itf-btn-icon-before"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path opacity=".4" d="M208 80a128 128 0 1 1-90.51 37.49A127.15 127.15 0 0 1 208 80m0-80C93.12 0 0 93.12 0 208s93.12 208 208 208 208-93.12 208-208S322.88 0 208 0z"/><path d="M504.9 476.7L476.6 505a23.9 23.9 0 0 1-33.9 0L343 405.3a24 24 0 0 1-7-17V372l36-36h16.3a24 24 0 0 1 17 7l99.7 99.7a24.11 24.11 0 0 1-.1 34z"/></svg></span>';
        echo '<span class="itf-btn-text">' . esc_html__('Search', 'intufind') . '</span>';
        echo '</button>';
        echo '<button type="button" class="itf-btn itf-btn-secondary itf-btn-medium" id="clear-filters">';
        echo '<span class="itf-btn-icon itf-btn-icon-before"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path opacity=".4" d="M8 454.06V320a24 24 0 0 1 24-24h134.06c21.38 0 32.09 25.85 17 41l-41.75 41.75A166.82 166.82 0 0 0 256.16 424c77.41-.07 144.31-53.14 162.78-126.85a12 12 0 0 1 11.65-9.15h57.31a12 12 0 0 1 11.81 14.18C478.07 417.08 377.19 504 256 504a247.14 247.14 0 0 1-171.31-68.69L49 471c-15.15 15.15-41 4.44-41-16.94z"/><path d="M12.3 209.82C33.93 94.92 134.81 8 256 8a247.14 247.14 0 0 1 171.31 68.69L463 41c15.12-15.12 41-4.41 41 17v134a24 24 0 0 1-24 24H345.94c-21.38 0-32.09-25.85-17-41l41.75-41.75A166.8 166.8 0 0 0 255.85 88c-77.46.07-144.33 53.18-162.79 126.85A12 12 0 0 1 81.41 224H24.1a12 12 0 0 1-11.8-14.18z"/></svg></span>';
        echo '<span class="itf-btn-text">' . esc_html__('Clear', 'intufind') . '</span>';
        echo '</button>';
        echo '</div>';
        echo '<div class="filter-actions-right">';
        echo '<button type="button" class="itf-btn itf-btn-secondary itf-btn-medium" id="export-conversations">';
        echo '<span class="itf-btn-icon itf-btn-icon-before"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 640 512"><path opacity=".4" d="M537.6 226.6A96.11 96.11 0 0 0 448 96a95.51 95.51 0 0 0-53.3 16.2A160 160 0 0 0 96 192c0 2.7.1 5.4.2 8.1A144 144 0 0 0 144 480h368a128 128 0 0 0 25.6-253.4zm-139.9 93L305 412.3a23.9 23.9 0 0 1-33.9 0l-92.7-92.7a23.9 23.9 0 0 1 0-33.9l10.8-10.8a24.09 24.09 0 0 1 34.5.5l32.4 34.5V184a23.94 23.94 0 0 1 24-24h16a23.94 23.94 0 0 1 24 24v125.9l32.4-34.5a24 24 0 0 1 34.5-.5l10.8 10.8a24.18 24.18 0 0 1-.1 33.9z"/><path d="M397.7 319.6L305 412.3a23.9 23.9 0 0 1-33.9 0l-92.7-92.7a23.9 23.9 0 0 1 0-33.9l10.8-10.8a24.09 24.09 0 0 1 34.5.5l32.4 34.5V184a23.94 23.94 0 0 1 24-24h16a23.94 23.94 0 0 1 24 24v125.9l32.4-34.5a24 24 0 0 1 34.5-.5l10.8 10.8a24.18 24.18 0 0 1-.1 33.9z"/></svg></span>';
        echo '<span class="itf-btn-text">' . esc_html__('Export', 'intufind') . '</span>';
        echo '</button>';
        echo '<span class="export-info-tooltip" style="margin-left: 4px; color: #646970; cursor: pointer; font-size: 13px;" data-popover="<strong>' . esc_html__('Export Information:', 'intufind') . '</strong><br/>' . esc_html__('Downloads a CSV file containing up to 100 conversations that match your current search and filter criteria.', 'intufind') . '<br/><br/><strong>' . esc_html__('Included Data:', 'intufind') . '</strong><br/>• ' . esc_html__('Conversation ID', 'intufind') . '<br/>• ' . esc_html__('User ID', 'intufind') . '<br/>• ' . esc_html__('Created Date', 'intufind') . '<br/>• ' . esc_html__('Message Count', 'intufind') . '<br/>• ' . esc_html__('Token Count', 'intufind') . '<br/>• ' . esc_html__('Content Preview', 'intufind') . '<br/><br/>' . esc_html__('The export respects all active filters including date range, user selection, and message count limits.', 'intufind') . '" title="' . esc_attr__('Export details', 'intufind') . '"><span class="dashicons dashicons-info" style="font-size: 16px; vertical-align: middle; opacity: 0.7; transition: opacity 0.2s ease;"></span></span>';
        echo '</div>';
        echo '</div>';
        
        echo '</form>';
        echo '</div>'; // .conversation-filters

        echo '</div>'; // .intucart-content-panel
    }

    /**
     * Render conversation analytics summary
     *
     * @return void
     */
    private function renderConversationAnalytics(): void
    {
        echo '<div class="intucart-content-panel" style="margin-bottom: 24px;">';
        echo '<h3 class="intucart-panel-header">';
        echo '<span class="dashicons dashicons-chart-bar"></span>';
        echo esc_html__('Conversation Analytics', 'intufind');
        echo '</h3>';

        echo '<div id="conversation-analytics-content">';
        echo '<div class="loading-spinner">' . esc_html__('Loading analytics...', 'intufind') . '</div>';
        echo '</div>';

        echo '</div>'; // .intucart-content-panel
    }

    /**
     * Render conversations table
     *
     * @return void
     */
    private function renderConversationsTable(): void
    {
        echo '<div class="intucart-content-panel">';
        echo '<h3 class="intucart-panel-header">';
        echo '<span class="dashicons dashicons-format-chat"></span>';
        echo esc_html__('Recent Conversations', 'intufind');
        echo '</h3>';

        echo '<div id="conversations-table-container">';
        echo '<div class="loading-spinner">' . esc_html__('Loading conversations...', 'intufind') . '</div>';
        echo '</div>';

        // Pagination container
        echo '<div id="conversations-pagination" class="tablenav bottom">';
        echo '<div class="tablenav-pages"></div>';
        echo '</div>';

        echo '</div>'; // .intucart-content-panel
    }

    /**
     * Handle AJAX request to search conversations
     *
     * @return void
     */
    public function handleAjaxSearchConversations(): void
    {
        // Check nonce for security
        check_ajax_referer('intucart_analytics_nonce', 'nonce');

        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('You do not have permission to perform this action.', 'intufind')]);
            return;
        }

        try {
            // Get search parameters
            $searchParams = [
                'query' => sanitize_text_field($_POST['query'] ?? ''),
                'userId' => sanitize_text_field($_POST['userId'] ?? ''),
                'dateFrom' => sanitize_text_field($_POST['dateFrom'] ?? ''),
                'dateTo' => sanitize_text_field($_POST['dateTo'] ?? ''),
                'minMessages' => (int) ($_POST['minMessages'] ?? 0),
                'maxMessages' => (int) ($_POST['maxMessages'] ?? 0),
                'sortBy' => sanitize_text_field($_POST['sortBy'] ?? 'timestamp'),
                'sortOrder' => sanitize_text_field($_POST['sortOrder'] ?? 'desc'),
                'limit' => (int) ($_POST['limit'] ?? 10),
                'offset' => (int) ($_POST['offset'] ?? 0)
            ];

            // Remove empty parameters
            $searchParams = array_filter($searchParams, function($value) {
                return $value !== '' && $value !== 0;
            });

            // Search conversations using ThreadService
            $result = $this->threadService->searchThreads($searchParams);

            // ThreadService returns raw search results, not wrapped response  
            $html = $this->renderConversationsTableHTML($result['results'], $result['total']);
            wp_send_json_success([
                'html' => $html,
                'total' => $result['total'],
                'pagination' => [
                    'total' => $result['total'],
                    'limit' => $result['limit'] ?? 10,
                    'offset' => $result['offset'] ?? 0
                ]
            ]);
        } catch (\Exception $e) {
            $this->logger->error('Failed to search conversations', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            wp_send_json_error(['message' => __('An error occurred while searching conversations', 'intufind')]);
        }
    }

    /**
     * Handle AJAX request to get conversation analytics
     *
     * @return void
     */
    public function handleAjaxGetConversationAnalytics(): void
    {
        // Check nonce for security
        check_ajax_referer('intucart_analytics_nonce', 'nonce');

        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('You do not have permission to perform this action.', 'intufind')]);
            return;
        }

        try {
            // Get analytics using ThreadService
            $result = $this->threadService->getThreadAnalytics();

            // Validate that we got analytics data
            if (empty($result)) {
                $this->logger->warning('Thread analytics returned empty result');
                wp_send_json_error(['message' => __('No analytics data available', 'intufind')]);
                return;
            }

            // ThreadService returns raw analytics data, not wrapped response
            $html = $this->renderConversationAnalyticsHTML($result);
            wp_send_json_success(['html' => $html]);
        } catch (\InvalidArgumentException $e) {
            // Handle license/permission errors specifically
            $this->logger->warning('Thread analytics access denied', [
                'error' => $e->getMessage()
            ]);
            wp_send_json_error(['message' => $e->getMessage()]);
        } catch (\Exception $e) {
            $this->logger->error('Failed to get conversation analytics', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            wp_send_json_error(['message' => sprintf(__('An error occurred while getting conversation analytics: %s', 'intufind'), $e->getMessage())]);
        }
    }

    /**
     * Handle AJAX request to delete a conversation
     *
     * @return void
     */
    public function handleAjaxDeleteConversation(): void
    {
        // Check nonce for security
        check_ajax_referer('intucart_analytics_nonce', 'nonce');

        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('You do not have permission to perform this action.', 'intufind')]);
            return;
        }

        try {
            $threadId = sanitize_text_field($_POST['threadId'] ?? '');
            
            if (empty($threadId)) {
                wp_send_json_error(['message' => __('Thread ID is required', 'intufind')]);
                return;
            }

            // Delete conversation using ThreadService
            $result = $this->threadService->deleteThread($threadId);

            // SDK returns {id, message} on success
            if (isset($result['id'])) {
                wp_send_json_success(['message' => $result['message'] ?? __('Conversation deleted successfully', 'intufind')]);
            } else {
                wp_send_json_error(['message' => $result['message'] ?? __('Failed to delete conversation', 'intufind')]);
            }
        } catch (\Exception $e) {
            $this->logger->error('Failed to delete conversation', [
                'threadId' => $_POST['threadId'] ?? 'unknown',
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            wp_send_json_error(['message' => __('An error occurred while deleting the conversation', 'intufind')]);
        }
    }

    /**
     * Handle AJAX request to get conversation details
     *
     * @return void
     */
    public function handleAjaxGetConversationDetails(): void
    {
        // Check nonce for security
        check_ajax_referer('intucart_analytics_nonce', 'nonce');

        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('You do not have permission to perform this action.', 'intufind')]);
            return;
        }

        try {
            $threadId = sanitize_text_field($_POST['threadId'] ?? '');
            
            if (empty($threadId)) {
                wp_send_json_error(['message' => __('Thread ID is required', 'intufind')]);
                return;
            }

            // Get conversation details using ThreadService
            $result = $this->threadService->getThreadById($threadId);

            // ThreadService returns raw thread data, not wrapped response
            $html = $this->renderConversationDetailsHTML($result);
            wp_send_json_success(['html' => $html]);
        } catch (\Exception $e) {
            $this->logger->error('Failed to get conversation details', [
                'threadId' => $_POST['threadId'] ?? 'unknown',
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            wp_send_json_error(['message' => __('An error occurred while getting conversation details', 'intufind')]);
        }
    }

    /**
     * Render conversations table HTML
     *
     * @param array $conversations Array of conversation data
     * @param int $total Total number of conversations
     * @return string HTML content
     */
    private function renderConversationsTableHTML(array $conversations, int $total): string
    {
        if (empty($conversations)) {
            return '<div class="no-conversations"><p>No conversations found matching your criteria.</p></div>';
        }

        $html = '<table class="wp-list-table widefat fixed striped">';
        $html .= '<thead>';
        $html .= '<tr>';
        $html .= '<th class="manage-column">Date</th>';
        $html .= '<th class="manage-column">' . esc_html__('User', 'intufind') . '</th>';
        $html .= '<th class="manage-column">' . esc_html__('Messages', 'intufind') . '</th>';
        $html .= '<th class="manage-column">' . esc_html__('Preview', 'intufind') . '</th>';
        $html .= '<th class="manage-column">' . esc_html__('Actions', 'intufind') . '</th>';
        $html .= '</tr>';
        $html .= '</thead>';
        $html .= '<tbody>';

        foreach ($conversations as $conversation) {
            $html .= '<tr>';
            $html .= '<td>' . esc_html(date('M j, Y H:i', strtotime($conversation['timestamp']))) . '</td>';
            $html .= '<td>' . esc_html($conversation['user_id'] ?? __('Unknown', 'intufind')) . '</td>';
            
            // Messages count with badge styling
            $messageCount = $conversation['message_count'] ?? __('N/A', 'intufind');
            if ($messageCount !== 'N/A') {
                $html .= '<td>' . esc_html($messageCount) . '</td>';
            } else {
                $html .= '<td>' . esc_html($messageCount) . '</td>';
            }
            
            $html .= '<td>' . esc_html(wp_trim_words($conversation['text_preview'] ?? $conversation['text'] ?? '', 15)) . '</td>';
            $html .= '<td style="white-space: nowrap;">';
            $html .= '<button type="button" class="itf-btn itf-btn-secondary itf-btn-small view-conversation" data-id="' . esc_attr($conversation['external_id']) . '">';
            $html .= '<span class="itf-btn-icon itf-btn-icon-before"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 576 512"><path opacity=".4" d="M208 352c-41 0-79.1-9.3-111.3-25-21.8 12.7-52.1 25-88.7 25a7.83 7.83 0 0 1-7.3-4.8 8 8 0 0 1 1.5-8.7c.3-.3 22.4-24.3 35.8-54.5-23.9-26.1-38-57.7-38-92C0 103.6 93.1 32 208 32s208 71.6 208 160-93.1 160-208 160z"/><path d="M576 320c0 34.3-14.1 66-38 92 13.4 30.3 35.5 54.2 35.8 54.5a8 8 0 0 1 1.5 8.7 7.88 7.88 0 0 1-7.3 4.8c-36.6 0-66.9-12.3-88.7-25-32.2 15.8-70.3 25-111.3 25-86.2 0-160.2-40.4-191.7-97.9A299.82 299.82 0 0 0 208 384c132.3 0 240-86.1 240-192a148.61 148.61 0 0 0-1.3-20.1C522.5 195.8 576 253.1 576 320z"/></svg></span>';
            $html .= '<span class="itf-btn-text">' . esc_html__('View', 'intufind') . '</span>';
            $html .= '</button> ';
            $html .= '<button type="button" class="itf-btn itf-btn-danger itf-btn-small delete-conversation" data-id="' . esc_attr($conversation['external_id']) . '">';
            $html .= '<span class="itf-btn-icon itf-btn-icon-before"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path opacity=".4" d="M53.2 467L32 96h384l-21.2 371a48 48 0 0 1-47.9 45H101.1a48 48 0 0 1-47.9-45z"/><path d="M0 80V48a16 16 0 0 1 16-16h120l9.4-18.7A23.72 23.72 0 0 1 166.8 0h114.3a24 24 0 0 1 21.5 13.3L312 32h120a16 16 0 0 1 16 16v32a16 16 0 0 1-16 16H16A16 16 0 0 1 0 80z"/></svg></span>';
            $html .= '<span class="itf-btn-text">' . esc_html__('Delete', 'intufind') . '</span>';
            $html .= '</button>';
            $html .= '</td>';
            $html .= '</tr>';
        }

        $html .= '</tbody>';
        $html .= '</table>';

        return $html;
    }

    /**
     * Render conversation analytics HTML
     *
     * @param array $analytics Analytics data
     * @return string HTML content
     */
    private function renderConversationAnalyticsHTML(array $analytics): string
    {
        $html = '<div class="conversation-analytics-grid">';
        
        $html .= '<div class="analytics-card">';
        $html .= '<div class="analytics-number">' . number_format($analytics['totalConversations'] ?? 0) . '</div>';
        $html .= '<div class="analytics-label">' . esc_html__('Total Conversations', 'intufind') . '</div>';
        $html .= '</div>';
        
        $html .= '<div class="analytics-card">';
        $html .= '<div class="analytics-number">' . number_format($analytics['activeUsers'] ?? 0) . '</div>';
        $html .= '<div class="analytics-label">' . esc_html__('Active Users', 'intufind') . '</div>';
        $html .= '</div>';
        
        $html .= '<div class="analytics-card">';
        $html .= '<div class="analytics-number">' . number_format($analytics['averageMessagesPerConversation'] ?? 0, 1) . '</div>';
        $html .= '<div class="analytics-label">' . esc_html__('Avg Messages', 'intufind') . '</div>';
        $html .= '</div>';
        
        $html .= '<div class="analytics-card">';
        $html .= '<div class="analytics-number">' . number_format($analytics['conversationsToday'] ?? 0) . '</div>';
        $html .= '<div class="analytics-label">' . esc_html__('Conversations Today', 'intufind') . '</div>';
        $html .= '</div>';
        
        $html .= '</div>';

        // Top users if available
        if (!empty($analytics['topUsers'])) {
            $html .= '<div class="top-users-section">';
            $html .= '<h4>' . esc_html__('Most Active Users', 'intufind') . '</h4>';
            $html .= '<table class="wp-list-table widefat fixed striped">';
            $html .= '<thead><tr><th>' . esc_html__('User ID', 'intufind') . '</th><th>' . esc_html__('Conversations', 'intufind') . '</th></tr></thead>';
            $html .= '<tbody>';
            
            foreach (array_slice($analytics['topUsers'], 0, 5) as $user) {
                $html .= '<tr>';
                $html .= '<td>' . esc_html($user['user_id']) . '</td>';
                $html .= '<td>' . esc_html($user['conversation_count']) . '</td>';
                $html .= '</tr>';
            }
            
            $html .= '</tbody></table>';
            $html .= '</div>';
        }

        return $html;
    }

    /**
     * Render conversation details HTML with individual messages
     *
     * @param array $conversation Conversation data
     * @return string HTML content
     */
    private function renderConversationDetailsHTML(array $conversation): string
    {
        $html = '<div class="conversation-details">';
        
        // Conversation header with better visual hierarchy
        $html .= '<div class="conversation-header">';
        
        // Primary info line - User prominently displayed
        $html .= '<div class="conversation-primary">';
        $html .= '<div class="conversation-user-primary">👤 ' . esc_html($conversation['user_id'] ?? __('Anonymous User', 'intufind')) . '</div>';
        $html .= '<span class="meta-item message-count-placeholder">💬 ' . esc_html__('Loading...', 'intufind') . '</span>';
        $html .= '</div>';
        
        // Secondary metadata line - Creation time
        $html .= '<div class="conversation-meta">';
        $html .= '<span class="meta-item">📅 ' . esc_html(date('M j, Y H:i', strtotime($conversation['timestamp']))) . '</span>';
        $html .= '</div>';
        
        $html .= '</div>';

        // Parse and display messages from content JSON
        if (!empty($conversation['content'])) {
            $messages = json_decode($conversation['content'], true);
            
            if (is_array($messages) && !empty($messages)) {
                $html .= '<div class="conversation-messages">';
                
                foreach ($messages as $index => $message) {
                    $roleClass = ($message['role'] ?? 'unknown') === 'user' ? 'user-message' : 'assistant-message';
                    $roleIcon = ($message['role'] ?? 'unknown') === 'user' ? '👤' : '🤖';
                    $roleLabel = strtoupper($message['role'] ?? 'unknown');
                    
                    $html .= '<div class="message-item ' . $roleClass . '">';
                    $html .= '<div class="message-header">';
                    $html .= '<span class="message-role">' . $roleIcon . ' ' . esc_html($roleLabel) . '</span>';
                    $html .= '<span class="message-number">' . sprintf(esc_html__('Message %d', 'intufind'), $index + 1) . '</span>';
                    $html .= '</div>';
                    
                    // Handle different content formats and extract products/posts
                    $content = '';
                    $products = [];
                    $posts = [];
                    
                    if (is_string($message['content'] ?? null)) {
                        $content = $message['content'];
                    } elseif (is_array($message['content'] ?? null)) {
                        // Handle array of content objects (like OpenAI format)
                        $contentParts = [];
                        foreach ($message['content'] as $item) {
                            if (is_string($item)) {
                                $contentParts[] = $item;
                            } elseif (is_array($item) && isset($item['type'])) {
                                if ($item['type'] === 'text' && !empty($item['text'])) {
                                    $contentParts[] = $item['text'];
                                } elseif ($item['type'] === 'output_text' && !empty($item['text'])) {
                                    // Try to parse the text as JSON to extract content, products, posts
                                    $parsed = json_decode($item['text'], true);
                                    if (json_last_error() === JSON_ERROR_NONE) {
                                        if (!empty($parsed['intro'])) {
                                            $contentParts[] = $parsed['intro'];
                                        }
                                        if (!empty($parsed['products']) && is_array($parsed['products'])) {
                                            $products = array_merge($products, $parsed['products']);
                                        }
                                        if (!empty($parsed['posts']) && is_array($parsed['posts'])) {
                                            $posts = array_merge($posts, $parsed['posts']);
                                        }
                                    } else {
                                        $contentParts[] = $item['text'];
                                    }
                                } else {
                                    $contentParts[] = json_encode($item, JSON_PRETTY_PRINT);
                                }
                            } else {
                                $contentParts[] = json_encode($item, JSON_PRETTY_PRINT);
                            }
                        }
                        $content = implode("\n", $contentParts);
                    } elseif (is_array($message['content'] ?? null) && isset($message['content']['type'])) {
                        if ($message['content']['type'] === 'text' && !empty($message['content']['text'])) {
                            $content = $message['content']['text'];
                        } elseif ($message['content']['type'] === 'output_text' && !empty($message['content']['text'])) {
                            // Try to parse the text as JSON to extract content, products, posts
                            $parsed = json_decode($message['content']['text'], true);
                            if (json_last_error() === JSON_ERROR_NONE) {
                                if (!empty($parsed['intro'])) {
                                    $content = $parsed['intro'];
                                }
                                if (!empty($parsed['products']) && is_array($parsed['products'])) {
                                    $products = $parsed['products'];
                                }
                                if (!empty($parsed['posts']) && is_array($parsed['posts'])) {
                                    $posts = $parsed['posts'];
                                }
                            } else {
                                $content = $message['content']['text'];
                            }
                        } else {
                            $content = json_encode($message['content'], JSON_PRETTY_PRINT);
                        }
                    } elseif (is_array($message['content'] ?? null) || is_object($message['content'] ?? null)) {
                        $content = json_encode($message['content'], JSON_PRETTY_PRINT);
                    } elseif (!empty($message['text'])) {
                        $content = $message['text'];
                    } else {
                        $content = json_encode($message, JSON_PRETTY_PRINT);
                    }
                    
                    // Also check for products/posts at message level
                    if (empty($products) && !empty($message['products']) && is_array($message['products'])) {
                        $products = $message['products'];
                    }
                    if (empty($posts) && !empty($message['posts']) && is_array($message['posts'])) {
                        $posts = $message['posts'];
                    }
                    
                    $html .= '<div class="message-content">' . esc_html($content) . '</div>';
                    
                    // Display products if present
                    if (!empty($products)) {
                        $html .= '<div class="message-attachments">';
                        $html .= '<div class="attachments-label">📦 ' . sprintf(esc_html__('Products shown (%d)', 'intufind'), count($products)) . '</div>';
                        $html .= '<div class="attachments-list">';
                        foreach ($products as $product) {
                            $productName = $product['name'] ?? $product['title'] ?? __('Unknown Product', 'intufind');
                            $productPrice = $product['price'] ?? '';
                            $productUrl = $product['url'] ?? $product['permalink'] ?? '';
                            
                            $html .= '<div class="attachment-item attachment-product">';
                            $html .= '<span class="attachment-name">' . esc_html($productName) . '</span>';
                            if ($productPrice) {
                                $html .= '<span class="attachment-price">' . esc_html($productPrice) . '</span>';
                            }
                            if ($productUrl) {
                                $html .= '<a href="' . esc_url($productUrl) . '" target="_blank" class="attachment-link">↗</a>';
                            }
                            $html .= '</div>';
                        }
                        $html .= '</div></div>';
                    }
                    
                    // Display posts if present
                    if (!empty($posts)) {
                        $html .= '<div class="message-attachments">';
                        $html .= '<div class="attachments-label">📄 ' . sprintf(esc_html__('Content shown (%d)', 'intufind'), count($posts)) . '</div>';
                        $html .= '<div class="attachments-list">';
                        foreach ($posts as $post) {
                            $postTitle = $post['title'] ?? $post['name'] ?? __('Unknown Post', 'intufind');
                            $postUrl = $post['url'] ?? $post['permalink'] ?? '';
                            $postType = $post['type'] ?? $post['post_type'] ?? '';
                            
                            $html .= '<div class="attachment-item attachment-post">';
                            $html .= '<span class="attachment-name">' . esc_html($postTitle) . '</span>';
                            if ($postType) {
                                $html .= '<span class="attachment-type">' . esc_html($postType) . '</span>';
                            }
                            if ($postUrl) {
                                $html .= '<a href="' . esc_url($postUrl) . '" target="_blank" class="attachment-link">↗</a>';
                            }
                            $html .= '</div>';
                        }
                        $html .= '</div></div>';
                    }
                    
                    // Show additional metadata if available
                    if (!empty($message['model'])) {
                        $html .= '<div class="message-meta">' . sprintf(esc_html__('Model: %s', 'intufind'), esc_html($message['model'])) . '</div>';
                    }
                    if (!empty($message['token_count'])) {
                        $html .= '<div class="message-meta">' . sprintf(esc_html__('Tokens: %s', 'intufind'), esc_html($message['token_count'])) . '</div>';
                    }
                    
                    $html .= '</div>'; // .message-item
                }
                
                $html .= '</div>'; // .conversation-messages
            } else {
                // Fallback: show readable text if JSON parsing fails
                if (!empty($conversation['text'])) {
                    $html .= '<div class="conversation-text">';
                    $html .= '<h5>' . esc_html__('Conversation Text', 'intufind') . '</h5>';
                    $html .= '<div class="text-content">' . esc_html($conversation['text']) . '</div>';
                    $html .= '</div>';
                }
            }
        } elseif (!empty($conversation['text'])) {
            // Fallback: show readable text if no content JSON
            $html .= '<div class="conversation-text">';
            $html .= '<h5>' . esc_html__('Conversation Text', 'intufind') . '</h5>';
            $html .= '<div class="text-content">' . esc_html($conversation['text']) . '</div>';
            $html .= '</div>';
        }

        $html .= '</div>'; // .conversation-details
        
        return $html;
    }

    /**
     * Handle AJAX validate export request
     *
     * @return void
     */
    public function handleAjaxValidateExport(): void
    {
        // Check nonce for security
        check_ajax_referer('intucart_analytics_nonce', 'nonce');

        try {
            // Get search parameters from request
            $searchParams = array_filter([
                'query' => sanitize_text_field($_POST['query'] ?? ''),
                'user_id' => sanitize_text_field($_POST['user_id'] ?? ''),
                'date_from' => sanitize_text_field($_POST['date_from'] ?? ''),
                'date_to' => sanitize_text_field($_POST['date_to'] ?? ''),
                'message_count_min' => (int) ($_POST['message_count_min'] ?? 0),
                'message_count_max' => (int) ($_POST['message_count_max'] ?? 0),
                'sort_by' => sanitize_text_field($_POST['sort_by'] ?? 'timestamp'),
                'sort_order' => sanitize_text_field($_POST['sort_order'] ?? 'desc'),
                'limit' => 100 // Same limit as export for proper validation
            ], function ($value) {
                return $value !== '' && $value !== 0;
            });

            // Validate by attempting a small search
            $result = $this->threadService->searchThreads($searchParams);
            
            if (isset($result['error'])) {
                wp_send_json_error(['message' => $result['error']]);
                return;
            }

            if (!isset($result['results'])) {
                wp_send_json_error(['message' => __('Failed to validate export parameters', 'intufind')]);
                return;
            }

            // Validation passed
            wp_send_json_success(['message' => __('Export validation successful', 'intufind')]);

        } catch (\Exception $e) {
            $this->logger->error('Export validation error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            wp_send_json_error(['message' => $e->getMessage()]);
        }
    }

    /**
     * Handle AJAX export conversations request
     *
     * @return void
     */
    public function handleAjaxExportConversations(): void
    {
        // Check nonce for security
        check_ajax_referer('intucart_analytics_nonce', 'nonce');

        try {
            // Get search parameters from request
            $searchParams = array_filter([
                'query' => sanitize_text_field($_POST['query'] ?? ''),
                'user_id' => sanitize_text_field($_POST['user_id'] ?? ''),
                'date_from' => sanitize_text_field($_POST['date_from'] ?? ''),
                'date_to' => sanitize_text_field($_POST['date_to'] ?? ''),
                'message_count_min' => (int) ($_POST['message_count_min'] ?? 0),
                'message_count_max' => (int) ($_POST['message_count_max'] ?? 0),
                'sort_by' => sanitize_text_field($_POST['sort_by'] ?? 'timestamp'),
                'sort_order' => sanitize_text_field($_POST['sort_order'] ?? 'desc'),
                'limit' => 100 // Export limit (max allowed)
            ], function ($value) {
                return $value !== '' && $value !== 0;
            });

            // Search conversations using ThreadService
            $result = $this->threadService->searchThreads($searchParams);
            
            if (isset($result['error'])) {
                // Handle API error - don't redirect, just show error message
                http_response_code(400);
                echo 'Export failed: ' . esc_html($result['error']);
                exit;
            }
            
            if (!isset($result['results']) || !is_array($result['results'])) {
                http_response_code(500);
                echo 'Export failed: Unable to retrieve conversations';
                exit;
            }

            $conversations = $result['results'];
            
            // Set headers for CSV download
            header('Content-Type: text/csv; charset=utf-8');
            header('Content-Disposition: attachment; filename=conversations_export_' . date('Y-m-d_H-i-s') . '.csv');
            header('Pragma: no-cache');
            header('Expires: 0');

            // Create output stream
            $output = fopen('php://output', 'w');

            // Add BOM for UTF-8
            fwrite($output, "\xEF\xBB\xBF");

            // CSV headers
            $headers = [
                'ID',
                'User ID', 
                'Created',
                'Messages',
                'Tokens',
                'Preview'
            ];
            fputcsv($output, $headers);

            // Export conversation data
            foreach ($conversations as $conversation) {
                // Handle timestamp conversion - could be Unix timestamp (int) or ISO string
                $createdDate = '';
                if (!empty($conversation['timestamp'])) {
                    if (is_numeric($conversation['timestamp'])) {
                        // Unix timestamp
                        $createdDate = date('Y-m-d H:i:s', (int)$conversation['timestamp']);
                    } else {
                        // ISO string - convert to readable format
                        $timestamp = strtotime($conversation['timestamp']);
                        $createdDate = $timestamp ? date('Y-m-d H:i:s', $timestamp) : $conversation['timestamp'];
                    }
                }
                
                $row = [
                    $conversation['external_id'] ?? '',
                    $conversation['user_id'] ?? '',
                    $createdDate,
                    $conversation['message_count'] ?? 0,
                    $conversation['token_count'] ?? 0,
                    wp_trim_words($conversation['text_preview'] ?? $conversation['text'] ?? '', 20)
                ];
                fputcsv($output, $row);
            }

            fclose($output);
            exit;

        } catch (\Exception $e) {
            $this->logger->error('Export conversations error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            // Return error response
            http_response_code(500);
            echo 'Export failed: ' . esc_html($e->getMessage());
            exit;
        }
    }
}
