<?php

namespace Intucart\Pages;

use Intucart\Services\Logger;
use Intucart\Services\AIClientManager;
use Intucart\Services\Cache\CacheService;
use Intucart\Services\Constants;
use Intucart\Components\Button;

class Prompts
{
    private Logger $logger;
    private AIClientManager $aiClientManager;
    private CacheService $cache;

    public function __construct(Logger $logger, AIClientManager $aiClientManager, CacheService $cache)
    {
        $this->logger = $logger;
        $this->aiClientManager = $aiClientManager;
        $this->cache = $cache;
        add_action('admin_post_intucart_save_prompt', [$this, 'handleSavePrompt']);
        add_action('admin_post_intucart_delete_prompt', [$this, 'handleDeletePrompt']);
        add_action('wp_ajax_intucart_search_products', [$this, 'handleAjaxSearchProducts']);
        add_action('wp_ajax_intucart_search_posts', [$this, 'handleAjaxSearchPosts']);
        add_action('wp_ajax_intucart_search_prompts', [$this, 'handleAjaxSearchPrompts']);
    }

    // Embedded mode used inside Chatbot tab
    public function renderEmbedded(): void
    {
        if (!current_user_can('manage_options')) {
            return;
        }
        $this->enqueueSelectAssets();

        // Content panel
        echo '<div class="intucart-content-panel">';

        $editingId = isset($_GET['prompt_id']) ? sanitize_text_field($_GET['prompt_id']) : '';
        $editingPrompt = $editingId ? $this->aiClientManager->getClient()->prompts()->get($editingId) : null;
        $showCreate = isset($_GET['create']) && $_GET['create'] === '1';

        if ($editingId || $showCreate) {
            echo '<h3 style="margin-top: 0; margin-bottom: 20px; color: #495057; display: flex; align-items: center;">';
            echo '<span class="dashicons dashicons-edit" style="margin-right: 8px;"></span>';
            echo ($editingId ? esc_html__('Edit Prompt', 'intufind') : esc_html__('Create New Prompt', 'intufind'));
            echo '</h3>';
            $this->renderPromptForm($editingPrompt);
        } else {
            echo '<h3 class="intucart-panel-header">';
            echo '<span class="dashicons dashicons-admin-comments"></span>';
            echo esc_html__('Manage Prompts', 'intufind');
            echo '</h3>';
            echo '<p class="intucart-panel-description">' . esc_html__('Search and manage your chat prompts. These appear as suggested questions in your chat widget.', 'intufind') . '</p>';
            $this->renderPromptSearch();
        }

        echo '</div>'; // Close content panel
    }

    private function renderPromptForm(?array $prompt): void
    {
        $id = $prompt['id'] ?? '';
        $title = $prompt['title'] ?? '';
        $promptText = html_entity_decode($prompt['prompt'] ?? '', ENT_QUOTES, 'UTF-8');
        $description = html_entity_decode($prompt['description'] ?? '', ENT_QUOTES, 'UTF-8');
        $canned = html_entity_decode($prompt['canned_response'] ?? '', ENT_QUOTES, 'UTF-8');
        $overrideMode = isset($prompt['override_mode']) && in_array($prompt['override_mode'], ['override', 'fallback']) ? (string)$prompt['override_mode'] : 'override';
        $productIds = isset($prompt['product_ids']) && is_array($prompt['product_ids']) ? $prompt['product_ids'] : [];
        $postIds = isset($prompt['post_ids']) && is_array($prompt['post_ids']) ? $prompt['post_ids'] : [];
        $followUpIds = isset($prompt['follow_up_ids']) && is_array($prompt['follow_up_ids']) ? $prompt['follow_up_ids'] : [];
        $tags = isset($prompt['tags']) && is_array($prompt['tags']) ? implode(',', $prompt['tags']) : '';
        $categories = isset($prompt['categories']) && is_array($prompt['categories']) ? implode(',', $prompt['categories']) : '';
        $icon = $prompt['icon'] ?? '';
        $metadata = (isset($prompt['metadata']) && is_array($prompt['metadata'])) ? $prompt['metadata'] : [];
        $isInitial = isset($metadata['initial']) ? (bool)$metadata['initial'] : false;
        $priority = isset($prompt['priority']) ? intval($prompt['priority']) : '';

        $embedded = isset($_GET['tab']) && $_GET['tab'] === 'chatbot' && isset($_GET['chatbot_subtab']) && $_GET['chatbot_subtab'] === 'prompts';
        echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '">';
        echo '<input type="hidden" name="action" value="intucart_save_prompt" />';
        if ($embedded) {
            echo '<input type="hidden" name="_intucart_embedded" value="1" />';
        }
        // Preserve ID only when editing; omit on create so server can auto-generate
        if (!empty($id)) {
            echo '<input type="hidden" name="id" value="' . esc_attr($id) . '" />';
        }
        wp_nonce_field('intucart_save_prompt', 'intucart_save_prompt_nonce');

        echo '<table class="form-table">';
        echo '<tr><th scope="row">' . esc_html__('Title', 'intufind') . '</th><td><input type="text" name="title" value="' . esc_attr($title) . '" class="regular-text" /></td></tr>';
        echo '<tr><th scope="row">' . esc_html__('Description', 'intufind') . '</th><td><textarea name="description" rows="3" class="large-text">' . esc_textarea($description) . '</textarea></td></tr>';
        echo '<tr><th scope="row">' . esc_html__('Prompt Text', 'intufind') . '</th><td><textarea name="prompt" rows="3" class="large-text" required>' . esc_textarea($promptText) . '</textarea></td></tr>';
        echo '<tr><th scope="row">' . esc_html__('Initial Prompt', 'intufind') . '</th><td>';
        echo '<label><input type="checkbox" name="is_initial" value="1" ' . checked(true, $isInitial, false) . ' /> ' . esc_html__('Show when chat opens', 'intufind') . '</label>';
        echo '<p class="description">' . esc_html__('If checked, this prompt is eligible to appear in the initial suggestions when the chat widget opens.', 'intufind') . '</p>';
        echo '</td></tr>';

        echo '<tr><th scope="row">' . esc_html__('Canned Response', 'intufind') . '</th><td><textarea name="canned_response" rows="5" class="large-text" placeholder="' . esc_attr__('Optional pre-written response that bypasses AI processing', 'intufind') . '">' . esc_textarea($canned) . '</textarea>';
        echo '<p class="description">' . esc_html__('If provided, this response will be used instead of AI-generated content. Markdown is supported.', 'intufind') . '</p></td></tr>';

        // Override mode - hidden field defaulting to 'override' until fallback mode is implemented
        echo '<input type="hidden" name="override_mode" value="override" />';
        
        // Commented out UI for future fallback implementation
        /*
        echo '<tr><th scope="row">Override Mode</th><td>';
        echo '<select name="override_mode">';
        $modes = ['override', 'fallback'];
        foreach ($modes as $mode) {
            echo '<option value="' . esc_attr($mode) . '" ' . selected($overrideMode, $mode, false) . '>' . esc_html(ucfirst($mode)) . '</option>';
        }
        echo '</select>';
        echo '<p class="description">Override: Use canned response exclusively. Fallback: Use canned response only if AI fails.</p>';
        echo '</td></tr>';
        */

        echo '<tr><th scope="row">' . esc_html__('Products', 'intufind') . '</th><td>';
        $productsSelectId = 'intucart-products-select';
        echo '<select id="' . esc_attr($productsSelectId) . '" name="product_ids[]" multiple class="intucart-select" style="width: 100%" data-action="intucart_search_products" data-placeholder="' . esc_attr__('Search products...', 'intufind') . '" data-nonce="' . esc_attr(wp_create_nonce('intucart_prompts_search')) . '">';
        if (!empty($productIds)) {
            foreach ((array)$productIds as $pid) {
                $p = get_post((int)$pid);
                if ($p && $p->post_type === 'product') {
                    $label = esc_html(wp_specialchars_decode(get_the_title($p), ENT_QUOTES));
                    echo '<option value="' . esc_attr((string)$pid) . '" selected>' . $label . '</option>';
                }
            }
        }
        echo '</select>';
        echo '<p class="description">' . esc_html__('Search and select products to attach to this prompt.', 'intufind') . '</p>';
        echo '</td></tr>';

        echo '<tr><th scope="row">' . esc_html__('Posts', 'intufind') . '</th><td>';
        $postsSelectId = 'intucart-posts-select';
        echo '<select id="' . esc_attr($postsSelectId) . '" name="post_ids[]" multiple class="intucart-select" style="width: 100%" data-action="intucart_search_posts" data-placeholder="' . esc_attr__('Search posts/pages...', 'intufind') . '" data-nonce="' . esc_attr(wp_create_nonce('intucart_prompts_search')) . '">';
        if (!empty($postIds)) {
            foreach ((array)$postIds as $poid) {
                $postObj = get_post((int)$poid);
                if ($postObj && in_array($postObj->post_type, ['post', 'page'], true)) {
                    $label = esc_html(wp_specialchars_decode(get_the_title($postObj), ENT_QUOTES));
                    echo '<option value="' . esc_attr((string)$poid) . '" selected>' . $label . '</option>';
                }
            }
        }
        echo '</select>';
        echo '<p class="description">' . esc_html__('Search and select posts/pages to attach to this prompt.', 'intufind') . '</p>';
        echo '</td></tr>';
        echo '<tr><th scope="row">' . esc_html__('Follow-up Prompts', 'intufind') . '</th><td>';
        $followupsSelectId = 'intucart-followups-select';
        echo '<select id="' . esc_attr($followupsSelectId) . '" name="follow_up_ids[]" multiple class="intucart-select" style="width: 100%" data-action="intucart_search_prompts" data-placeholder="' . esc_attr__('Search prompts...', 'intufind') . '" data-current-id="' . esc_attr((string)$id) . '" data-nonce="' . esc_attr(wp_create_nonce('intucart_prompts_search')) . '">';
        if (!empty($followUpIds)) {
            foreach ((array)$followUpIds as $fid) {
                $p = $this->aiClientManager->getClient()->prompts()->get((string)$fid);
                $label = $p && isset($p['title']) && $p['title'] !== '' ? (string)$p['title'] : (string)$fid;
                $label = esc_html(wp_specialchars_decode($label, ENT_QUOTES));
                echo '<option value="' . esc_attr((string)$fid) . '" selected>' . $label . '</option>';
            }
        }
        echo '</select>';
        echo '<p class="description">' . esc_html__('Select prompts to suggest as follow-ups.', 'intufind') . '</p>';
        echo '</td></tr>';

        echo '<tr><th scope="row">' . esc_html__('Tags', 'intufind') . '</th><td><input type="text" name="tags" value="' . esc_attr($tags) . '" class="regular-text" placeholder="' . esc_attr__('comma-separated', 'intufind') . '" /></td></tr>';
        echo '<tr><th scope="row">' . esc_html__('Categories', 'intufind') . '</th><td><input type="text" name="categories" value="' . esc_attr($categories) . '" class="regular-text" placeholder="' . esc_attr__('comma-separated', 'intufind') . '" /></td></tr>';
        echo '<tr><th scope="row">' . esc_html__('Icon', 'intufind') . '</th><td>';
        echo '<input type="text" id="intucart-icon-input" name="icon" value="' . esc_attr($icon) . '" class="regular-text" placeholder="' . esc_attr__('dashicon slug or URL', 'intufind') . '" /> ';
        Button::secondary(__('Select Image', 'intufind'), [
            'type' => 'button',
            'size' => 'small',
            'attributes' => [
                'class' => 'intucart-icon-select'
            ],
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path opacity=".4" d="M448 384H64v-48l71.51-71.52a12 12 0 0 1 17 0L208 320l135.51-135.52a12 12 0 0 1 17 0L448 272z"/><path d="M464 64H48a48 48 0 0 0-48 48v288a48 48 0 0 0 48 48h416a48 48 0 0 0 48-48V112a48 48 0 0 0-48-48zm-352 56a56 56 0 1 1-56 56 56 56 0 0 1 56-56zm336 264H64v-48l71.51-71.52a12 12 0 0 1 17 0L208 320l135.51-135.52a12 12 0 0 1 17 0L448 272z"/></svg>'
        ]);
        echo ' ';
        Button::secondary(__('Remove', 'intufind'), [
            'type' => 'button',
            'size' => 'small',
            'attributes' => [
                'class' => 'intucart-icon-remove',
                'style' => $icon !== '' ? '' : 'display: none;'
            ],
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 352 512"><path d="M242.72 256l100.07-100.07c12.28-12.28 12.28-32.19 0-44.48l-22.24-22.24c-12.28-12.28-32.19-12.28-44.48 0L176 189.28 75.93 89.21c-12.28-12.28-32.19-12.28-44.48 0L9.21 111.45c-12.28 12.28-12.28 32.19 0 44.48L109.28 256 9.21 356.07c-12.28 12.28-12.28 32.19 0 44.48l22.24 22.24c12.28 12.28 32.2 12.28 44.48 0L176 322.72l100.07 100.07c12.28 12.28 32.2 12.28 44.48 0l22.24-22.24c12.28-12.28 12.28-32.19 0-44.48L242.72 256z"/></svg>'
        ]);
        $isImageUrl = is_string($icon) && preg_match('/^https?:\/\//i', $icon);
        echo '<div class="intucart-icon-preview" style="margin-top:6px;">';
        if ($isImageUrl) {
            echo '<img src="' . esc_url($icon) . '" alt="' . esc_attr__('Icon preview', 'intufind') . '" style="max-width:80px;height:auto;display:block;border:1px solid #ddd;padding:2px;border-radius:2px;" />';
        }
        echo '</div>';
        echo '<p class="description">' . esc_html__('Enter a Dashicons slug (e.g. "dashicons-info") or choose an image from the media library.', 'intufind') . '</p>';
        echo '</td></tr>';
        echo '<tr><th scope="row">' . esc_html__('Priority', 'intufind') . '</th><td><input type="number" name="priority" value="' . esc_attr($priority) . '" class="small-text" /></td></tr>';

        echo '</table>';
        echo '<div class="submit" style="display: flex; gap: 10px; align-items: center;">';
        Button::primary($id ? __('Update Prompt', 'intufind') : __('Create Prompt', 'intufind'), [
            'type' => 'submit',
            'attributes' => [
                'class' => 'intucart-btn-save'
            ],
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path opacity=".4" d="M400 480H48a48 48 0 0 1-48-48V80a48 48 0 0 1 48-48h245.5a48 48 0 0 1 33.94 14.06l106.53 106.53A48 48 0 0 1 448 186.5V432a48 48 0 0 1-48 48zM96 96v128h160V96z"/><path d="M352 480V296a24 24 0 0 0-24-24H120a24 24 0 0 0-24 24v184z"/></svg>'
        ]);
        $embeddedForm = isset($_GET['tab']) && $_GET['tab'] === 'chatbot' && isset($_GET['chatbot_subtab']) && $_GET['chatbot_subtab'] === 'prompts';
        $cancelUrl = $embeddedForm
            ? admin_url('admin.php?page=intucart&tab=chatbot&chatbot_subtab=prompts')
            : admin_url('admin.php?page=intucart-prompts');
        Button::secondary(__('Cancel', 'intufind'), [
            'href' => $cancelUrl,
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 352 512"><path d="M242.72 256l100.07-100.07c12.28-12.28 12.28-32.19 0-44.48l-22.24-22.24c-12.28-12.28-32.19-12.28-44.48 0L176 189.28 75.93 89.21c-12.28-12.28-32.19-12.28-44.48 0L9.21 111.45c-12.28 12.28-12.28 32.19 0 44.48L109.28 256 9.21 356.07c-12.28 12.28-12.28 32.19 0 44.48l22.24 22.24c12.28 12.28 32.2 12.28 44.48 0L176 322.72l100.07 100.07c12.28 12.28 32.2 12.28 44.48 0l22.24-22.24c12.28-12.28 12.28-32.19 0-44.48L242.72 256z"/></svg>'
        ]);
        echo '</div>';
        echo '</form>';
    }

    private function renderPromptSearch(): void
    {
        $embedded = isset($_GET['tab']) && $_GET['tab'] === 'chatbot' && isset($_GET['chatbot_subtab']) && $_GET['chatbot_subtab'] === 'prompts';
        echo '<form method="get" style="display:flex; gap:8px; align-items:center;">';
        if ($embedded) {
            echo '<input type="hidden" name="page" value="intucart" />';
            echo '<input type="hidden" name="tab" value="chatbot" />';
            echo '<input type="hidden" name="chatbot_subtab" value="prompts" />';
        } else {
            echo '<input type="hidden" name="page" value="intucart-prompts" />';
        }
        echo '<input type="text" name="q" value="' . esc_attr($_GET['q'] ?? '') . '" class="regular-text" placeholder="' . esc_attr__('Search prompts by text...', 'intufind') . '" /> ';
        Button::secondary(__('Search', 'intufind'), [
            'type' => 'submit',
            'attributes' => [
                'class' => 'intucart-btn-search'
            ],
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path opacity=".4" d="M208 80a128 128 0 1 1-90.51 37.49A127.15 127.15 0 0 1 208 80m0-80C93.12 0 0 93.12 0 208s93.12 208 208 208 208-93.12 208-208S322.88 0 208 0z"/><path d="M504.9 476.7L476.6 505a23.9 23.9 0 0 1-33.9 0L343 405.3a24 24 0 0 1-7-17V372l36-36h16.3a24 24 0 0 1 17 7l99.7 99.7a24.11 24.11 0 0 1-.1 34z"/></svg>'
        ]);
        $createUrl = $embedded
            ? admin_url('admin.php?page=intucart&tab=chatbot&chatbot_subtab=prompts&create=1')
            : admin_url('admin.php?page=intucart-prompts&create=1');
        Button::primary(__('Create Prompt', 'intufind'), [
            'href' => $createUrl,
            'attributes' => [
                'style' => 'margin-left:auto'
            ],
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path opacity=".4" d="M176 448a32 32 0 0 0 32 32h32a32 32 0 0 0 32-32V304h-96zm64-416h-32a32 32 0 0 0-32 32v144h96V64a32 32 0 0 0-32-32z"/><path d="M448 240v32a32 32 0 0 1-32 32H32a32 32 0 0 1-32-32v-32a32 32 0 0 1 32-32h384a32 32 0 0 1 32 32z"/></svg>'
        ]);
        $refreshUrl = $embedded
            ? admin_url('admin.php?page=intucart&tab=chatbot&chatbot_subtab=prompts' . (!empty($_GET['q']) ? '&q=' . urlencode($_GET['q']) : ''))
            : admin_url('admin.php?page=intucart-prompts' . (!empty($_GET['q']) ? '&q=' . urlencode($_GET['q']) : ''));
        Button::secondary(__('Refresh', 'intufind'), [
            'href' => $refreshUrl,
            'attributes' => [
                'style' => 'margin-left: 8px',
                'title' => __('Refresh prompt list', 'intufind')
            ],
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path opacity=".4" d="M0 500V299.67a12 12 0 0 1 12-12h200.33a12 12 0 0 1 12 12v47.41a12 12 0 0 1-12.57 12l-101.87-4.88a176.07 176.07 0 0 0 317.25-56.94 12 12 0 0 1 11.67-9.26h49.09a12 12 0 0 1 11.8 14.18C478.07 417.08 377.19 504 256 504a247.43 247.43 0 0 1-188.76-87.17l4.13 82.57a12 12 0 0 1-12 12.6H12a12 12 0 0 1-12-12z"/><path d="M12.3 209.82C33.93 94.92 134.81 8 256 8a247.4 247.4 0 0 1 188.9 87.34l-4-82.77A12 12 0 0 1 452.92 0h47.41a12 12 0 0 1 12 12v200.33a12 12 0 0 1-12 12H300a12 12 0 0 1-12-12v-47.41a12 12 0 0 1 12.57-12l101.53 4.88a176.07 176.07 0 0 0-317.24 56.94A12 12 0 0 1 73.19 224H24.1a12 12 0 0 1-11.8-14.18z"/></svg>'
        ]);
        echo '</form>';

        // Add indexing notification (dismissible)
        echo '<div id="intucart-prompt-indexing-notice" class="intucart-notice intucart-notice-info" style="margin: 16px 0; position: relative;">';
        echo '<p><strong>📝 ' . esc_html__('Prompt Indexing:', 'intufind') . '</strong> ' . esc_html__('New or updated prompts may take a few moments to appear in search results and become available in your chat widget. Use the refresh button above to check for the latest changes.', 'intufind') . '</p>';
        echo '<button type="button" class="intucart-notice-dismiss" style="position: absolute; top: 8px; right: 8px; border: none; margin: 0; padding: 4px; background: none; color: currentColor; opacity: 0.6; cursor: pointer; font-size: 13px;" title="' . esc_attr__('Dismiss this notice', 'intufind') . '">';
        echo '<span class="screen-reader-text">' . esc_html__('Dismiss this notice.', 'intufind') . '</span>';
        echo '<span class="dashicons dashicons-dismiss" style="font-size: 16px; width: 16px; height: 16px;"></span>';
        echo '</button>';
        echo '</div>';

        $page = isset($_GET['paged']) ? max(1, (int) $_GET['paged']) : 1;
        $per_page = 20;
        if (!empty($_GET['q'])) {
            $q = sanitize_text_field($_GET['q']);
            $results = [];
            $searchError = null;
            
            try {
                $response = $this->aiClientManager->getClient()->prompts()->search($q, [], $per_page);
                // Search response contains { results: [...], total: N, query: {...} }
                $results = isset($response['results']) && is_array($response['results']) ? $response['results'] : [];
            } catch (\Exception $e) {
                $this->logger->error('Prompt search failed', [
                    'query' => $q,
                    'error' => $e->getMessage()
                ]);
                $searchError = $e->getMessage();
            }
            
            if ($searchError) {
                echo '<div class="intucart-notice intucart-notice-error" style="margin: 16px 0;"><p><strong>' . esc_html__('Search failed:', 'intufind') . '</strong> ' . esc_html($searchError) . '</p></div>';
            }
            
            if (!empty($results)) {
                echo '<table class="widefat intucart-prompts-table" style="margin-top:10px">';
                echo '<thead><tr><th>' . esc_html__('Title', 'intufind') . '</th><th>' . esc_html__('Prompt', 'intufind') . '</th><th>' . esc_html__('Priority', 'intufind') . '</th><th>' . esc_html__('Response', 'intufind') . '</th><th>' . esc_html__('Follow-ups', 'intufind') . '</th><th>' . esc_html__('Actions', 'intufind') . '</th></tr></thead><tbody>';
                foreach ($results as $hit) {
                    $src = is_array($hit) ? $hit : [];
                    $eid = isset($src['id']) ? (string)$src['id'] : (string)($src['external_id'] ?? '');
                    echo '<tr>';
                    echo '<td>' . esc_html($src['title'] ?? '') . '</td>';
                    $promptText = html_entity_decode((string)($src['prompt'] ?? ''), ENT_QUOTES, 'UTF-8');
                    $promptText = mb_strimwidth($promptText, 0, 80, '…');
                    echo '<td>' . esc_html($promptText) . '</td>';
                    echo '<td>' . esc_html((string)($src['priority'] ?? '')) . '</td>';
                    $hasCanned = isset($src['canned_response']) && is_string($src['canned_response']) && trim((string)$src['canned_response']) !== '';
                    $mode = isset($src['override_mode']) ? (string)$src['override_mode'] : '';
                    $cannedLabel = $hasCanned ? ('✓ ' . esc_html(ucfirst($mode !== '' ? $mode : 'override'))) : '—';
                    echo '<td>' . $cannedLabel . '</td>';
                    $fuCount = isset($src['follow_up_ids']) && is_array($src['follow_up_ids']) ? count($src['follow_up_ids']) : 0;
                    echo '<td>' . (int)$fuCount . '</td>';
                    echo '<td style="white-space: nowrap;">';
                    $editUrl = $embedded
                        ? admin_url('admin.php?page=intucart&tab=chatbot&chatbot_subtab=prompts&prompt_id=' . urlencode($eid))
                        : admin_url('admin.php?page=intucart-prompts&prompt_id=' . urlencode($eid));
                    Button::secondary(__('Edit', 'intufind'), [
                        'href' => $editUrl,
                        'size' => 'small',
                        'attributes' => [
                            'class' => 'intucart-btn-edit'
                        ],
                        'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 576 512"><path opacity=".4" d="M564.6 60.2l-48.8-48.8a39.11 39.11 0 0 0-55.2 0l-35.4 35.4a9.78 9.78 0 0 0 0 13.8l90.2 90.2a9.78 9.78 0 0 0 13.8 0l35.4-35.4a39.11 39.11 0 0 0 0-55.2zM427.5 297.6l-40 40a12.3 12.3 0 0 0-3.5 8.5v101.8H64v-320h229.8a12.3 12.3 0 0 0 8.5-3.5l40-40a12 12 0 0 0-8.5-20.5H48a48 48 0 0 0-48 48v352a48 48 0 0 0 48 48h352a48 48 0 0 0 48-48V306.1a12 12 0 0 0-20.5-8.5z"/><path d="M492.8 173.3a9.78 9.78 0 0 1 0 13.8L274.4 405.5l-92.8 10.3a19.45 19.45 0 0 1-21.5-21.5l10.3-92.8L388.8 83.1a9.78 9.78 0 0 1 13.8 0z"/></svg>'
                    ]);
                    echo ' ';
                    echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" style="display:inline">';
                    echo '<input type="hidden" name="action" value="intucart_delete_prompt" />';
                    if ($embedded) {
                        echo '<input type="hidden" name="_intucart_embedded" value="1" />';
                    }
                    wp_nonce_field('intucart_delete_prompt', 'intucart_delete_prompt_nonce');
                    echo '<input type="hidden" name="id" value="' . esc_attr($eid) . '" />';
                    Button::danger(__('Delete', 'intufind'), [
                        'type' => 'submit',
                        'size' => 'small',
                        'attributes' => [
                            'class' => 'intucart-btn-delete',
                            'onclick' => "return confirm('Delete this prompt?')"
                        ],
                        'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path opacity=".4" d="M53.2 467L32 96h384l-21.2 371a48 48 0 0 1-47.9 45H101.1a48 48 0 0 1-47.9-45z"/><path d="M0 80V48a16 16 0 0 1 16-16h120l9.4-18.7A23.72 23.72 0 0 1 166.8 0h114.3a24 24 0 0 1 21.5 13.3L312 32h120a16 16 0 0 1 16 16v32a16 16 0 0 1-16 16H16A16 16 0 0 1 0 80z"/></svg>'
                    ]);
                    echo '</form>';
                    echo '</td>';
                    echo '</tr>';
                }
                echo '</tbody></table>';
                // No built-in pagination for search path here (API returns only limited results)
            } else {
                echo '<table class="widefat" style="margin-top:10px">';
                echo '<thead><tr><th>Title</th><th>Prompt</th><th>Priority</th><th>Response</th><th>Follow-ups</th><th>Actions</th></tr></thead>';
                echo '<tbody><tr><td colspan="6" style="text-align:center; color:#666; padding:16px;">No prompts found.</td></tr></tbody>';
                echo '</table>';
            }
        } else {
            // Paginated list when no search query provided
            $list = $this->aiClientManager->getClient()->prompts()->list($page, $per_page);
            $rows = $list['results'] ?? [];
            $total = (int) ($list['total'] ?? 0);
            $page = (int) ($list['page'] ?? 1);
            $per_page = (int) ($list['perPage'] ?? $per_page);
            if (!empty($rows)) {
                echo '<table class="widefat intucart-prompts-table" style="margin-top:10px">';
                echo '<thead><tr><th>Title</th><th>Prompt</th><th>Initial</th><th>Priority</th><th>Canned</th><th>Follow-ups</th><th>Actions</th></tr></thead><tbody>';
                foreach ($rows as $row) {
                    $eid = $row['id'] ?? '';
                    echo '<tr>';
                    echo '<td>' . esc_html($row['title'] ?? '') . '</td>';
                    $promptText = html_entity_decode((string)($row['prompt'] ?? ''), ENT_QUOTES, 'UTF-8');
                    $promptText = mb_strimwidth($promptText, 0, 80, '…');
                    echo '<td>' . esc_html($promptText) . '</td>';
                    $isInitial = !empty($row['metadata']['initial']);
                    echo '<td><input type="checkbox" disabled ' . ($isInitial ? 'checked' : '') . ' /></td>';
                    echo '<td>' . esc_html((string)($row['priority'] ?? '')) . '</td>';
                    $hasCannedRow = isset($row['canned_response']) && is_string($row['canned_response']) && trim((string)$row['canned_response']) !== '';
                    $modeRow = isset($row['override_mode']) ? (string)$row['override_mode'] : '';
                    $cannedLabelRow = $hasCannedRow ? ('✓ ' . esc_html(ucfirst($modeRow !== '' ? $modeRow : 'override'))) : '—';
                    echo '<td>' . $cannedLabelRow . '</td>';
                    $fuCountRow = isset($row['follow_up_ids']) && is_array($row['follow_up_ids']) ? count($row['follow_up_ids']) : 0;
                    echo '<td>' . (int)$fuCountRow . '</td>';
                    echo '<td style="white-space: nowrap;">';
                    $editUrl2 = $embedded
                        ? admin_url('admin.php?page=intucart&tab=chatbot&chatbot_subtab=prompts&prompt_id=' . urlencode((string)$eid))
                        : admin_url('admin.php?page=intucart-prompts&prompt_id=' . urlencode((string)$eid));
                    Button::secondary('Edit', [
                        'href' => $editUrl2,
                        'size' => 'small',
                        'attributes' => [
                            'class' => 'intucart-btn-edit'
                        ],
                        'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 576 512"><path opacity=".4" d="M564.6 60.2l-48.8-48.8a39.11 39.11 0 0 0-55.2 0l-35.4 35.4a9.78 9.78 0 0 0 0 13.8l90.2 90.2a9.78 9.78 0 0 0 13.8 0l35.4-35.4a39.11 39.11 0 0 0 0-55.2zM427.5 297.6l-40 40a12.3 12.3 0 0 0-3.5 8.5v101.8H64v-320h229.8a12.3 12.3 0 0 0 8.5-3.5l40-40a12 12 0 0 0-8.5-20.5H48a48 48 0 0 0-48 48v352a48 48 0 0 0 48 48h352a48 48 0 0 0 48-48V306.1a12 12 0 0 0-20.5-8.5z"/><path d="M492.8 173.3a9.78 9.78 0 0 1 0 13.8L274.4 405.5l-92.8 10.3a19.45 19.45 0 0 1-21.5-21.5l10.3-92.8L388.8 83.1a9.78 9.78 0 0 1 13.8 0z"/></svg>'
                    ]);
                    echo ' ';
                    echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" style="display:inline">';
                    echo '<input type="hidden" name="action" value="intucart_delete_prompt" />';
                    if ($embedded) {
                        echo '<input type="hidden" name="_intucart_embedded" value="1" />';
                    }
                    wp_nonce_field('intucart_delete_prompt', 'intucart_delete_prompt_nonce');
                    echo '<input type="hidden" name="id" value="' . esc_attr((string)$eid) . '" />';
                    Button::danger('Delete', [
                        'type' => 'submit',
                        'size' => 'small',
                        'attributes' => [
                            'class' => 'intucart-btn-delete',
                            'onclick' => "return confirm('Delete this prompt?')"
                        ],
                        'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path opacity=".4" d="M53.2 467L32 96h384l-21.2 371a48 48 0 0 1-47.9 45H101.1a48 48 0 0 1-47.9-45z"/><path d="M0 80V48a16 16 0 0 1 16-16h120l9.4-18.7A23.72 23.72 0 0 1 166.8 0h114.3a24 24 0 0 1 21.5 13.3L312 32h120a16 16 0 0 1 16 16v32a16 16 0 0 1-16 16H16A16 16 0 0 1 0 80z"/></svg>'
                    ]);
                    echo '</form>';
                    echo '</td>';
                    echo '</tr>';
                }
                echo '</tbody></table>';
                // Pagination links
                if ($total > $per_page) {
                    $total_pages = (int) ceil($total / $per_page);
                    $base_url = remove_query_arg('paged');
                    echo '<div class="tablenav"><div class="tablenav-pages">';
                    echo paginate_links([
                        'base' => add_query_arg('paged', '%#%', $base_url),
                        'format' => '',
                        'current' => $page,
                        'total' => $total_pages,
                        'prev_text' => '&laquo;',
                        'next_text' => '&raquo;',
                    ]);
                    echo '</div></div>';
                }
            } else {
                echo '<table class="widefat intucart-prompts-table" style="margin-top:10px">';
                echo '<thead><tr><th>Title</th><th>Prompt</th><th>Initial</th><th>Priority</th><th>Canned</th><th>Follow-ups</th><th>Actions</th></tr></thead>';
                echo '<tbody><tr><td colspan="7" style="text-align:center; color:#666; padding:16px;">No prompts found.</td></tr></tbody>';
                echo '</table>';
            }
        }
    }

    public function handleSavePrompt(): void
    {
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        check_admin_referer('intucart_save_prompt', 'intucart_save_prompt_nonce');

        $payload = [
            'id' => isset($_POST['id']) ? sanitize_text_field($_POST['id']) : '',
            'title' => sanitize_text_field($_POST['title'] ?? ''),
            'prompt' => wp_unslash(sanitize_textarea_field($_POST['prompt'] ?? '')),
            'description' => wp_unslash(sanitize_textarea_field($_POST['description'] ?? '')),
            'canned_response' => wp_unslash(sanitize_textarea_field($_POST['canned_response'] ?? '')),
            'response_format' => 'markdown',
            'product_ids' => isset($_POST['product_ids']) ? array_map('strval', (array) $_POST['product_ids']) : [],
            'post_ids' => isset($_POST['post_ids']) ? array_map('strval', (array) $_POST['post_ids']) : [],
            'follow_up_ids' => isset($_POST['follow_up_ids']) ? array_map('strval', (array) $_POST['follow_up_ids']) : [],
            'tags' => $this->parseCsv($_POST['tags'] ?? ''),
            'categories' => $this->parseCsv($_POST['categories'] ?? ''),
            'icon' => sanitize_text_field($_POST['icon'] ?? ''),
            'priority' => intval($_POST['priority'] ?? 0),
            'metadata' => [
                'initial' => isset($_POST['is_initial']) && $_POST['is_initial'] === '1',
            ],
        ];

        // Always set override_mode to 'override' until fallback mode is implemented
        $payload['override_mode'] = 'override';

        $result = $this->aiClientManager->getClient()->prompts()->upsert($payload);
        $ok = $result['success'] ?? false;
        // Invalidate initial prompts cache on save
        // Invalidate initial prompts cache via injected cache service
        $this->cache->delete(Constants::INITIAL_PROMPTS_CACHE_KEY, Constants::PROMPTS_CACHE_GROUP);
        // Redirect based on current location (embedded or standalone)
        $redirect = isset($_POST['_intucart_embedded']) && $_POST['_intucart_embedded'] === '1'
            ? admin_url('admin.php?page=intucart&tab=chatbot&chatbot_subtab=prompts')
            : admin_url('admin.php?page=intucart-prompts');
        if ($ok) {
            wp_safe_redirect(add_query_arg(['updated' => '1'], $redirect));
        } else {
            wp_safe_redirect(add_query_arg(['error' => '1'], $redirect));
        }
        exit;
    }

    public function handleDeletePrompt(): void
    {
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        check_admin_referer('intucart_delete_prompt', 'intucart_delete_prompt_nonce');
        $id = sanitize_text_field($_POST['id'] ?? '');
        if ($id) {
            $this->aiClientManager->getClient()->prompts()->delete($id);
            // Invalidate initial prompts cache on delete via injected cache service
            $this->cache->delete(Constants::INITIAL_PROMPTS_CACHE_KEY, Constants::PROMPTS_CACHE_GROUP);
        }
        $redirect = isset($_POST['_intucart_embedded']) && $_POST['_intucart_embedded'] === '1'
            ? admin_url('admin.php?page=intucart&tab=chatbot&chatbot_subtab=prompts')
            : admin_url('admin.php?page=intucart-prompts');
        wp_safe_redirect($redirect);
        exit;
    }

    private function parseCsv(string $input): array
    {
        $input = trim($input);
        if ($input === '') {
            return [];
        }
        $parts = array_filter(array_map('trim', explode(',', $input)), static function ($v) {
            return $v !== '';
        });
        // Normalize to strings
        return array_map('strval', $parts);
    }

    private function enqueueSelectAssets(): void
    {
        // WooCommerce admin bundles selectWoo (Select2). If not present, enqueue WP core select2 if available.
        // We defensively enqueue styles/scripts if not yet enqueued.
        if (class_exists('WooCommerce')) {
            // WooCommerce uses selectWoo handle
            if (!wp_script_is('selectWoo', 'enqueued')) {
                wp_enqueue_script('selectWoo');
            }
            if (!wp_style_is('select2', 'enqueued')) {
                wp_enqueue_style('select2');
            }
        } else {
            // Fallback: try core select2 handles if registered by theme/plugins
            if (wp_script_is('select2', 'registered') && !wp_script_is('select2', 'enqueued')) {
                wp_enqueue_script('select2');
            }
            if (wp_style_is('select2', 'registered') && !wp_style_is('select2', 'enqueued')) {
                wp_enqueue_style('select2');
            }
        }
    }

    public function handleAjaxSearchProducts(): void
    {
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Unauthorized', 'intufind'), 403);
        }
        check_ajax_referer('intucart_prompts_search', 'nonce');
        $q = isset($_GET['q']) ? sanitize_text_field(wp_unslash($_GET['q'])) : '';
        $args = [
            'post_type' => 'product',
            's' => $q,
            'posts_per_page' => 20,
            'post_status' => 'publish',
            'fields' => 'ids',
        ];
        $ids = get_posts($args);
        $results = [];
        foreach ((array) $ids as $id) {
            $results[] = [ 'id' => (string) $id, 'text' => get_the_title($id) ];
        }
        wp_send_json($results);
    }

    public function handleAjaxSearchPosts(): void
    {
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Unauthorized', 'intufind'), 403);
        }
        check_ajax_referer('intucart_prompts_search', 'nonce');
        $q = isset($_GET['q']) ? sanitize_text_field(wp_unslash($_GET['q'])) : '';
        $args = [
            'post_type' => ['post', 'page'],
            's' => $q,
            'posts_per_page' => 20,
            'post_status' => 'publish',
            'fields' => 'ids',
        ];
        $ids = get_posts($args);
        $results = [];
        foreach ((array) $ids as $id) {
            $results[] = [ 'id' => (string) $id, 'text' => get_the_title($id) ];
        }
        wp_send_json($results);
    }

    public function handleAjaxSearchPrompts(): void
    {
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Unauthorized', 'intufind'), 403);
        }
        check_ajax_referer('intucart_prompts_search', 'nonce');
        $q = isset($_GET['q']) ? sanitize_text_field(wp_unslash($_GET['q'])) : '';

        // Use cloud search to find prompts by text
        $results = [];
        try {
            if (trim($q) === '') {
                wp_send_json($results);
                return;
            }
            
            $response = $this->aiClientManager->getClient()->prompts()->search($q, [], 20);
            // Search response contains { results: [...], total: N, query: {...} }
            $hits = isset($response['results']) && is_array($response['results']) ? $response['results'] : [];
            $exclude = isset($_GET['exclude_id']) ? sanitize_text_field(wp_unslash($_GET['exclude_id'])) : '';
            foreach ($hits as $hit) {
                $src = is_array($hit) ? $hit : [];
                $id = (string)($src['id'] ?? $src['external_id'] ?? '');
                if ($id === '') {
                    continue;
                }
                if ($exclude !== '' && $id === $exclude) {
                    continue; // don't suggest the current prompt as its own follow-up
                }
                $title = (string)($src['title'] ?? $src['prompt'] ?? $id);
                $results[] = [ 'id' => $id, 'text' => $title ];
            }
        } catch (\Exception $e) {
            $this->logger->error('Prompt search failed', [
                'query' => $q,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            // Return error in a way that Select2 can understand
            wp_send_json_error([
                'message' => 'Search failed: ' . $e->getMessage()
            ], 500);
            return;
        }
        wp_send_json($results);
    }
}
