<?php

namespace Intucart\Pages\Settings;

use Intucart\Services\Logger;

/**
 * Abstract base class for settings with UI helper methods
 */
abstract class AbstractSettings
{
    protected Logger $logger;

    public function __construct(Logger $logger)
    {
        $this->logger = $logger;
    }

    /**
     * Register settings
     *
     * @return void
     */
    abstract public function registerSettings(): void;

    // ========================================
    // UI HELPER METHODS
    // ========================================

    /**
     * Render a settings panel container
     *
     * @param string $title Panel title
     * @param string $description Panel description
     * @param string $icon Dashicon class (without 'dashicons-')
     * @param callable $content_callback Callback to render panel content
     * @return void
     */
    protected function renderSettingsPanel(string $title, string $description, string $icon, callable $content_callback): void
    {
        echo '<div class="intucart-content-panel">';
        echo '<h3 class="intucart-panel-header">';
        echo '<span class="dashicons dashicons-' . esc_attr($icon) . '"></span>';
        echo esc_html($title);
        echo '</h3>';
        echo '<p class="intucart-panel-description">' . esc_html($description) . '</p>';
        
        call_user_func($content_callback);
        
        echo '</div>';
    }

    /**
     * Render a form table row
     *
     * @param string $label Row label
     * @param callable $content_callback Callback to render row content
     * @return void
     */
    protected function renderTableRow(string $label, callable $content_callback): void
    {
        echo '<tr>';
        echo '<th scope="row">' . esc_html($label) . '</th>';
        echo '<td>';
        call_user_func($content_callback);
        echo '</td>';
        echo '</tr>';
    }

    /**
     * Render a color picker input
     *
     * @param string $name Input name
     * @param string $value Current value
     * @param string $label Optional label
     * @param bool $disabled Whether the input is disabled
     * @return void
     */
    protected function renderColorPicker(string $name, string $value, string $label = '', bool $disabled = false): void
    {
        if ($label) {
            echo '<label style="font-weight: 500;">' . esc_html($label) . '</label>';
        }
        
        $disabled_attr = $disabled ? 'disabled' : '';
        $disabled_style = $disabled ? 'style="opacity: 0.5;"' : '';
        
        // Use WordPress color picker (allows both clicking and manual hex input)
        echo '<input type="text" name="' . esc_attr($name) . '" value="' . esc_attr($value) . '" class="itf-color-picker" ' . $disabled_attr . ' ' . $disabled_style . ' />';
    }

    /**
     * Render a color picker with optional "None" checkbox
     *
     * @param string $name Input name
     * @param string $value Current value
     * @param string $label Input label
     * @param string $default_color Default color when None is unchecked
     * @param bool $is_none Whether None is currently selected
     * @return void
     */
    protected function renderOptionalColorPicker(string $name, string $value, string $label, string $default_color = '#ffffff', bool $is_none = false): void
    {
        echo '<div style="display: flex; align-items: center; gap: 8px;">';
        echo '<label style="font-weight: 500;">' . esc_html($label) . ':</label>';
        
        $display_value = $is_none ? $default_color : $value;
        $this->renderColorPicker($name, $display_value, '', $is_none);
        
        echo '<label style="margin-left: 4px;">';
        echo '<input type="checkbox" ' . ($is_none ? 'checked' : '') . ' onchange="toggleColorPicker(this, \'' . esc_attr($name) . '\', \'' . esc_attr($default_color) . '\')">';
        echo ' None</label>';
        echo '</div>';
    }

    /**
     * Render multiple color pickers in a row
     *
     * @param array $colors Array of ['name' => string, 'value' => string, 'label' => string]
     * @return void
     */
    protected function renderColorPickerGroup(array $colors): void
    {
        echo '<div style="display: flex; gap: 20px; align-items: center; flex-wrap: wrap;">';
        
        foreach ($colors as $color) {
            echo '<div style="display: flex; align-items: center; gap: 8px;">';
            $this->renderColorPicker($color['name'], $color['value'], $color['label']);
            echo '</div>';
        }
        
        echo '</div>';
    }

    /**
     * Render a number input
     *
     * @param string $name Input name
     * @param int $value Current value
     * @param int $min Minimum value
     * @param int $max Maximum value
     * @param string $unit Unit suffix (e.g., 'px', '%')
     * @param string $class CSS class
     * @return void
     */
    protected function renderNumberInput(string $name, int $value, int $min = 0, int $max = 100, string $unit = '', string $class = 'small-text'): void
    {
        echo '<input type="number" name="' . esc_attr($name) . '" value="' . esc_attr($value) . '" ';
        echo 'min="' . esc_attr($min) . '" max="' . esc_attr($max) . '" class="' . esc_attr($class) . '" />';
        
        if ($unit) {
            echo ' ' . esc_html($unit);
        }
    }

    /**
     * Render a checkbox input
     *
     * @param string $name Input name
     * @param string $value Input value (usually '1')
     * @param bool $checked Whether checkbox is checked
     * @param string $label Checkbox label
     * @return void
     */
    protected function renderCheckbox(string $name, string $value, bool $checked, string $label): void
    {
        echo '<label>';
        echo '<input type="checkbox" name="' . esc_attr($name) . '" value="' . esc_attr($value) . '" ' . checked($checked, true, false) . ' />';
        echo ' ' . esc_html($label);
        echo '</label>';
    }

    /**
     * Render a select dropdown
     *
     * @param string $name Select name
     * @param array $options Array of value => label pairs
     * @param string $selected Currently selected value
     * @return void
     */
    protected function renderSelect(string $name, array $options, string $selected): void
    {
        echo '<select name="' . esc_attr($name) . '">';
        
        foreach ($options as $value => $label) {
            echo '<option value="' . esc_attr($value) . '" ' . selected($selected, $value, false) . '>';
            echo esc_html($label);
            echo '</option>';
        }
        
        echo '</select>';
    }

    /**
     * Render a hint/description box
     *
     * @param string $text Hint text
     * @param array $examples Optional array of example strings
     * @return void
     */
    protected function renderHint(string $text, array $examples = []): void
    {
        echo '<div class="intucart-hint">';
        echo '<div class="intucart-hint-section">' . wp_kses_post($text) . '</div>';
        
        if (!empty($examples)) {
            echo '<div class="intucart-hint-examples">';
            echo '💡 <strong>Examples:</strong> ';
            echo implode(', ', array_map('esc_html', $examples));
            echo '</div>';
        }
        
        echo '</div>';
    }

    /**
     * Render a textarea input
     *
     * @param string $name Input name
     * @param string $value Current value
     * @param int $rows Number of rows
     * @param string $placeholder Placeholder text
     * @param string $class CSS class
     * @return void
     */
    protected function renderTextarea(string $name, string $value, int $rows = 3, string $placeholder = '', string $class = 'large-text'): void
    {
        echo '<textarea name="' . esc_attr($name) . '" rows="' . esc_attr($rows) . '" class="' . esc_attr($class) . '"';
        
        if ($placeholder) {
            echo ' placeholder="' . esc_attr($placeholder) . '"';
        }
        
        echo '>' . esc_textarea($value) . '</textarea>';
    }

    /**
     * Render a text input
     *
     * @param string $name Input name
     * @param string $value Current value
     * @param string $placeholder Placeholder text
     * @param string $class CSS class
     * @return void
     */
    protected function renderTextInput(string $name, string $value, string $placeholder = '', string $class = 'regular-text'): void
    {
        echo '<input type="text" name="' . esc_attr($name) . '" value="' . esc_attr($value) . '" class="' . esc_attr($class) . '"';
        
        if ($placeholder) {
            echo ' placeholder="' . esc_attr($placeholder) . '"';
        }
        
        echo ' />';
    }

    /**
     * Render a section header within a panel
     *
     * @param string $title Section title
     * @param string $icon Dashicon class (without 'dashicons-')
     * @return void
     */
    protected function renderSectionHeader(string $title, string $icon): void
    {
        echo '<tr><th scope="row" colspan="2" style="background: #ffffff; padding: 12px; border-radius: 4px; border: 1px solid #e1e5e9;">';
        echo '<h4 style="margin: 0; color: #495057; display: flex; align-items: center;">';
        echo '<span class="dashicons dashicons-' . esc_attr($icon) . '" style="margin-right: 8px; font-size: 16px;"></span>';
        echo esc_html($title);
        echo '</h4>';
        echo '</th></tr>';
    }

}
