<?php

namespace Intucart\Pages\Settings\Chat;

use Intucart\Pages\Settings\AbstractSettings;
use Intucart\Services\AIClientManager;
use Intucart\Services\Logger;
use Intucart\Services\Constants;
use Intucart\Components\Button;

/**
 * Live Agent Handoff Settings Page
 * 
 * Allows admins to configure live agent handoff with providers:
 * - Slack (FREE with existing workspace)
 * - Zendesk (Enterprise)
 * - Intercom (Enterprise)
 * - Drift (Enterprise)
 */
class LiveAgentSettings extends AbstractSettings
{
    private AIClientManager $aiClientManager;

    public function __construct(Logger $logger, AIClientManager $aiClientManager)
    {
        parent::__construct($logger);
        $this->aiClientManager = $aiClientManager;
        
        // Register early hooks for form handling (must run before any output)
        $this->init();
    }

    public function registerSettings(): void
    {
        // Register provider configuration settings with sanitization callback
        register_setting(Constants::SETTINGS_GROUP_LIVE_AGENT_PROVIDER, Constants::OPTION_LIVE_AGENT_PROVIDER, [
            'sanitize_callback' => [$this, 'sanitizeProviderChange']
        ]);
        
        // Register behavior settings
        register_setting(Constants::SETTINGS_GROUP_LIVE_AGENT_BEHAVIOR, Constants::OPTION_LIVE_AGENT_ENABLED);
        register_setting(Constants::SETTINGS_GROUP_LIVE_AGENT_BEHAVIOR, Constants::OPTION_LIVE_AGENT_AUTO_HANDOFF);
        register_setting(Constants::SETTINGS_GROUP_LIVE_AGENT_BEHAVIOR, Constants::OPTION_LIVE_AGENT_MANUAL_HANDOFF);
        register_setting(Constants::SETTINGS_GROUP_LIVE_AGENT_BEHAVIOR, Constants::OPTION_LIVE_AGENT_BUSINESS_HOURS);
        register_setting(Constants::SETTINGS_GROUP_LIVE_AGENT_BEHAVIOR, Constants::OPTION_LIVE_AGENT_AVAILABILITY_SCHEDULE, [
            'sanitize_callback' => [$this, 'syncAvailabilitySchedule']
        ]);
        register_setting(Constants::SETTINGS_GROUP_LIVE_AGENT_BEHAVIOR, Constants::OPTION_LIVE_AGENT_ESCALATION_PROMPT);
        register_setting(Constants::SETTINGS_GROUP_LIVE_AGENT_BEHAVIOR, Constants::OPTION_LIVE_AGENT_INACTIVITY_TIMEOUT, [
            'type' => 'integer',
            'default' => Constants::DEFAULT_LIVE_AGENT_INACTIVITY_TIMEOUT,
            'sanitize_callback' => [$this, 'sanitizeInactivityTimeout']
        ]);
        register_setting(Constants::SETTINGS_GROUP_LIVE_AGENT_BEHAVIOR, Constants::OPTION_CONTACT_FORM_ENABLED, [
            'type' => 'boolean',
            'default' => false,
            'sanitize_callback' => 'rest_sanitize_boolean'
        ]);
        
        // Add settings sections
        add_settings_section(
            'intucart_live_agent_provider_section',
            'Provider Configuration',
            [$this, 'renderProviderSectionDescription'],
            Constants::SETTINGS_GROUP_LIVE_AGENT_PROVIDER
        );
        
        add_settings_section(
            'intucart_live_agent_behavior_section',
            'Handoff Behavior',
            [$this, 'renderBehaviorSectionDescription'],
            Constants::SETTINGS_GROUP_LIVE_AGENT_BEHAVIOR
        );
    }

    public function renderProviderSectionDescription(): void
    {
        echo '<p class="description">';
        echo esc_html__('Select your preferred live chat provider.', 'intufind');
        echo '</p>';
    }

    public function renderBehaviorSectionDescription(): void
    {
        echo '<p class="description">';
        echo esc_html__('Configure when and how to handoff conversations to human agents.', 'intufind');
        echo '</p>';
    }

    /**
     * Sanitize provider change and delete old credentials if provider changed
     */
    public function sanitizeProviderChange($newProvider): string
    {
        $oldProvider = get_option(Constants::OPTION_LIVE_AGENT_PROVIDER, Constants::LIVE_AGENT_PROVIDER_NONE);
        $newProvider = sanitize_text_field($newProvider);
        
        // If provider changed and old provider had credentials, delete them
        if ($oldProvider !== $newProvider && $oldProvider !== Constants::LIVE_AGENT_PROVIDER_NONE) {
            $licenseKey = get_option(Constants::LICENSE_KEY_OPTION, '');
            
            if (!empty($licenseKey)) {
                try {
                    $response = $this->aiClientManager->deleteLiveAgentCredentials($oldProvider);
                    
                    // SDK unwraps 'data' envelope, so check for 'message' (indicates success)
                    if (isset($response['message'])) {
                        // Clear the cached configured status and disable feature when provider changes
                        update_option(Constants::OPTION_LIVE_AGENT_CONFIGURED, false);
                        update_option(Constants::OPTION_LIVE_AGENT_ENABLED, false);
                        
                        add_settings_error(
                            'intucart_live_agent_messages',
                            'intucart_live_agent_provider_change',
                            sprintf(__('Provider changed to %s. Previous %s credentials have been deleted.', 'intufind'), 
                                ucfirst($newProvider === Constants::LIVE_AGENT_PROVIDER_NONE ? __('None', 'intufind') : $newProvider),
                                ucfirst($oldProvider)
                            ),
                            'success'
                        );
                    }
                } catch (\Exception $e) {
                    $this->logger->error('Failed to delete old provider credentials during provider change', [
                        'old_provider' => $oldProvider,
                        'new_provider' => $newProvider,
                        'error' => $e->getMessage()
                    ]);
                    // Don't show error to user - just log it. The provider change should still succeed.
                }
            }
        }
        
        // If changing to 'none', also clear configured status and disable feature
        if ($newProvider === Constants::LIVE_AGENT_PROVIDER_NONE) {
            update_option(Constants::OPTION_LIVE_AGENT_CONFIGURED, false);
            update_option(Constants::OPTION_LIVE_AGENT_ENABLED, false);
        }
        
        return $newProvider;
    }

    /**
     * Sanitize availability schedule and sync to cloud API
     */
    public function syncAvailabilitySchedule($value): string
    {
        $schedule = sanitize_textarea_field($value);
        
        // Sync to cloud API (non-blocking, log errors but don't fail the save)
        $this->saveAvailabilitySchedule($schedule);
        
        return $schedule;
    }

    /**
     * Sanitize inactivity timeout
     * Ensures value is between 1 minute (60s) and 60 minutes (3600s)
     */
    public function sanitizeInactivityTimeout($value): int
    {
        $value = absint($value);
        
        // Minimum 1 minute, maximum 60 minutes
        if ($value < 60) {
            $value = 60;
        } elseif ($value > 3600) {
            $value = 3600;
        }
        
        return $value;
    }

    /**
     * Initialize hooks for handling form submissions
     * Called early (admin_init) before any HTML output, so redirects work
     */
    public function init(): void
    {
        add_action('admin_init', [$this, 'handleFormSubmissions']);
    }

    /**
     * Handle form submissions - must run on admin_init before any output
     */
    public function handleFormSubmissions(): void
    {
        // Only process on our page
        if (!isset($_GET['page']) || $_GET['page'] !== 'intucart') {
            return;
        }
        if (!isset($_GET['tab']) || $_GET['tab'] !== 'chatbot') {
            return;
        }
        if (!isset($_GET['chatbot_subtab']) || $_GET['chatbot_subtab'] !== 'liveagent') {
            return;
        }
        
        // Only process POST requests with our action
        if (!isset($_POST['action'])) {
            return;
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Verify nonce
        if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'intucart_live_agent_credentials')) {
            return;
        }
        
        // Build redirect URL
        $redirect_url = admin_url('admin.php?page=intucart&tab=chatbot&chatbot_subtab=liveagent');
        
        if ($_POST['action'] === 'save_credentials') {
            $this->handleSaveCredentials();
            set_transient('settings_errors', get_settings_errors(), 30);
            wp_safe_redirect(add_query_arg('settings-updated', 'credentials', $redirect_url));
            exit;
        } elseif ($_POST['action'] === 'delete_credentials') {
            $this->handleDeleteCredentials();
            set_transient('settings_errors', get_settings_errors(), 30);
            wp_safe_redirect(add_query_arg('settings-updated', 'credentials', $redirect_url));
            exit;
        } elseif ($_POST['action'] === 'save_channel') {
            $this->handleSaveChannel();
            set_transient('settings_errors', get_settings_errors(), 30);
            wp_safe_redirect(add_query_arg('settings-updated', 'channel', $redirect_url));
            exit;
        }
    }

    /**
     * Render the Live Agent settings page
     */
    public function render(): void
    {
        // Check permissions
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Get current settings
        $enabled = get_option(Constants::OPTION_LIVE_AGENT_ENABLED, false);
        $provider = get_option(Constants::OPTION_LIVE_AGENT_PROVIDER, Constants::LIVE_AGENT_PROVIDER_NONE);
        $autoHandoff = get_option(Constants::OPTION_LIVE_AGENT_AUTO_HANDOFF, true);
        $manualHandoff = get_option(Constants::OPTION_LIVE_AGENT_MANUAL_HANDOFF, true);
        $contactFormEnabled = get_option(Constants::OPTION_CONTACT_FORM_ENABLED, false);
        $businessHours = get_option(Constants::OPTION_LIVE_AGENT_BUSINESS_HOURS, '');
        $availabilitySchedule = get_option(Constants::OPTION_LIVE_AGENT_AVAILABILITY_SCHEDULE, '');
        
        // Get license key (needed below for credential status and schedule loading)
        $license = get_option(Constants::LICENSE_KEY_OPTION, '');
        
        // Load schedule from cloud API if we have a license and provider configured
        if (!empty($license) && $provider !== Constants::LIVE_AGENT_PROVIDER_NONE && empty($availabilitySchedule)) {
            $availabilitySchedule = $this->loadAvailabilitySchedule();
        }
        
        // Get escalation prompt with smart default
        $escalationPrompt = get_option(Constants::OPTION_LIVE_AGENT_ESCALATION_PROMPT);
        
        // If not set (first time), provide a smart default based on MCP configuration
        if ($escalationPrompt === false) {
            // Check if MCP Order Support is enabled
            $hasMcpOrderSupport = get_option(Constants::OPTION_ENABLE_MCP_ORDERS, true) && class_exists('WooCommerce');
            
            if ($hasMcpOrderSupport) {
                // MCP enabled - exclude order/billing lookup questions from escalation (AI will handle via MCP)
                $escalationPrompt = 'Escalate requests that require human judgment, empathy, or actions the AI cannot perform (e.g., refunds, cancellations, account closures, complex policy exceptions, sensitive customer situations).';
            } else {
                // No MCP - include order/billing issues in escalation
                $escalationPrompt = 'Always escalate: order inquiries, refunds, cancellations, order disputes, billing issues, account problems, or any request requiring system access the AI cannot provide.';
            }
        }
        
        $inactivityTimeout = get_option(Constants::OPTION_LIVE_AGENT_INACTIVITY_TIMEOUT, Constants::DEFAULT_LIVE_AGENT_INACTIVITY_TIMEOUT);

        // Check credential status (only if we have a license and provider is selected)
        $credentialStatus = ['configured' => false, 'configurationType' => 'none'];
        if (!empty($license) && $provider !== Constants::LIVE_AGENT_PROVIDER_NONE) {
            $credentialStatus = $this->getCredentialStatus($provider, $license);
        }
        
        $isConfigured = $credentialStatus['configured'];
        $configurationType = $credentialStatus['configurationType'];
        $teamName = $credentialStatus['teamName'] ?? null;
        
        // Sync the cached configured status with the actual status from the API
        // This ensures the widget has the correct value (e.g., after OAuth callback)
        $cachedConfigured = (bool) get_option(Constants::OPTION_LIVE_AGENT_CONFIGURED, false);
        if ($cachedConfigured !== $isConfigured) {
            update_option(Constants::OPTION_LIVE_AGENT_CONFIGURED, $isConfigured);
            
            // If credentials were removed externally, also disable the feature
            if (!$isConfigured && $enabled) {
                update_option(Constants::OPTION_LIVE_AGENT_ENABLED, false);
                $enabled = false;
            }
        }

        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>

            <?php
            // Show success message after credential save/delete
            if (isset($_GET['settings-updated'])) {
                if ($_GET['settings-updated'] === 'credentials') {
                    echo '<div class="intucart-notice intucart-notice-success"><p>✓ ' . esc_html__('Credentials updated successfully!', 'intufind') . '</p></div>';
                } elseif ($_GET['settings-updated'] === 'channel') {
                    echo '<div class="intucart-notice intucart-notice-success"><p>✓ ' . esc_html__('Channel configuration updated successfully!', 'intufind') . '</p></div>';
                }
            }
            ?>

            <?php settings_errors('intucart_live_agent_messages'); ?>

            <?php if (empty($license)): ?>
                <div class="intucart-notice intucart-notice-warning">
                    <p><strong>⚠️ <?php echo esc_html__('License Key Required', 'intufind'); ?></strong></p>
                    <p><?php echo esc_html__('Please configure your license key in Settings → General before setting up live agent handoff.', 'intufind'); ?></p>
                </div>
            <?php endif; ?>

            <!-- Provider Selection -->
            <div class="intucart-settings-section">
                <h2><?php echo esc_html__('Provider Configuration', 'intufind'); ?></h2>
                <form method="post" action="options.php">
                    <?php settings_fields(Constants::SETTINGS_GROUP_LIVE_AGENT_PROVIDER); ?>
                    
                    <table class="form-table" role="presentation">
                        <tr>
                            <th scope="row">
                                <label for="intucart_live_agent_provider"><?php echo esc_html__('Live Agent Provider', 'intufind'); ?></label>
                            </th>
                            <td>
                                <select name="<?php echo esc_attr(Constants::OPTION_LIVE_AGENT_PROVIDER); ?>" id="intucart_live_agent_provider">
                                    <option value="<?php echo esc_attr(Constants::LIVE_AGENT_PROVIDER_NONE); ?>" <?php selected($provider, Constants::LIVE_AGENT_PROVIDER_NONE); ?>><?php echo esc_html__('None (Disabled)', 'intufind'); ?></option>
                                    <option value="<?php echo esc_attr(Constants::LIVE_AGENT_PROVIDER_SLACK); ?>" <?php selected($provider, Constants::LIVE_AGENT_PROVIDER_SLACK); ?>><?php echo esc_html__('Slack (FREE with workspace)', 'intufind'); ?></option>
                                    <option value="" disabled><?php echo esc_html__('WhatsApp (Coming Soon)', 'intufind'); ?></option>
                                    <option value="" disabled><?php echo esc_html__('Zendesk (Coming Soon)', 'intufind'); ?></option>
                                </select>
                                <p class="description"><?php echo esc_html__('Choose which platform to use for live agent conversations', 'intufind'); ?></p>

                                <?php if ($provider !== Constants::LIVE_AGENT_PROVIDER_NONE): ?>
                                    <p style="margin-top: 10px;">
                                        <strong><?php echo esc_html__('Status:', 'intufind'); ?></strong>
                                        <?php if ($isConfigured): ?>
                                            <span class="intucart-badge intucart-badge-success">
                                                ✓ <?php echo esc_html__('Configured', 'intufind'); ?>
                                                <?php if ($configurationType === 'oauth'): ?>
                                                    (<?php echo esc_html__('App', 'intufind'); ?>)
                                                <?php elseif ($configurationType === 'manual'): ?>
                                                    (<?php echo esc_html__('Custom Bot', 'intufind'); ?>)
                                                <?php endif; ?>
                                            </span>
                                            <?php if ($teamName): ?>
                                                <span style="margin-left: 8px; color: #666;"><?php echo esc_html__('Workspace:', 'intufind'); ?> <?php echo esc_html($teamName); ?></span>
                                            <?php endif; ?>
                                        <?php else: ?>
                                            <span class="intucart-badge intucart-badge-warning"><?php echo esc_html__('Not Configured', 'intufind'); ?></span>
                                        <?php endif; ?>
                                    </p>
                                <?php endif; ?>
                            </td>
                        </tr>
                    </table>

                    <div style="margin-top: 20px;">
                        <?php
                        Button::primary(__('Save Provider', 'intufind'), ['type' => 'submit']);
                        ?>
                    </div>
                </form>
            </div>

            <!-- Provider-Specific Credentials -->
            <?php if ($provider === Constants::LIVE_AGENT_PROVIDER_SLACK): ?>
                <?php $this->renderSlackCredentials($license); ?>
            <?php elseif ($provider === Constants::LIVE_AGENT_PROVIDER_ZENDESK): ?>
                <?php $this->renderZendeskCredentials($license); ?>
            <?php endif; ?>

            <!-- Customer Support Options (always visible) -->
            <div class="intucart-settings-section">
                <h2><?php echo esc_html__('Customer Support Options', 'intufind'); ?></h2>
                <p class="description" style="margin-bottom: 16px;">
                    <?php echo esc_html__('Choose how customers can get help when the AI cannot fully address their needs.', 'intufind'); ?>
                </p>
                
                <form method="post" action="options.php">
                    <?php settings_fields(Constants::SETTINGS_GROUP_LIVE_AGENT_BEHAVIOR); ?>
                    
                    <table class="form-table" role="presentation">
                        <tr>
                            <th scope="row"><?php echo esc_html__('Live Agent Handoff', 'intufind'); ?></th>
                            <td>
                                <label>
                                    <input type="checkbox" name="<?php echo esc_attr(Constants::OPTION_LIVE_AGENT_ENABLED); ?>" value="1" <?php checked($enabled, true); ?> <?php disabled(!$isConfigured); ?>>
                                    <?php echo esc_html__('Connect customers to live agents', 'intufind'); ?>
                                </label>
                                <?php if ($provider === Constants::LIVE_AGENT_PROVIDER_NONE): ?>
                                    <p class="description" style="color: #d63638;">
                                        <?php echo esc_html__('Select a provider above to enable live agent handoff.', 'intufind'); ?>
                                    </p>
                                <?php elseif (!$isConfigured): ?>
                                    <p class="description" style="color: #d63638;">
                                        <?php echo sprintf(esc_html__('Complete %s setup above to enable.', 'intufind'), esc_html(ucfirst($provider))); ?>
                                    </p>
                                <?php else: ?>
                                    <p class="description">
                                        <?php echo esc_html__('When enabled, customers can be connected to live agents via your configured provider.', 'intufind'); ?>
                                    </p>
                                <?php endif; ?>
                            </td>
                        </tr>

                        <tr>
                            <th scope="row"><?php echo esc_html__('Contact Form', 'intufind'); ?></th>
                            <td>
                                <label>
                                    <input type="checkbox" name="<?php echo esc_attr(Constants::OPTION_CONTACT_FORM_ENABLED); ?>" value="1" <?php checked($contactFormEnabled, true); ?>>
                                    <?php echo esc_html__('Enable contact form', 'intufind'); ?>
                                </label>
                                <p class="description">
                                    <?php echo esc_html__('Allow customers to leave their contact details and a message. Works as a fallback when live agents are unavailable, or as a standalone lead capture option.', 'intufind'); ?>
                                </p>
                            </td>
                        </tr>
                    </table>

                    <?php if ($provider !== Constants::LIVE_AGENT_PROVIDER_NONE && $isConfigured): ?>
                    <div style="margin-top: 24px; padding-top: 24px; border-top: 1px solid #e0e0e3;">
                        <h3 style="margin-top: 0; margin-bottom: 16px; font-size: 14px; font-weight: 600; color: #1d2327;">
                            <?php echo esc_html__('Live Agent Settings', 'intufind'); ?>
                        </h3>
                        
                        <table class="form-table" role="presentation">
                            <tr>
                                <th scope="row"><?php echo esc_html__('AI-Suggested Handoff', 'intufind'); ?></th>
                                <td>
                                    <label>
                                        <input type="checkbox" name="<?php echo esc_attr(Constants::OPTION_LIVE_AGENT_AUTO_HANDOFF); ?>" value="1" <?php checked($autoHandoff, true); ?>>
                                        <?php echo esc_html__('AI can suggest handoff (when detecting frustration or complex issues)', 'intufind'); ?>
                                    </label>
                                </td>
                            </tr>

                            <tr>
                                <th scope="row">
                                    <label for="intucart_live_agent_escalation_prompt"><?php echo esc_html__('Custom Escalation Criteria', 'intufind'); ?></label>
                                </th>
                                <td>
                                    <?php
                                    // Show info about MCP Order Support status
                                    $hasMcpOrderSupport = get_option(Constants::OPTION_ENABLE_MCP_ORDERS, true) && class_exists('WooCommerce');
                                    if ($hasMcpOrderSupport):
                                    ?>
                                        <div class="intucart-notice intucart-notice-success" style="margin: 0 0 12px 0;">
                                            <p><strong>✓ <?php echo esc_html__('AI Order Inquiries Enabled', 'intufind'); ?></strong></p>
                                            <p><?php echo esc_html__('The AI can look up order information for logged-in customers. Your escalation criteria should focus on actions the AI cannot perform (refunds, cancellations, etc.).', 'intufind'); ?></p>
                                            <p><?php echo sprintf(esc_html__('To disable AI order access, go to: %s', 'intufind'), '<strong>' . esc_html__('Chatbot → WooCommerce → AI Capabilities', 'intufind') . '</strong>'); ?></p>
                                        </div>
                                    <?php else: ?>
                                        <div class="intucart-notice intucart-notice-warning" style="margin: 0 0 12px 0;">
                                            <p><strong>⚠ <?php echo esc_html__('AI Order Inquiries Disabled', 'intufind'); ?></strong></p>
                                            <?php if (!class_exists('WooCommerce')): ?>
                                                <p><?php echo esc_html__('WooCommerce is not installed. Order-related questions will be escalated to live agents.', 'intufind'); ?></p>
                                            <?php else: ?>
                                                <p><?php echo esc_html__('Your escalation criteria should include order inquiries since the AI cannot access order data.', 'intufind'); ?></p>
                                                <p><?php echo sprintf(esc_html__('To enable AI order lookups, go to: %s', 'intufind'), '<strong>' . esc_html__('Chatbot → WooCommerce → AI Capabilities', 'intufind') . '</strong>'); ?></p>
                                            <?php endif; ?>
                                        </div>
                                    <?php endif; ?>
                                    
                                    <textarea 
                                        name="<?php echo esc_attr(Constants::OPTION_LIVE_AGENT_ESCALATION_PROMPT); ?>" 
                                        id="intucart_live_agent_escalation_prompt" 
                                        rows="4" 
                                        class="large-text"
                                        maxlength="1500"
                                        placeholder="<?php echo esc_attr__('e.g., Always escalate questions about custom orders, bulk purchases, or wholesale inquiries', 'intufind'); ?>"
                                    ><?php echo esc_textarea($escalationPrompt ?? ''); ?></textarea>
                                    <p class="description">
                                        <?php echo esc_html__('Define specific situations when the AI should escalate to a live agent. The default criteria cover common escalation scenarios, but you can customize this based on your business needs. Leave empty to disable custom criteria.', 'intufind'); ?>
                                        <br><strong><?php echo esc_html__('Maximum 1,500 characters', 'intufind'); ?></strong> <?php echo esc_html__('(approximately 3-4 sentences or a short bulleted list).', 'intufind'); ?>
                                    </p>
                                </td>
                            </tr>

                            <tr>
                                <th scope="row"><?php echo esc_html__('Manual Handoff Button', 'intufind'); ?></th>
                                <td>
                                    <label>
                                        <input type="checkbox" name="<?php echo esc_attr(Constants::OPTION_LIVE_AGENT_MANUAL_HANDOFF); ?>" value="1" <?php checked($manualHandoff, true); ?>>
                                        <?php echo esc_html__('Show "Talk to a Human" button', 'intufind'); ?>
                                    </label>
                                </td>
                            </tr>

                            <tr>
                                <th scope="row">
                                    <label for="intucart_live_agent_availability_schedule"><?php echo esc_html__('Availability Schedule', 'intufind'); ?></label>
                                </th>
                                <td>
                                    <textarea 
                                        name="<?php echo esc_attr(Constants::OPTION_LIVE_AGENT_AVAILABILITY_SCHEDULE); ?>" 
                                        id="intucart_live_agent_availability_schedule" 
                                        rows="3" 
                                        class="large-text"
                                        placeholder="<?php echo esc_attr__('e.g., Monday-Friday 9am-5pm EST. Closed on US holidays. After hours support available via email.', 'intufind'); ?>"
                                    ><?php echo esc_textarea($availabilitySchedule ?? ''); ?></textarea>
                                    <p class="description">
                                        <?php echo esc_html__('Define your live agent availability schedule in natural language. The AI will interpret this schedule to determine if agents are available. Include time zones, holidays, and any exceptions.', 'intufind'); ?>
                                        <br><strong><?php echo esc_html__('Examples:', 'intufind'); ?></strong>
                                        <br>• <?php echo esc_html__('Mon-Fri 8AM to 5PM EST, closed 12/25 and 1/1', 'intufind'); ?>
                                        <br>• <?php echo esc_html__('24/7 except major US holidays', 'intufind'); ?>
                                        <br>• <?php echo esc_html__('Weekdays 9-6 PST, Saturdays 10-2 PST', 'intufind'); ?>
                                    </p>
                                </td>
                            </tr>

                            <tr>
                                <th scope="row">
                                    <label for="intucart_live_agent_business_hours"><?php echo esc_html__('Display Hours', 'intufind'); ?></label>
                                </th>
                                <td>
                                    <input type="text" 
                                           name="<?php echo esc_attr(Constants::OPTION_LIVE_AGENT_BUSINESS_HOURS); ?>" 
                                           id="intucart_live_agent_business_hours" 
                                           value="<?php echo esc_attr($businessHours); ?>" 
                                           class="regular-text"
                                           placeholder="<?php echo esc_attr__('Mon-Fri 9am-5pm EST', 'intufind'); ?>">
                                    <p class="description"><?php echo esc_html__('Short text displayed to customers when agents are unavailable (optional, for display only)', 'intufind'); ?></p>
                                </td>
                            </tr>

                            <tr>
                                <th scope="row">
                                    <label for="intucart_live_agent_inactivity_timeout"><?php echo esc_html__('Inactivity Timeout', 'intufind'); ?></label>
                                </th>
                                <td>
                                    <select name="<?php echo esc_attr(Constants::OPTION_LIVE_AGENT_INACTIVITY_TIMEOUT); ?>" id="intucart_live_agent_inactivity_timeout">
                                        <option value="60" <?php selected($inactivityTimeout, 60); ?>><?php echo esc_html__('1 minute', 'intufind'); ?></option>
                                        <option value="300" <?php selected($inactivityTimeout, 300); ?>><?php echo esc_html__('5 minutes', 'intufind'); ?></option>
                                        <option value="600" <?php selected($inactivityTimeout, 600); ?>><?php echo esc_html__('10 minutes (recommended)', 'intufind'); ?></option>
                                        <option value="900" <?php selected($inactivityTimeout, 900); ?>><?php echo esc_html__('15 minutes', 'intufind'); ?></option>
                                        <option value="1800" <?php selected($inactivityTimeout, 1800); ?>><?php echo esc_html__('30 minutes', 'intufind'); ?></option>
                                        <option value="3600" <?php selected($inactivityTimeout, 3600); ?>><?php echo esc_html__('60 minutes', 'intufind'); ?></option>
                                    </select>
                                    <p class="description"><?php echo esc_html__('Automatically end live agent session after this period of inactivity', 'intufind'); ?></p>
                                </td>
                            </tr>
                        </table>
                    </div>
                    <?php endif; ?>

                    <div style="margin-top: 20px;">
                        <?php Button::primary(__('Save Settings', 'intufind'), ['type' => 'submit']); ?>
                    </div>
                </form>
            </div>

            <!-- Information -->
            <div class="intucart-settings-section">
                <h2><?php echo esc_html__('About Live Agent Handoff', 'intufind'); ?></h2>
                <p>
                    <?php echo esc_html__('When the AI chatbot encounters a question it can\'t answer or detects customer frustration, it can seamlessly hand off the conversation to a human agent through your chosen platform.', 'intufind'); ?>
                </p>
                <h4><?php echo esc_html__('How it works:', 'intufind'); ?></h4>
                <ol>
                    <li><?php echo esc_html__('Customer requests human help (or AI detects need)', 'intufind'); ?></li>
                    <li><?php echo sprintf(esc_html__('Conversation is forwarded to your %s', 'intufind'), $provider === Constants::LIVE_AGENT_PROVIDER_SLACK ? esc_html__('Slack channel', 'intufind') : esc_html__('live agent platform', 'intufind')); ?></li>
                    <li><?php echo esc_html__('Agent responds through the platform', 'intufind'); ?></li>
                    <li><?php echo esc_html__('Messages appear in the chatbot widget in real-time', 'intufind'); ?></li>
                </ol>
            </div>
        </div>

        <style>
            .intucart-settings-section {
                background: #fff;
                border: 1px solid #e0e0e3;
                padding: 24px;
                margin: 20px 0;
                border-radius: 12px;
                box-shadow: 0 1px 3px rgba(0,0,0,.05);
            }
            .intucart-settings-section h2 {
                margin-top: 0;
                margin-bottom: 16px;
                color: #1d2327;
                font-size: 18px;
                font-weight: 600;
            }
            /* Remove borders from form table rows in Live Agent settings */
            .intucart-settings-section .form-table tr {
                border-bottom: none;
            }
            .intucart-badge {
                display: inline-block;
                padding: 4px 10px;
                font-size: 12px;
                font-weight: 600;
                border-radius: 20px;
                letter-spacing: 0.3px;
            }
            .intucart-badge-success {
                background: linear-gradient(135deg, #d7f5dc 0%, #c8f0d2 100%);
                color: #1e7e34;
                border: 1px solid #4caf50;
            }
            .intucart-badge-warning {
                background: linear-gradient(135deg, #fff9e6 0%, #fff3cd 100%);
                color: #856404;
                border: 1px solid #ffc107;
            }
            /* Details/Summary accordion styling */
            .intucart-details {
                margin-top: 20px;
            }
            .intucart-details summary {
                cursor: pointer;
                font-weight: 600;
                font-size: 14px;
                padding: 14px 18px;
                background: #f6f7f7;
                border-radius: 8px;
                border: 1px solid #e0e0e3;
                color: #1d2327;
                display: flex;
                align-items: center;
                gap: 8px;
                transition: all 0.2s ease;
                list-style: none;
            }
            .intucart-details summary::-webkit-details-marker {
                display: none;
            }
            .intucart-details summary::before {
                content: '▸';
                font-size: 12px;
                color: #646970;
                transition: transform 0.2s ease;
            }
            .intucart-details[open] summary::before {
                transform: rotate(90deg);
            }
            .intucart-details summary:hover {
                background: #f0f0f1;
                border-color: #c3c4c7;
            }
            .intucart-details-content {
                margin-top: 16px;
                padding: 20px;
                border: 1px solid #e0e0e3;
                border-radius: 8px;
                background: #fff;
            }
        </style>
        <?php
    }

    /**
     * Render Slack credentials form
     */
    private function renderSlackCredentials(string $license): void
    {
        // Get OAuth endpoint (defined in intucart.php with default)
        $cloudApiUrl = INTUFIND_OAUTH_URL;
        $workspaceId = $this->aiClientManager->getWorkspaceId();
        
        // Check credential status and get configuration type
        $status = $this->getCredentialStatus('slack', $license);
        $isConfigured = $status['configured'];
        $configurationType = $status['configurationType'];
        $teamName = $status['teamName'] ?? null;
        $hasChannelId = $status['hasChannelId'] ?? false;
        
        ?>
        <div class="intucart-settings-section">
            <h2><?php echo esc_html__('Slack Integration', 'intufind'); ?></h2>

            <?php if ($isConfigured): ?>
            <?php if ($configurationType === 'oauth' && !$hasChannelId): ?>
                <!-- App connected but channel not configured -->
                <div class="intucart-notice intucart-notice-warning">
                    <p>
                        <strong>⚠ <?php echo esc_html__('App Connected - Channel Setup Required', 'intufind'); ?></strong>
                        <?php if ($teamName): ?>
                            — <?php echo esc_html__('Workspace:', 'intufind'); ?> <strong><?php echo esc_html($teamName); ?></strong>
                        <?php endif; ?>
                    </p>
                    <p><?php echo esc_html__('Please configure a Slack channel below to complete the setup.', 'intufind'); ?></p>
                </div>
            <?php else: ?>
                <!-- Fully configured - simple confirmation (details shown in Provider Configuration above) -->
                <div class="intucart-notice intucart-notice-success">
                    <p><strong>✓ <?php echo esc_html__('Ready to receive live agent requests', 'intufind'); ?></strong></p>
                </div>
            <?php endif; ?>
            <?php endif; ?>

            <!-- Easy Installation Option (hide if already OAuth configured) -->
            <?php if ($configurationType !== 'oauth'): ?>
            <div class="intucart-info-box">
                <h3>✨ <?php echo esc_html__('Easy Setup (Recommended)', 'intufind'); ?></h3>
                <p><?php echo esc_html__('Click the button below to install the Intufind Slack App in one click. No manual configuration needed!', 'intufind'); ?></p>
                <p>
                    <a href="<?php echo esc_url($cloudApiUrl); ?>/slack/install?api_key=<?php echo esc_attr($license); ?>&workspace_id=<?php echo esc_attr($workspaceId); ?>&return_url=<?php echo urlencode(admin_url('admin.php?page=intucart&tab=chatbot&chatbot_subtab=liveagent')); ?>" 
                       style="display: inline-block;">
                        <img alt="<?php echo esc_attr__('Add to Slack', 'intufind'); ?>" 
                             height="40" 
                             width="139" 
                             src="https://platform.slack-edge.com/img/add_to_slack.png" 
                             srcset="https://platform.slack-edge.com/img/add_to_slack.png 1x, https://platform.slack-edge.com/img/add_to_slack@2x.png 2x">
                    </a>
                </p>
                <p class="description"><?php echo esc_html__('After installation, you\'ll just need to select which Slack channel to use for customer support.', 'intufind'); ?></p>
            </div>
            <?php endif; // End OAuth installation option ?>

            <!-- Manual Configuration (collapsed) -->
            <?php if ($configurationType !== 'oauth'): ?>
            <details class="intucart-details">
                <summary><?php echo esc_html__('Advanced: Manual Configuration', 'intufind'); ?></summary>
                <div class="intucart-details-content">
            <div class="intucart-info-box intucart-info-box-secondary">
                <h3><?php echo esc_html__('Setup Instructions', 'intufind'); ?></h3>
                <ol>
                    <li><?php echo sprintf(esc_html__('Go to %s and create a new app', 'intufind'), '<code>api.slack.com/apps</code>'); ?></li>
                    <li><?php echo sprintf(esc_html__('Add bot scopes: %s, %s, %s', 'intufind'), '<code>chat:write</code>', '<code>channels:read</code>', '<code>users:read</code>'); ?></li>
                    <li><?php echo esc_html__('Install app to workspace and copy the Bot Token', 'intufind'); ?></li>
                    <li><?php echo sprintf(esc_html__('Enable Event Subscriptions with URL: %s', 'intufind'), '<code>' . esc_html($cloudApiUrl) . '/live-agent/webhook/slack</code>'); ?></li>
                    <li><?php echo esc_html__('Create a support channel (e.g., #customer-support) and invite your bot', 'intufind'); ?></li>
                </ol>
            </div>

            <form method="post">
                <?php wp_nonce_field('intucart_live_agent_credentials'); ?>
                <input type="hidden" name="action" value="save_credentials">
                <input type="hidden" name="provider" value="slack">
                <input type="hidden" name="license_key" value="<?php echo esc_attr($license); ?>">

                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row">
                            <label for="slack_bot_token"><?php echo esc_html__('Bot User OAuth Token', 'intufind'); ?> <span style="color: #d63638;">*</span></label>
                        </th>
                        <td>
                            <input type="password" 
                                   name="bot_token" 
                                   id="slack_bot_token" 
                                   class="regular-text" 
                                   placeholder="<?php echo $isConfigured ? '••••••••••••••••' : 'xoxb-...'; ?>" 
                                   autocomplete="off"
                                   <?php echo $isConfigured ? '' : 'required'; ?>>
                            <p class="description">
                                <?php if ($isConfigured): ?>
                                    <span style="color: #46b450;">✓ <?php echo esc_html__('Configured', 'intufind'); ?></span> - <?php echo esc_html__('Enter a new value to update, or leave empty to keep current', 'intufind'); ?>
                                <?php else: ?>
                                    <?php echo esc_html__('Found in OAuth & Permissions after installing app', 'intufind'); ?>
                                <?php endif; ?>
                            </p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="slack_channel_id"><?php echo esc_html__('Channel ID', 'intufind'); ?> <span style="color: #d63638;">*</span></label>
                        </th>
                        <td>
                            <input type="text" 
                                   name="channel_id" 
                                   id="slack_channel_id" 
                                   class="regular-text" 
                                   placeholder="<?php echo $isConfigured ? '••••••••••' : 'C01234567'; ?>"
                                   <?php echo $isConfigured ? '' : 'required'; ?>>
                            <p class="description">
                                <?php if ($isConfigured): ?>
                                    <span style="color: #46b450;">✓ <?php echo esc_html__('Configured', 'intufind'); ?></span> - <?php echo esc_html__('Enter a new value to update, or leave empty to keep current', 'intufind'); ?>
                                <?php else: ?>
                                    <?php echo esc_html__('After adding your bot to the channel, find the ID: Right-click channel → View channel details → scroll to bottom for ID', 'intufind'); ?>
                                <?php endif; ?>
                            </p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="slack_signing_secret"><?php echo esc_html__('Signing Secret', 'intufind'); ?> <span style="color: #d63638;">*</span></label>
                        </th>
                        <td>
                            <input type="password" 
                                   name="signing_secret" 
                                   id="slack_signing_secret" 
                                   class="regular-text" 
                                   placeholder="<?php echo $isConfigured ? '••••••••••••••••' : ''; ?>"
                                   autocomplete="off"
                                   <?php echo $isConfigured ? '' : 'required'; ?>>
                            <p class="description">
                                <?php if ($isConfigured): ?>
                                    <span style="color: #46b450;">✓ <?php echo esc_html__('Configured', 'intufind'); ?></span> - <?php echo esc_html__('Enter a new value to update, or leave empty to keep current', 'intufind'); ?>
                                <?php else: ?>
                                    <?php echo esc_html__('Found in Basic Information → App Credentials', 'intufind'); ?>
                                <?php endif; ?>
                            </p>
                        </td>
                    </tr>
                </table>

                <div style="margin-top: 20px;">
                    <?php
                    Button::primary($isConfigured ? __('Update Credentials', 'intufind') : __('Save & Validate Credentials', 'intufind'), ['type' => 'submit']);
                    
                    if ($isConfigured):
                        echo ' ';
                        Button::danger(__('Delete Credentials', 'intufind'), [
                            'type' => 'submit',
                            'attributes' => [
                                'name' => 'action',
                                'value' => 'delete_credentials',
                                'onclick' => 'return confirm(\'' . esc_js(__('Are you sure you want to delete your Slack credentials?', 'intufind')) . '\');'
                            ]
                        ]);
                    endif;
                    ?>
                </div>
            </form>
        </div>
            </details>
            <?php endif; // End manual configuration ?>

            <!-- Channel Configuration (ONLY for OAuth installations) -->
            <?php if ($configurationType === 'oauth'): ?>
        <?php
                // Fetch current channel ID if configured
                $currentChannelId = '';
                if ($hasChannelId) {
                    try {
                        $client = $this->aiClientManager->getClient();
                        // We can't retrieve the actual channel ID from backend (it's encrypted)
                        // So we just show that it's configured
                    } catch (\Exception $e) {
                        // Ignore error
                    }
                }
                ?>
                <div style="margin-top: 20px;">
                    <h3><?php echo esc_html__('Channel Configuration', 'intufind'); ?></h3>
                    
                    <?php if ($hasChannelId): ?>
                        <div class="intucart-notice intucart-notice-success">
                            <p><strong>✓ <?php echo esc_html__('Channel Configured', 'intufind'); ?></strong></p>
                            <p><?php echo esc_html__('A Slack channel is currently configured. Enter a new Channel ID below to change it.', 'intufind'); ?></p>
                            <p style="margin-top: 8px; color: #1d2327;">
                                <strong>💡 <?php echo esc_html__('Reminder:', 'intufind'); ?></strong>
                                <?php echo esc_html__('The Intufind app must be a member of this channel. Type', 'intufind'); ?> <code>/invite @Intufind</code> <?php echo esc_html__('in the channel if you haven\'t already.', 'intufind'); ?>
                            </p>
                        </div>
                    <?php endif; ?>
                    
                    <form method="post">
                        <?php wp_nonce_field('intucart_live_agent_credentials'); ?>
                        <input type="hidden" name="action" value="save_channel">
                        <input type="hidden" name="provider" value="slack">
                        <input type="hidden" name="license_key" value="<?php echo esc_attr($license); ?>">

                        <table class="form-table" role="presentation">
                            <tr>
                                <th scope="row">
                                    <label for="slack_channel_id_update"><?php echo esc_html__('Support Channel ID', 'intufind'); ?> <span style="color: #d63638;">*</span></label>
                                </th>
                                <td>
                                    <input type="text" 
                                           name="channel_id" 
                                           id="slack_channel_id_update" 
                                           class="regular-text" 
                                           placeholder="<?php echo $hasChannelId ? '••••••••••' : 'C01234567'; ?>"
                                           <?php echo $hasChannelId ? '' : 'required'; ?>>
                                    <p class="description">
                                        <?php if ($hasChannelId): ?>
                                            <span style="color: #46b450;">✓ <?php echo esc_html__('Configured', 'intufind'); ?></span> - <?php echo esc_html__('Enter a new Channel ID to update, or leave empty to keep current', 'intufind'); ?>
                                        <?php else: ?>
                                            <strong><?php echo esc_html__('Setup Steps:', 'intufind'); ?></strong><br>
                                            1) <?php echo esc_html__('Create a channel in Slack (e.g., #customer-support)', 'intufind'); ?><br>
                                            2) <?php echo esc_html__('Add the Intufind app to the channel: Right-click channel → View channel details → Integrations → Add Apps → Select Intufind', 'intufind'); ?><br>
                                            3) <?php echo esc_html__('Find the Channel ID: Still in channel details, scroll down to the bottom to find the ID', 'intufind'); ?>
                                        <?php endif; ?>
                                    </p>
                                </td>
                            </tr>
                        </table>

                        <div style="margin-top: 20px;">
                            <?php Button::primary($hasChannelId ? __('Update Channel', 'intufind') : __('Save Channel', 'intufind'), ['type' => 'submit']); ?>
                        </div>
                    </form>

                    <!-- Disconnect OAuth Integration -->
                    <form method="post" style="margin-top: 30px; padding-top: 20px; border-top: 1px solid #ddd;">
                        <?php wp_nonce_field('intucart_live_agent_credentials'); ?>
                        <input type="hidden" name="action" value="delete_credentials">
                        <input type="hidden" name="provider" value="slack">
                        <input type="hidden" name="license_key" value="<?php echo esc_attr($license); ?>">
                        
                        <h3><?php echo esc_html__('Disconnect Slack App', 'intufind'); ?></h3>
                        <p class="description" style="margin-bottom: 12px;">
                            <?php echo esc_html__('This will remove the Intufind Slack App connection and all associated credentials. You\'ll need to reconnect to use Slack again.', 'intufind'); ?>
                        </p>
                        
                        <?php 
                        Button::danger(__('Disconnect Slack', 'intufind'), [
                            'type' => 'submit',
                            'attributes' => [
                                'onclick' => 'return confirm(\'' . esc_js(__('Are you sure you want to disconnect Slack? This will remove all OAuth credentials and channel configuration.', 'intufind')) . '\');'
                            ]
                        ]); 
                        ?>
                    </form>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Render Zendesk credentials form
     */
    private function renderZendeskCredentials(string $license): void
    {
        // Check if credentials are already configured
        $status = $this->getCredentialStatus('zendesk', $license);
        $isConfigured = $status['configured'];
        
        ?>
        <div class="intucart-settings-section">
            <h2><?php echo esc_html__('Zendesk Credentials', 'intufind'); ?></h2>

            <?php if ($isConfigured): ?>
                <div class="intucart-notice intucart-notice-success">
                    <p><strong>✓ <?php echo esc_html__('Credentials Configured', 'intufind'); ?></strong> — <?php echo esc_html__('Your Zendesk credentials are securely stored. Re-enter values below to update them.', 'intufind'); ?></p>
                </div>
            <?php endif; ?>

            <form method="post">
                <?php wp_nonce_field('intucart_live_agent_credentials'); ?>
                <input type="hidden" name="action" value="save_credentials">
                <input type="hidden" name="provider" value="zendesk">
                <input type="hidden" name="license_key" value="<?php echo esc_attr($license); ?>">

                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row">
                            <label for="zendesk_subdomain"><?php echo esc_html__('Zendesk Subdomain', 'intufind'); ?> <span style="color: #d63638;">*</span></label>
                        </th>
                        <td>
                            <input type="text" 
                                   name="subdomain" 
                                   id="zendesk_subdomain" 
                                   class="regular-text" 
                                   placeholder="<?php echo $isConfigured ? '••••••••••' : 'mycompany'; ?>"
                                   <?php echo $isConfigured ? '' : 'required'; ?>>
                            <p class="description">
                                <?php if ($isConfigured): ?>
                                    <span style="color: #46b450;">✓ <?php echo esc_html__('Configured', 'intufind'); ?></span> - <?php echo esc_html__('Enter a new value to update, or leave empty to keep current', 'intufind'); ?>
                                <?php else: ?>
                                    <?php echo esc_html__('Your Zendesk URL: mycompany.zendesk.com', 'intufind'); ?>
                                <?php endif; ?>
                            </p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="zendesk_api_token"><?php echo esc_html__('API Token', 'intufind'); ?> <span style="color: #d63638;">*</span></label>
                        </th>
                        <td>
                            <input type="password" 
                                   name="api_token" 
                                   id="zendesk_api_token" 
                                   class="regular-text" 
                                   placeholder="<?php echo $isConfigured ? '••••••••••••••••' : ''; ?>"
                                   autocomplete="off"
                                   <?php echo $isConfigured ? '' : 'required'; ?>>
                            <p class="description">
                                <?php if ($isConfigured): ?>
                                    <span style="color: #46b450;">✓ <?php echo esc_html__('Configured', 'intufind'); ?></span> - <?php echo esc_html__('Enter a new value to update, or leave empty to keep current', 'intufind'); ?>
                                <?php else: ?>
                                    <?php echo esc_html__('Generate in Zendesk Admin → Channels → API', 'intufind'); ?>
                                <?php endif; ?>
                            </p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="zendesk_webhook_secret"><?php echo esc_html__('Webhook Secret', 'intufind'); ?> <span style="color: #d63638;">*</span></label>
                        </th>
                        <td>
                            <input type="password" 
                                   name="webhook_secret" 
                                   id="zendesk_webhook_secret" 
                                   class="regular-text" 
                                   placeholder="<?php echo $isConfigured ? '••••••••••••••••' : ''; ?>"
                                   autocomplete="off"
                                   <?php echo $isConfigured ? '' : 'required'; ?>>
                            <p class="description">
                                <?php if ($isConfigured): ?>
                                    <span style="color: #46b450;">✓ <?php echo esc_html__('Configured', 'intufind'); ?></span> - <?php echo esc_html__('Enter a new value to update, or leave empty to keep current', 'intufind'); ?>
                                <?php else: ?>
                                    <?php echo esc_html__('Used to verify webhook authenticity', 'intufind'); ?>
                                <?php endif; ?>
                            </p>
                        </td>
                    </tr>
                </table>

                <div style="margin-top: 20px;">
                    <?php
                    Button::primary($isConfigured ? __('Update Credentials', 'intufind') : __('Save & Validate Credentials', 'intufind'), ['type' => 'submit']);
                    
                    if ($isConfigured):
                        echo ' ';
                        Button::danger(__('Delete Credentials', 'intufind'), [
                            'type' => 'submit',
                            'attributes' => [
                                'name' => 'action',
                                'value' => 'delete_credentials',
                                'onclick' => 'return confirm(\'' . esc_js(__('Are you sure you want to delete your Zendesk credentials?', 'intufind')) . '\');'
                            ]
                        ]);
                    endif;
                    ?>
                </div>
            </form>
        </div>
        <?php
    }

    /**
     * Handle saving credentials
     */
    private function handleSaveCredentials(): void
    {
        $provider = sanitize_text_field($_POST['provider'] ?? '');
        $licenseKey = sanitize_text_field($_POST['license_key'] ?? '');

        if (empty($licenseKey)) {
            add_settings_error(
                'intucart_live_agent_messages',
                'intucart_live_agent_error',
                'License key is required',
                'error'
            );
            return;
        }

        // Build credentials object
        $credentials = ['provider' => $provider];

        if ($provider === Constants::LIVE_AGENT_PROVIDER_SLACK) {
            $credentials['botToken'] = sanitize_text_field($_POST['bot_token'] ?? '');
            $credentials['channelId'] = sanitize_text_field($_POST['channel_id'] ?? '');
            $credentials['signingSecret'] = sanitize_text_field($_POST['signing_secret'] ?? '');
        } elseif ($provider === Constants::LIVE_AGENT_PROVIDER_ZENDESK) {
            $credentials['subdomain'] = sanitize_text_field($_POST['subdomain'] ?? '');
            $credentials['apiToken'] = sanitize_text_field($_POST['api_token'] ?? '');
            $credentials['webhookSecret'] = sanitize_text_field($_POST['webhook_secret'] ?? '');
        }

        // Send to backend API
        try {
            $client = $this->aiClientManager->getClient();
            $response = $client->tenant()->storeLiveAgentCredentials($credentials);

            // SDK unwraps 'data' envelope, so check for 'message' (indicates success)
            if (isset($response['message'])) {
                // Cache the configured status locally for the widget
                update_option(Constants::OPTION_LIVE_AGENT_CONFIGURED, true);
                
                add_settings_error(
                    'intucart_live_agent_messages',
                    'intucart_live_agent_success',
                    __('Credentials saved and validated successfully!', 'intufind'),
                    'success'
                );
            } else {
                $error = $response['error'] ?? __('Failed to save credentials', 'intufind');
                $details = $response['details'] ?? '';
                add_settings_error(
                    'intucart_live_agent_messages',
                    'intucart_live_agent_error',
                    $error . ($details ? ' - ' . $details : ''),
                    'error'
                );
            }
        } catch (\Throwable $e) {
            // Catch both Exception and Error (PHP 7+)
            add_settings_error(
                'intucart_live_agent_messages',
                'intucart_live_agent_error',
                sprintf(__('Failed to connect to API: %s', 'intufind'), $e->getMessage()),
                'error'
            );
        }
    }

    /**
     * Handle deleting credentials
     */
    private function handleDeleteCredentials(): void
    {
        $provider = sanitize_text_field($_POST['provider'] ?? '');
        $licenseKey = sanitize_text_field($_POST['license_key'] ?? '');

        try {
            $client = $this->aiClientManager->getClient();
            $response = $client->tenant()->deleteLiveAgentCredentials($provider);

            // SDK unwraps 'data' envelope, so check for 'message' (indicates success)
            if (isset($response['message'])) {
                // Clear the cached configured status and disable the feature
                update_option(Constants::OPTION_LIVE_AGENT_CONFIGURED, false);
                update_option(Constants::OPTION_LIVE_AGENT_ENABLED, false);
                
                add_settings_error(
                    'intucart_live_agent_messages',
                    'intucart_live_agent_success',
                    __('Credentials deleted successfully!', 'intufind'),
                    'success'
                );
            } else {
                add_settings_error(
                    'intucart_live_agent_messages',
                    'intucart_live_agent_error',
                    $response['error'] ?? __('Failed to delete credentials', 'intufind'),
                    'error'
                );
            }
        } catch (\Throwable $e) {
            // Catch both Exception and Error (PHP 7+)
            add_settings_error(
                'intucart_live_agent_messages',
                'intucart_live_agent_error',
                sprintf(__('Failed to connect to API: %s', 'intufind'), $e->getMessage()),
                'error'
            );
        }
    }

    /**
     * Handle saving channel configuration (for OAuth installations)
     */
    private function handleSaveChannel(): void
    {
        $provider = sanitize_text_field($_POST['provider'] ?? '');
        $licenseKey = sanitize_text_field($_POST['license_key'] ?? '');
        $channelId = sanitize_text_field($_POST['channel_id'] ?? '');

        if (empty($licenseKey)) {
            add_settings_error(
                'intucart_live_agent_messages',
                'intucart_live_agent_error',
                __('License key is required', 'intufind'),
                'error'
            );
            return;
        }

        if (empty($channelId)) {
            // If channel ID is empty, it means user wants to keep the current one
            // Only show error if this is the first time setting it
            $status = $this->getCredentialStatus('slack', $licenseKey);
            if (!($status['hasChannelId'] ?? false)) {
                add_settings_error(
                    'intucart_live_agent_messages',
                    'intucart_live_agent_error',
                    __('Channel ID is required', 'intufind'),
                    'error'
                );
                return;
            }
            // Channel already configured and user left it empty - keep current
            return;
        }

        // Send only channel ID to backend API (for OAuth-based installations)
        try {
            $client = $this->aiClientManager->getClient();
            $credentials = [
                'provider' => $provider,
                'channelId' => $channelId,
                'updateChannelOnly' => true  // Flag to indicate partial update
            ];
            
            $response = $client->tenant()->storeLiveAgentCredentials($credentials);

            // SDK unwraps 'data' envelope, so check for 'message' (indicates success)
            if (isset($response['message'])) {
                // Cache the configured status locally for the widget
                update_option(Constants::OPTION_LIVE_AGENT_CONFIGURED, true);
                
                add_settings_error(
                    'intucart_live_agent_messages',
                    'intucart_live_agent_success',
                    __('Channel configuration updated successfully!', 'intufind'),
                    'success'
                );
            } else {
                $error = $response['error'] ?? __('Failed to update channel configuration', 'intufind');
                $details = $response['details'] ?? '';
                add_settings_error(
                    'intucart_live_agent_messages',
                    'intucart_live_agent_error',
                    $error . ($details ? ' - ' . $details : ''),
                    'error'
                );
            }
        } catch (\Throwable $e) {
            // Catch both Exception and Error (PHP 7+)
            add_settings_error(
                'intucart_live_agent_messages',
                'intucart_live_agent_error',
                sprintf(__('Failed to connect to API: %s', 'intufind'), $e->getMessage()),
                'error'
            );
        }
    }

    /**
     * Check credential status for a provider
     */
    private function getCredentialStatus(string $provider, string $license): array
    {
        if (empty($license) || $provider === Constants::LIVE_AGENT_PROVIDER_NONE) {
            return ['configured' => false, 'configurationType' => 'none', 'hasChannelId' => false];
        }

        try {
            $client = $this->aiClientManager->getClient();
            $response = $client->tenant()->checkLiveAgentCredentialsStatus($provider);
            
            // Ensure response is an array before accessing
            if (!is_array($response)) {
                $this->logger->error('Invalid credential status response', ['response' => $response]);
                return ['configured' => false, 'configurationType' => 'none', 'hasChannelId' => false];
            }
            
            // SDK already unwraps the 'data' field, so access directly
            $details = $response['details'] ?? [];
            
            return [
                'configured' => $response['configured'] ?? false,
                'configurationType' => $response['configurationType'] ?? 'none',
                'teamName' => is_array($details) ? ($details['teamName'] ?? null) : null,
                'hasChannelId' => is_array($details) ? ($details['hasChannelId'] ?? false) : false,
            ];
        } catch (\Throwable $e) {
            // Catch both Exception and Error (PHP 7+) to handle TypeError, etc.
            $this->logger->error('Failed to check credential status', ['error' => $e->getMessage()]);
            return ['configured' => false, 'configurationType' => 'none', 'hasChannelId' => false];
        }
    }

    /**
     * Load availability schedule from the cloud API
     */
    private function loadAvailabilitySchedule(): string
    {
        try {
            $client = $this->aiClientManager->getClient();
            $response = $client->tenant()->getLiveAgentSchedule();
            
            // Ensure response is an array before accessing
            if (!is_array($response)) {
                return '';
            }
            
            return $response['schedule'] ?? '';
        } catch (\Throwable $e) {
            // Catch both Exception and Error (PHP 7+)
            $this->logger->debug('Failed to load availability schedule from API', ['error' => $e->getMessage()]);
            return '';
        }
    }

    /**
     * Save availability schedule to the cloud API
     */
    private function saveAvailabilitySchedule(string $schedule): bool
    {
        try {
            $client = $this->aiClientManager->getClient();
            
            if (empty($schedule)) {
                $client->tenant()->deleteLiveAgentSchedule();
            } else {
                $client->tenant()->updateLiveAgentSchedule($schedule);
            }
            
            return true;
        } catch (\Throwable $e) {
            // Catch both Exception and Error (PHP 7+)
            $this->logger->error('Failed to save availability schedule to API', ['error' => $e->getMessage()]);
            return false;
        }
    }
}

