<?php

/**
 * Chat Main Settings - Behavior & Content
 *
 * Handles the main settings tab for chat configuration
 */

namespace Intucart\Pages\Settings\Chat;

use Intucart\Pages\Settings\AbstractSettings;
use Intucart\Services\Constants;
use Intucart\Services\Logger;
use Intucart\Components\Field;
use Intucart\Components\ImageUploader;

class MainSettings extends AbstractSettings
{
    public function __construct(Logger $logger)
    {
        parent::__construct($logger);
    }

    /**
     * Settings registration handled by parent ChatSettings class
     */
    public function registerSettings(): void
    {
        // No-op: Registration handled by ChatSettings controller
    }

    /**
     * Render main chat settings (Behavior & Content)
     */
    public function render(): void
    {
        // ===================================================================
        // BEHAVIOR SETTINGS
        // ===================================================================
        $this->renderSettingsPanel(
            __('Behavior', 'intufind'),
            __('Configure how the chat widget behaves on your site.', 'intufind'),
            'admin-settings',
            function () {
                echo '<table class="form-table">';

                // Enable Chat Widget
                $this->renderTableRow(__('Enable Chat Widget', 'intufind'), function () {
                    $enabled = get_option(Constants::OPTION_ENABLE_CHAT_ASSISTANT, true);
                    Field::checkbox(Constants::OPTION_ENABLE_CHAT_ASSISTANT, __('Show the floating chat widget on your site', 'intufind'), $enabled);
                });

                // Widget Position
                $this->renderTableRow(__('Widget Position', 'intufind'), function () {
                    $pos = get_option(Constants::OPTION_CHAT_WIDGET_POSITION, 'bottom-right');
                    Field::select(Constants::OPTION_CHAT_WIDGET_POSITION, '', [
                        'bottom-right' => __('Bottom Right', 'intufind'),
                        'bottom-left' => __('Bottom Left', 'intufind'),
                        'top-right' => __('Top Right', 'intufind'),
                        'top-left' => __('Top Left', 'intufind'),
                    ], $pos, [
                        'hint' => [[
                            'content' => __('Where the chat trigger button appears on your site.', 'intufind')
                        ]]
                    ]);
                });

                // Card Layout
                $this->renderTableRow(__('Product/Post Card Layout', 'intufind'), function () {
                    $layout = get_option(Constants::OPTION_CHAT_CARD_LAYOUT, 'carousel');
                    Field::select(Constants::OPTION_CHAT_CARD_LAYOUT, '', [
                        'carousel' => __('Carousel (Swipeable)', 'intufind'),
                        'grid' => __('Grid (Static)', 'intufind'),
                    ], $layout, [
                        'hint' => [[
                            'content' => __('How product and post cards are displayed. Carousel allows horizontal swiping (recommended for mobile), Grid shows all cards in a static layout.', 'intufind')
                        ]]
                    ]);
                });

                // Prompt Style
                $this->renderTableRow(__('Prompt Style', 'intufind'), function () {
                    $style = get_option(Constants::OPTION_CHAT_PROMPT_STYLE, 'footer');
                    Field::select(Constants::OPTION_CHAT_PROMPT_STYLE, '', [
                        'footer' => __('Footer Carousel', 'intufind'),
                        'chips' => __('Chat Area Chips', 'intufind'),
                        'both' => __('Both', 'intufind'),
                    ], $style, [
                        'hint' => [[
                            'content' => __('How suggested prompts are displayed. Footer shows prompts in a horizontal carousel above the input. Chat Area shows prompts as chips inside the conversation flow.', 'intufind')
                        ]]
                    ]);
                });

                // Show Message Actions
                $this->renderTableRow(__('Show Message Actions', 'intufind'), function () {
                    $enabled = get_option(Constants::OPTION_CHAT_SHOW_MESSAGE_ACTIONS, false);
                    Field::checkbox(Constants::OPTION_CHAT_SHOW_MESSAGE_ACTIONS, __('Show feedback buttons and copy button on assistant messages', 'intufind'), $enabled);
                });

                // Show User Copy Button
                $this->renderTableRow(__('Show User Copy Button', 'intufind'), function () {
                    $enabled = get_option(Constants::OPTION_CHAT_SHOW_USER_COPY_BUTTON, false);
                    Field::checkbox(Constants::OPTION_CHAT_SHOW_USER_COPY_BUTTON, __('Show copy button on user messages', 'intufind'), $enabled);
                });

                // Show Message Timestamps
                $this->renderTableRow(__('Show Message Timestamps', 'intufind'), function () {
                    $enabled = get_option(Constants::OPTION_CHAT_SHOW_MESSAGE_TIMESTAMPS, false);
                    Field::checkbox(Constants::OPTION_CHAT_SHOW_MESSAGE_TIMESTAMPS, __('Show relative timestamps on messages (e.g., "2 minutes ago")', 'intufind'), $enabled);
                });

                // Show Post Image
                $this->renderTableRow(__('Show Post Image', 'intufind'), function () {
                    $enabled = get_option(Constants::OPTION_CHAT_SHOW_POST_IMAGE, false);
                    Field::checkbox(Constants::OPTION_CHAT_SHOW_POST_IMAGE, __('Display the image in post cards (enable if your posts have featured images)', 'intufind'), $enabled);
                });

                // Show Post Author
                $this->renderTableRow(__('Show Post Author', 'intufind'), function () {
                    $enabled = get_option(Constants::OPTION_CHAT_SHOW_POST_AUTHOR, true);
                    Field::checkbox(Constants::OPTION_CHAT_SHOW_POST_AUTHOR, __('Display the author name in post cards (e.g., "by Author Name")', 'intufind'), $enabled);
                });

                // Show Post Date
                $this->renderTableRow(__('Show Post Date', 'intufind'), function () {
                    $enabled = get_option(Constants::OPTION_CHAT_SHOW_POST_DATE, true);
                    Field::checkbox(Constants::OPTION_CHAT_SHOW_POST_DATE, __('Display the publication date in post cards', 'intufind'), $enabled);
                });

                // Show Post Category
                $this->renderTableRow(__('Show Post Category', 'intufind'), function () {
                    $enabled = get_option(Constants::OPTION_CHAT_SHOW_POST_CATEGORY, true);
                    Field::checkbox(Constants::OPTION_CHAT_SHOW_POST_CATEGORY, __('Display the category/tags in post cards', 'intufind'), $enabled);
                });

                echo '</table>';
            }
        );

        // ===================================================================
        // PRIVACY & ANALYTICS SETTINGS
        // ===================================================================
        $this->renderSettingsPanel(
            __('Privacy & Analytics', 'intufind'),
            __('Configure visitor tracking and analytics for the chat widget.', 'intufind'),
            'visibility',
            function () {
                echo '<table class="form-table">';

                // Enable Analytics/Visitor Tracking
                $this->renderTableRow(__('Enable Visitor Tracking', 'intufind'), function () {
                    $enabled = get_option(Constants::OPTION_CHAT_ANALYTICS_ENABLED, true);
                    Field::checkbox(
                        Constants::OPTION_CHAT_ANALYTICS_ENABLED, 
                        __('Track unique visitors for analytics (uses device fingerprinting)', 'intufind'), 
                        $enabled
                    );
                    echo '<p class="description" style="margin-top: 8px; max-width: 600px;">';
                    echo esc_html__('When enabled, the widget generates a pseudonymous visitor ID using browser characteristics to count unique visitors. This data is not linked to personal information and is used solely for analytics in your Intufind dashboard.', 'intufind');
                    echo '</p>';
                });

                // Require Consent Before Tracking
                $this->renderTableRow(__('Require Consent', 'intufind'), function () {
                    $required = get_option(Constants::OPTION_CHAT_ANALYTICS_CONSENT_REQUIRED, false);
                    Field::checkbox(
                        Constants::OPTION_CHAT_ANALYTICS_CONSENT_REQUIRED, 
                        __('Wait for user consent before tracking visitors (GDPR compliance)', 'intufind'), 
                        $required
                    );
                    echo '<p class="description" style="margin-top: 8px; max-width: 600px;">';
                    echo '<strong>' . esc_html__('For GDPR/Cookie Consent:', 'intufind') . '</strong> ';
                    echo esc_html__('When enabled, visitor tracking will only activate after consent is granted. You can integrate with your cookie consent tool using:', 'intufind');
                    echo '<br><br>';
                    echo '<code style="display: block; padding: 8px; background: #f0f0f0; border-radius: 4px; font-size: 12px;">';
                    echo '// Option 1: Set consent signal before widget loads<br>';
                    echo 'window.intufindConsent = true;<br><br>';
                    echo '// Option 2: Grant consent after user accepts<br>';
                    echo 'window.IntufindChatbot.grantAnalyticsConsent();';
                    echo '</code>';
                    echo '</p>';
                });

                echo '</table>';
            }
        );

        // ===================================================================
        // CONTENT SETTINGS
        // ===================================================================
        $this->renderSettingsPanel(
            __('Content', 'intufind'),
            __('Configure the text and messages shown in your chat widget.', 'intufind'),
            'format-chat',
            function () {
                echo '<table class="form-table">';

                // Header Title
                $this->renderTableRow(__('Header Title', 'intufind'), function () {
                    $header_text = get_option(Constants::OPTION_CHAT_HEADER_TEXT, Constants::ASSISTANT_DEFAULT_CHAT_HEADER_TEXT);
                    Field::text(Constants::OPTION_CHAT_HEADER_TEXT, '', $header_text, [
                        'hint' => [[
                            'content' => __('Text shown in the chat header. Supports HTML: &lt;strong&gt;, &lt;em&gt;, &lt;b&gt;, &lt;i&gt;, &lt;br&gt;', 'intufind')
                        ]]
                    ]);
                });

                // Header Subtitle
                $this->renderTableRow(__('Header Subtitle', 'intufind'), function () {
                    $subtitle = get_option(Constants::OPTION_CHAT_HEADER_SUBTITLE, '');
                    Field::text(Constants::OPTION_CHAT_HEADER_SUBTITLE, '', $subtitle, [
                        'attributes' => [
                            'placeholder' => __('e.g., We typically reply in a few minutes', 'intufind')
                        ],
                        'hint' => [[
                            'content' => __('Optional subtitle below header. Supports HTML: &lt;strong&gt;, &lt;em&gt;, &lt;b&gt;, &lt;i&gt;, &lt;br&gt;', 'intufind')
                        ]]
                    ]);
                });

                // Greeting (Empty State Title)
                $this->renderTableRow(__('Greeting', 'intufind'), function () {
                    $greeting = get_option(Constants::OPTION_ASSISTANT_WELCOME_MESSAGE, Constants::ASSISTANT_DEFAULT_WELCOME_MESSAGE);
                    Field::text(Constants::OPTION_ASSISTANT_WELCOME_MESSAGE, '', $greeting, [
                        'attributes' => [
                            'placeholder' => __('e.g., How can I help you today?', 'intufind')
                        ],
                        'hint' => [[
                            'content' => __('Main greeting text shown when chat is empty (before first message).', 'intufind')
                        ]]
                    ]);
                });

                // Greeting Subtext
                $this->renderTableRow(__('Greeting Subtext', 'intufind'), function () {
                    $subtext = get_option(Constants::OPTION_CHAT_GREETING_SUBTEXT, '');
                    Field::text(Constants::OPTION_CHAT_GREETING_SUBTEXT, '', $subtext, [
                        'attributes' => [
                            'placeholder' => __('e.g., Ask me anything about products or orders', 'intufind')
                        ],
                        'hint' => [[
                            'content' => __('Optional text shown below the greeting (leave empty for default).', 'intufind')
                        ]]
                    ]);
                });

                // Greeting Icon (uses standard media.js system)
                $this->renderTableRow(__('Greeting Icon', 'intufind'), function () {
                    $icon_id = get_option(Constants::OPTION_CHAT_GREETING_ICON_ID, 0);
                    
                    ImageUploader::icon(Constants::OPTION_CHAT_GREETING_ICON_ID, $icon_id, [
                        'hint' => __('Optional icon/image shown in empty state (48x48px recommended). Leave empty to use default chat icon.', 'intufind')
                    ]);
                });

                echo '</table>';
            }
        );

        // ===================================================================
        // SITE CONTEXT & INSTRUCTIONS
        // ===================================================================
        $this->renderSettingsPanel(
            __('Site Context & Instructions', 'intufind'),
            __('Provide important facts, corrections, and business-specific knowledge to guide the AI assistant.', 'intufind'),
            'admin-generic',
            function () {
                echo '<table class="form-table">';

                // Site Description
                $this->renderTableRow(__('Site Description', 'intufind'), function () {
                    $description = get_option(Constants::OPTION_CHAT_SITE_DESCRIPTION, '');
                    Field::render([
                        'type' => 'textarea',
                        'name' => Constants::OPTION_CHAT_SITE_DESCRIPTION,
                        'value' => $description,
                        'attributes' => ['rows' => 3, 'maxlength' => 500],
                        'hint' => [
                            [
                                'title' => __('Describe your website in 1-2 sentences (max 500 characters).', 'intufind'),
                                'content' => __('This helps the AI understand your site\'s purpose and provide more relevant responses.', 'intufind')
                            ],
                            [
                                'title' => __('Examples:', 'intufind'),
                                'content' => '• "' . esc_html__('An online outdoor gear retailer specializing in camping and hiking equipment', 'intufind') . '"<br>• "' . esc_html__('A specialty coffee roastery with brewing guides and educational content', 'intufind') . '"<br>• "' . esc_html__('A personal blog about sustainable living and zero-waste lifestyle tips', 'intufind') . '"<br>• "' . esc_html__('A SaaS platform providing project management tools for creative agencies', 'intufind') . '"<br>• "' . esc_html__('A community forum and resource hub for woodworking enthusiasts', 'intufind') . '"<br><br><em>' . esc_html__('Leave empty to auto-detect based on your content (uses AI inference).', 'intufind') . '</em>'
                            ]
                        ]
                    ]);
                });

                // Custom Instructions
                $this->renderTableRow(__('Custom Instructions', 'intufind'), function () {
                    $instructions = get_option(Constants::OPTION_CHAT_CUSTOM_INSTRUCTIONS, '');
                    
                    Field::render([
                        'type' => 'textarea',
                        'name' => Constants::OPTION_CHAT_CUSTOM_INSTRUCTIONS,
                        'value' => $instructions,
                        'attributes' => [
                            'rows' => 12,
                            'maxlength' => 5000,
                            'style' => 'font-family: ui-monospace, Menlo, Monaco, "Cascadia Code", "Courier New", monospace;'
                        ],
                        'hint' => []
                    ]);
                    
                    // Important notice
                    echo '<div class="intucart-notice intucart-notice-info" style="margin-top: 12px; max-width: 800px;">';
                    echo '<p><strong>' . esc_html__('Note:', 'intufind') . '</strong> ' . esc_html__('The AI assistant is already fully configured with comprehensive instructions.', 'intufind') . ' <strong>' . esc_html__('Only add information that\'s unique to your business.', 'intufind') . '</strong> ' . esc_html__('This is not the main system prompt—it\'s for supplemental facts and corrections only.', 'intufind') . '</p>';
                    echo '</div>';
                    
                    echo '<p class="description" style="max-width: 800px; line-height: 1.6; margin-top: 12px;">';
                    echo '<strong>' . esc_html__('Add important facts, corrections, or business-specific knowledge that the AI should always reference (max 5,000 characters).', 'intufind') . '</strong><br><br>';
                    echo '<strong>' . esc_html__('Use cases:', 'intufind') . '</strong><br>';
                    echo '• <strong>' . esc_html__('Correct hallucinations:', 'intufind') . '</strong> "' . esc_html__('Our return policy is 60 days, not 30 days', 'intufind') . '"<br>';
                    echo '• <strong>' . esc_html__('Add missing facts:', 'intufind') . '</strong> "' . esc_html__('Product X is now gluten-free (updated formulation)', 'intufind') . '"<br>';
                    echo '• <strong>' . esc_html__('Business policies:', 'intufind') . '</strong> "' . esc_html__('We ship to Canada and Mexico only. No other international shipping.', 'intufind') . '"<br>';
                    echo '• <strong>' . esc_html__('Hours & contact:', 'intufind') . '</strong> "' . esc_html__('Customer service: M-F 9am-6pm EST. Email: support@example.com', 'intufind') . '"<br>';
                    echo '• <strong>' . esc_html__('Specific guidance:', 'intufind') . '</strong> "' . esc_html__('When users ask about sizing, always mention our size chart link', 'intufind') . '"<br><br>';
                    echo '<strong>' . esc_html__('Formatting tips:', 'intufind') . '</strong><br>';
                    echo '• ' . esc_html__('Use **bold** for emphasis', 'intufind') . '<br>';
                    echo '• ' . esc_html__('Use bullet points (-) for lists', 'intufind') . '<br>';
                    echo '• ' . esc_html__('Use headings (###) to organize sections', 'intufind') . '<br>';
                    echo '• ' . esc_html__('Keep entries clear and concise', 'intufind') . '<br>';
                    echo '</p>';
                });

                echo '</table>';
            }
        );
        
        // Note: Character counters are now handled by Field component
        // Note: Greeting icon uploader handled by assets/js/admin/media.js
    }
}
