<?php

/**
 * Chat Styling Settings - Colors, Avatars, Advanced Options
 *
 * Handles the styling tab for chat visual customization
 */

namespace Intucart\Pages\Settings\Chat;

use Intucart\Pages\Settings\AbstractSettings;
use Intucart\Services\Constants;
use Intucart\Services\Logger;
use Intucart\Components\ImageUploader;
use Intucart\Components\ColorPicker;
use Intucart\Components\Field;
use Intucart\Components\Button;

class StylingSettings extends AbstractSettings
{
    public function __construct(Logger $logger)
    {
        parent::__construct($logger);
    }

    /**
     * Settings registration handled by parent ChatSettings class
     */
    public function registerSettings(): void
    {
        // No-op: Registration handled by ChatSettings controller
    }

    /**
     * Render chat styling/branding settings (Styling tab)
     */
    public function render(): void
    {
        // Branding Panel
        $this->renderBrandingPanel();

        // Trigger Button Panel
        $this->renderTriggerPanel();

        // Header Panel
        $this->renderHeaderPanel();

        // Message Bubbles Panel
        $this->renderMessageBubblesPanel();

        // Cards & Content Panel
        $this->renderCardsContentPanel();

        // Prompts & Suggestions Panel
        $this->renderPromptsPanel();

        // Styling Overrides Panel
        $this->renderStylingOverridesPanel();

        // Reset Button
        $this->renderResetButton();

        // JavaScript for interactions
        $this->renderJavaScript();
    }

    /**
     * Render branding panel (colors, avatars, content)
     */
    private function renderBrandingPanel(): void
    {
        $this->renderSettingsPanel(
            __('Branding', 'intufind'),
            __('Customize the visual appearance and branding of your chat widget.', 'intufind'),
            'art',
            function () {
                // AI Color Generator Section (NEW)
                $this->renderColorGeneratorPanel();

                echo '<table class="form-table">';

                // Color Scheme (Light/Dark/Auto)
                $this->renderTableRow(__('Color Scheme', 'intufind'), function () {
                    Field::select(
                        Constants::OPTION_CHAT_COLOR_SCHEME,
                        '',
                        [
                            'auto' => __('Auto (Follow System Preference)', 'intufind'),
                            'light' => __('Light Mode (Always)', 'intufind'),
                            'dark' => __('Dark Mode (Always)', 'intufind'),
                        ],
                        get_option(Constants::OPTION_CHAT_COLOR_SCHEME, 'auto'),
                        [
                            'description' => __('Choose how the chatbot adapts to light/dark mode. "Auto" respects the user\'s system preference.', 'intufind')
                        ]
                    );
                });

                // Primary Color
                $this->renderTableRow(__('Primary Color', 'intufind'), function () {
                    ColorPicker::dual([
                        'light_name' => Constants::OPTION_CHAT_PRIMARY_COLOR,
                        'light_value' => get_option(Constants::OPTION_CHAT_PRIMARY_COLOR, '#0084ff'),
                        'dark_name' => Constants::OPTION_CHAT_DARK_PRIMARY_COLOR,
                        'dark_value' => get_option(Constants::OPTION_CHAT_DARK_PRIMARY_COLOR, '#0084ff'),
                        'description' => __('Main brand color - used for user message bubbles and primary buttons.', 'intufind')
                    ]);
                });

                // Accent Color
                $this->renderTableRow(__('Accent Color', 'intufind'), function () {
                    ColorPicker::dual([
                        'light_name' => Constants::OPTION_CHAT_ACCENT_COLOR,
                        'light_value' => get_option(Constants::OPTION_CHAT_ACCENT_COLOR, '#dadce0'),
                        'dark_name' => Constants::OPTION_CHAT_DARK_ACCENT_COLOR,
                        'dark_value' => get_option(Constants::OPTION_CHAT_DARK_ACCENT_COLOR, '#404040'),
                        'description' => __('Used for borders, dividers, and subtle UI elements.', 'intufind')
                    ]);
                });

                // Send Button Color
                $this->renderTableRow(__('Send Button', 'intufind'), function () {
                    ColorPicker::dual([
                        'light_name' => Constants::OPTION_CHAT_SEND_BUTTON_COLOR,
                        'light_value' => get_option(Constants::OPTION_CHAT_SEND_BUTTON_COLOR, '#0084ff'),
                        'dark_name' => Constants::OPTION_CHAT_DARK_SEND_BUTTON_COLOR,
                        'dark_value' => get_option(Constants::OPTION_CHAT_DARK_SEND_BUTTON_COLOR, '#0084ff'),
                        'description' => __('Color for the message send button (separate from primary for more control).', 'intufind')
                    ]);
                });

                echo '</table>';
            }
        );
    }

    /**
     * Render AI Color Generator Panel
     */
    private function renderColorGeneratorPanel(): void
    {
        ?>
        <div class="intucart-color-generator-panel" style="background: white; border: 2px solid #6101fe; border-radius: 8px; padding: 24px; margin-bottom: 24px; box-shadow: 0 2px 8px rgba(97, 1, 254, 0.08);">
            <div style="display: flex; align-items: center; gap: 12px; margin-bottom: 8px;">
                <div style="width: 24px; height: 24px; background-image: url('data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHZpZXdCb3g9IjAgMCAyNCAyNCIgd2lkdGg9IjIwIiBoZWlnaHQ9IjIwIiBmaWxsPSIjNjEwMWZlIj4KPHBhdGggZD0iTTE5IDlsMS4yNS0yLjc1TDIzIDVsLTIuNzUtMS4yNUwxOSAxbC0xLjI1IDIuNzVMMTUgNWwyLjc1IDEuMjVMMTkgOXptLTcuNS41TDkgNCA2LjUgOS41IDEgMTJsNS41IDIuNUw5IDIwbDIuNS01LjVMMTcgMTJsLTUuNS0yLjV6TTE5IDE1bC0xLjI1IDIuNzVMMTUgMTlsMi43NSAxLjI1TDE5IDIzbDEuMjUtMi43NUwyMyAxOWwtMi43NS0xLjI1TDE5IDE1eiIvPgo8L3N2Zz4g'); background-size: contain; background-repeat: no-repeat; background-position: center;"></div>
                <h3 style="margin: 0; color: #6101fe; font-size: 18px; font-weight: 600;"><?php echo esc_html__('AI Color Generator', 'intufind'); ?></h3>
            </div>
            <p style="margin: 0 0 20px; color: #666; font-size: 14px; line-height: 1.5;">
                <?php echo sprintf(
                    esc_html__('Let AI create a complete, accessible color scheme for your chat widget. Our AI considers %s and design harmony.', 'intufind'),
                    '<strong>' . esc_html__('WCAG 2.1 AA compliance', 'intufind') . '</strong>'
                ); ?>
            </p>
            
            <div style="display: flex; gap: 20px; align-items: flex-start; margin-bottom: 16px; flex-wrap: wrap;">
                <!-- Primary Color -->
                <div class="intucart-dual-color-picker" style="flex: 0 0 auto;">
                    <div class="intucart-color-picker-item">
                        <label class="intucart-color-picker-label" style="color: #333; font-weight: 600; font-size: 13px; display: block; margin-bottom: 6px;">
                            <?php echo esc_html__('Primary Color', 'intufind'); ?> *
                        </label>
                        <?php
                        ColorPicker::single(
                            'intucart_ai_primary_color',
                            '#6101fe',
                            ['id' => 'intucart-ai-primary-color']
                        );
                        ?>
                    </div>
                </div>
                
                <!-- Secondary Color (Optional) -->
                <div class="intucart-dual-color-picker" style="flex: 0 0 auto;">
                    <div class="intucart-color-picker-item">
                        <label class="intucart-color-picker-label" style="color: #333; font-weight: 600; font-size: 13px; display: flex; align-items: center; gap: 6px; margin-bottom: 6px; cursor: pointer;">
                            <input type="checkbox" id="intucart-use-secondary-color" style="margin: 0; cursor: pointer;">
                            <span><?php echo esc_html__('Secondary Color', 'intufind'); ?></span>
                        </label>
                        <?php
                        ColorPicker::single(
                            'intucart_ai_secondary_color',
                            '',
                            [
                                'id' => 'intucart-ai-secondary-color',
                                'attributes' => ['disabled' => 'disabled', 'placeholder' => '#ffffff']
                            ]
                        );
                        ?>
                    </div>
                </div>
            </div>
            
            <!-- Generate Button -->
            <div style="margin-bottom: 16px;">
                <?php
                Button::primary(
                    __('Generate Color Scheme', 'intufind'),
                    [
                        'id' => 'intucart-generate-colors',
                        'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512" style="width: 16px; height: 16px;"><path d="M204.3 5C104.9 24.4 24.8 104.3 5.2 203.4c-37 187 131.7 326.4 258.8 306.7 41.2-6.4 61.4-54.6 42.5-91.7-23.1-45.4 9.9-98.4 60.9-98.4h79.7c35.8 0 64.8-29.6 64.9-65.3C511.5 97.1 368.1-26.9 204.3 5zM96 320c-17.7 0-32-14.3-32-32s14.3-32 32-32 32 14.3 32 32-14.3 32-32 32zm32-128c-17.7 0-32-14.3-32-32s14.3-32 32-32 32 14.3 32 32-14.3 32-32 32zm128-64c-17.7 0-32-14.3-32-32s14.3-32 32-32 32 14.3 32 32-14.3 32-32 32zm128 64c-17.7 0-32-14.3-32-32s14.3-32 32-32 32 14.3 32 32-14.3 32-32 32z"/></svg>',
                        'size' => 'medium',
                        'class' => 'intucart-ai-generate-btn'
                    ]
                );
                ?>
            </div>
            
            <!-- Status Message -->
            <div id="intucart-color-generator-status" style="display: none; margin-top: 16px;"></div>
        </div>
        <style>
            .intucart-ai-generate-btn {
                background: #6101fe !important;
                color: white !important;
                border: none !important;
                font-weight: 600 !important;
                padding: 10px 24px !important;
                height: auto !important;
                font-size: 14px !important;
                transition: all 0.2s ease !important;
            }
            .intucart-ai-generate-btn:hover {
                background: #4e01cc !important;
                transform: translateY(-1px);
                box-shadow: 0 4px 12px rgba(97, 1, 254, 0.3) !important;
            }
            .intucart-ai-generate-btn:active {
                transform: translateY(0);
            }
        </style>
        <?php
    }

    /**
     * Render trigger button panel (floating button styling)
     */
    private function renderTriggerPanel(): void
    {
        $this->renderSettingsPanel(
            __('Trigger Button', 'intufind'),
            __('Customize the appearance of the floating chat trigger button.', 'intufind'),
            'format-chat',
            function () {
                echo '<table class="form-table">';

                // Trigger Button Colors
                $this->renderTableRow(__('Button Colors', 'intufind'), function () {
                    echo '<div style="display: flex; flex-direction: column; gap: 16px;">';
                    
                    // Background Color (dual)
                    echo '<div>';
                    echo '<label style="font-weight: 600; display: block; margin-bottom: 8px;">' . esc_html__('Background:', 'intufind') . '</label>';
                    ColorPicker::dual([
                        'light_name' => Constants::OPTION_CHAT_TRIGGER_BUTTON_COLOR,
                        'light_value' => get_option(Constants::OPTION_CHAT_TRIGGER_BUTTON_COLOR, '#0084ff'),
                        'dark_name' => Constants::OPTION_CHAT_DARK_TRIGGER_BUTTON_COLOR,
                        'dark_value' => get_option(Constants::OPTION_CHAT_DARK_TRIGGER_BUTTON_COLOR, '#0084ff')
                    ]);
                    echo '</div>';

                    // Hover Color (dual)
                    echo '<div>';
                    echo '<label style="font-weight: 600; display: block; margin-bottom: 8px;">' . esc_html__('Hover:', 'intufind') . '</label>';
                    ColorPicker::dual([
                        'light_name' => Constants::OPTION_CHAT_TRIGGER_BUTTON_HOVER_COLOR,
                        'light_value' => get_option(Constants::OPTION_CHAT_TRIGGER_BUTTON_HOVER_COLOR, '#0073e6'),
                        'dark_name' => Constants::OPTION_CHAT_DARK_TRIGGER_BUTTON_HOVER_COLOR,
                        'dark_value' => get_option(Constants::OPTION_CHAT_DARK_TRIGGER_BUTTON_HOVER_COLOR, '#0073e6')
                    ]);
                    echo '</div>';

                    // Border Color (dual)
                    echo '<div>';
                    echo '<label style="font-weight: 600; display: block; margin-bottom: 8px;">' . esc_html__('Border:', 'intufind') . '</label>';
                    ColorPicker::dual([
                        'light_name' => Constants::OPTION_CHAT_TRIGGER_BUTTON_BORDER_COLOR,
                        'light_value' => get_option(Constants::OPTION_CHAT_TRIGGER_BUTTON_BORDER_COLOR, '#dadce0'),
                        'dark_name' => Constants::OPTION_CHAT_DARK_TRIGGER_BUTTON_BORDER_COLOR,
                        'dark_value' => get_option(Constants::OPTION_CHAT_DARK_TRIGGER_BUTTON_BORDER_COLOR, '#404040')
                    ]);
                    echo '</div>';

                    echo '</div>';
                    $this->renderHint(__('Customize background, hover, and border colors for the trigger button.', 'intufind'));
                });

                // Trigger Avatar
                $this->renderTableRow(__('Button Icon', 'intufind'), function () {
                    $this->renderAvatarUploader(
                        'trigger',
                        Constants::OPTION_TOGGLE_ICON_ID,
                        __('Avatar/icon for the chat trigger button (40x40px recommended).', 'intufind')
                    );
                });

                // Trigger Button Background Toggle
                $this->renderTableRow(__('Icon Background', 'intufind'), function () {
                    Field::checkbox(
                        Constants::OPTION_CHAT_TRIGGER_BUTTON_SHOW_BACKGROUND,
                        __('Show background circle behind trigger button icon', 'intufind'),
                        get_option(Constants::OPTION_CHAT_TRIGGER_BUTTON_SHOW_BACKGROUND, true),
                        [
                            'description' => __('Toggle the circular background behind the trigger button icon/avatar. When disabled, the button becomes transparent and displays the icon in its natural shape.', 'intufind')
                        ]
                    );
                });

                // Trigger Icon Border Toggle
                $this->renderTableRow(__('Icon Border', 'intufind'), function () {
                    Field::checkbox(
                        Constants::OPTION_CHAT_TRIGGER_ICON_SHOW_BORDER,
                        __('Show border around trigger button icon/avatar', 'intufind'),
                        get_option(Constants::OPTION_CHAT_TRIGGER_ICON_SHOW_BORDER, true),
                        [
                            'description' => __('Toggle the border around the trigger button icon/avatar. When disabled, the icon displays in its natural shape without circular cropping.', 'intufind')
                        ]
                    );
                });

                echo '</table>';
            }
        );
    }

    /**
     * Render header panel (header-specific styling)
     */
    private function renderHeaderPanel(): void
    {
        $this->renderSettingsPanel(
            __('Header', 'intufind'),
            __('Customize the appearance of the chat widget header bar.', 'intufind'),
            'admin-appearance',
            function () {
                echo '<table class="form-table">';

                // Header Background
                $this->renderTableRow(__('Header Background', 'intufind'), function () {
                    ColorPicker::dual([
                        'light_name' => Constants::OPTION_CHAT_HEADER_BACKGROUND,
                        'light_value' => get_option(Constants::OPTION_CHAT_HEADER_BACKGROUND, '#ffffff'),
                        'dark_name' => Constants::OPTION_CHAT_DARK_HEADER_BACKGROUND,
                        'dark_value' => get_option(Constants::OPTION_CHAT_DARK_HEADER_BACKGROUND, '#1f1f1f'),
                        'description' => __('Background color for the chat header bar.', 'intufind')
                    ]);
                });

                // Header Title Color
                $this->renderTableRow(__('Header Title', 'intufind'), function () {
                    ColorPicker::dual([
                        'light_name' => Constants::OPTION_CHAT_HEADER_TITLE_COLOR,
                        'light_value' => get_option(Constants::OPTION_CHAT_HEADER_TITLE_COLOR, '#0f172a'),
                        'dark_name' => Constants::OPTION_CHAT_DARK_HEADER_TITLE_COLOR,
                        'dark_value' => get_option(Constants::OPTION_CHAT_DARK_HEADER_TITLE_COLOR, '#e8eaed'),
                        'description' => __('Color for the header title text.', 'intufind')
                    ]);
                });

                // Header Subtitle Color
                $this->renderTableRow(__('Header Subtitle', 'intufind'), function () {
                    ColorPicker::dual([
                        'light_name' => Constants::OPTION_CHAT_HEADER_SUBTITLE_COLOR,
                        'light_value' => get_option(Constants::OPTION_CHAT_HEADER_SUBTITLE_COLOR, '#64748b'),
                        'dark_name' => Constants::OPTION_CHAT_DARK_HEADER_SUBTITLE_COLOR,
                        'dark_value' => get_option(Constants::OPTION_CHAT_DARK_HEADER_SUBTITLE_COLOR, '#9aa0a6'),
                        'description' => __('Color for the header subtitle/status text.', 'intufind')
                    ]);
                });

                // Header Icon Color
                $this->renderTableRow(__('Header Icons', 'intufind'), function () {
                    ColorPicker::dual([
                        'light_name' => Constants::OPTION_CHAT_HEADER_ICON_COLOR,
                        'light_value' => get_option(Constants::OPTION_CHAT_HEADER_ICON_COLOR, '#64748b'),
                        'dark_name' => Constants::OPTION_CHAT_DARK_HEADER_ICON_COLOR,
                        'dark_value' => get_option(Constants::OPTION_CHAT_DARK_HEADER_ICON_COLOR, '#9aa0a6'),
                        'description' => __('Color for header buttons (new chat, close).', 'intufind')
                    ]);
                });

                // Header Avatar
                $this->renderTableRow(__('Header Avatar', 'intufind'), function () {
                    $this->renderAvatarUploader(
                        'header',
                        Constants::OPTION_CHAT_HEADER_ICON_ID,
                        __('Avatar/icon shown in the chat header (40x40px recommended).', 'intufind')
                    );
                });

                // Header Avatar Background Toggle
                $this->renderTableRow(__('Avatar Background', 'intufind'), function () {
                    Field::checkbox(
                        Constants::OPTION_CHAT_HEADER_AVATAR_SHOW_BACKGROUND,
                        __('Show background circle behind header avatar', 'intufind'),
                        get_option(Constants::OPTION_CHAT_HEADER_AVATAR_SHOW_BACKGROUND, true),
                        [
                            'description' => __('Toggle the circular background behind the header avatar/icon. When disabled, the avatar displays in its natural shape without circular cropping.', 'intufind')
                        ]
                    );
                });

                // Header Avatar Border Toggle
                $this->renderTableRow(__('Avatar Border', 'intufind'), function () {
                    Field::checkbox(
                        Constants::OPTION_CHAT_HEADER_AVATAR_SHOW_BORDER,
                        __('Show border around header avatar', 'intufind'),
                        get_option(Constants::OPTION_CHAT_HEADER_AVATAR_SHOW_BORDER, true),
                        [
                            'description' => __('Toggle the border outline around the header avatar/icon.', 'intufind')
                        ]
                    );
                });

                echo '</table>';
            }
        );
    }

    /**
     * Render message bubbles styling panel
     */
    private function renderMessageBubblesPanel(): void
    {
        $this->renderSettingsPanel(
            __('Message Bubbles', 'intufind'),
            __('Customize the appearance of chat message bubbles for both user and assistant messages.', 'intufind'),
            'admin-comments',
            function () {
                echo '<table class="form-table">';

                // Message Font Size
                $this->renderTableRow(__('Font Size', 'intufind'), function () {
                    Field::number(
                        Constants::OPTION_CHAT_MESSAGE_FONT_SIZE,
                        '',
                        get_option(Constants::OPTION_CHAT_MESSAGE_FONT_SIZE, 14),
                        [
                            'attributes' => ['min' => '12', 'max' => '20'],
                            'suffix' => __('px', 'intufind'),
                            'description' => __('Font size for message text (12-20px). This overrides theme font sizes to ensure consistent readability.', 'intufind')
                        ]
                    );
                });

                // Message Line Height
                $this->renderTableRow(__('Line Height', 'intufind'), function () {
                    Field::number(
                        Constants::OPTION_CHAT_MESSAGE_LINE_HEIGHT,
                        '',
                        get_option(Constants::OPTION_CHAT_MESSAGE_LINE_HEIGHT, '1.5'),
                        [
                            'attributes' => ['min' => '1.0', 'max' => '2.5', 'step' => '0.1'],
                            'description' => __('Line height for message text (1.0-2.5). Controls spacing between lines. 1.5 is standard for readable text.', 'intufind')
                        ]
                    );
                });

                // Message Font Family
                $this->renderTableRow(__('Font Family', 'intufind'), function () {
                    $message_font_family = get_option(Constants::OPTION_CHAT_MESSAGE_FONT_FAMILY, 'inherit');
                    $message_custom_font = get_option(Constants::OPTION_CHAT_MESSAGE_CUSTOM_FONT_FAMILY, '');

                    Field::select(
                        Constants::OPTION_CHAT_MESSAGE_FONT_FAMILY,
                        '',
                        [
                            'inherit' => __('Inherit from Widget (Default)', 'intufind'),
                            'system' => __('System Default', 'intufind'),
                            'inter' => __('Inter (Google Font)', 'intufind'),
                            'roboto' => __('Roboto (Google Font)', 'intufind'),
                            'open-sans' => __('Open Sans (Google Font)', 'intufind'),
                            'lato' => __('Lato (Google Font)', 'intufind'),
                            'custom' => __('Custom Font', 'intufind'),
                        ],
                        $message_font_family,
                        [
                            'description' => __('Font family for message bubbles. Choose "Inherit" to use the global widget font, or select a specific font.', 'intufind')
                        ]
                    );

                    echo '<div id="message_custom_font_wrapper" style="margin-top: 12px; ' . ($message_font_family !== 'custom' ? 'display:none;' : '') . '">';
                    Field::text(
                        Constants::OPTION_CHAT_MESSAGE_CUSTOM_FONT_FAMILY,
                        '',
                        $message_custom_font,
                        [
                            'id' => 'message_custom_font_family',
                            'attributes' => ['placeholder' => __('e.g., Montserrat, Arial, sans-serif', 'intufind')],
                            'class' => 'regular-text',
                            'wrapper_class' => 'no-margin'
                        ]
                    );
                    echo '</div>';
                });

                // Assistant Bubble Background
                $this->renderTableRow(__('Assistant Bubble Background', 'intufind'), function () {
                    ColorPicker::dual([
                        'light_name' => Constants::OPTION_CHAT_ASSISTANT_BUBBLE_BG,
                        'light_value' => get_option(Constants::OPTION_CHAT_ASSISTANT_BUBBLE_BG, '#f8f9fa'),
                        'dark_name' => Constants::OPTION_CHAT_DARK_ASSISTANT_BUBBLE_BG,
                        'dark_value' => get_option(Constants::OPTION_CHAT_DARK_ASSISTANT_BUBBLE_BG, '#2d2d2d'),
                        'description' => __('Background color for AI assistant message bubbles.', 'intufind')
                    ]);
                });

                // Link Color
                $this->renderTableRow(__('Link Color', 'intufind'), function () {
                    ColorPicker::dual([
                        'light_name' => Constants::OPTION_CHAT_LINK_COLOR,
                        'light_value' => get_option(Constants::OPTION_CHAT_LINK_COLOR, '#0084ff'),
                        'dark_name' => Constants::OPTION_CHAT_DARK_LINK_COLOR,
                        'dark_value' => get_option(Constants::OPTION_CHAT_DARK_LINK_COLOR, '#6495ed'),
                        'description' => __('Color for clickable links in messages and cards.', 'intufind')
                    ]);
                });

                echo '</table>';
            }
        );
    }

    /**
     * Render cards & content styling panel
     */
    private function renderCardsContentPanel(): void
    {
        $this->renderSettingsPanel(
            __('Cards & Content', 'intufind'),
            __('Customize the appearance of product and post cards displayed in chat responses.', 'intufind'),
            'id-alt',
            function () {
                echo '<table class="form-table">';

                // Card Background
                $this->renderTableRow(__('Card Background', 'intufind'), function () {
                    ColorPicker::dual([
                        'light_name' => Constants::OPTION_CHAT_CARD_BACKGROUND,
                        'light_value' => get_option(Constants::OPTION_CHAT_CARD_BACKGROUND, '#ffffff'),
                        'dark_name' => Constants::OPTION_CHAT_DARK_CARD_BACKGROUND,
                        'dark_value' => get_option(Constants::OPTION_CHAT_DARK_CARD_BACKGROUND, '#2d2d2d'),
                        'description' => __('Background color for product and post cards.', 'intufind')
                    ]);
                });

                // Card Border
                $this->renderTableRow(__('Card Border', 'intufind'), function () {
                    ColorPicker::dual([
                        'light_name' => Constants::OPTION_CHAT_CARD_BORDER,
                        'light_value' => get_option(Constants::OPTION_CHAT_CARD_BORDER, '#e1e5e9'),
                        'dark_name' => Constants::OPTION_CHAT_DARK_CARD_BORDER,
                        'dark_value' => get_option(Constants::OPTION_CHAT_DARK_CARD_BORDER, '#404040'),
                        'description' => __('Border color around cards.', 'intufind')
                    ]);
                });

                // Card Hover
                $this->renderTableRow(__('Card Hover', 'intufind'), function () {
                    ColorPicker::dual([
                        'light_name' => Constants::OPTION_CHAT_CARD_HOVER_BG,
                        'light_value' => get_option(Constants::OPTION_CHAT_CARD_HOVER_BG, '#f8f9fa'),
                        'dark_name' => Constants::OPTION_CHAT_DARK_CARD_HOVER_BG,
                        'dark_value' => get_option(Constants::OPTION_CHAT_DARK_CARD_HOVER_BG, '#383838'),
                        'description' => __('Background color when hovering over cards.', 'intufind')
                    ]);
                });

                // Card Shape
                $this->renderTableRow(__('Card Shape', 'intufind'), function () {
                    $border_radius = get_option(Constants::OPTION_CHAT_CARD_BORDER_RADIUS, 8);
                    $shadow = get_option(Constants::OPTION_CHAT_CARD_SHADOW, 'light');

                    echo '<div style="display: flex; flex-direction: column; gap: 16px;">';
                    
                    // Border Radius
                    echo '<div>';
                    Field::range(
                        Constants::OPTION_CHAT_CARD_BORDER_RADIUS,
                        __('Corner Radius', 'intufind'),
                        $border_radius,
                        [
                            'id' => 'card_border_radius_slider',
                            'attributes' => ['min' => '0', 'max' => '24'],
                            'suffix' => __('px', 'intufind'),
                            'wrapper_class' => 'no-margin'
                        ]
                    );
                    echo '</div>';

                    // Shadow
                    echo '<div>';
                    Field::select(
                        Constants::OPTION_CHAT_CARD_SHADOW,
                        __('Drop Shadow', 'intufind'),
                        [
                            'none' => __('None', 'intufind'),
                            'light' => __('Light', 'intufind'),
                            'medium' => __('Medium', 'intufind'),
                            'heavy' => __('Heavy', 'intufind'),
                        ],
                        $shadow,
                        ['wrapper_class' => 'no-margin']
                    );
                    echo '</div>';

                    echo '</div>';
                    $this->renderHint(__('Adjust card corner roundness and shadow depth.', 'intufind'));
                });

                // Title Color
                $this->renderTableRow(__('Title Color', 'intufind'), function () {
                    ColorPicker::dual([
                        'light_name' => Constants::OPTION_CHAT_CARD_TITLE_COLOR,
                        'light_value' => get_option(Constants::OPTION_CHAT_CARD_TITLE_COLOR, '#1a1a1a'),
                        'dark_name' => Constants::OPTION_CHAT_DARK_CARD_TITLE_COLOR,
                        'dark_value' => get_option(Constants::OPTION_CHAT_DARK_CARD_TITLE_COLOR, '#f5f5f5'),
                        'description' => __('Color for card titles (product names, post titles).', 'intufind')
                    ]);
                });

                // Title Font Size
                $this->renderTableRow(__('Title Font', 'intufind'), function () {
                    $title_size = get_option(Constants::OPTION_CHAT_CARD_TITLE_SIZE, 16);
                    $title_weight = get_option(Constants::OPTION_CHAT_CARD_TITLE_WEIGHT, '600');

                    echo '<div style="display: flex; flex-direction: column; gap: 16px;">';
                    
                    echo '<div>';
                    Field::number(
                        Constants::OPTION_CHAT_CARD_TITLE_SIZE,
                        __('Size', 'intufind'),
                        $title_size,
                        [
                            'attributes' => ['min' => '12', 'max' => '24'],
                            'suffix' => __('px', 'intufind'),
                            'wrapper_class' => 'no-margin'
                        ]
                    );
                    echo '</div>';

                    echo '<div>';
                    Field::select(
                        Constants::OPTION_CHAT_CARD_TITLE_WEIGHT,
                        __('Weight', 'intufind'),
                        [
                            '400' => __('Normal (400)', 'intufind'),
                            '500' => __('Medium (500)', 'intufind'),
                            '600' => __('Semi Bold (600)', 'intufind'),
                            '700' => __('Bold (700)', 'intufind'),
                        ],
                        $title_weight,
                        ['wrapper_class' => 'no-margin']
                    );
                    echo '</div>';

                    echo '</div>';
                    $this->renderHint(__('Font styling for card titles.', 'intufind'));
                });

                // Meta Text (price, date, author)
                $this->renderTableRow(__('Meta Text', 'intufind'), function () {
                    ColorPicker::dual([
                        'light_name' => Constants::OPTION_CHAT_CARD_META_COLOR,
                        'light_value' => get_option(Constants::OPTION_CHAT_CARD_META_COLOR, '#6c757d'),
                        'dark_name' => Constants::OPTION_CHAT_DARK_CARD_META_COLOR,
                        'dark_value' => get_option(Constants::OPTION_CHAT_DARK_CARD_META_COLOR, '#a0a0a0'),
                        'description' => __('Color for secondary information (price, date, author).', 'intufind')
                    ]);
                });

                // Description Text
                $this->renderTableRow(__('Description Text', 'intufind'), function () {
                    ColorPicker::dual([
                        'light_name' => Constants::OPTION_CHAT_CARD_DESC_COLOR,
                        'light_value' => get_option(Constants::OPTION_CHAT_CARD_DESC_COLOR, '#495057'),
                        'dark_name' => Constants::OPTION_CHAT_DARK_CARD_DESC_COLOR,
                        'dark_value' => get_option(Constants::OPTION_CHAT_DARK_CARD_DESC_COLOR, '#d0d0d0'),
                        'description' => __('Color for card description text.', 'intufind')
                    ]);
                });

                // Description Length
                $this->renderTableRow(__('Description Length', 'intufind'), function () {
                    Field::number(
                        Constants::OPTION_CHAT_CARD_DESC_LENGTH,
                        '',
                        get_option(Constants::OPTION_CHAT_CARD_DESC_LENGTH, 150),
                        [
                            'attributes' => ['min' => '50', 'max' => '300'],
                            'suffix' => __('characters', 'intufind'),
                            'description' => __('Maximum length for card descriptions (50-300 characters).', 'intufind')
                        ]
                    );
                });

                // Card Font Size
                $this->renderTableRow(__('Font Size', 'intufind'), function () {
                    Field::number(
                        Constants::OPTION_CHAT_CARD_FONT_SIZE,
                        '',
                        get_option(Constants::OPTION_CHAT_CARD_FONT_SIZE, 14),
                        [
                            'attributes' => ['min' => '12', 'max' => '18'],
                            'suffix' => __('px', 'intufind'),
                            'description' => __('Font size for card text (12-18px). Controls the size of text in product and post cards.', 'intufind')
                        ]
                    );
                });

                // Card Line Height
                $this->renderTableRow(__('Line Height', 'intufind'), function () {
                    Field::number(
                        Constants::OPTION_CHAT_CARD_LINE_HEIGHT,
                        '',
                        get_option(Constants::OPTION_CHAT_CARD_LINE_HEIGHT, '1.4'),
                        [
                            'attributes' => ['min' => '1.0', 'max' => '2.0', 'step' => '0.1'],
                            'description' => __('Line height for card text (1.0-2.0). Controls spacing between lines in cards. 1.4 provides compact but readable text.', 'intufind')
                        ]
                    );
                });

                echo '</table>';
            }
        );
    }

    /**
     * Render prompts & suggestions styling panel
     */
    private function renderPromptsPanel(): void
    {
        $this->renderSettingsPanel(
            __('Prompts & Suggestions', 'intufind'),
            __('Customize the appearance of prompt suggestion chips displayed to users.', 'intufind'),
            'lightbulb',
            function () {
                echo '<table class="form-table">';

                // Prompt Chip Background
                $this->renderTableRow(__('Chip Background', 'intufind'), function () {
                    ColorPicker::dual([
                        'light_name' => Constants::OPTION_CHAT_PROMPT_CHIP_BACKGROUND,
                        'light_value' => get_option(Constants::OPTION_CHAT_PROMPT_CHIP_BACKGROUND, '#f8f9fa'),
                        'dark_name' => Constants::OPTION_CHAT_DARK_PROMPT_CHIP_BACKGROUND,
                        'dark_value' => get_option(Constants::OPTION_CHAT_DARK_PROMPT_CHIP_BACKGROUND, '#2d2d2d'),
                        'description' => __('Background color for prompt suggestion chips.', 'intufind')
                    ]);
                });

                // Prompt Chip Text Color
                $this->renderTableRow(__('Chip Text', 'intufind'), function () {
                    ColorPicker::dual([
                        'light_name' => Constants::OPTION_CHAT_PROMPT_CHIP_TEXT_COLOR,
                        'light_value' => get_option(Constants::OPTION_CHAT_PROMPT_CHIP_TEXT_COLOR, '#202124'),
                        'dark_name' => Constants::OPTION_CHAT_DARK_PROMPT_CHIP_TEXT_COLOR,
                        'dark_value' => get_option(Constants::OPTION_CHAT_DARK_PROMPT_CHIP_TEXT_COLOR, '#ffffff'),
                        'description' => __('Text color for prompt chips.', 'intufind')
                    ]);
                });

                // Prompt Chip Border
                $this->renderTableRow(__('Chip Border', 'intufind'), function () {
                    ColorPicker::dual([
                        'light_name' => Constants::OPTION_CHAT_PROMPT_CHIP_BORDER,
                        'light_value' => get_option(Constants::OPTION_CHAT_PROMPT_CHIP_BORDER, '#d1d5db'),
                        'dark_name' => Constants::OPTION_CHAT_DARK_PROMPT_CHIP_BORDER,
                        'dark_value' => get_option(Constants::OPTION_CHAT_DARK_PROMPT_CHIP_BORDER, '#404040'),
                        'description' => __('Border color around prompt chips.', 'intufind')
                    ]);
                });

                // Prompt Chip Hover Background
                $this->renderTableRow(__('Chip Hover', 'intufind'), function () {
                    ColorPicker::dual([
                        'light_name' => Constants::OPTION_CHAT_PROMPT_CHIP_HOVER_BG,
                        'light_value' => get_option(Constants::OPTION_CHAT_PROMPT_CHIP_HOVER_BG, '#f1f3f4'),
                        'dark_name' => Constants::OPTION_CHAT_DARK_PROMPT_CHIP_HOVER_BG,
                        'dark_value' => get_option(Constants::OPTION_CHAT_DARK_PROMPT_CHIP_HOVER_BG, '#383838'),
                        'description' => __('Background color when hovering over prompt chips.', 'intufind')
                    ]);
                });

                echo '</table>';
            }
        );
    }

    /**
     * Render styling overrides panel (advanced options)
     */
    private function renderStylingOverridesPanel(): void
    {
        $this->renderSettingsPanel(
            __('Styling Overrides', 'intufind'),
            __('Fine-tune the widget appearance with advanced options.', 'intufind'),
            'admin-customizer',
            function () {
                echo '<table class="form-table">';

                // Font Family
                $this->renderTableRow(__('Font Family', 'intufind'), function () {
                    $font_family = get_option(Constants::OPTION_CHAT_FONT_FAMILY, 'system');
                    $custom_font = get_option(Constants::OPTION_CHAT_CUSTOM_FONT_FAMILY, '');

                    Field::select(
                        Constants::OPTION_CHAT_FONT_FAMILY,
                        '',
                        [
                            'system' => __('System Default (Recommended)', 'intufind'),
                            'inter' => __('Inter (Google Font)', 'intufind'),
                            'roboto' => __('Roboto (Google Font)', 'intufind'),
                            'open-sans' => __('Open Sans (Google Font)', 'intufind'),
                            'lato' => __('Lato (Google Font)', 'intufind'),
                            'custom' => __('Custom Font', 'intufind'),
                        ],
                        $font_family,
                        [
                            'description' => __('Choose a font to match your website. Custom font requires the font to be loaded on your site.', 'intufind')
                        ]
                    );

                    echo '<div id="custom_font_wrapper" style="margin-top: 12px; ' . ($font_family !== 'custom' ? 'display:none;' : '') . '">';
                    Field::text(
                        Constants::OPTION_CHAT_CUSTOM_FONT_FAMILY,
                        '',
                        $custom_font,
                        [
                            'id' => 'custom_font_family',
                            'attributes' => ['placeholder' => __('e.g., Montserrat, Arial, sans-serif', 'intufind')],
                            'class' => 'regular-text',
                            'wrapper_class' => 'no-margin'
                        ]
                    );
                    echo '</div>';
                });

                // Widget Shadow
                $this->renderTableRow(__('Widget Shadow', 'intufind'), function () {
                    Field::select(
                        Constants::OPTION_CHAT_WIDGET_SHADOW,
                        '',
                        [
                            'none' => __('None (Flat)', 'intufind'),
                            'light' => __('Light', 'intufind'),
                            'medium' => __('Medium (Recommended)', 'intufind'),
                            'heavy' => __('Heavy', 'intufind'),
                        ],
                        get_option(Constants::OPTION_CHAT_WIDGET_SHADOW, 'medium'),
                        [
                            'description' => __('Adjust the shadow depth around the widget (affects both floating button and chat window).', 'intufind')
                        ]
                    );
                });

                // Input Field Colors
                $this->renderTableRow(__('Input Field', 'intufind'), function () {
                    echo '<div style="display: flex; flex-direction: column; gap: 16px;">';
                    
                    // Background (dual)
                    echo '<div>';
                    echo '<label style="font-weight: 600; display: block; margin-bottom: 8px;">' . esc_html__('Background:', 'intufind') . '</label>';
                    ColorPicker::dual([
                        'light_name' => Constants::OPTION_CHAT_INPUT_BACKGROUND,
                        'light_value' => get_option(Constants::OPTION_CHAT_INPUT_BACKGROUND, '#ffffff'),
                        'dark_name' => Constants::OPTION_CHAT_DARK_INPUT_BACKGROUND,
                        'dark_value' => get_option(Constants::OPTION_CHAT_DARK_INPUT_BACKGROUND, '#2d2d2d')
                    ]);
                    echo '</div>';
                    
                    // Border (dual)
                    echo '<div>';
                    echo '<label style="font-weight: 600; display: block; margin-bottom: 8px;">' . esc_html__('Border:', 'intufind') . '</label>';
                    ColorPicker::dual([
                        'light_name' => Constants::OPTION_CHAT_INPUT_BORDER,
                        'light_value' => get_option(Constants::OPTION_CHAT_INPUT_BORDER, '#dadce0'),
                        'dark_name' => Constants::OPTION_CHAT_DARK_INPUT_BORDER,
                        'dark_value' => get_option(Constants::OPTION_CHAT_DARK_INPUT_BORDER, '#404040')
                    ]);
                    echo '</div>';
                    
                    echo '</div>';
                    $this->renderHint(__('Customize the message input field appearance.', 'intufind'));
                });

                // Widget Dimensions
                $this->renderTableRow(__('Widget Dimensions', 'intufind'), function () {
                    $max_width = get_option(Constants::OPTION_CHAT_WIDGET_MAX_WIDTH, 800);
                    $max_height = get_option(Constants::OPTION_CHAT_WIDGET_MAX_HEIGHT, 900);

                    echo '<div style="display: flex; flex-direction: column; gap: 16px;">';
                    
                    echo '<div>';
                    Field::range(
                        Constants::OPTION_CHAT_WIDGET_MAX_WIDTH,
                        __('Max Width', 'intufind'),
                        $max_width,
                        [
                            'id' => 'max_width_slider',
                            'attributes' => ['min' => '300', 'max' => '1400', 'step' => '10'],
                            'suffix' => __('px', 'intufind'),
                            'wrapper_class' => 'no-margin'
                        ]
                    );
                    echo '</div>';
                    
                    echo '<div>';
                    Field::range(
                        Constants::OPTION_CHAT_WIDGET_MAX_HEIGHT,
                        __('Max Height', 'intufind'),
                        $max_height,
                        [
                            'id' => 'max_height_slider',
                            'attributes' => ['min' => '400', 'max' => '1000', 'step' => '10'],
                            'suffix' => __('px', 'intufind'),
                            'wrapper_class' => 'no-margin'
                        ]
                    );
                    echo '</div>';
                    
                    echo '</div>';
                    $this->renderHint(__('Control the maximum size of the chat widget (especially useful on desktop).', 'intufind'));
                });

                // Widget Background
                $this->renderTableRow(__('Widget Background', 'intufind'), function () {
                    $bg_mode = get_option(Constants::OPTION_CHAT_WIDGET_BACKGROUND_MODE, 'auto');

                    Field::select(
                        Constants::OPTION_CHAT_WIDGET_BACKGROUND_MODE,
                        '',
                        [
                            'auto' => __('Auto (theme default)', 'intufind'),
                            'custom' => __('Custom Color', 'intufind'),
                        ],
                        $bg_mode,
                        [
                            'description' => __('Override the automatic background color if needed.', 'intufind')
                        ]
                    );

                    echo '<div id="widget_bg_custom_wrapper" style="margin-top: 12px; ' . ($bg_mode === 'auto' ? 'display:none;' : '') . '">';
                    ColorPicker::dual([
                        'light_name' => Constants::OPTION_CHAT_WIDGET_BACKGROUND_CUSTOM,
                        'light_value' => get_option(Constants::OPTION_CHAT_WIDGET_BACKGROUND_CUSTOM, '#ffffff'),
                        'dark_name' => Constants::OPTION_CHAT_DARK_WIDGET_BACKGROUND_CUSTOM,
                        'dark_value' => get_option(Constants::OPTION_CHAT_DARK_WIDGET_BACKGROUND_CUSTOM, '#1f1f1f')
                    ]);
                    echo '</div>';
                });

                // Border Radius
                $this->renderTableRow(__('Border Radius', 'intufind'), function () {
                    Field::range(
                        Constants::OPTION_CHAT_BORDER_RADIUS,
                        '',
                        get_option(Constants::OPTION_CHAT_BORDER_RADIUS, 16),
                        [
                            'id' => 'border_radius_slider',
                            'attributes' => ['min' => '0', 'max' => '50'],
                            'suffix' => __('px', 'intufind'),
                            'description' => __('Adjust the roundness of widget corners (0 = square, 50 = very round).', 'intufind')
                        ]
                    );
                });

                // Custom CSS
                $this->renderTableRow(__('Custom CSS', 'intufind'), function () {
                    $custom_css = get_option(Constants::OPTION_CHAT_CUSTOM_CSS, '');
                    $placeholder = "/* " . esc_html__('Example: Customize message bubble styling', 'intufind') . " */\n"
                                 . "[data-itf-part=\"assistant-message\"] {\n"
                                 . "  --itf-assistant-bubble-bg: #f0f4f8;\n"
                                 . "  border-left: 3px solid var(--itf-primary);\n"
                                 . "}\n\n"
                                 . "/* " . esc_html__('Override header padding', 'intufind') . " */\n"
                                 . "[data-itf-part=\"header\"] {\n"
                                 . "  padding: 20px;\n"
                                 . "}";
                    echo '<textarea name="' . esc_attr(Constants::OPTION_CHAT_CUSTOM_CSS) . '" rows="8" style="width: 100%; max-width: 600px; font-family: monospace; font-size: 12px;" placeholder="' . esc_attr($placeholder) . '">';
                    echo esc_textarea($custom_css);
                    echo '</textarea>';
                    echo '<br>';
                    $this->renderHint(__('For developers: Add custom CSS to override any widget styles. Target elements using [data-itf-part="..."] attributes or CSS variables (--itf-*).', 'intufind'));
                });

                echo '</table>';
            }
        );
    }

    /**
     * Render avatar uploader UI (uses standard media.js system)
     */
    private function renderAvatarUploader(string $type, string $option_name, string $hint): void
    {
        $icon_id = get_option($option_name, 0);
        
        ImageUploader::avatar($option_name, $icon_id, [
            'preview_style' => 'border-radius: var(--itf-avatar-border-radius, 50%);',
            'hint' => $hint
        ]);
    }

    /**
     * Render reset button
     */
    private function renderResetButton(): void
    {
        ?>
        <div class="intucart-notice intucart-notice-warning">
                <p><strong>⚠️ <?php echo esc_html__('Reset to Defaults', 'intufind'); ?></strong></p>
                <p>
                    <?php echo esc_html__('Reset all styling settings to their original theme defaults. This will clear custom colors, avatars, and CSS.', 'intufind'); ?> 
                    <strong><?php echo esc_html__('This action cannot be undone!', 'intufind'); ?></strong>
                </p>
                <p style="margin-top: 12px;">
                <?php
                Button::danger(
                    __('Reset All Styling Settings', 'intufind'),
                    [
                        'id' => 'intucart-reset-styling',
                        'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path opacity=".4" d="M8 454.06V320a24 24 0 0 1 24-24h134.06c21.38 0 32.09 25.85 17 41l-41.75 41.75A166.82 166.82 0 0 0 256.16 424c77.41-.07 144.31-53.14 162.78-126.85a12 12 0 0 1 11.65-9.15h57.31a12 12 0 0 1 11.81 14.18C478.07 417.08 377.19 504 256 504a247.14 247.14 0 0 1-171.31-68.69L49 471c-15.15 15.15-41 4.44-41-16.94z"/><path d="M12.3 209.82C33.93 94.92 134.81 8 256 8a247.14 247.14 0 0 1 171.31 68.69L463 41c15.12-15.12 41-4.41 41 17v134a24 24 0 0 1-24 24H345.94c-21.38 0-32.09-25.85-17-41l41.75-41.75A166.8 166.8 0 0 0 255.85 88c-77.46.07-144.33 53.18-162.79 126.85A12 12 0 0 1 81.41 224H24.1a12 12 0 0 1-11.8-14.18z"/></svg>',
                        'size' => 'medium'
                    ]
                );
                ?>
                <span id="intucart-reset-status" style="margin-left: 12px; display: none;"></span>
                </p>
        </div>
        <?php
    }

    /**
     * Render JavaScript for UI interactions
     * Note: All JavaScript has been moved to assets/js/admin/chatbot/styling.js
     */
    private function renderJavaScript(): void
    {
        ?>
        <style>
        /* Ensure dashicons in buttons are vertically aligned */
        .button .dashicons {
            vertical-align: middle;
            margin-top: -6px;
        }
        /* Reset button hover state */
        #intucart-reset-styling:hover {
            background: #c82333 !important;
            border-color: #bd2130 !important;
        }
        /* Rotation animation for loading spinners */
        @keyframes rotation {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }
        </style>
        <?php
    }
}
