<?php

/**
 * WooCommerce Settings - AI features for e-commerce
 *
 * Handles WooCommerce-specific AI configuration including MCP capabilities,
 * product recommendations, and shopping assistant behavior.
 */

namespace Intucart\Pages\Settings\Chat;

use Intucart\Pages\Settings\AbstractSettings;
use Intucart\Services\Constants;
use Intucart\Services\Logger;
use Intucart\Services\AIClientManager;
use Intucart\Components\Button;
use Intucart\Components\Field;

class WooCommerceSettings extends AbstractSettings
{
    private AIClientManager $aiClientManager;

    public function __construct(Logger $logger, AIClientManager $aiClientManager)
    {
        parent::__construct($logger);
        $this->aiClientManager = $aiClientManager;
    }

    /**
     * Register settings and sync hooks
     */
    public function registerSettings(): void
    {
        // Hook into option updates to sync MCP config to cloud
        add_action('update_option_' . Constants::OPTION_ENABLE_MCP_ORDERS, [$this, 'onMcpSettingChanged'], 10, 3);
        
        // Register AJAX handler for manual sync
        add_action('wp_ajax_intucart_sync_mcp_config', [$this, 'handleMcpSyncAjax']);
    }

    /**
     * Handle MCP setting change - sync to cloud
     * 
     * @param mixed $old_value Previous value
     * @param mixed $new_value New value
     * @param string $option Option name
     */
    public function onMcpSettingChanged($old_value, $new_value, $option): void
    {
        // Skip if value hasn't actually changed
        if ($old_value === $new_value) {
            return;
        }

        // Sync MCP config to cloud
        $result = $this->aiClientManager->syncMcpConfig();
        
        if (defined('WP_DEBUG') && WP_DEBUG) {
            if ($result['success']) {
                error_log('Intucart: MCP config synced to cloud - ' . ($result['action'] ?? 'unknown'));
            } else {
                error_log('Intucart: Failed to sync MCP config - ' . ($result['error'] ?? 'unknown error'));
            }
        }
    }

    /**
     * AJAX handler for manual MCP config sync
     */
    public function handleMcpSyncAjax(): void
    {
        // Verify nonce
        if (!check_ajax_referer('intucart_admin_nonce', 'nonce', false)) {
            wp_send_json_error(['message' => __('Invalid security token', 'intufind')]);
            return;
        }

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'intufind')]);
            return;
        }

        $result = $this->aiClientManager->syncMcpConfig();

        if ($result['success']) {
            wp_send_json_success([
                'message' => __('MCP configuration synced successfully', 'intufind'),
                'action' => $result['action'] ?? 'synced',
                'config' => $result['config'] ?? []
            ]);
        } else {
            wp_send_json_error([
                'message' => sprintf(__('Failed to sync MCP configuration: %s', 'intufind'), $result['error'] ?? __('Unknown error', 'intufind'))
            ]);
        }
    }

    /**
     * Render MCP connection status indicator
     */
    private function renderMcpSyncStatus(): void
    {
        $mcpStatus = $this->aiClientManager->getMcpConfig();
        $isConfigured = $mcpStatus['configured'] ?? false;
        $localEnabled = get_option(Constants::OPTION_ENABLE_MCP_ORDERS, true);
        $woocommerceActive = class_exists('WooCommerce');
        
        // Determine connection state
        $expectedConfigured = $woocommerceActive && $localEnabled;
        $inSync = $isConfigured === $expectedConfigured;
        
        if ($inSync && $isConfigured) {
            // Connected and working
            $statusClass = 'intucart-notice-success';
            $statusIcon = '✓';
            $statusTitle = __('Connected', 'intufind');
            $statusText = __('The AI assistant can access your WooCommerce order data.', 'intufind');
            $showReconnect = false;
        } elseif ($inSync && !$isConfigured) {
            // Intentionally disconnected
            $statusClass = 'intucart-notice-info';
            $statusIcon = '';
            $statusTitle = __('Not Connected', 'intufind');
            $statusText = __('Enable "Order Inquiries" below to let the AI help customers with their orders.', 'intufind');
            $showReconnect = false;
        } else {
            // Out of sync - needs reconnection
            $statusClass = 'intucart-notice-warning';
            $statusIcon = '⚠';
            $statusTitle = __('Connection Issue', 'intufind');
            $statusText = __('Your settings have changed. Click "Reconnect" to update the AI assistant\'s access.', 'intufind');
            $showReconnect = true;
        }
        
        ?>
        <div class="intucart-notice <?php echo esc_attr($statusClass); ?>" style="margin: 0 0 20px 0;">
            <div style="display: flex; align-items: flex-start; justify-content: space-between; gap: 16px;">
                <div style="flex: 1;">
                    <p style="margin: 0 0 4px 0;">
                        <strong><?php echo esc_html($statusIcon ? $statusIcon . ' ' : ''); ?><?php echo esc_html($statusTitle); ?></strong>
                    </p>
                    <p style="margin: 0;">
                        <?php echo esc_html($statusText); ?>
                    </p>
                </div>
                <?php if ($showReconnect): ?>
                <div>
                    <?php Button::primary(__('Reconnect', 'intufind'), [
                        'id' => 'intucart-mcp-sync-btn',
                        'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512" style="width: 14px; height: 14px;"><path d="M500.33 0h-47.41a12 12 0 0 0-12 12.57l4 82.76A247.42 247.42 0 0 0 256 8C119.34 8 7.9 119.53 8 256.19 8.1 393.07 119.1 504 256 504a247.1 247.1 0 0 0 166.18-63.91 12 12 0 0 0 .48-17.43l-34-34a12 12 0 0 0-16.38-.55A176 176 0 1 1 402.1 157.8l-101.53-4.87a12 12 0 0 0-12.57 12v47.41a12 12 0 0 0 12 12H500.33a12 12 0 0 0 12-12V12a12 12 0 0 0-12-12z"/></svg>',
                        'size' => 'medium'
                    ]); ?>
                </div>
                <?php endif; ?>
            </div>
        </div>
        
        <?php if ($showReconnect):
            wp_enqueue_script(
                'intucart-mcp-sync',
                INTUCART_PLUGIN_URL . 'assets/js/admin/mcp-sync.js',
                ['jquery'],
                INTUCART_VERSION,
                true
            );
            wp_localize_script('intucart-mcp-sync', 'intucartMcpSync', [
                'nonce' => wp_create_nonce('intucart_admin_nonce')
            ]);
        ?>
        <style>
            .itf-btn-icon svg.spin {
                animation: intucart-spin 1s linear infinite;
            }
            @keyframes intucart-spin {
                from { transform: rotate(0deg); }
                to { transform: rotate(360deg); }
            }
        </style>
        <?php endif;
    }

    /**
     * Render WooCommerce-specific AI settings
     */
    public function render(): void
    {
        // Check if WooCommerce is installed
        $woocommerceActive = class_exists('WooCommerce');
        
        if (!$woocommerceActive) {
            $this->renderWooCommerceNotInstalledNotice();
            return;
        }

        // ===================================================================
        // AI CAPABILITIES (MCP)
        // ===================================================================
        $this->renderSettingsPanel(
            __('AI Capabilities', 'intufind'),
            __('Enable real-time access to WooCommerce data for customer service.', 'intufind'),
            'admin-network',
            function () {
                // Show sync status
                $this->renderMcpSyncStatus();
                
                echo '<table class="form-table">';

                // Enable Order Inquiries (MCP)
                $this->renderTableRow(__('Order Inquiries', 'intufind'), function () {
                    $enabled = get_option(Constants::OPTION_ENABLE_MCP_ORDERS, true);
                    
                    Field::checkbox(
                        Constants::OPTION_ENABLE_MCP_ORDERS, 
                        __('Let the AI assistant help customers with their orders', 'intufind'), 
                        $enabled,
                        [
                            'description' => __('When enabled, logged-in customers can ask the AI about their order status, tracking info, and order history.', 'intufind'),
                            'hint' => [[
                                'title' => __('What customers can ask:', 'intufind'),
                                'content' => __('"Where is my order?" • "What did I order last month?" • "Has my order shipped yet?" • "Show me my recent purchases"', 'intufind')
                            ], [
                                'title' => __('Privacy & Security:', 'intufind'),
                                'content' => __('Customers can only see their own orders. Order data stays on your WordPress site and is accessed securely via authenticated API calls.', 'intufind')
                            ]]
                        ]
                    );
                });

                // Future MCP capabilities
                echo '<tr><td colspan="2" style="padding-top: 20px;">';
                echo '<div style="background: #f9f9f9; border: 1px solid #ddd; padding: 12px 16px; border-radius: 4px;">';
                echo '<strong style="font-size: 13px;">🚀 ' . esc_html__('Coming Soon: Additional Customer Service Capabilities', 'intufind') . '</strong>';
                echo '<ul style="margin: 8px 0 0 0; font-size: 13px; line-height: 1.6;">';
                echo '<li><strong>' . esc_html__('Inventory Management:', 'intufind') . '</strong> ' . esc_html__('Real-time stock level checking and low-stock alerts', 'intufind') . '</li>';
                echo '<li><strong>' . esc_html__('Shipping & Tracking:', 'intufind') . '</strong> ' . esc_html__('Live shipping rate estimates and carrier tracking', 'intufind') . '</li>';
                echo '<li><strong>' . esc_html__('Returns & Refunds:', 'intufind') . '</strong> ' . esc_html__('Automated return policy guidance and refund status lookups', 'intufind') . '</li>';
                echo '<li><strong>' . esc_html__('Customer Account:', 'intufind') . '</strong> ' . esc_html__('View subscriptions, payment methods, and account preferences', 'intufind') . '</li>';
                echo '</ul>';
                echo '</div>';
                echo '</td></tr>';

                echo '</table>';
                
                echo '<div style="background: #f9f9f9; border: 1px solid #ddd; padding: 12px 16px; margin-top: 20px; border-radius: 4px;">';
                echo '<strong style="font-size: 13px;">🔒 ' . esc_html__('Security & Privacy', 'intufind') . '</strong>';
                echo '<ul style="margin: 8px 0 0 0; font-size: 13px; line-height: 1.6;">';
                echo '<li>' . esc_html__('All MCP requests are authenticated with JWT tokens (signed with your license key)', 'intufind') . '</li>';
                echo '<li>' . esc_html__('Users can only access their own data (customer orders, account info, etc.)', 'intufind') . '</li>';
                echo '<li>' . esc_html__('Admins with proper WooCommerce capabilities can access broader data for support purposes', 'intufind') . '</li>';
                echo '<li>' . esc_html__('All data remains on your WordPress site - nothing is stored in the cloud', 'intufind') . '</li>';
                echo '<li>' . esc_html__('Order data is transmitted securely over HTTPS and never persisted in AI conversation history', 'intufind') . '</li>';
                echo '</ul>';
                echo '</div>';
            }
        );

        // ===================================================================
        // PRODUCT DISCOVERY (Already Active)
        // ===================================================================
        $this->renderSettingsPanel(
            __('Product Discovery', 'intufind'),
            __('AI-powered product search and recommendations are active and working.', 'intufind'),
            'products',
            function () {
                echo '<div class="intucart-notice intucart-notice-success" style="margin-bottom: 20px;">';
                echo '<p><strong>✓ ' . esc_html__('Active Feature', 'intufind') . '</strong></p>';
                echo '<p>' . esc_html__('The AI assistant can already help customers discover products using natural language queries. No configuration needed—it works automatically!', 'intufind') . '</p>';
                echo '</div>';
                
                echo '<table class="form-table">';
                
                echo '<tr><td colspan="2">';
                echo '<strong style="font-size: 13px;">✓ ' . esc_html__('Current Capabilities', 'intufind') . '</strong>';
                echo '<ul style="margin: 8px 0 0 0; font-size: 13px; line-height: 1.6;">';
                echo '<li><strong>' . esc_html__('Semantic Search:', 'intufind') . '</strong> ' . esc_html__('Understands natural language product queries (e.g., "comfortable running shoes")', 'intufind') . '</li>';
                echo '<li><strong>' . esc_html__('Intelligent Filtering:', 'intufind') . '</strong> ' . esc_html__('Automatically applies filters for price, color, size, categories, and more', 'intufind') . '</li>';
                echo '<li><strong>' . esc_html__('Context Awareness:', 'intufind') . '</strong> ' . esc_html__('References previous products shown in the conversation', 'intufind') . '</li>';
                echo '<li><strong>' . esc_html__('Product Details:', 'intufind') . '</strong> ' . esc_html__('Fetches and displays detailed product information on request', 'intufind') . '</li>';
                echo '<li><strong>' . esc_html__('Recommendations:', 'intufind') . '</strong> ' . esc_html__('Suggests related or alternative products based on user preferences', 'intufind') . '</li>';
                echo '</ul>';
                echo '</td></tr>';
                
                echo '<tr><td colspan="2" style="padding-top: 16px;">';
                echo '<strong style="font-size: 13px;">💡 ' . esc_html__('Example Queries (Already Working)', 'intufind') . '</strong>';
                echo '<ul style="margin: 8px 0 0 0; font-size: 13px; line-height: 1.6;">';
                echo '<li>"' . esc_html__('Show me baseball caps under $30', 'intufind') . '"</li>';
                echo '<li>"' . esc_html__('I need a blue medium t-shirt', 'intufind') . '"</li>';
                echo '<li>"' . esc_html__('What running shoes do you have in my size?', 'intufind') . '"</li>';
                echo '<li>"' . esc_html__('Show me products similar to the last one', 'intufind') . '"</li>';
                echo '<li>"' . esc_html__('Tell me more about product #1234', 'intufind') . '"</li>';
                echo '</ul>';
                echo '</td></tr>';
                
                echo '<tr><td colspan="2" style="padding-top: 16px;">';
                echo '<strong style="font-size: 13px;">📝 ' . esc_html__('Pro Tip', 'intufind') . '</strong>';
                echo '<p style="margin: 8px 0 0 0; font-size: 13px; line-height: 1.6;">' . esc_html__('The AI uses your WooCommerce product data including titles, descriptions, categories, attributes, and pricing. Keep your product information up-to-date for best results!', 'intufind') . '</p>';
                echo '</td></tr>';
                
                echo '</table>';
            }
        );
    }

    /**
     * Render notice when WooCommerce is not installed
     */
    private function renderWooCommerceNotInstalledNotice(): void
    {
        ?>
        <div class="intucart-notice intucart-notice-warning" style="margin: 20px 0;">
            <h3>⚠️ <?php echo esc_html__('WooCommerce Not Detected', 'intufind'); ?></h3>
            <p>
                <?php echo esc_html__('The WooCommerce AI features require WooCommerce to be installed and activated. These features enable the AI assistant to help customers with:', 'intufind'); ?>
            </p>
            <ul>
                <li><strong><?php echo esc_html__('Order status and tracking inquiries', 'intufind'); ?></strong> — <?php echo esc_html__('Look up order details for logged-in customers', 'intufind'); ?></li>
                <li><strong><?php echo esc_html__('Product discovery and search', 'intufind'); ?></strong> — <?php echo esc_html__('Already active! Natural language product queries', 'intufind'); ?></li>
                <li><strong><?php echo esc_html__('Intelligent filtering', 'intufind'); ?></strong> — <?php echo esc_html__('Automatic price, color, size, and category filters', 'intufind'); ?></li>
                <li><strong><?php echo esc_html__('Inventory availability', 'intufind'); ?></strong> — <?php echo esc_html__('Real-time stock level checking (coming soon)', 'intufind'); ?></li>
                <li><strong><?php echo esc_html__('Returns and refunds', 'intufind'); ?></strong> — <?php echo esc_html__('Automated policy guidance (coming soon)', 'intufind'); ?></li>
            </ul>
            <p>
                <strong><?php echo esc_html__('To enable these features:', 'intufind'); ?></strong>
            </p>
            <ol>
                <li><?php echo sprintf(esc_html__('Install and activate the %sWooCommerce plugin%s', 'intufind'), '<a href="' . esc_url(admin_url('plugin-install.php?s=woocommerce&tab=search&type=term')) . '" target="_blank">', '</a>'); ?></li>
                <li><?php echo esc_html__('Complete the WooCommerce setup wizard', 'intufind'); ?></li>
                <li><?php echo esc_html__('Return to this page to configure AI shopping assistant features', 'intufind'); ?></li>
            </ol>
            <p style="margin-top: 12px;">
                <a href="<?php echo admin_url('plugin-install.php?s=woocommerce&tab=search&type=term'); ?>" class="button button-primary">
                    <?php echo esc_html__('Install WooCommerce', 'intufind'); ?>
                </a>
            </p>
        </div>
        
        <?php
    }
}

