<?php

/**
 * Chat Settings Controller
 * 
 * Coordinates chat settings registration and delegates rendering to specialized classes
 */

namespace Intucart\Pages\Settings;

use Intucart\Pages\Settings\Chat\MainSettings;
use Intucart\Pages\Settings\Chat\StylingSettings;
use Intucart\Pages\Settings\Chat\LiveAgentSettings;
use Intucart\Pages\Settings\Chat\WooCommerceSettings;
use Intucart\Services\Logger;
use Intucart\Services\AIClientManager;
use Intucart\Services\Constants;

class ChatSettings extends AbstractSettings
{
    private MainSettings $mainSettings;
    private StylingSettings $stylingSettings;
    private LiveAgentSettings $liveAgentSettings;
    private WooCommerceSettings $wooCommerceSettings;
    private AIClientManager $aiClientManager;

    public function __construct(Logger $logger, AIClientManager $aiClientManager)
    {
        parent::__construct($logger);
        $this->aiClientManager = $aiClientManager;
        $this->mainSettings = new MainSettings($logger);
        $this->stylingSettings = new StylingSettings($logger);
        $this->liveAgentSettings = new LiveAgentSettings($logger, $aiClientManager);
        $this->wooCommerceSettings = new WooCommerceSettings($logger, $aiClientManager);
    }

    public function registerSettings(): void
    {
        add_settings_section('intucart_chat_section', '', null, 'intucart_chat_settings');

        // Register live agent settings
        $this->liveAgentSettings->registerSettings();
        
        // Register WooCommerce/MCP settings hooks
        $this->wooCommerceSettings->registerSettings();

        register_setting('intucart_chat_settings', Constants::OPTION_ENABLE_CHAT_ASSISTANT, ['type' => 'boolean', 'default' => true, 'sanitize_callback' => 'rest_sanitize_boolean']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_WIDGET_POSITION, ['type' => 'string', 'default' => 'bottom-right', 'sanitize_callback' => [$this, 'sanitizePosition']]);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_CARD_LAYOUT, ['type' => 'string', 'default' => 'carousel', 'sanitize_callback' => [$this, 'sanitizeCardLayout']]);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_PROMPT_STYLE, ['type' => 'string', 'default' => 'footer', 'sanitize_callback' => [$this, 'sanitizePromptStyle']]);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_COLOR_SCHEME, ['type' => 'string', 'default' => 'auto', 'sanitize_callback' => [$this, 'sanitizeColorScheme']]);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_PRIMARY_COLOR, ['type' => 'string', 'default' => '#0084ff', 'sanitize_callback' => 'sanitize_hex_color']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_ASSISTANT_BUBBLE_BG, ['type' => 'string', 'default' => '#f8f9fa', 'sanitize_callback' => 'sanitize_hex_color']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_HEADER_BACKGROUND, ['type' => 'string', 'default' => '#ffffff', 'sanitize_callback' => 'sanitize_hex_color']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_ACCENT_COLOR, ['type' => 'string', 'default' => '#dadce0', 'sanitize_callback' => 'sanitize_hex_color']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_SEND_BUTTON_COLOR, ['type' => 'string', 'default' => '#0084ff', 'sanitize_callback' => 'sanitize_hex_color']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_LINK_COLOR, ['type' => 'string', 'default' => '#0084ff', 'sanitize_callback' => 'sanitize_hex_color']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_TRIGGER_BUTTON_COLOR, ['type' => 'string', 'default' => '#0084ff', 'sanitize_callback' => 'sanitize_hex_color']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_TRIGGER_BUTTON_HOVER_COLOR, ['type' => 'string', 'default' => '#0073e6', 'sanitize_callback' => 'sanitize_hex_color']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_TRIGGER_BUTTON_BORDER_COLOR, ['type' => 'string', 'default' => '#dadce0', 'sanitize_callback' => 'sanitize_hex_color']);
        register_setting('intucart_chat_settings', Constants::OPTION_TOGGLE_ICON_ID, ['type' => 'integer', 'default' => 0, 'sanitize_callback' => 'absint']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_TRIGGER_BUTTON_SHOW_BACKGROUND, ['type' => 'boolean', 'default' => true, 'sanitize_callback' => 'rest_sanitize_boolean']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_TRIGGER_ICON_SHOW_BORDER, ['type' => 'boolean', 'default' => true, 'sanitize_callback' => 'rest_sanitize_boolean']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_HEADER_ICON_ID, ['type' => 'integer', 'default' => 0, 'sanitize_callback' => 'absint']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_HEADER_AVATAR_SHOW_BACKGROUND, ['type' => 'boolean', 'default' => true, 'sanitize_callback' => 'rest_sanitize_boolean']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_HEADER_AVATAR_SHOW_BORDER, ['type' => 'boolean', 'default' => true, 'sanitize_callback' => 'rest_sanitize_boolean']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_HEADER_TEXT, ['type' => 'string', 'default' => Constants::ASSISTANT_DEFAULT_CHAT_HEADER_TEXT, 'sanitize_callback' => 'sanitize_text_field']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_HEADER_SUBTITLE, ['type' => 'string', 'default' => '', 'sanitize_callback' => 'sanitize_text_field']);
        register_setting('intucart_chat_settings', Constants::OPTION_ASSISTANT_WELCOME_MESSAGE, ['type' => 'string', 'default' => Constants::ASSISTANT_DEFAULT_WELCOME_MESSAGE, 'sanitize_callback' => 'sanitize_textarea_field']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_GREETING_SUBTEXT, ['type' => 'string', 'default' => '', 'sanitize_callback' => 'sanitize_textarea_field']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_GREETING_ICON_ID, ['type' => 'integer', 'default' => 0, 'sanitize_callback' => 'absint']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_CUSTOM_INSTRUCTIONS, ['type' => 'string', 'default' => '', 'sanitize_callback' => [$this, 'sanitizeCustomInstructions']]);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_SITE_DESCRIPTION, ['type' => 'string', 'default' => '', 'sanitize_callback' => [$this, 'sanitizeSiteDescription']]);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_WIDGET_BACKGROUND_MODE, ['type' => 'string', 'default' => 'auto', 'sanitize_callback' => [$this, 'sanitizeBackgroundMode']]);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_WIDGET_BACKGROUND_CUSTOM, ['type' => 'string', 'default' => '#ffffff', 'sanitize_callback' => 'sanitize_hex_color']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_BORDER_RADIUS, ['type' => 'integer', 'default' => 16, 'sanitize_callback' => [$this, 'sanitizeBorderRadius']]);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_CUSTOM_CSS, ['type' => 'string', 'default' => '', 'sanitize_callback' => 'wp_strip_all_tags']);
        
        // High-priority branding settings
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_FONT_FAMILY, ['type' => 'string', 'default' => 'system', 'sanitize_callback' => 'sanitize_text_field']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_CUSTOM_FONT_FAMILY, ['type' => 'string', 'default' => '', 'sanitize_callback' => 'sanitize_text_field']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_MESSAGE_FONT_SIZE, ['type' => 'integer', 'default' => 14, 'sanitize_callback' => 'absint']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_MESSAGE_LINE_HEIGHT, ['type' => 'string', 'default' => '1.5', 'sanitize_callback' => [$this, 'sanitizeLineHeight']]);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_MESSAGE_FONT_FAMILY, ['type' => 'string', 'default' => 'inherit', 'sanitize_callback' => 'sanitize_text_field']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_MESSAGE_CUSTOM_FONT_FAMILY, ['type' => 'string', 'default' => '', 'sanitize_callback' => 'sanitize_text_field']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_CARD_FONT_SIZE, ['type' => 'integer', 'default' => 14, 'sanitize_callback' => 'absint']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_CARD_LINE_HEIGHT, ['type' => 'string', 'default' => '1.4', 'sanitize_callback' => [$this, 'sanitizeLineHeight']]);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_WIDGET_SHADOW, ['type' => 'string', 'default' => 'medium', 'sanitize_callback' => [$this, 'sanitizeShadow']]);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_INPUT_BACKGROUND, ['type' => 'string', 'default' => '#ffffff', 'sanitize_callback' => 'sanitize_hex_color']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_INPUT_BORDER, ['type' => 'string', 'default' => '#dadce0', 'sanitize_callback' => 'sanitize_hex_color']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_WIDGET_MAX_WIDTH, ['type' => 'string', 'default' => 800, 'sanitize_callback' => [$this, 'sanitizeMaxWidth']]);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_WIDGET_MAX_HEIGHT, ['type' => 'string', 'default' => 900, 'sanitize_callback' => [$this, 'sanitizeMaxHeight']]);
        
        // Dark Mode Color Variants
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_DARK_PRIMARY_COLOR, ['type' => 'string', 'default' => '#0084ff', 'sanitize_callback' => 'sanitize_hex_color']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_DARK_ASSISTANT_BUBBLE_BG, ['type' => 'string', 'default' => '#2d2d2d', 'sanitize_callback' => 'sanitize_hex_color']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_DARK_HEADER_BACKGROUND, ['type' => 'string', 'default' => '#1f1f1f', 'sanitize_callback' => 'sanitize_hex_color']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_DARK_ACCENT_COLOR, ['type' => 'string', 'default' => '#404040', 'sanitize_callback' => 'sanitize_hex_color']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_DARK_SEND_BUTTON_COLOR, ['type' => 'string', 'default' => '#0084ff', 'sanitize_callback' => 'sanitize_hex_color']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_DARK_LINK_COLOR, ['type' => 'string', 'default' => '#6495ed', 'sanitize_callback' => 'sanitize_hex_color']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_DARK_TRIGGER_BUTTON_COLOR, ['type' => 'string', 'default' => '#0084ff', 'sanitize_callback' => 'sanitize_hex_color']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_DARK_TRIGGER_BUTTON_HOVER_COLOR, ['type' => 'string', 'default' => '#0073e6', 'sanitize_callback' => 'sanitize_hex_color']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_DARK_TRIGGER_BUTTON_BORDER_COLOR, ['type' => 'string', 'default' => '#404040', 'sanitize_callback' => 'sanitize_hex_color']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_DARK_INPUT_BACKGROUND, ['type' => 'string', 'default' => '#2d2d2d', 'sanitize_callback' => 'sanitize_hex_color']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_DARK_INPUT_BORDER, ['type' => 'string', 'default' => '#404040', 'sanitize_callback' => 'sanitize_hex_color']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_DARK_WIDGET_BACKGROUND_CUSTOM, ['type' => 'string', 'default' => '#1f1f1f', 'sanitize_callback' => 'sanitize_hex_color']);
        
        // Message Actions & Timestamps
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_SHOW_MESSAGE_ACTIONS, ['type' => 'boolean', 'default' => false, 'sanitize_callback' => 'rest_sanitize_boolean']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_SHOW_USER_COPY_BUTTON, ['type' => 'boolean', 'default' => false, 'sanitize_callback' => 'rest_sanitize_boolean']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_SHOW_MESSAGE_TIMESTAMPS, ['type' => 'boolean', 'default' => false, 'sanitize_callback' => 'rest_sanitize_boolean']);
        
        // Post Card Display Options
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_SHOW_POST_IMAGE, ['type' => 'boolean', 'default' => false, 'sanitize_callback' => 'rest_sanitize_boolean']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_SHOW_POST_AUTHOR, ['type' => 'boolean', 'default' => true, 'sanitize_callback' => 'rest_sanitize_boolean']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_SHOW_POST_DATE, ['type' => 'boolean', 'default' => true, 'sanitize_callback' => 'rest_sanitize_boolean']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_SHOW_POST_CATEGORY, ['type' => 'boolean', 'default' => true, 'sanitize_callback' => 'rest_sanitize_boolean']);
        
        // MCP (Model Context Protocol) Feature Toggles
        register_setting('intucart_chat_settings', Constants::OPTION_ENABLE_MCP_ORDERS, ['type' => 'boolean', 'default' => true, 'sanitize_callback' => 'rest_sanitize_boolean']);
        
        // Privacy & Analytics
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_ANALYTICS_ENABLED, ['type' => 'boolean', 'default' => true, 'sanitize_callback' => 'rest_sanitize_boolean']);
        register_setting('intucart_chat_settings', Constants::OPTION_CHAT_ANALYTICS_CONSENT_REQUIRED, ['type' => 'boolean', 'default' => false, 'sanitize_callback' => 'rest_sanitize_boolean']);
    }

    public function sanitizePosition($value): string
    {
        $valid = ['bottom-right', 'bottom-left', 'top-right', 'top-left'];
        return in_array($value, $valid, true) ? $value : 'bottom-right';
    }

    public function sanitizeCardLayout($value): string
    {
        $valid = ['carousel', 'grid'];
        return in_array($value, $valid, true) ? $value : 'carousel';
    }

    public function sanitizePromptStyle($value): string
    {
        $valid = ['footer', 'chips', 'both'];
        return in_array($value, $valid, true) ? $value : 'footer';
    }

    public function sanitizeColorScheme($value): string
    {
        $valid = ['auto', 'light', 'dark'];
        return in_array($value, $valid, true) ? $value : 'auto';
    }

    public function sanitizeBackgroundMode($value): string
    {
        $valid = ['auto', 'custom'];
        return in_array($value, $valid, true) ? $value : 'auto';
    }

    public function sanitizeBorderRadius($value): int
    {
        $value = (int) $value;
        return max(0, min(50, $value));
    }

    public function sanitizeShadow($value): string
    {
        $valid = ['none', 'light', 'medium', 'heavy'];
        return in_array($value, $valid, true) ? $value : 'medium';
    }

    public function sanitizeMaxWidth($value): int
    {
        $value = (int) $value;
        return max(300, min(800, $value));
    }

    public function sanitizeMaxHeight($value): int
    {
        $value = (int) $value;
        return max(400, min(1000, $value));
    }

    public function sanitizeLineHeight($value): string
    {
        $value = (float) $value;
        // Clamp between 1.0 and 2.5 (reasonable range for line height)
        $value = max(1.0, min(2.5, $value));
        return (string) $value;
    }

    public function sanitizeCustomInstructions($value): string
    {
        // Max 5000 characters for custom instructions
        $value = wp_kses_post($value);
        if (strlen($value) > 5000) {
            $value = substr($value, 0, 5000);
            add_settings_error(
                Constants::OPTION_CHAT_CUSTOM_INSTRUCTIONS,
                'custom_instructions_too_long',
                'Custom Instructions was truncated to 5,000 characters maximum.',
                'warning'
            );
        }
        return $value;
    }

    public function sanitizeSiteDescription($value): string
    {
        // Max 500 characters for site description (1-2 sentences)
        $value = sanitize_textarea_field($value);
        if (strlen($value) > 500) {
            $value = substr($value, 0, 500);
            add_settings_error(
                Constants::OPTION_CHAT_SITE_DESCRIPTION,
                'site_description_too_long',
                'Site Description was truncated to 500 characters maximum.',
                'warning'
            );
        }
        return $value;
    }

    public function renderChatSettings(): void
    {
        $this->mainSettings->render();
    }

    public function renderChatStylingSettings(): void
    {
        $this->stylingSettings->render();
    }

    /**
     * Render live agent settings
     */
    public function renderLiveAgentSettings(): void
    {
        $this->liveAgentSettings->render();
    }

    /**
     * Render WooCommerce settings
     */
    public function renderWooCommerceSettings(): void
    {
        $this->wooCommerceSettings->render();
    }

    /**
     * Reset all styling settings to their theme defaults
     * Called via AJAX from the styling settings page
     *
     * @return void
     */
    public function resetStylingSettings(): void
    {
        // Check nonce
        check_ajax_referer('intucart_admin_nonce', 'nonce');

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'intufind')]);
            return;
        }

        // Define all styling settings and their defaults (from theme.css)
        $defaults = [
            // Light Mode Colors
            Constants::OPTION_CHAT_PRIMARY_COLOR => '#0084ff',
            Constants::OPTION_CHAT_ASSISTANT_BUBBLE_BG => '#f8f9fa',
            Constants::OPTION_CHAT_HEADER_BACKGROUND => '#ffffff',
            Constants::OPTION_CHAT_ACCENT_COLOR => '#dadce0',
            Constants::OPTION_CHAT_SEND_BUTTON_COLOR => '#0084ff',
            Constants::OPTION_CHAT_LINK_COLOR => '#0084ff',
            Constants::OPTION_CHAT_TRIGGER_BUTTON_COLOR => '#0084ff',
            Constants::OPTION_CHAT_TRIGGER_BUTTON_HOVER_COLOR => '#0073e6',
            Constants::OPTION_CHAT_TRIGGER_BUTTON_BORDER_COLOR => '#dadce0',
            Constants::OPTION_CHAT_INPUT_BACKGROUND => '#ffffff',
            Constants::OPTION_CHAT_INPUT_BORDER => '#dadce0',
            Constants::OPTION_CHAT_WIDGET_BACKGROUND_CUSTOM => '#ffffff',
            // Dark Mode Colors
            Constants::OPTION_CHAT_DARK_PRIMARY_COLOR => '#0084ff',
            Constants::OPTION_CHAT_DARK_ASSISTANT_BUBBLE_BG => '#2d2d2d',
            Constants::OPTION_CHAT_DARK_HEADER_BACKGROUND => '#1f1f1f',
            Constants::OPTION_CHAT_DARK_ACCENT_COLOR => '#404040',
            Constants::OPTION_CHAT_DARK_SEND_BUTTON_COLOR => '#0084ff',
            Constants::OPTION_CHAT_DARK_LINK_COLOR => '#6495ed',
            Constants::OPTION_CHAT_DARK_TRIGGER_BUTTON_COLOR => '#0084ff',
            Constants::OPTION_CHAT_DARK_TRIGGER_BUTTON_HOVER_COLOR => '#0073e6',
            Constants::OPTION_CHAT_DARK_TRIGGER_BUTTON_BORDER_COLOR => '#404040',
            Constants::OPTION_CHAT_DARK_INPUT_BACKGROUND => '#2d2d2d',
            Constants::OPTION_CHAT_DARK_INPUT_BORDER => '#404040',
            Constants::OPTION_CHAT_DARK_WIDGET_BACKGROUND_CUSTOM => '#1f1f1f',
            // Widget Settings
            Constants::OPTION_CHAT_BORDER_RADIUS => 16,
            Constants::OPTION_CHAT_WIDGET_BACKGROUND_MODE => 'auto',
            Constants::OPTION_CHAT_COLOR_SCHEME => 'auto',
            Constants::OPTION_CHAT_CUSTOM_CSS => '',
            // Reset avatars to default (remove custom uploads)
            Constants::OPTION_TOGGLE_ICON_ID => 0,
            Constants::OPTION_CHAT_HEADER_ICON_ID => 0,
            // Avatar/Icon Display Settings
            Constants::OPTION_CHAT_HEADER_AVATAR_SHOW_BACKGROUND => true,
            Constants::OPTION_CHAT_HEADER_AVATAR_SHOW_BORDER => true,
            Constants::OPTION_CHAT_TRIGGER_BUTTON_SHOW_BACKGROUND => true,
            Constants::OPTION_CHAT_TRIGGER_ICON_SHOW_BORDER => true,
            // High-priority branding settings
            Constants::OPTION_CHAT_FONT_FAMILY => 'system',
            Constants::OPTION_CHAT_CUSTOM_FONT_FAMILY => '',
            Constants::OPTION_CHAT_MESSAGE_FONT_SIZE => 14,
            Constants::OPTION_CHAT_MESSAGE_LINE_HEIGHT => '1.5',
            Constants::OPTION_CHAT_MESSAGE_FONT_FAMILY => 'inherit',
            Constants::OPTION_CHAT_MESSAGE_CUSTOM_FONT_FAMILY => '',
            Constants::OPTION_CHAT_CARD_FONT_SIZE => 14,
            Constants::OPTION_CHAT_CARD_LINE_HEIGHT => '1.4',
            Constants::OPTION_CHAT_WIDGET_SHADOW => 'medium',
            Constants::OPTION_CHAT_WIDGET_MAX_WIDTH => 400,
            Constants::OPTION_CHAT_WIDGET_MAX_HEIGHT => 700,
            // Message Actions (disabled by default)
            Constants::OPTION_CHAT_SHOW_MESSAGE_ACTIONS => false,
            Constants::OPTION_CHAT_SHOW_USER_COPY_BUTTON => false,
            Constants::OPTION_CHAT_SHOW_MESSAGE_TIMESTAMPS => false,
            // MCP Features (enabled by default if dependencies available)
            Constants::OPTION_ENABLE_MCP_ORDERS => true,
            // Privacy & Analytics (enabled by default, consent not required by default)
            Constants::OPTION_CHAT_ANALYTICS_ENABLED => true,
            Constants::OPTION_CHAT_ANALYTICS_CONSENT_REQUIRED => false,
        ];

        // Reset all settings
        $reset_count = 0;
        foreach ($defaults as $option_name => $default_value) {
            update_option($option_name, $default_value);
            $reset_count++;
        }

        wp_send_json_success([
            'message' => sprintf('Successfully reset %d styling settings to defaults!', $reset_count),
            'reset_count' => $reset_count
        ]);
    }

    /**
     * Generate AI-powered color scheme
     * Called via AJAX from the styling settings page
     *
     * @return void
     */
    public function generateColorScheme(): void
    {
        // Check nonce
        check_ajax_referer('intucart_admin_nonce', 'nonce');

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'intufind')]);
            return;
        }

        // Get and validate input colors
        $primary_color = sanitize_text_field($_POST['primaryColor'] ?? '');
        $secondary_color = !empty($_POST['secondaryColor']) ? sanitize_text_field($_POST['secondaryColor']) : null;

        // Validate hex color format
        if (!preg_match('/^#[0-9A-Fa-f]{6}$/', $primary_color)) {
            wp_send_json_error(['message' => __('Invalid primary color format. Please use hex format (e.g., #0084ff).', 'intufind')]);
            return;
        }

        if ($secondary_color !== null && !preg_match('/^#[0-9A-Fa-f]{6}$/', $secondary_color)) {
            wp_send_json_error(['message' => __('Invalid secondary color format. Please use hex format (e.g., #0084ff).', 'intufind')]);
            return;
        }

        try {
            // Call AI service to generate color scheme
            $result = $this->aiClientManager->generateColorScheme($primary_color, $secondary_color);

            if (!$result['success']) {
                wp_send_json_error([
                    'message' => $result['error'] ?? 'Failed to generate color scheme. Please try again.'
                ]);
                return;
            }

            wp_send_json_success([
                'message' => 'Color scheme generated successfully!',
                'colors' => $result['data']
            ]);
        } catch (\Exception $e) {
            wp_send_json_error([
                'message' => 'Error generating color scheme: ' . $e->getMessage()
            ]);
        }
    }
}
