<?php

namespace Intucart\Pages\Settings;

use Intucart\Services\Logger;
use Intucart\Services\Licensing\License;
use Intucart\Services\Constants;
use Intucart\Components\Field;

class EcommerceSettings extends AbstractSettings
{
    private License $license;
    private bool $recommendationsAvailable;

    public function __construct(Logger $logger, License $license, bool $recommendationsAvailable = false)
    {
        parent::__construct($logger);
        $this->license = $license;
        $this->recommendationsAvailable = $recommendationsAvailable;
    }

    public function registerSettings(): void
    {
        // E-commerce platform section removed - always using WooCommerce



        // WooCommerce section
        if (class_exists('WooCommerce')) {
            add_settings_section(
                'intucart_woocommerce_section',
                __('WooCommerce Settings', 'intufind'),
                [$this, 'renderWooCommerceSection'],
                Constants::SETTINGS_GROUP_WOOCOMMERCE
            );

            // Semantic search settings moved to SearchSettings

            // Add all recommendation settings to WooCommerce section (Pro only)
            if ($this->recommendationsAvailable) {
                // Register Time Decay Half-Life setting
                register_setting(
                    Constants::SETTINGS_GROUP_WOOCOMMERCE,
                    Constants::RECOMMENDATION_HALF_LIFE_OPTION,
                    [
                        'type' => 'number',
                        'default' => 30,
                        'sanitize_callback' => [$this, 'sanitizeHalfLife']
                    ]
                );

                // Register recommendation context settings
                register_setting(
                    Constants::SETTINGS_GROUP_WOOCOMMERCE,
                    Constants::ENABLE_RELATED_PRODUCTS_OPTION,
                    [
                        'type' => 'boolean',
                        'default' => true
                    ]
                );

                register_setting(
                    Constants::SETTINGS_GROUP_WOOCOMMERCE,
                    Constants::ENABLE_UPSELLS_OPTION,
                    [
                        'type' => 'boolean',
                        'default' => true
                    ]
                );

                register_setting(
                    Constants::SETTINGS_GROUP_WOOCOMMERCE,
                    Constants::ENABLE_CROSS_SELLS_OPTION,
                    [
                        'type' => 'boolean',
                        'default' => true
                    ]
                );

                // Add settings fields
                add_settings_field(
                    'intucart_recommendation_contexts',
                    __('Recommendation Contexts', 'intufind'),
                    [$this, 'renderRecommendationContextsField'],
                    Constants::SETTINGS_GROUP_WOOCOMMERCE,
                    'intucart_woocommerce_section'
                );

                // Time Decay Half-Life setting is registered but hidden from UI
                // TODO: Move to advanced settings section when ready
                // add_settings_field(
                //     Constants::RECOMMENDATION_HALF_LIFE_OPTION,
                //     'Time Decay Half-Life (days)',
                //     [$this, 'renderHalfLifeField'],
                //     'intucart_plugin',
                //     'intucart_woocommerce_section'
                // );
            }
        }
    }

    public function renderSection(): void
    {
        // This method is required by the parent class but not used in this implementation
    }

    public function renderPlatformField(): void
    {
        // Platform section and field methods removed
    }

    public function renderWooCommerceSection(): void
    {
        echo '<div class="intucart-settings-section-description">';
        echo '<p>' . esc_html__('Configure WooCommerce-specific settings for AI-powered recommendations.', 'intufind') . '</p>';
        echo '</div>';
    }

    public function renderRecommendationContextsField(): void
    {
        $contexts = [
            'related_products' => [
                'label' => __('Related Products', 'intufind'),
                'description' => __('Replace related products section on single product pages', 'intufind')
            ],
            'upsells' => [
                'label' => __('Upsells', 'intufind'),
                'description' => __('Enhance "You may also like..." products on single product pages. Manually defined upsells (in Product Data → Linked Products) will be shown first, followed by AI recommendations in any remaining slots.', 'intufind')
            ],
            'cross_sells' => [
                'label' => __('Cross-sells', 'intufind'),
                'description' => __('Replace "You may be interested in..." products in cart', 'intufind')
            ]
        ];

        echo '<div class="intucart-checkbox-group">';
        foreach ($contexts as $context => $config) {
            $option_name = "intucart_enable_{$context}";
            $value = get_option($option_name, true);

            echo '<div class="intucart-checkbox-item">';
            Field::checkbox($option_name, $config['label'], $value, [
                'description' => $config['description']
            ]);
            echo '</div>';
        }
        echo '</div>';
        echo '<div class="intucart-hint">';
        echo '<div class="intucart-hint-section">' . esc_html__('Select which WooCommerce recommendation types to enhance with AI. Unselected types will use WooCommerce\'s default behavior.', 'intufind') . '</div>';
        echo '</div>';
    }

    public function renderHalfLifeField(): void
    {
        $value = get_option(Constants::RECOMMENDATION_HALF_LIFE_OPTION, 30);

        Field::number('intucart_recommendation_half_life', '', $value, [
            'suffix' => __('days', 'intufind'),
            'attributes' => [
                'min' => '1',
                'max' => '365',
                'step' => '1'
            ],
            'hint' => [[
                'content' => sprintf(
                    __('The half-life parameter controls how quickly the influence of past purchases decays over time when calculating product recommendations. %s', 'intufind'),
                    '<a href="' . esc_url(Constants::WEBSITE_URL) . '/docs/configure-recommendation-settings/" target="_blank" class="intucart-help-link">' . esc_html__('Learn more about recommendation settings', 'intufind') . '</a>'
                )
            ]]
        ]);
    }

    /**
     * Sanitize half-life value
     *
     * @param mixed $value The value to sanitize
     * @return int
     */
    public function sanitizeHalfLife($value): int
    {
        if ($value === null) {
            return 30; // Default to 30 days
        }

        $value = absint($value);
        // Ensure value is between 1 and 365 days (1 day to 1 year)
        return max(1, min(365, $value));
    }
}
