<?php

namespace Intucart\Pages\Settings;

use Intucart\Services\Logger;
use Intucart\Services\Constants;
use Intucart\Components\Field;

class SearchSettings extends AbstractSettings
{
    public function __construct(Logger $logger)
    {
        parent::__construct($logger);
    }

    public function registerSettings(): void
    {
        // Search settings section
        add_settings_section(
            'intucart_search_section',
            __('Search Settings', 'intufind'),
            [$this, 'renderSearchSection'],
            Constants::SETTINGS_GROUP_SEARCH
        );

        // Semantic search settings - available in all tiers
        register_setting(
            Constants::SETTINGS_GROUP_SEARCH,
            Constants::ENABLE_SEMANTIC_SEARCH_OPTION,
            [
                'type' => 'boolean',
                'default' => true
            ]
        );

        add_settings_field(
            Constants::ENABLE_SEMANTIC_SEARCH_OPTION,
            __('Enable Semantic Search', 'intufind'),
            [$this, 'renderEnableSemanticSearchField'],
            Constants::SETTINGS_GROUP_SEARCH,
            'intucart_search_section'
        );

        // Mixed content search setting
        register_setting(
            Constants::SETTINGS_GROUP_SEARCH,
            Constants::ENABLE_MIXED_CONTENT_SEARCH_OPTION,
            [
                'type' => 'boolean',
                'default' => true
            ]
        );

        add_settings_field(
            Constants::ENABLE_MIXED_CONTENT_SEARCH_OPTION,
            __('Enable Mixed Content Search', 'intufind'),
            [$this, 'renderMixedContentSearchField'],
            Constants::SETTINGS_GROUP_SEARCH,
            'intucart_search_section'
        );

        // Enhanced AI search setting
        register_setting(
            Constants::SETTINGS_GROUP_SEARCH,
            Constants::ENABLE_ENHANCED_AI_SEARCH_OPTION,
            [
                'type' => 'boolean',
                'default' => false
            ]
        );

        add_settings_field(
            Constants::ENABLE_ENHANCED_AI_SEARCH_OPTION,
            __('Enable Enhanced AI Search', 'intufind'),
            [$this, 'renderEnhancedAISearchField'],
            Constants::SETTINGS_GROUP_SEARCH,
            'intucart_search_section'
        );

        // Search cache duration setting
        register_setting(
            Constants::SETTINGS_GROUP_SEARCH,
            Constants::SEARCH_CACHE_DURATION_OPTION,
            [
                'type' => 'number',
                'default' => 3600,
                'sanitize_callback' => [$this, 'sanitizeCacheDuration']
            ]
        );

        add_settings_field(
            Constants::SEARCH_CACHE_DURATION_OPTION,
            __('Search Cache Duration', 'intufind'),
            [$this, 'renderCacheDurationField'],
            Constants::SETTINGS_GROUP_SEARCH,
            'intucart_search_section'
        );
    }

    public function renderSection(): void
    {
        // This method is required by the parent class but not used in this implementation
    }

    public function renderSearchSection(): void
    {
        echo '<div class="intucart-settings-section-description">';
        echo '<p>' . esc_html__('Configure AI-powered search behavior and performance settings for your website.', 'intufind') . '</p>';
        echo '</div>';
    }

    public function renderEnableSemanticSearchField(): void
    {
        $option_name = Constants::ENABLE_SEMANTIC_SEARCH_OPTION;
        $value = get_option($option_name, true);
        
        Field::checkbox($option_name, __('Use AI-powered search that understands context and intent', 'intufind'), $value, [
            'description' => __('Enable AI-powered semantic search that understands context and intent, not just keywords.', 'intufind')
        ]);
    }

    // Fallback behavior removed - always uses silent fallback with intelligent admin notices

    public function renderMixedContentSearchField(): void
    {
        $option_name = Constants::ENABLE_MIXED_CONTENT_SEARCH_OPTION;
        $value = get_option($option_name, true);
        
        Field::checkbox($option_name, __('Include products alongside posts and pages in general searches', 'intufind'), $value, [
            'description' => __('Allow general site searches to include products alongside posts and pages. When disabled, general searches will only return posts and pages (product-specific searches still work normally).', 'intufind')
        ]);
    }

    public function renderEnhancedAISearchField(): void
    {
        $option_name = Constants::ENABLE_ENHANCED_AI_SEARCH_OPTION;
        $value = get_option($option_name, false);
        
        Field::checkbox($option_name, __('Use advanced AI agents for complex, multi-criteria queries', 'intufind'), $value, [
            'description' => __('Intufind search is already very powerful for most queries! This enhancement adds AI agents for complex queries that need advanced filtering and understanding. Costs 3x more credits only when AI agents are triggered by complex searches.', 'intufind'),
            'hint' => [
                [
                    'title' => __('Think of it like this:', 'intufind'),
                    'content' => __('Standard search is like having a smart librarian who understands what you mean. Enhanced AI search adds a team of specialists who can break down complex requests, apply multiple filters, and reason through comparisons - but you only pay for the specialists when you need them.', 'intufind')
                ],
                [
                    'title' => __('When Enhanced AI Search helps:', 'intufind'),
                    'content' => '<ul style="margin: 8px 0; padding-left: 20px;"><li><strong>' . esc_html__('Complex filters:', 'intufind') . '</strong> "' . esc_html__('Show me red Nike shoes under $100 in stock', 'intufind') . '"</li><li><strong>' . esc_html__('Comparative queries:', 'intufind') . '</strong> "' . esc_html__('Compare wireless headphones vs wired ones', 'intufind') . '"</li><li><strong>' . esc_html__('Multi-criteria:', 'intufind') . '</strong> "' . esc_html__('Best rated laptops for gaming under $1500', 'intufind') . '"</li></ul><div class="intucart-hint-section"><strong>' . esc_html__('Note:', 'intufind') . '</strong> ' . esc_html__('Simple searches like "laptop" or "red shoes" work perfectly with standard Intucart search and don\'t trigger the 3x cost.', 'intufind') . '</div>'
                ]
            ]
        ]);
    }

    public function renderCacheDurationField(): void
    {
        $option_name = Constants::SEARCH_CACHE_DURATION_OPTION;
        $value = get_option($option_name, 3600);
        
        Field::number($option_name, '', $value, [
            'suffix' => __('seconds', 'intufind'),
            'attributes' => [
                'min' => '60',
                'max' => '86400',
                'step' => '60'
            ],
            'hint' => [[
                'content' => __('How long to cache search results (60 seconds to 24 hours). Longer durations improve performance but may show outdated results.', 'intufind')
            ]]
        ]);
    }

    /**
     * Sanitize cache duration setting
     */
    public function sanitizeCacheDuration($value): int
    {
        if ($value === null) {
            return 3600;
        }

        $value = (int) $value;
        return max(60, min($value, 86400)); // Between 1 minute and 24 hours
    }
}
