<?php

declare(strict_types=1);

namespace Intucart\Pages\Settings;

use Intucart\Services\Logger;
use Intucart\Services\Constants;
use Intucart\Components\Field;
use Intucart\Components\ColorPicker;
use Intucart\Components\ImageUploader;

/**
 * Settings page for the Search Widget configuration.
 */
class SearchWidgetSettings extends AbstractSettings
{
    public function __construct(Logger $logger)
    {
        parent::__construct($logger);
    }

    public function registerSettings(): void
    {
        // Register settings group
        $this->registerWidgetSettings();
    }

    /**
     * Register all widget settings.
     */
    private function registerWidgetSettings(): void
    {
        // === Enable Search Widget ===
        register_setting(
            Constants::SETTINGS_GROUP_SEARCH_WIDGET,
            Constants::OPTION_SEARCH_WIDGET_ENABLED,
            [
                'type' => 'string',
                'default' => '0',
                'sanitize_callback' => 'sanitize_text_field'
            ]
        );

        // === Search Scope Settings ===
        register_setting(
            Constants::SETTINGS_GROUP_SEARCH_WIDGET,
            Constants::OPTION_SEARCH_WIDGET_PRODUCTS,
            [
                'type' => 'string',
                'default' => '1',
                'sanitize_callback' => 'sanitize_text_field'
            ]
        );

        register_setting(
            Constants::SETTINGS_GROUP_SEARCH_WIDGET,
            Constants::OPTION_SEARCH_WIDGET_POSTS,
            [
                'type' => 'string',
                'default' => '1',
                'sanitize_callback' => 'sanitize_text_field'
            ]
        );

        // === Trigger Settings ===
        register_setting(
            Constants::SETTINGS_GROUP_SEARCH_WIDGET,
            Constants::OPTION_SEARCH_WIDGET_KEYBOARD_SHORTCUT,
            [
                'type' => 'string',
                'default' => Constants::DEFAULT_SEARCH_WIDGET_KEYBOARD_SHORTCUT,
                'sanitize_callback' => 'sanitize_text_field'
            ]
        );

        register_setting(
            Constants::SETTINGS_GROUP_SEARCH_WIDGET,
            Constants::OPTION_SEARCH_WIDGET_SHOW_TRIGGER,
            [
                'type' => 'string',
                'default' => '1',
                'sanitize_callback' => 'sanitize_text_field'
            ]
        );

        register_setting(
            Constants::SETTINGS_GROUP_SEARCH_WIDGET,
            Constants::OPTION_SEARCH_WIDGET_TRIGGER_POSITION,
            [
                'type' => 'string',
                'default' => Constants::DEFAULT_SEARCH_WIDGET_TRIGGER_POSITION,
                'sanitize_callback' => 'sanitize_text_field'
            ]
        );

        // === Theme Settings ===
        register_setting(
            Constants::SETTINGS_GROUP_SEARCH_WIDGET,
            Constants::OPTION_SEARCH_WIDGET_PRIMARY_COLOR,
            [
                'type' => 'string',
                'default' => '#0084ff',
                'sanitize_callback' => 'sanitize_hex_color'
            ]
        );

        register_setting(
            Constants::SETTINGS_GROUP_SEARCH_WIDGET,
            Constants::OPTION_SEARCH_WIDGET_DARK_PRIMARY_COLOR,
            [
                'type' => 'string',
                'default' => '#0084ff',
                'sanitize_callback' => 'sanitize_hex_color'
            ]
        );

        register_setting(
            Constants::SETTINGS_GROUP_SEARCH_WIDGET,
            Constants::OPTION_SEARCH_WIDGET_COLOR_SCHEME,
            [
                'type' => 'string',
                'default' => 'auto',
                'sanitize_callback' => 'sanitize_text_field'
            ]
        );

        // === Trigger Button Styling ===
        register_setting(
            Constants::SETTINGS_GROUP_SEARCH_WIDGET,
            Constants::OPTION_SEARCH_WIDGET_TRIGGER_BG,
            [
                'type' => 'string',
                'default' => Constants::DEFAULT_SEARCH_WIDGET_TRIGGER_BG,
                'sanitize_callback' => 'sanitize_text_field'
            ]
        );

        register_setting(
            Constants::SETTINGS_GROUP_SEARCH_WIDGET,
            Constants::OPTION_SEARCH_WIDGET_TRIGGER_TEXT,
            [
                'type' => 'string',
                'default' => Constants::DEFAULT_SEARCH_WIDGET_TRIGGER_TEXT,
                'sanitize_callback' => 'sanitize_hex_color'
            ]
        );

        register_setting(
            Constants::SETTINGS_GROUP_SEARCH_WIDGET,
            Constants::OPTION_SEARCH_WIDGET_TRIGGER_BORDER,
            [
                'type' => 'string',
                'default' => Constants::DEFAULT_SEARCH_WIDGET_TRIGGER_BORDER,
                'sanitize_callback' => 'sanitize_text_field'
            ]
        );

        register_setting(
            Constants::SETTINGS_GROUP_SEARCH_WIDGET,
            Constants::OPTION_SEARCH_WIDGET_DARK_TRIGGER_BG,
            [
                'type' => 'string',
                'default' => Constants::DEFAULT_SEARCH_WIDGET_DARK_TRIGGER_BG,
                'sanitize_callback' => 'sanitize_text_field'
            ]
        );

        register_setting(
            Constants::SETTINGS_GROUP_SEARCH_WIDGET,
            Constants::OPTION_SEARCH_WIDGET_DARK_TRIGGER_TEXT,
            [
                'type' => 'string',
                'default' => Constants::DEFAULT_SEARCH_WIDGET_DARK_TRIGGER_TEXT,
                'sanitize_callback' => 'sanitize_hex_color'
            ]
        );

        register_setting(
            Constants::SETTINGS_GROUP_SEARCH_WIDGET,
            Constants::OPTION_SEARCH_WIDGET_DARK_TRIGGER_BORDER,
            [
                'type' => 'string',
                'default' => Constants::DEFAULT_SEARCH_WIDGET_DARK_TRIGGER_BORDER,
                'sanitize_callback' => 'sanitize_text_field'
            ]
        );

        // === Trigger Button Icon ===
        register_setting(
            Constants::SETTINGS_GROUP_SEARCH_WIDGET,
            Constants::OPTION_SEARCH_WIDGET_TRIGGER_ICON_ID,
            [
                'type' => 'integer',
                'default' => 0,
                'sanitize_callback' => 'absint'
            ]
        );
    }

    public function renderSection(): void
    {
        // Required by parent class
    }

    /**
     * Render the full widget settings panel.
     */
    public function renderWidgetSettings(): void
    {
        $this->renderEnableWidget();
        $this->renderSearchScope();
        $this->renderTriggerSettings();
        $this->renderThemeSettings();
        $this->renderShortcodeHelp();
    }

    /**
     * Render enable widget toggle.
     */
    private function renderEnableWidget(): void
    {
        echo '<div class="intucart-settings-section">';
        echo '<h3>' . esc_html__('Enable Search Widget', 'intufind') . '</h3>';
        
        $enabled = get_option(Constants::OPTION_SEARCH_WIDGET_ENABLED, '0');
        
        Field::checkbox(
            Constants::OPTION_SEARCH_WIDGET_ENABLED,
            __('Enable AI-powered search widget on your site', 'intufind'),
            $enabled === '1',
            [
                'description' => __('Adds an instant search overlay that users can access via keyboard shortcut or trigger button. Searches your indexed products and content.', 'intufind')
            ]
        );
        
        echo '</div>';
    }

    /**
     * Render search scope settings.
     */
    private function renderSearchScope(): void
    {
        echo '<div class="intucart-settings-section">';
        echo '<h3>' . esc_html__('Search Scope', 'intufind') . '</h3>';
        echo '<p class="description">' . esc_html__('Choose what content types to include in search results.', 'intufind') . '</p>';

        $search_products = get_option(Constants::OPTION_SEARCH_WIDGET_PRODUCTS, '1');
        $search_posts = get_option(Constants::OPTION_SEARCH_WIDGET_POSTS, '1');

        echo '<table class="form-table">';
        
        // Products toggle
        echo '<tr>';
        echo '<th scope="row">' . esc_html__('Search Products', 'intufind') . '</th>';
        echo '<td>';
        Field::checkbox(
            Constants::OPTION_SEARCH_WIDGET_PRODUCTS,
            __('Include products in search results', 'intufind'),
            $search_products === '1',
            [
                'description' => class_exists('WooCommerce')
                    ? __('Search your WooCommerce products with faceted filtering.', 'intufind')
                    : __('WooCommerce not detected. This option will have no effect.', 'intufind')
            ]
        );
        echo '</td>';
        echo '</tr>';

        // Posts toggle
        echo '<tr>';
        echo '<th scope="row">' . esc_html__('Search Posts', 'intufind') . '</th>';
        echo '<td>';
        Field::checkbox(
            Constants::OPTION_SEARCH_WIDGET_POSTS,
            __('Include posts and pages in search results', 'intufind'),
            $search_posts === '1',
            [
                'description' => __('Search your blog posts, pages, and other indexed content.', 'intufind')
            ]
        );
        echo '</td>';
        echo '</tr>';

        echo '</table>';
        echo '</div>';
    }

    /**
     * Render trigger settings.
     */
    private function renderTriggerSettings(): void
    {
        echo '<div class="intucart-settings-section">';
        echo '<h3>' . esc_html__('Trigger Settings', 'intufind') . '</h3>';
        echo '<p class="description">' . esc_html__('Configure how users open the search widget.', 'intufind') . '</p>';

        $shortcut = get_option(
            Constants::OPTION_SEARCH_WIDGET_KEYBOARD_SHORTCUT,
            Constants::DEFAULT_SEARCH_WIDGET_KEYBOARD_SHORTCUT
        );
        $show_trigger = get_option(Constants::OPTION_SEARCH_WIDGET_SHOW_TRIGGER, '1');
        $trigger_position = get_option(
            Constants::OPTION_SEARCH_WIDGET_TRIGGER_POSITION,
            Constants::DEFAULT_SEARCH_WIDGET_TRIGGER_POSITION
        );
        $chatbot_enabled = get_option(Constants::OPTION_ENABLE_CHAT_ASSISTANT, '1') === '1';

        echo '<table class="form-table">';

        // Keyboard shortcut
        echo '<tr>';
        echo '<th scope="row"><label for="' . esc_attr(Constants::OPTION_SEARCH_WIDGET_KEYBOARD_SHORTCUT) . '">' . esc_html__('Keyboard Shortcut', 'intufind') . '</label></th>';
        echo '<td>';
        Field::select(
            Constants::OPTION_SEARCH_WIDGET_KEYBOARD_SHORTCUT,
            '',
            [
                'ctrl+k' => 'Ctrl+K / ⌘K',
                'ctrl+/' => 'Ctrl+/ / ⌘/',
                'ctrl+shift+f' => 'Ctrl+Shift+F / ⌘⇧F',
            ],
            $shortcut,
            [
                'description' => __('Users can press this keyboard combination to open search. This is always available when the widget is enabled.', 'intufind')
            ]
        );
        echo '</td>';
        echo '</tr>';

        // Show floating trigger
        echo '<tr>';
        echo '<th scope="row">' . esc_html__('Floating Trigger Button', 'intufind') . '</th>';
        echo '<td>';
        
        if ($chatbot_enabled) {
            // Chatbot is enabled - explain the behavior
            echo '<div class="intucart-notice intucart-notice-info" style="margin: 0 0 10px 0;">';
            echo '<p>' . esc_html__('The AI Chatbot is enabled, so the floating search button is automatically hidden to avoid clutter. Users can still use the keyboard shortcut or shortcode buttons to access search.', 'intufind') . '</p>';
            echo '</div>';
            
            Field::checkbox(
                Constants::OPTION_SEARCH_WIDGET_SHOW_TRIGGER,
                __('Show floating search button (only when chatbot is disabled)', 'intufind'),
                $show_trigger === '1',
                [
                    'description' => __('This setting will take effect if you disable the chatbot.', 'intufind'),
                    'attributes' => ['disabled' => 'disabled']
                ]
            );
        } else {
            Field::checkbox(
                Constants::OPTION_SEARCH_WIDGET_SHOW_TRIGGER,
                __('Show floating search button', 'intufind'),
                $show_trigger === '1',
                [
                    'description' => __('Display a floating search icon button for easy access.', 'intufind')
                ]
            );
        }
        
        echo '</td>';
        echo '</tr>';

        // Trigger position (only if trigger is shown)
        echo '<tr id="search-widget-trigger-position-row"' . ($chatbot_enabled ? ' style="display:none;"' : '') . '>';
        echo '<th scope="row"><label for="' . esc_attr(Constants::OPTION_SEARCH_WIDGET_TRIGGER_POSITION) . '">' . esc_html__('Button Position', 'intufind') . '</label></th>';
        echo '<td>';
        Field::select(
            Constants::OPTION_SEARCH_WIDGET_TRIGGER_POSITION,
            '',
            [
                'bottom-left' => __('Bottom Left (recommended)', 'intufind'),
                'bottom-right' => __('Bottom Right', 'intufind'),
                'top-left' => __('Top Left', 'intufind'),
                'top-right' => __('Top Right', 'intufind'),
            ],
            $trigger_position,
            [
                'description' => __('Position of the floating search button. Bottom Left is recommended to avoid overlapping with the chatbot.', 'intufind')
            ]
        );
        echo '</td>';
        echo '</tr>';

        echo '</table>';
        echo '</div>';
    }

    /**
     * Render theme settings.
     */
    private function renderThemeSettings(): void
    {
        echo '<div class="intucart-settings-section">';
        echo '<h3>' . esc_html__('Appearance', 'intufind') . '</h3>';

        $primary_color = get_option(Constants::OPTION_SEARCH_WIDGET_PRIMARY_COLOR, '#0084ff');
        $dark_primary_color = get_option(Constants::OPTION_SEARCH_WIDGET_DARK_PRIMARY_COLOR, '#0084ff');
        $color_scheme = get_option(Constants::OPTION_SEARCH_WIDGET_COLOR_SCHEME, 'auto');

        echo '<table class="form-table">';

        // Color scheme
        echo '<tr>';
        echo '<th scope="row"><label for="' . esc_attr(Constants::OPTION_SEARCH_WIDGET_COLOR_SCHEME) . '">' . esc_html__('Color Scheme', 'intufind') . '</label></th>';
        echo '<td>';
        Field::select(
            Constants::OPTION_SEARCH_WIDGET_COLOR_SCHEME,
            '',
            [
                'auto' => __('Auto (follow system preference)', 'intufind'),
                'light' => __('Light', 'intufind'),
                'dark' => __('Dark', 'intufind'),
            ],
            $color_scheme,
            [
                'description' => __('Choose the widget color scheme. Auto will match the user\'s system preference.', 'intufind')
            ]
        );
        echo '</td>';
        echo '</tr>';

        // Primary color (dual picker for light/dark modes)
        echo '<tr>';
        echo '<th scope="row"><label>' . esc_html__('Primary Color', 'intufind') . '</label></th>';
        echo '<td>';
        ColorPicker::dual([
            'light_name' => Constants::OPTION_SEARCH_WIDGET_PRIMARY_COLOR,
            'light_value' => $primary_color,
            'dark_name' => Constants::OPTION_SEARCH_WIDGET_DARK_PRIMARY_COLOR,
            'dark_value' => $dark_primary_color,
            'description' => __('Accent color used for search highlighting, active filters, and input focus styling.', 'intufind')
        ]);
        echo '</td>';
        echo '</tr>';

        echo '</table>';
        echo '</div>';

        // Trigger Button Styling Section
        $this->renderTriggerButtonStyling();
    }

    /**
     * Render trigger button styling settings.
     */
    private function renderTriggerButtonStyling(): void
    {
        echo '<div class="intucart-settings-section">';
        echo '<h3>' . esc_html__('Trigger Button Styling', 'intufind') . '</h3>';
        echo '<p class="description">' . esc_html__('Customize the appearance of the floating search trigger button.', 'intufind') . '</p>';

        $trigger_bg = get_option(Constants::OPTION_SEARCH_WIDGET_TRIGGER_BG, Constants::DEFAULT_SEARCH_WIDGET_TRIGGER_BG);
        $trigger_text = get_option(Constants::OPTION_SEARCH_WIDGET_TRIGGER_TEXT, Constants::DEFAULT_SEARCH_WIDGET_TRIGGER_TEXT);
        $trigger_border = get_option(Constants::OPTION_SEARCH_WIDGET_TRIGGER_BORDER, Constants::DEFAULT_SEARCH_WIDGET_TRIGGER_BORDER);
        $dark_trigger_bg = get_option(Constants::OPTION_SEARCH_WIDGET_DARK_TRIGGER_BG, Constants::DEFAULT_SEARCH_WIDGET_DARK_TRIGGER_BG);
        $dark_trigger_text = get_option(Constants::OPTION_SEARCH_WIDGET_DARK_TRIGGER_TEXT, Constants::DEFAULT_SEARCH_WIDGET_DARK_TRIGGER_TEXT);
        $dark_trigger_border = get_option(Constants::OPTION_SEARCH_WIDGET_DARK_TRIGGER_BORDER, Constants::DEFAULT_SEARCH_WIDGET_DARK_TRIGGER_BORDER);

        echo '<table class="form-table">';

        // Background Color
        echo '<tr>';
        echo '<th scope="row"><label>' . esc_html__('Background', 'intufind') . '</label></th>';
        echo '<td>';
        ColorPicker::dual([
            'light_name' => Constants::OPTION_SEARCH_WIDGET_TRIGGER_BG,
            'light_value' => $trigger_bg,
            'dark_name' => Constants::OPTION_SEARCH_WIDGET_DARK_TRIGGER_BG,
            'dark_value' => $dark_trigger_bg,
            'description' => __('Background color of the trigger button.', 'intufind')
        ]);
        echo '</td>';
        echo '</tr>';

        // Text/Icon Color
        echo '<tr>';
        echo '<th scope="row"><label>' . esc_html__('Text & Icon', 'intufind') . '</label></th>';
        echo '<td>';
        ColorPicker::dual([
            'light_name' => Constants::OPTION_SEARCH_WIDGET_TRIGGER_TEXT,
            'light_value' => $trigger_text,
            'dark_name' => Constants::OPTION_SEARCH_WIDGET_DARK_TRIGGER_TEXT,
            'dark_value' => $dark_trigger_text,
            'description' => __('Color for the button text and search icon.', 'intufind')
        ]);
        echo '</td>';
        echo '</tr>';

        // Border Color
        echo '<tr>';
        echo '<th scope="row"><label>' . esc_html__('Border', 'intufind') . '</label></th>';
        echo '<td>';
        ColorPicker::dual([
            'light_name' => Constants::OPTION_SEARCH_WIDGET_TRIGGER_BORDER,
            'light_value' => $trigger_border,
            'dark_name' => Constants::OPTION_SEARCH_WIDGET_DARK_TRIGGER_BORDER,
            'dark_value' => $dark_trigger_border,
            'description' => __('Border color of the trigger button.', 'intufind')
        ]);
        echo '</td>';
        echo '</tr>';

        // Custom Icon
        $trigger_icon_id = get_option(Constants::OPTION_SEARCH_WIDGET_TRIGGER_ICON_ID, 0);
        echo '<tr>';
        echo '<th scope="row"><label>' . esc_html__('Custom Icon', 'intufind') . '</label></th>';
        echo '<td>';
        ImageUploader::icon(
            Constants::OPTION_SEARCH_WIDGET_TRIGGER_ICON_ID,
            (int) $trigger_icon_id,
            [
                'hint' => __('Upload a custom icon to replace the default search icon (40x40px recommended). Leave empty to use the default magnifying glass.', 'intufind'),
                'preview_size' => [48, 48],
                'preview_style' => 'border-radius: 8px;'
            ]
        );
        echo '</td>';
        echo '</tr>';

        echo '</table>';
        echo '</div>';
    }

    /**
     * Render shortcode help section.
     */
    private function renderShortcodeHelp(): void
    {
        echo '<div class="intucart-settings-section">';
        echo '<h3>' . esc_html__('Shortcode', 'intufind') . '</h3>';
        echo '<p class="description">' . esc_html__('Use the shortcode to place a search button anywhere in your theme.', 'intufind') . '</p>';

        echo '<div class="intucart-code-examples">';
        
        echo '<div class="intucart-code-example">';
        echo '<h4>' . esc_html__('Basic Usage', 'intufind') . '</h4>';
        echo '<code>[intufind_search]</code>';
        echo '<p class="description">' . esc_html__('Renders a "Search" button with an icon.', 'intufind') . '</p>';
        echo '</div>';

        echo '<div class="intucart-code-example">';
        echo '<h4>' . esc_html__('Custom Text', 'intufind') . '</h4>';
        echo '<code>[intufind_search text="Find Products"]</code>';
        echo '<p class="description">' . esc_html__('Customize the button text.', 'intufind') . '</p>';
        echo '</div>';

        echo '<div class="intucart-code-example">';
        echo '<h4>' . esc_html__('Icon Only', 'intufind') . '</h4>';
        echo '<code>[intufind_search icon_only="true"]</code>';
        echo '<p class="description">' . esc_html__('Show only the search icon, ideal for navigation bars.', 'intufind') . '</p>';
        echo '</div>';

        echo '<div class="intucart-code-example">';
        echo '<h4>' . esc_html__('With Custom Class', 'intufind') . '</h4>';
        echo '<code>[intufind_search text="Search" class="my-custom-btn"]</code>';
        echo '<p class="description">' . esc_html__('Add custom CSS classes for styling.', 'intufind') . '</p>';
        echo '</div>';

        echo '<div class="intucart-code-example">';
        echo '<h4>' . esc_html__('Pre-filled Query', 'intufind') . '</h4>';
        echo '<code>[intufind_search text="Find Shoes" query="shoes"]</code>';
        echo '<p class="description">' . esc_html__('Open search with a pre-filled query.', 'intufind') . '</p>';
        echo '</div>';

        echo '</div>'; // .intucart-code-examples

        // JavaScript API
        echo '<h4 style="margin-top: 20px;">' . esc_html__('JavaScript API', 'intufind') . '</h4>';
        echo '<p class="description">' . esc_html__('For theme developers, the search widget exposes a JavaScript API:', 'intufind') . '</p>';
        echo '<pre style="background: #f5f5f5; padding: 12px; border-radius: 4px; overflow-x: auto;">';
        echo esc_html("// Open search overlay\nwindow.IntufindSearch.open();\n\n// Open with pre-filled query\nwindow.IntufindSearch.open('running shoes');\n\n// Close search overlay\nwindow.IntufindSearch.close();\n\n// Toggle search overlay\nwindow.IntufindSearch.toggle();");
        echo '</pre>';

        echo '</div>';
    }
}

