<?php

namespace Intucart\Pages\Settings;

use Intucart\Services\Constants;
use Intucart\Components\Field;

class SyncSettings extends AbstractSettings
{
    public function registerSettings(): void
    {
        add_settings_section(
            'intucart_sync_section',
            __('Sync Settings', 'intufind'),
            [$this, 'renderSection'],
            Constants::SETTINGS_GROUP_SYNC
        );

        register_setting(
            Constants::SETTINGS_GROUP_SYNC,
            Constants::SYNC_INTERVAL_OPTION,
            [
                'type' => 'integer',
                'default' => Constants::DEFAULT_SYNC_INTERVAL,
                'sanitize_callback' => [$this, 'sanitizeSyncInterval']
            ]
        );

        register_setting(
            Constants::SETTINGS_GROUP_SYNC,
            Constants::SYNC_HOUR_OPTION,
            [
                'type' => 'integer',
                'default' => Constants::DEFAULT_SYNC_HOUR,
                'sanitize_callback' => [$this, 'sanitizeSyncHour']
            ]
        );

        add_settings_field(
            Constants::SYNC_INTERVAL_OPTION,
            __('Interval (hours)', 'intufind'),
            [$this, 'renderSyncIntervalField'],
            Constants::SETTINGS_GROUP_SYNC,
            'intucart_sync_section'
        );

        add_settings_field(
            Constants::SYNC_HOUR_OPTION,
            __('Time', 'intufind'),
            [$this, 'renderSyncHourField'],
            Constants::SETTINGS_GROUP_SYNC,
            'intucart_sync_section'
        );
    }

    public function renderSection(): void
    {
        echo '<p>' . sprintf(
            esc_html__('Sync settings control how often your website data is automatically updated with %s\'s AI system to keep recommendations and search results accurate and up to date. The sync happens automatically in the background.', 'intufind'),
            Constants::PLUGIN_SHORT_NAME
        ) . '</p>';
    }

    public function renderSyncIntervalField(): void
    {
        $value = get_option(Constants::SYNC_INTERVAL_OPTION, Constants::DEFAULT_SYNC_INTERVAL);
        $hours = $value / 3600;
        Field::number(
            Constants::SYNC_INTERVAL_OPTION,
            '',
            $hours,
            [
                'min' => 1,
                'max' => 168,
                'suffix' => __('hours', 'intufind'),
                'description' => __('How often to sync (1-168 hours)', 'intufind')
            ]
        );
    }

    public function renderSyncHourField(): void
    {
        $value = get_option(Constants::SYNC_HOUR_OPTION, Constants::DEFAULT_SYNC_HOUR);
        
        // Build options array for select
        $options = [];
        for ($i = 0; $i < 24; $i++) {
            $hour = str_pad($i, 2, '0', STR_PAD_LEFT);
            $options[$i] = $hour . ':00';
        }
        
        Field::select(
            Constants::SYNC_HOUR_OPTION,
            '',
            $options,
            $value,
            [
                'description' => __('The hour of the day to run the sync (24-hour format)', 'intufind')
            ]
        );
    }

    public function sanitizeSyncInterval($interval): int
    {
        // Handle null or empty values
        if ($interval === null || $interval === '') {
            return Constants::DEFAULT_SYNC_INTERVAL; // Default to 24 hours (1 day) in seconds
        }

        // Convert hours to seconds and ensure it's within bounds
        $interval = intval($interval) * 3600;
        if ($interval < 3600) {
            return 3600; // Minimum 1 hour
        }
        if ($interval > 604800) {
            return 604800; // Maximum 1 week
        }
        return $interval;
    }

    public function sanitizeSyncHour($hour): int
    {
        // Handle null or empty values
        if ($hour === null || $hour === '') {
            return Constants::DEFAULT_SYNC_HOUR; // Default to 2 AM
        }

        $hour = intval($hour);
        if ($hour < 0) {
            return 0;
        }
        if ($hour > 23) {
            return 23;
        }
        return $hour;
    }
}
