<?php

namespace Intucart\Pages\Settings;

use Intucart\Services\Logger;
use Intucart\Services\Constants;
use Intucart\Services\AIClientManager;
use Intufind\AI\Client;
use Intucart\Components\Field;
use Intucart\Components\Button;

/**
 * Webhook Settings Registration and UI
 * Handles registration and rendering of webhook configuration settings
 */
class WebhookSettings extends AbstractSettings
{
    private AIClientManager $aiClientManager;

    /**
     * Constructor
     *
     * @param Logger $logger Logger instance
     * @param AIClientManager $aiClientManager AI Client Manager service
     */
    public function __construct(Logger $logger, AIClientManager $aiClientManager)
    {
        parent::__construct($logger);
        $this->aiClientManager = $aiClientManager;
    }

    /**
     * Get the AI client instance
     *
     * @return \Intufind\AI\Client|null
     */
    private function getAiClient(): ?Client
    {
        try {
            /** @var \Intufind\AI\Client $client */
            $client = $this->aiClientManager->getClient();
            
            // Debug: Log the API endpoint being used
            $config = $this->aiClientManager->getConfig();
            $this->logger->debug('AI Client configuration', [
                'api_endpoint' => $config->getApiEndpoint(),
                'has_api_key' => $config->hasAuthentication(),
                'has_license_key' => $this->aiClientManager->hasLicenseKey(),
            ]);
            
            return $client;
        } catch (\Exception $e) {
            $this->logger->error('Failed to get AI client', [
                'error' => $e->getMessage()
            ]);
            
            return null;
        }
    }

    /**
     * Initialize webhook settings (register AJAX handlers)
     *
     * @return void
     */
    public function initialize(): void
    {
        // Register webhook AJAX handlers
        add_action('wp_ajax_intucart_test_webhook', [$this, 'handleAjaxTestWebhook']);
        add_action('wp_ajax_intucart_save_webhook_endpoint', [$this, 'handleAjaxSaveWebhookEndpoint']);
        add_action('wp_ajax_intucart_delete_webhook_endpoint', [$this, 'handleAjaxDeleteWebhookEndpoint']);
        add_action('wp_ajax_intucart_refresh_webhooks_table', [$this, 'handleAjaxRefreshWebhooksTable']);
    }

    /**
     * Register settings for webhooks
     *
     * @return void
     */
    public function registerSettings(): void
    {
        // Register settings section
        add_settings_section('intucart_webhook_section', '', null, Constants::SETTINGS_GROUP_WEBHOOKS);

        // === BASIC WEBHOOK SETTINGS ===

        // Enable/Disable Webhooks
        register_setting(
            Constants::SETTINGS_GROUP_WEBHOOKS,
            Constants::OPTION_WEBHOOKS_ENABLED,
            [
                'type' => 'boolean',
                'default' => false,
                'sanitize_callback' => [$this, 'sanitizeBoolean']
            ]
        );

        // Global Webhook Secret (optional)
        register_setting(
            Constants::SETTINGS_GROUP_WEBHOOKS,
            Constants::OPTION_WEBHOOK_GLOBAL_SECRET,
            [
                'type' => 'string',
                'default' => '',
                'sanitize_callback' => [$this, 'sanitizeSecret']
            ]
        );

        // Webhook Timeout
        register_setting(
            Constants::SETTINGS_GROUP_WEBHOOKS,
            Constants::OPTION_WEBHOOK_TIMEOUT,
            [
                'type' => 'integer',
                'default' => Constants::DEFAULT_WEBHOOK_TIMEOUT,
                'sanitize_callback' => [$this, 'sanitizeTimeout']
            ]
        );

        // Retry Attempts
        register_setting(
            Constants::SETTINGS_GROUP_WEBHOOKS,
            Constants::OPTION_WEBHOOK_RETRY_ATTEMPTS,
            [
                'type' => 'integer',
                'default' => Constants::DEFAULT_WEBHOOK_RETRY_ATTEMPTS,
                'sanitize_callback' => [$this, 'sanitizeRetryAttempts']
            ]
        );

        // Retry Delay
        register_setting(
            Constants::SETTINGS_GROUP_WEBHOOKS,
            Constants::OPTION_WEBHOOK_RETRY_DELAY,
            [
                'type' => 'integer',
                'default' => Constants::DEFAULT_WEBHOOK_RETRY_DELAY,
                'sanitize_callback' => [$this, 'sanitizeRetryDelay']
            ]
        );

        // Note: Webhook endpoints are now managed via the cloud API, not stored locally
    }

    // ====================================================================
    // SANITIZATION METHODS
    // ====================================================================

    /**
     * Sanitize boolean value
     *
     * @param mixed $value
     * @return bool
     */
    public function sanitizeBoolean($value): bool
    {
        return (bool) $value;
    }

    /**
     * Sanitize webhook secret
     *
     * @param string|null $value
     * @return string
     */
    public function sanitizeSecret(?string $value): string
    {
        if (empty($value)) {
            return '';
        }

        // Remove whitespace and ensure reasonable length
        $secret = trim($value);
        if (strlen($secret) < 8) {
            add_settings_error(
                Constants::OPTION_WEBHOOK_GLOBAL_SECRET,
                'webhook_secret_too_short',
                'Webhook secret must be at least 8 characters long.'
            );
            return '';
        }
        return sanitize_text_field($secret);
    }

    /**
     * Sanitize timeout value (5-300 seconds)
     *
     * @param mixed $value
     * @return int
     */
    public function sanitizeTimeout($value): int
    {
        $timeout = intval($value);
        return max(5, min(300, $timeout));
    }

    /**
     * Sanitize retry attempts (1-10)
     *
     * @param mixed $value
     * @return int
     */
    public function sanitizeRetryAttempts($value): int
    {
        $attempts = intval($value);
        return max(1, min(10, $attempts));
    }

    /**
     * Sanitize retry delay (1-60 seconds)
     *
     * @param mixed $value
     * @return int
     */
    public function sanitizeRetryDelay($value): int
    {
        $delay = intval($value);
        return max(1, min(60, $delay));
    }



    // ====================================================================
    // API INTEGRATION METHODS
    // ====================================================================

    /**
     * Fetch webhooks from the API
     *
     * @return array|null Array of webhooks or null if failed
     */
    private function fetchWebhooksFromApi(): ?array
    {
        $client = $this->getAiClient();
        if (!$client) {
            $this->logger->debug('AI client is null in fetchWebhooksFromApi');
            return null;
        }

        try {
            $this->logger->debug('Calling API to list webhooks');
            $result = $client->webhooks()->list();
            $this->logger->debug('API response received', [
                'result_type' => get_class($result),
                'results_count' => count($result->results ?? []),
                'full_result' => $result
            ]);
            
            $webhooks = [];

            $this->logger->debug('Processing webhook results', [
                'results_array' => $result->results ?? []
            ]);

            foreach ($result->results as $index => $webhook) {
                $this->logger->debug("Processing webhook $index", [
                    'webhook_type' => get_class($webhook),
                    'webhook_data' => $webhook
                ]);
                $webhooks[] = [
                    'id' => $webhook->id,
                    'name' => $webhook->name,
                    'url' => $webhook->url,
                    'events' => $webhook->events,
                    'secret' => !empty($webhook->secret) ? '[CONFIGURED]' : '',
                    'enabled' => $webhook->isActive(),
                    'status' => $webhook->status ?? 'active',
                    'method' => $webhook->method,
                    'description' => $webhook->description,
                    'total_deliveries' => $webhook->totalDeliveries,
                    'successful_deliveries' => $webhook->successfulDeliveries,
                    'failed_deliveries' => $webhook->failedDeliveries,
                    'success_rate' => $webhook->getSuccessRate(),
                    'last_delivery_at' => $webhook->lastDeliveryAt,
                    'last_delivery_error' => $webhook->lastDeliveryError ?? '',
                ];
            }

            return $webhooks;
        } catch (\Exception $e) {
            $this->logger->error('Failed to fetch webhooks from API', [
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }


    // ====================================================================
    // UI RENDERING METHODS
    // ====================================================================

    /**
     * Render webhook basic settings section
     *
     * @return void
     */
    public function renderWebhookBasicSettings(): void
    {
        $this->renderSettingsPanel(
            __('Webhook Configuration', 'intufind'),
            __('Configure global webhook settings including timeouts, retry policies, and security options.', 'intufind'),
            'admin-generic',
            function() {
                echo '<table class="form-table">';

                // Enable Webhooks
                $webhooks_enabled = get_option(Constants::OPTION_WEBHOOKS_ENABLED, false);
                $this->renderTableRow(__('Enable Webhooks', 'intufind'), function() use ($webhooks_enabled) {
                    Field::checkbox(
                        Constants::OPTION_WEBHOOKS_ENABLED,
                        __('Enable webhook notifications for events', 'intufind'),
                        $webhooks_enabled,
                        [
                            'description' => __('When enabled, configured webhook endpoints will receive HTTP notifications for selected events.', 'intufind')
                        ]
                    );
                });

                // Global Secret (Optional)
                $global_secret = get_option(Constants::OPTION_WEBHOOK_GLOBAL_SECRET, '');
                $this->renderTableRow(__('Global Secret', 'intufind'), function() use ($global_secret) {
                    Field::text(
                        Constants::OPTION_WEBHOOK_GLOBAL_SECRET,
                        '',
                        $global_secret,
                        [
                            'placeholder' => __('Optional shared secret for all webhooks', 'intufind'),
                            'description' => __('Optional shared secret used to sign webhook payloads. Leave empty to disable signing. Minimum 8 characters.', 'intufind'),
                            'type' => 'password'
                        ]
                    );
                });

                // Timeout Settings
                $timeout = get_option(Constants::OPTION_WEBHOOK_TIMEOUT, Constants::DEFAULT_WEBHOOK_TIMEOUT);
                $this->renderTableRow(__('Request Timeout', 'intufind'), function() use ($timeout) {
                    Field::number(
                        Constants::OPTION_WEBHOOK_TIMEOUT,
                        '',
                        $timeout,
                        [
                            'min' => 5,
                            'max' => 300,
                            'suffix' => __('seconds', 'intufind'),
                            'description' => __('Maximum time to wait for webhook endpoint response (5-300 seconds).', 'intufind')
                        ]
                    );
                });

                // Retry Settings
                $retry_attempts = get_option(Constants::OPTION_WEBHOOK_RETRY_ATTEMPTS, Constants::DEFAULT_WEBHOOK_RETRY_ATTEMPTS);
                $this->renderTableRow(__('Retry Attempts', 'intufind'), function() use ($retry_attempts) {
                    Field::number(
                        Constants::OPTION_WEBHOOK_RETRY_ATTEMPTS,
                        '',
                        $retry_attempts,
                        [
                            'min' => 1,
                            'max' => 10,
                            'suffix' => __('attempts', 'intufind'),
                            'description' => __('Number of times to retry failed webhook deliveries (1-10 attempts).', 'intufind')
                        ]
                    );
                });

                $retry_delay = get_option(Constants::OPTION_WEBHOOK_RETRY_DELAY, Constants::DEFAULT_WEBHOOK_RETRY_DELAY);
                $this->renderTableRow(__('Retry Delay', 'intufind'), function() use ($retry_delay) {
                    Field::number(
                        Constants::OPTION_WEBHOOK_RETRY_DELAY,
                        '',
                        $retry_delay,
                        [
                            'min' => 1,
                            'max' => 60,
                            'suffix' => __('seconds', 'intufind'),
                            'description' => __('Delay between retry attempts (1-60 seconds).', 'intufind')
                        ]
                    );
                });

                echo '</table>';
            }
        );
    }

    /**
     * Render webhook endpoints management section
     *
     * @return void
     */
    public function renderWebhookEndpoints(): void
    {
        // Fetch webhooks from API
        $this->logger->debug('Initial page load - fetching webhooks from API');
        $endpoints = $this->fetchWebhooksFromApi();
        
        $this->logger->debug('Initial webhook fetch result', [
            'endpoints_is_null' => $endpoints === null,
            'endpoints_count' => $endpoints ? count($endpoints) : 0,
            'endpoints_data' => $endpoints
        ]);
        
        if ($endpoints === null) {
            $this->logger->error('Unable to connect to webhook API on initial page load');
            echo '<div class="intucart-notice intucart-notice-error"><p><strong>' . esc_html__('Error:', 'intufind') . '</strong> ' . esc_html__('Unable to connect to webhook API. Please check your license key and API endpoint configuration.', 'intufind') . '</p>';
            echo '</div>';
            $endpoints = [];
        }

        echo '<div class="intucart-content-panel">';
        echo '<h3 class="intucart-panel-header">';
        echo '<span class="dashicons dashicons-admin-links"></span>';
        echo esc_html__('Webhook Endpoints', 'intufind');
        echo '</h3>';
        echo '<p class="intucart-panel-description">' . esc_html__('Configure webhook endpoints that will receive event notifications. Each endpoint can subscribe to specific events.', 'intufind') . '</p>';

        // Add New Endpoint Button
        echo '<div style="margin-bottom: 20px;">';
        Button::primary(__('Add Webhook Endpoint', 'intufind'), [
            'type' => 'button',
            'id' => 'add-webhook-endpoint',
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path d="M416 208H272V64c0-17.67-14.33-32-32-32h-32c-17.67 0-32 14.33-32 32v144H32c-17.67 0-32 14.33-32 32v32c0 17.67 14.33 32 32 32h144v144c0 17.67 14.33 32 32 32h32c17.67 0 32-14.33 32-32V304h144c17.67 0 32-14.33 32-32v-32c0-17.67-14.33-32-32-32z"/></svg>'
        ]);
        echo '</div>';

        // Webhook Form (hidden by default)
        echo '<div id="webhook-form-container" style="display: none; margin-bottom: 20px;">';
        $this->renderWebhookForm();
        echo '</div>';

        // Endpoints Table
        echo '<div id="webhook-endpoints-container">';
        $this->renderEndpointsTable($endpoints);
        echo '</div>';

        echo '</div>';
    }

    /**
     * Render webhook form for adding/editing endpoints
     *
     * @return void
     */
    private function renderWebhookForm(): void
    {
        echo '<div class="intucart-webhook-form-panel">';
        echo '<div class="intucart-webhook-form-header">';
        echo '<h4 id="webhook-form-title">' . esc_html__('Add Webhook Endpoint', 'intufind') . '</h4>';
        Button::secondary(__('Cancel', 'intufind'), [
            'type' => 'button',
            'id' => 'cancel-webhook-form',
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 352 512"><path d="M242.72 256l100.07-100.07c12.28-12.28 12.28-32.19 0-44.48l-22.24-22.24c-12.28-12.28-32.19-12.28-44.48 0L176 189.28 75.93 89.21c-12.28-12.28-32.19-12.28-44.48 0L9.21 111.45c-12.28 12.28-12.28 32.19 0 44.48L109.28 256 9.21 356.07c-12.28 12.28-12.28 32.19 0 44.48l22.24 22.24c12.28 12.28 32.2 12.28 44.48 0L176 322.72l100.07 100.07c12.28 12.28 32.2 12.28 44.48 0l22.24-22.24c12.28-12.28 12.28-32.19 0-44.48L242.72 256z"/></svg>'
        ]);
        echo '</div>';

        echo '<div id="webhook-endpoint-form" style="margin-top: 15px;">';
        echo '<input type="hidden" id="webhook-endpoint-id" name="endpoint_id" value="">';
        echo '<input type="hidden" id="webhook-form-mode" name="form_mode" value="add">';

        echo '<table class="form-table">';

        // Name field
        echo '<tr>';
        echo '<th scope="row"><label for="webhook-name">' . esc_html__('Endpoint Name', 'intufind') . ' <span class="required">*</span></label></th>';
        echo '<td>';
        echo '<input type="text" id="webhook-name" name="name" class="regular-text" required>';
        echo '<p class="description">' . esc_html__('A descriptive name for this webhook endpoint', 'intufind') . '</p>';
        echo '</td>';
        echo '</tr>';

        // Description field
        echo '<tr>';
        echo '<th scope="row"><label for="webhook-description">' . esc_html__('Description', 'intufind') . '</label></th>';
        echo '<td>';
        echo '<textarea id="webhook-description" name="description" class="large-text" rows="2"></textarea>';
        echo '<p class="description">' . esc_html__('Optional description for this webhook endpoint', 'intufind') . '</p>';
        echo '</td>';
        echo '</tr>';

        // URL field
        echo '<tr>';
        echo '<th scope="row"><label for="webhook-url">' . esc_html__('Webhook URL', 'intufind') . ' <span class="required">*</span></label></th>';
        echo '<td>';
        echo '<input type="url" id="webhook-url" name="url" class="regular-text" required>';
        echo '<p class="description">' . esc_html__('The HTTP endpoint that will receive webhook notifications', 'intufind') . '</p>';
        echo '</td>';
        echo '</tr>';

        // Method field
        echo '<tr>';
        echo '<th scope="row"><label for="webhook-method">' . esc_html__('HTTP Method', 'intufind') . '</label></th>';
        echo '<td>';
        echo '<select id="webhook-method" name="method">';
        echo '<option value="POST" selected>POST</option>';
        echo '<option value="PUT">PUT</option>';
        echo '<option value="PATCH">PATCH</option>';
        echo '</select>';
        echo '<p class="description">' . esc_html__('HTTP method to use for webhook requests', 'intufind') . '</p>';
        echo '</td>';
        echo '</tr>';

        // Secret field
        echo '<tr>';
        echo '<th scope="row"><label for="webhook-secret">' . esc_html__('Secret', 'intufind') . '</label></th>';
        echo '<td>';
        echo '<input type="text" id="webhook-secret" name="secret" class="regular-text">';
        echo '<p class="description">' . esc_html__('Optional secret for signing webhook payloads (HMAC-SHA256). If specified, this overrides the global webhook secret for this endpoint.', 'intufind') . '</p>';
        echo '</td>';
        echo '</tr>';

        // Events selection
        echo '<tr>';
        echo '<th scope="row">' . esc_html__('Events', 'intufind') . ' <span class="required">*</span></th>';
        echo '<td>';
        echo '<div id="webhook-events-selection">';
        $this->renderEventCheckboxes();
        echo '</div>';
        echo '<p class="description">' . esc_html__('Select which events should trigger this webhook', 'intufind') . '</p>';
        echo '</td>';
        echo '</tr>';

        // Payload Template field
        echo '<tr>';
        echo '<th scope="row"><label for="webhook-payload-template">' . esc_html__('Payload Template', 'intufind') . '</label></th>';
        echo '<td>';
        echo '<textarea id="webhook-payload-template" name="payload_template" rows="4" class="large-text code" placeholder=\'{"text": "{{text}}"}\'></textarea>';
        echo '<p class="description">';
        echo esc_html__('Optional. Transform the webhook payload using a JSON template with {{placeholders}}.', 'intufind') . '<br>';
        echo '<strong>' . esc_html__('Available placeholders:', 'intufind') . '</strong> ';
        echo '<code>{{event_type}}</code>, <code>{{text}}</code>, <code>{{markdown}}</code>, <code>{{timestamp}}</code>, <code>{{data.field}}</code> ';
        echo '<span class="dashicons dashicons-info-outline" style="font-size: 14px; line-height: inherit; width: 14px; height: 14px; color: #2271b1; cursor: help; vertical-align: text-top;" data-popover="' . esc_attr($this->getDataFieldsPopoverContent()) . '"></span>';
        echo '</p>';
        echo '<p class="description" style="margin-top: 8px;">';
        echo '<strong>' . esc_html__('Examples:', 'intufind') . '</strong><br>';
        echo '<span style="display: inline-block; margin: 4px 0;">' . esc_html__('Slack:', 'intufind') . ' <code>{"text": "{{text}}"}</code></span><br>';
        echo '<span style="display: inline-block; margin: 4px 0;">' . esc_html__('Discord:', 'intufind') . ' <code>{"content": "{{text}}"}</code></span>';
        echo '</p>';
        echo '</td>';
        echo '</tr>';

        // Enabled checkbox
        echo '<tr>';
        echo '<th scope="row">' . esc_html__('Status', 'intufind') . '</th>';
        echo '<td>';
        echo '<label>';
        echo '<input type="checkbox" id="webhook-enabled" name="enabled" value="1" checked>';
        echo ' ' . esc_html__('Enabled', 'intufind');
        echo '</label>';
        echo '<p class="description">' . esc_html__('Enable or disable this webhook endpoint', 'intufind') . '</p>';
        echo '</td>';
        echo '</tr>';

        echo '</table>';

        // Form actions
        echo '<div class="intucart-webhook-form-actions" style="display: flex; gap: 10px;">';
        Button::primary(__('Save Endpoint', 'intufind'), [
            'type' => 'button',
            'id' => 'save-webhook-endpoint',
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path d="M433.941 129.941l-83.882-83.882A48 48 0 0 0 316.118 32H48C21.49 32 0 53.49 0 80v352c0 26.51 21.49 48 48 48h352c26.51 0 48-21.49 48-48V163.882a48 48 0 0 0-14.059-33.941zM272 80v80H144V80h128zm122 352H54a6 6 0 0 1-6-6V86a6 6 0 0 1 6-6h42v104c0 13.255 10.745 24 24 24h176c13.255 0 24-10.745 24-24V83.882l78.243 78.243a6 6 0 0 1 1.757 4.243V426a6 6 0 0 1-6 6zM224 232c-48.523 0-88 39.477-88 88s39.477 88 88 88 88-39.477 88-88-39.477-88-88-88zm0 128c-22.056 0-40-17.944-40-40s17.944-40 40-40 40 17.944 40 40-17.944 40-40 40z"/></svg>'
        ]);
        Button::secondary(__('Test Webhook', 'intufind'), [
            'type' => 'button',
            'id' => 'test-webhook-form',
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path d="M501.1 395.7L384 278.6c-23.1-23.1-57.6-27.6-85.4-13.9L192 158.1V96L64 0 0 64l96 128h62.1l106.6 106.6c-13.6 27.8-9.2 62.3 13.9 85.4l117.1 117.1c14.6 14.6 38.2 14.6 52.7 0l52.7-52.7c14.5-14.6 14.5-38.2 0-52.7z"/></svg>'
        ]);
        echo '</div>';

        echo '</div>';
        echo '</div>';
    }

    /**
     * Render event checkboxes for webhook form
     *
     * @return void
     */
    private function renderEventCheckboxes(): void
    {
        $events = [
            __('Product Events', 'intufind') => [
                'product.created' => __('Product Created', 'intufind'),
                'product.updated' => __('Product Updated', 'intufind'), 
                'product.deleted' => __('Product Deleted', 'intufind'),
                'product.bulk_deleted' => __('Products Bulk Deleted', 'intufind'),
            ],
            __('Post Events', 'intufind') => [
                'post.created' => __('Post Created', 'intufind'),
                'post.updated' => __('Post Updated', 'intufind'),
                'post.deleted' => __('Post Deleted', 'intufind'),
                'post.bulk_deleted' => __('Posts Bulk Deleted', 'intufind'),
            ],
            __('Chat Events', 'intufind') => [
                'chat.message.sent' => __('AI Message Sent', 'intufind'),
                'chat.message.received' => __('User Message Received', 'intufind'),
                'chat.conversation.started' => __('Conversation Started', 'intufind'),
            ],
            __('Prompt Events', 'intufind') => [
                'prompt.created' => __('Prompt Created', 'intufind'),
                'prompt.updated' => __('Prompt Updated', 'intufind'),
                'prompt.deleted' => __('Prompt Deleted', 'intufind'),
                'prompt.bulk_deleted' => __('Prompts Bulk Deleted', 'intufind'),
            ],
            __('Taxonomy Events', 'intufind') => [
                'taxonomy.created' => __('Taxonomy Created', 'intufind'),
                'taxonomy.updated' => __('Taxonomy Updated', 'intufind'),
                'taxonomy.deleted' => __('Taxonomy Deleted', 'intufind'),
                'taxonomy.bulk_deleted' => __('Taxonomies Bulk Deleted', 'intufind'),
            ],
            __('Feedback Events', 'intufind') => [
                'feedback.submitted' => __('Feedback Submitted', 'intufind'),
                'feedback.deleted' => __('Feedback Deleted', 'intufind'),
                'feedback.bulk_deleted' => __('Feedback Bulk Deleted', 'intufind'),
            ],
            __('Recommendation Events', 'intufind') => [
                'recommendation.generated' => __('Recommendation Generated', 'intufind'),
                'recommendation.requested' => __('Recommendation Requested', 'intufind'),
            ],
            __('Search Events', 'intufind') => [
                'search.performed' => __('Search Performed', 'intufind'),
                'search.failed' => __('Search Failed', 'intufind'),
            ],
        ];

        echo '<div class="intucart-webhook-events-grid">';
        
        foreach ($events as $category => $categoryEvents) {
            echo '<div class="intucart-webhook-event-category">';
            echo '<h5>' . esc_html($category) . '</h5>';
            
            foreach ($categoryEvents as $eventKey => $eventLabel) {
                echo '<label class="intucart-webhook-event-item">';
                echo '<input type="checkbox" name="events[]" value="' . esc_attr($eventKey) . '">';
                echo ' ' . esc_html($eventLabel);
                echo '</label>';
            }
            
            echo '</div>';
        }
        
        echo '</div>';
    }

    /**
     * Render the endpoints table
     *
     * @param array $endpoints
     * @return void
     */
    private function renderEndpointsTable(array $endpoints): void
    {
        if (empty($endpoints)) {
            echo '<div class="intucart-webhook-no-endpoints">';
            echo '<div class="dashicons dashicons-admin-links"></div>';
            echo '<h4>' . esc_html__('No webhook endpoints configured', 'intufind') . '</h4>';
            echo '<p>' . esc_html__('Click "Add Webhook Endpoint" above to create your first webhook endpoint and start receiving real-time notifications.', 'intufind') . '</p>';
            echo '</div>';
            return;
        }

        echo '<table class="widefat intucart-webhook-endpoints-table" id="webhook-endpoints-table">';
        echo '<thead>';
        echo '<tr>';
        echo '<th>' . esc_html__('Name', 'intufind') . '</th>';
        echo '<th>' . esc_html__('URL', 'intufind') . '</th>';
        echo '<th>' . esc_html__('Events', 'intufind') . '</th>';
        echo '<th>' . esc_html__('Status', 'intufind') . '</th>';
        echo '<th>' . esc_html__('Actions', 'intufind') . '</th>';
        echo '</tr>';
        echo '</thead>';
        echo '<tbody>';

        foreach ($endpoints as $index => $endpoint) {
            $this->renderEndpointRow($endpoint, $index);
        }

        echo '</tbody>';
        echo '</table>';
    }

    /**
     * Render a single endpoint row
     *
     * @param array $endpoint
     * @param int $index
     * @return void
     */
    private function renderEndpointRow(array $endpoint, int $index): void
    {
        $enabled = $endpoint['enabled'] ?? true;
        $events = $endpoint['events'] ?? [];
        $events_count = count($events);

        // Store endpoint data as JSON in data attribute for editing
        $endpoint_json = esc_attr(json_encode($endpoint));

        echo '<tr class="webhook-endpoint-row" data-index="' . esc_attr($index) . '" data-endpoint="' . $endpoint_json . '">';
        
        // Name
        echo '<td>';
        echo '<strong>' . esc_html($endpoint['name'] ?? '') . '</strong>';
        if (!empty($endpoint['secret']) && $endpoint['secret'] !== '[CONFIGURED]') {
            echo '<br><small class="intucart-webhook-secret-indicator">🔒 ' . esc_html__('Secret configured', 'intufind') . '</span>';
        } elseif ($endpoint['secret'] === '[CONFIGURED]') {
            echo '<br><small class="intucart-webhook-secret-indicator">🔒 ' . esc_html__('Secret configured (API)', 'intufind') . '</span>';
        }
        
        // Show description if available
        if (!empty($endpoint['description'])) {
            echo '<br><small style="color: #666;">' . esc_html($endpoint['description']) . '</small>';
        }
        echo '</td>';
        
        // URL
        echo '<td>';
        echo '<code style="font-size: 11px; word-break: break-all;">' . esc_html($endpoint['url'] ?? '') . '</code>';
        echo '<br><small style="color: #666;">' . sprintf(esc_html__('Method: %s', 'intufind'), esc_html($endpoint['method'] ?? 'POST')) . '</small>';
        echo '</td>';
        
        // Events
        echo '<td>';
        if ($events_count > 0) {
            echo '<span class="status-text-ok">' . sprintf(esc_html__('%d events', 'intufind'), $events_count) . '</span>';
            if ($events_count <= 3) {
                echo '<br><small style="color: #666;">' . esc_html(implode(', ', array_slice($events, 0, 3))) . '</small>';
            } else {
                echo '<br><small style="color: #666;">' . esc_html(implode(', ', array_slice($events, 0, 2))) . ', +' . sprintf(esc_html__('%d more', 'intufind'), ($events_count - 2)) . '</small>';
            }
        } else {
            echo '<span class="status-text-warning">' . esc_html__('No events', 'intufind') . '</span>';
        }
        echo '</td>';
        
        // Status with delivery info
        echo '<td>';
        
        // Determine status display
        $status = $endpoint['status'] ?? ($enabled ? 'active' : 'inactive');
        
        if ($status === 'active') {
            echo '<span class="status-indicator status-ok">' . esc_html__('Active', 'intufind') . '</span>';
        } elseif ($status === 'suspended') {
            echo '<span class="status-indicator status-error" title="' . esc_attr__('Auto-suspended due to failures', 'intufind') . '">' . esc_html__('Suspended', 'intufind') . '</span>';
        } else {
            echo '<span class="status-indicator status-neutral">' . esc_html__('Inactive', 'intufind') . '</span>';
        }
            
        // Show delivery stats if available (from API)
        if (isset($endpoint['total_deliveries'])) {
            $success_rate = $endpoint['success_rate'] ?? 0;
            $total = $endpoint['total_deliveries'];
            
            if ($total > 0) {
                $color = $success_rate >= 90 ? 'green' : ($success_rate >= 70 ? 'orange' : 'red');
                echo '<br><small style="color: ' . $color . ';">';
                echo sprintf(esc_html__('%.1f%% success (%d total)', 'intufind'), $success_rate, $total);
                echo '</small>';
            }
        }
        
        // Show auto-suspension message if suspended
        if ($status === 'suspended' && !empty($endpoint['last_delivery_error'])) {
            echo '<br><small style="color: #dc3232;">';
            echo esc_html($endpoint['last_delivery_error']);
            echo '</small>';
        }
        
        echo '</td>';
        
        // Actions
        echo '<td style="white-space: nowrap;">';
        echo '<div style="display: flex; gap: 6px; flex-wrap: nowrap;">';
        
        Button::secondary(__('Edit', 'intufind'), [
            'type' => 'button',
            'size' => 'small',
            'attributes' => [
                'class' => 'edit-endpoint',
                'data-index' => $index
            ]
        ]);
        
        Button::secondary(__('Test', 'intufind'), [
            'type' => 'button',
            'size' => 'small',
            'attributes' => [
                'class' => 'test-endpoint',
                'data-index' => $index
            ]
        ]);
        
        // Show "Reactivate" button for suspended webhooks
        if ($status === 'suspended') {
            Button::primary(__('Reactivate', 'intufind'), [
                'type' => 'button',
                'size' => 'small',
                'attributes' => [
                    'class' => 'reactivate-endpoint',
                    'data-index' => $index,
                    'title' => __('Reactivate this webhook after fixing the issue', 'intufind')
                ]
            ]);
        }
        
        Button::danger(__('Delete', 'intufind'), [
            'type' => 'button',
            'size' => 'small',
            'attributes' => [
                'class' => 'delete-endpoint',
                'data-index' => $index
            ]
        ]);
        
        echo '</div>';
        echo '</td>';
        
        echo '</tr>';
    }

    /**
     * Render webhook events reference section
     *
     * @return void
     */
    public function renderWebhookEventsReference(): void
    {
        $this->renderSettingsPanel(
            __('Available Events', 'intufind'),
            __('Reference of all available webhook events that can be configured for your endpoints.', 'intufind'),
            'info',
            function() {
                $events = [
                    __('Product Events', 'intufind') => [
                        'product.created' => __('Triggered when a new product is created', 'intufind'),
                        'product.updated' => __('Triggered when a product is updated', 'intufind'),
                        'product.deleted' => __('Triggered when a product is deleted', 'intufind'),
                        'product.bulk_deleted' => __('Triggered when products are bulk deleted', 'intufind'),
                    ],
                    __('Post Events', 'intufind') => [
                        'post.created' => __('Triggered when a new post/page is created', 'intufind'),
                        'post.updated' => __('Triggered when a post/page is updated', 'intufind'),
                        'post.deleted' => __('Triggered when a post/page is deleted', 'intufind'),
                        'post.bulk_deleted' => __('Triggered when posts are bulk deleted', 'intufind'),
                    ],
                    __('Chat Events', 'intufind') => [
                        'chat.message.sent' => __('Triggered when the AI assistant sends a message', 'intufind'),
                        'chat.message.received' => __('Triggered when a user sends a message', 'intufind'),
                        'chat.conversation.started' => __('Triggered when a new conversation begins', 'intufind'),
                    ],
                    __('Prompt Events', 'intufind') => [
                        'prompt.created' => __('Triggered when a new prompt is created', 'intufind'),
                        'prompt.updated' => __('Triggered when a prompt is updated', 'intufind'),
                        'prompt.deleted' => __('Triggered when a prompt is deleted', 'intufind'),
                        'prompt.bulk_deleted' => __('Triggered when prompts are bulk deleted', 'intufind'),
                    ],
                    __('Taxonomy Events', 'intufind') => [
                        'taxonomy.created' => __('Triggered when a new taxonomy term is created', 'intufind'),
                        'taxonomy.updated' => __('Triggered when a taxonomy term is updated', 'intufind'),
                        'taxonomy.deleted' => __('Triggered when a taxonomy term is deleted', 'intufind'),
                        'taxonomy.bulk_deleted' => __('Triggered when taxonomy terms are bulk deleted', 'intufind'),
                    ],
                    __('Feedback Events', 'intufind') => [
                        'feedback.submitted' => __('Triggered when user feedback is submitted', 'intufind'),
                        'feedback.deleted' => __('Triggered when feedback is deleted', 'intufind'),
                        'feedback.bulk_deleted' => __('Triggered when feedback is bulk deleted', 'intufind'),
                    ],
                    __('Recommendation Events', 'intufind') => [
                        'recommendation.generated' => __('Triggered when AI recommendations are generated', 'intufind'),
                        'recommendation.requested' => __('Triggered when AI recommendations are requested', 'intufind'),
                    ],
                    __('Search Events', 'intufind') => [
                        'search.performed' => __('Triggered when a search is performed', 'intufind'),
                        'search.failed' => __('Triggered when a search fails', 'intufind'),
                    ],
                ];

                echo '<div class="intucart-webhook-events-reference">';
                
                foreach ($events as $category => $categoryEvents) {
                    echo '<div>';
                    echo '<h4>' . esc_html($category) . '</h4>';
                    echo '<table class="widefat">';
                    echo '<thead><tr><th style="width: 200px;">' . esc_html__('Event Type', 'intufind') . '</th><th>' . esc_html__('Description', 'intufind') . '</th></tr></thead>';
                    echo '<tbody>';
                    
                    foreach ($categoryEvents as $event => $description) {
                        echo '<tr>';
                        echo '<td><code>' . esc_html($event) . '</code></td>';
                        echo '<td>' . esc_html($description) . '</td>';
                        echo '</tr>';
                    }
                    
                    echo '</tbody></table>';
                    echo '</div>';
                }
                
                echo '</div>';
            }
        );
    }

    /**
     * Get popover content explaining available data fields for different event types
     *
     * @return string HTML content for the popover
     */
    private function getDataFieldsPopoverContent(): string
    {
        $content = '<strong>' . __('Data Fields by Event Type:', 'intufind') . '</strong><br><br>';
        
        $content .= '<strong>product.*</strong><br>';
        $content .= '• {{data.product_id}}<br>';
        $content .= '• {{data.name}}<br>';
        $content .= '• {{data.price}}<br>';
        $content .= '• {{data.sku}}<br><br>';
        
        $content .= '<strong>chat.*</strong><br>';
        $content .= '• {{data.thread_id}}<br>';
        $content .= '• {{data.user_message}}<br>';
        $content .= '• {{data.assistant_message}}<br><br>';
        
        $content .= '<strong>feedback.*</strong><br>';
        $content .= '• {{data.thread_id}}<br>';
        $content .= '• {{data.rating}}<br>';
        $content .= '• {{data.comment}}<br><br>';
        
        $content .= '<em>' . __('Use {{text}} or {{markdown}} for pre-formatted summaries.', 'intufind') . '</em>';
        
        return $content;
    }

    // ====================================================================
    // WEBHOOK AJAX HANDLERS
    // ====================================================================

    /**
     * Handle AJAX request to test a webhook endpoint
     *
     * @return void
     */
    public function handleAjaxTestWebhook(): void
    {
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'intufind')]);
            return;
        }

        check_ajax_referer('intucart_admin_nonce', 'nonce');

        $webhook_id = sanitize_text_field($_POST['webhook_id'] ?? '');

        if (empty($webhook_id)) {
            wp_send_json_error(['message' => __('Webhook ID is required for testing.', 'intufind')]);
            return;
        }

        try {
            $client = $this->getAiClient();
            
            if (!$client) {
                wp_send_json_error(['message' => __('Unable to connect to webhook API. Please check your license key and API endpoint configuration.', 'intufind')]);
                return;
            }

            $result = $client->webhooks()->test($webhook_id);
            
            wp_send_json_success([
                'message' => $result->isSuccessful() ? __('Test webhook sent successfully!', 'intufind') : __('Webhook test failed', 'intufind'),
                'response_code' => $result->responseCode,
                'response_time' => $result->getFormattedResponseTime(),
                'response_body' => $result->responseBody ? substr($result->responseBody, 0, 500) : 'No response body',
                'error_message' => $result->errorMessage,
                'success' => $result->isSuccessful(),
            ]);

        } catch (\Exception $e) {
            $this->logger->error('Failed to test webhook', [
                'webhook_id' => $webhook_id,
                'error' => $e->getMessage()
            ]);
            wp_send_json_error(['message' => sprintf(__('Failed to test webhook: %s', 'intufind'), $e->getMessage())]);
        }
    }

    /**
     * Handle AJAX request to save a webhook endpoint
     *
     * @return void
     */
    public function handleAjaxSaveWebhookEndpoint(): void
    {
        // Debug logging
        $this->logger->debug('Webhook save AJAX handler called', [
            'post_data' => $_POST,
            'nonce_received' => $_POST['nonce'] ?? 'not_set',
            'user_can_manage' => current_user_can('manage_options')
        ]);
        
        if (!current_user_can('manage_options')) {
            $this->logger->error('Webhook save failed: Permission denied');
            wp_send_json_error(['message' => __('Permission denied.', 'intufind')]);
            return;
        }

        try {
            check_ajax_referer('intucart_admin_nonce', 'nonce');
            $this->logger->debug('Webhook save: Nonce verified successfully');
        } catch (\Exception $e) {
            $this->logger->error('Webhook save failed: Invalid nonce', [
                'error' => $e->getMessage(),
                'nonce_received' => $_POST['nonce'] ?? 'not_set'
            ]);
            wp_send_json_error(['message' => __('Invalid nonce', 'intufind')]);
            return;
        }

        $endpoint_data = [
            'id' => sanitize_text_field($_POST['id'] ?? ''),
            'name' => sanitize_text_field($_POST['name'] ?? ''),
            'url' => sanitize_url($_POST['url'] ?? ''),
            'events' => array_map('sanitize_text_field', $_POST['events'] ?? []),
            'secret' => sanitize_text_field($_POST['secret'] ?? ''),
            'payload_template' => wp_unslash($_POST['payload_template'] ?? ''), // JSON template, keep as-is
            'enabled' => (bool) ($_POST['enabled'] ?? true),
            'method' => in_array($_POST['method'] ?? 'POST', ['POST', 'PUT', 'PATCH']) 
                ? $_POST['method'] 
                : 'POST',
            'description' => sanitize_text_field($_POST['description'] ?? ''),
        ];

        // Validate required fields
        if (empty($endpoint_data['name'])) {
            wp_send_json_error(['message' => __('Endpoint name is required.', 'intufind')]);
            return;
        }

        if (empty($endpoint_data['url'])) {
            wp_send_json_error(['message' => __('Webhook URL is required.', 'intufind')]);
            return;
        }

        if (empty($endpoint_data['events'])) {
            wp_send_json_error(['message' => __('At least one event must be selected.', 'intufind')]);
            return;
        }

        // Validate URL format
        if (!filter_var($endpoint_data['url'], FILTER_VALIDATE_URL)) {
            wp_send_json_error(['message' => __('Invalid webhook URL format.', 'intufind')]);
            return;
        }

        try {
            $client = $this->getAiClient();
            
            if (!$client) {
                $this->logger->error('AI client is null in webhook save handler');
                wp_send_json_error(['message' => __('Unable to connect to webhook API. Please check your license key and API endpoint configuration.', 'intufind')]);
                return;
            }

            // Debug: Log the client configuration
            $config = $client->getConfig();
            $this->logger->debug('Client config for webhook save', [
                'api_endpoint' => $config->getApiEndpoint(),
                'has_api_key' => $config->hasAuthentication(),
                'has_license_key' => $this->aiClientManager->hasLicenseKey(),
                'workspace_id' => $config->getWorkspaceId(),
            ]);

            $webhookData = [
                'name' => $endpoint_data['name'],
                'description' => $endpoint_data['description'],
                'url' => $endpoint_data['url'],
                'method' => $endpoint_data['method'],
                'events' => $endpoint_data['events'], // Use standard dot notation format
                'status' => $endpoint_data['enabled'] ? 'active' : 'inactive',
            ];

            if (!empty($endpoint_data['secret'])) {
                $webhookData['secret'] = $endpoint_data['secret'];
            }

            // Add payload template if provided (for transforming payloads, e.g., Slack format)
            if (!empty($endpoint_data['payload_template'])) {
                $webhookData['payload_template'] = $endpoint_data['payload_template'];
            }

            $this->logger->debug('Attempting to save webhook', [
                'webhook_data' => $webhookData,
                'is_update' => !empty($endpoint_data['id']),
                'webhook_id' => $endpoint_data['id'] ?? 'new'
            ]);

            if (empty($endpoint_data['id'])) {
                // Create new webhook
                $this->logger->debug('Creating new webhook via API');
                $webhook = $client->webhooks()->create($webhookData);
                $message = __('Webhook endpoint created successfully.', 'intufind');
                $endpoint_data['id'] = $webhook->id;
                $this->logger->info('Webhook created successfully', ['webhook_id' => $webhook->id]);
            } else {
                // Update existing webhook
                $this->logger->debug('Updating existing webhook via API', ['webhook_id' => $endpoint_data['id']]);
                $webhook = $client->webhooks()->update($endpoint_data['id'], $webhookData);
                $message = __('Webhook endpoint updated successfully.', 'intufind');
                $this->logger->info('Webhook updated successfully', ['webhook_id' => $endpoint_data['id']]);
            }

            wp_send_json_success([
                'message' => $message,
                'endpoint' => $endpoint_data,
            ]);

        } catch (\Exception $e) {
            $this->logger->error('Failed to save webhook endpoint', [
                'endpoint_data' => $endpoint_data,
                'error' => $e->getMessage()
            ]);
            wp_send_json_error(['message' => sprintf(__('Failed to save webhook endpoint: %s', 'intufind'), $e->getMessage())]);
        }
    }

    /**
     * Handle AJAX request to delete a webhook endpoint
     *
     * @return void
     */
    public function handleAjaxDeleteWebhookEndpoint(): void
    {
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'intufind')]);
            return;
        }

        check_ajax_referer('intucart_admin_nonce', 'nonce');

        $endpoint_id = sanitize_text_field($_POST['endpoint_id'] ?? '');

        if (empty($endpoint_id)) {
            wp_send_json_error(['message' => __('Endpoint ID is required.', 'intufind')]);
            return;
        }

        try {
            $client = $this->getAiClient();
            
            if (!$client) {
                wp_send_json_error(['message' => __('Unable to connect to webhook API. Please check your license key and API endpoint configuration.', 'intufind')]);
                return;
            }

            $deleted = $client->webhooks()->delete($endpoint_id);
            
            if ($deleted) {
                wp_send_json_success([
                    'message' => __('Webhook endpoint deleted successfully.', 'intufind'),
                ]);
            } else {
                wp_send_json_error(['message' => __('Webhook endpoint not found or could not be deleted.', 'intufind')]);
            }

        } catch (\Exception $e) {
            $this->logger->error('Failed to delete webhook endpoint', [
                'endpoint_id' => $endpoint_id,
                'error' => $e->getMessage()
            ]);
            wp_send_json_error(['message' => sprintf(__('Failed to delete webhook endpoint: %s', 'intufind'), $e->getMessage())]);
        }
    }

    /**
     * Handle AJAX request to refresh the webhooks table
     *
     * @return void
     */
    public function handleAjaxRefreshWebhooksTable(): void
    {
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'intufind')]);
            return;
        }

        try {
            check_ajax_referer('intucart_admin_nonce', 'nonce');
        } catch (\Exception $e) {
            wp_send_json_error(['message' => __('Invalid nonce', 'intufind')]);
            return;
        }

        // Get updated webhooks from API
        $this->logger->debug('Refreshing webhooks table - fetching from API');
        $endpoints = $this->fetchWebhooksFromApi();
        
        $this->logger->debug('Fetched webhooks for table refresh', [
            'endpoints_count' => $endpoints ? count($endpoints) : 0,
            'endpoints' => $endpoints
        ]);
        
        if ($endpoints === null) {
            $this->logger->error('Unable to fetch webhooks from API for table refresh');
            wp_send_json_error(['message' => __('Unable to fetch webhooks from API.', 'intufind')]);
            return;
        }

        // Generate table rows HTML
        ob_start();
        
        if (empty($endpoints)) {
            echo '<tr><td colspan="8" style="text-align: center; padding: 20px; color: #666;">' . esc_html__('No webhook endpoints configured.', 'intufind') . '</td></tr>';
        } else {
            foreach ($endpoints as $index => $endpoint) {
                $this->renderEndpointRow($endpoint, $index);
            }
        }
        
        $tableHtml = ob_get_clean();

        wp_send_json_success(['html' => $tableHtml]);
    }
}
