<?php

namespace Intucart\Pages;

use Intucart\Services\Logger;
use Intucart\Services\Constants;
use Intucart\Services\Licensing\License;
use Intucart\Services\Cache\CacheService;
use Intucart\Services\AIClientManager;
use Intucart\Services\Mcp\McpServer;
use Intucart\Services\UsageService;
use Intucart\Services\Managers\PostTypeManager;
use Intucart\Services\Managers\TaxonomyManager;
use Intucart\Components\Tabs;
use Intucart\Components\Button;

/**
 * Status Page
 */
class Status
{
    private Logger $logger;
    private License $license;
    private CacheService $cacheService;
    private AIClientManager $aiClientManager;
    private McpServer $mcpServer;
    private UsageService $usageService;
    private PostTypeManager $postTypeManager;
    private TaxonomyManager $taxonomyManager;
    private bool $recommendationsAvailable = false;

    // Class-level cache to prevent multiple API calls during same request
    private ?array $systemHealthCache = null;

    /**
     * Constructor
     *
     * @param Logger           $logger           Logger instance
     * @param License          $license          License service
     * @param CacheService     $cacheService     Cache service instance
     * @param AIClientManager  $aiClientManager  AI Client Manager service
     * @param McpServer        $mcpServer        MCP server instance
     * @param UsageService     $usageService     Usage tracking service
     * @param PostTypeManager  $postTypeManager  Post type manager service
     * @param TaxonomyManager  $taxonomyManager  Taxonomy manager service
     */
    public function __construct(
        Logger $logger,
        License $license,
        CacheService $cacheService,
        AIClientManager $aiClientManager,
        McpServer $mcpServer,
        UsageService $usageService,
        PostTypeManager $postTypeManager,
        TaxonomyManager $taxonomyManager
    ) {
        $this->logger = $logger;
        $this->license = $license;
        $this->cacheService = $cacheService;
        $this->aiClientManager = $aiClientManager;
        $this->mcpServer = $mcpServer;
        $this->usageService = $usageService;
        $this->postTypeManager = $postTypeManager;
        $this->taxonomyManager = $taxonomyManager;

        // Calculate recommendations availability for conditional display
        $this->calculateRecommendationsAvailability();
    }

    /**
     * Calculate whether recommendations are available based on license tier
     *
     * @return void
     */
    private function calculateRecommendationsAvailability(): void
    {
        // Use the license service's proper hasRecommendations() method
        // This checks tier config: recommendations.enabled && recommendations.perMonth > 0
        $this->recommendationsAvailable = $this->license->hasRecommendations();
    }

    /**
     * Render the status dashboard with tabs
     *
     * @return void
     */
    public function render(): void
    {
        // Enqueue Chart.js (bundled locally for WordPress.org compliance)
        wp_enqueue_script(
            'chartjs',
            INTUCART_PLUGIN_URL . 'assets/js/vendor/chart.min.js',
            [],
            '4.4.1',
            true
        );

        // Enqueue our custom script that depends on Chart.js
        wp_enqueue_script(
            'intucart-status',
            INTUCART_PLUGIN_URL . 'assets/js/admin/status.js',
            ['chartjs', 'jquery'],
            filemtime(INTUCART_PLUGIN_DIR . 'assets/js/admin/status.js'),
            true
        );

        // Enqueue Floating UI (bundled locally for WordPress.org compliance)
        wp_enqueue_script(
            'floating-ui-core',
            INTUCART_PLUGIN_URL . 'assets/js/vendor/floating-ui.core.min.js',
            [],
            '1.7.3',
            true
        );
        
        wp_enqueue_script(
            'floating-ui-dom',
            INTUCART_PLUGIN_URL . 'assets/js/vendor/floating-ui.dom.min.js',
            ['floating-ui-core'],
            '1.7.3',
            true
        );

        // Enqueue popovers system (depends on Floating UI UMD)
        wp_enqueue_script(
            'intucart-popovers',
            INTUCART_PLUGIN_URL . 'assets/js/admin/popovers.js',
            ['jquery', 'floating-ui-dom'],
            filemtime(INTUCART_PLUGIN_DIR . 'assets/js/admin/popovers.js'),
            true
        );

        // Enqueue common admin CSS first
        if (!wp_style_is('intucart-admin-common', 'enqueued')) {
            wp_enqueue_style(
                'intucart-admin-common',
                INTUCART_PLUGIN_URL . 'assets/css/admin/common.css',
                [],
                filemtime(INTUCART_PLUGIN_DIR . 'assets/css/admin/common.css')
            );
        }

        // Enqueue component styles
        if (!wp_style_is('intucart-admin-components', 'enqueued')) {
            wp_enqueue_style(
                'intucart-admin-components',
                INTUCART_PLUGIN_URL . 'assets/css/admin/components.css',
                ['intucart-admin-common'],
                filemtime(INTUCART_PLUGIN_DIR . 'assets/css/admin/components.css')
            );
        }

        // Enqueue status CSS file
        wp_enqueue_style(
            'intucart-status',
            INTUCART_PLUGIN_URL . 'assets/css/admin/status.css',
            ['intucart-admin-common'],
            filemtime(INTUCART_PLUGIN_DIR . 'assets/css/admin/status.css')
        );

        // Localize the script with necessary data
        wp_localize_script(
            'intucart-status',
            'intucartStatus',
            [
                'ajaxUrl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('intucart_status_nonce'),
                'cacheNonce' => wp_create_nonce('intucart_cache_nonce')
            ]
        );

        echo '<div class="wrap">';
        
        // Page title and description before tabs
        echo '<h1>';
        echo '<span class="intucart-page-header-icon">';
        echo '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path opacity=".4" d="M320.21 243.85l-49.7 99.4a16 16 0 0 1-28.9-.6l-56.9-126.3-30 71.7h-94.1l182.5 186.5a17.82 17.82 0 0 0 25.7 0l182.61-186.5H342.31zM473.71 74l-2.4-2.5a131 131 0 0 0-187.39 0L256 100.05l-27.9-28.5a130.83 130.83 0 0 0-187.4 0L38.31 74c-48.7 49.8-50.8 129.1-7.3 182.1h102.41l35.89-86.2a16 16 0 0 1 29.4-.4l58.21 129.3 49-97.9a16 16 0 0 1 28.59 0l27.6 55.2H481c43.51-53.05 41.42-132.35-7.29-182.1z"/><path d="M451 288l-.1.05H342.31l-22.1-44.2-49.7 99.4a16 16 0 0 1-28.9-.6l-56.9-126.3-30 71.7H61V288l-29.95-31.95h102.37l35.89-86.2a16 16 0 0 1 29.4-.4l58.21 129.3 49-97.9a16 16 0 0 1 28.59 0l27.6 55.2H481z"/></svg>';
        echo '</span>';
        echo esc_html(get_admin_page_title());
        echo '</h1>';
        echo '<p class="intucart-page-description">' . sprintf(esc_html__('Monitor your %s system health, usage statistics, and background processes.', 'intufind'), Constants::PLUGIN_NAME) . '</p>';

        // Render tabbed interface
        $this->renderTabbedInterface();

        echo '</div>'; // Close wrap
    }

    /**
     * Render the tabbed interface
     *
     * @return void
     */
    private function renderTabbedInterface(): void
    {
        $current_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'plugin';

        // Check if API key is configured to determine if Usage tab should be shown
        $license_is_valid = $this->aiClientManager->hasApiKey();

        // Check if MCP has any domains available (e.g., WooCommerce orders)
        $mcp_status = $this->mcpServer->getStatus();
        $has_mcp_domains = !empty($mcp_status['domains']);

        // If usage tab is not available and user tries to access it, redirect to plugin tab
        if (!$license_is_valid && $current_tab === 'usage') {
            $current_tab = 'plugin';
        }

        // If MCP tab is not available and user tries to access it, redirect to plugin tab
        if (!$has_mcp_domains && $current_tab === 'mcp') {
            $current_tab = 'plugin';
        }

        // Configure tabs
        $tabs = [
            'plugin' => [
                'url' => '?page=intucart-status&tab=plugin',
                'label' => __('Plugin', 'intufind'),
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 640 512"><path opacity=".4" d="M635.3 177.9l-34.2 34c-6.2 6.1-16 6.2-22.4.4-146-133.7-371.3-133.7-517.2 0-6.4 5.9-16.2 5.7-22.4-.4l-34.2-34-.5-.5c-6-6.4-5.6-16.5.9-22.5C182.2-8.9 457.7-9 634.9 154.9c.2.2.4.3.5.5 6.2 6.3 6.1 16.3-.1 22.5z"/><path d="M320 352c-35.3 0-64 28.7-64 64s28.7 64 64 64 64-28.7 64-64-28.7-64-64-64zm203.5-82.8l-.8-.8c-115.3-101.9-290.2-101.8-405.3 0-6.5 5.8-7.1 15.8-1.4 22.3.3.3.5.6.8.8l34.4 34c6 5.9 15.6 6.3 22 .8 84-72.6 209.7-72.4 293.5 0 6.4 5.5 16 5.2 22-.8l34.4-34c6.4-6 6.5-16 .4-22.3z"/></svg>'
            ],
            'index' => [
                'url' => '?page=intucart-status&tab=index',
                'label' => __('Index & Sync', 'intufind'),
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path opacity=".4" d="M8 454.06V320a24 24 0 0 1 24-24h134.06c21.38 0 32.09 25.85 17 41l-41.75 41.75A166.82 166.82 0 0 0 256.16 424c77.41-.07 144.31-53.14 162.78-126.85a12 12 0 0 1 11.65-9.15h57.31a12 12 0 0 1 11.81 14.18C478.07 417.08 377.19 504 256 504a247.14 247.14 0 0 1-171.31-68.69L49 471c-15.15 15.15-41 4.44-41-16.94z"/><path d="M12.3 209.82C33.93 94.92 134.81 8 256 8a247.14 247.14 0 0 1 171.31 68.69L463 41c15.12-15.12 41-4.41 41 17v134a24 24 0 0 1-24 24H345.94c-21.38 0-32.09-25.85-17-41l41.75-41.75A166.8 166.8 0 0 0 255.85 88c-77.46.07-144.33 53.18-162.79 126.85A12 12 0 0 1 81.41 224H24.1a12 12 0 0 1-11.8-14.18z"/></svg>'
            ],
            'usage' => [
                'url' => '?page=intucart-status&tab=usage',
                'label' => __('Usage', 'intufind'),
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 576 512"><path opacity=".4" d="M288 32C128.94 32 0 160.94 0 320a286.5 286.5 0 0 0 39.06 144.8c5.61 9.62 16.3 15.2 27.44 15.2h443c11.14 0 21.83-5.58 27.44-15.2A286.5 286.5 0 0 0 576 320c0-159.06-128.94-288-288-288zm0 64c14.71 0 26.58 10.13 30.32 23.65-1.11 2.26-2.64 4.23-3.45 6.67L305.65 154c-5.13 3.49-11 6-17.64 6a32 32 0 0 1 0-64zM96 384a32 32 0 1 1 32-32 32 32 0 0 1-32 32zm48-160a32 32 0 1 1 32-32 32 32 0 0 1-32 32zm246.77-72.41l-61.33 184A62.8 62.8 0 0 1 343.12 416H232.88a63.34 63.34 0 0 1 51-95.59l61.34-184a24 24 0 0 1 45.53 15.19zm14.66 57.2L421 162.24a31.54 31.54 0 0 1 11-2.24 32 32 0 0 1 0 64c-11.38 0-20.89-6.27-26.57-15.21zM480 384a32 32 0 1 1 32-32 32 32 0 0 1-32 32z"/><path d="M283.9 320.41l61.34-184a24 24 0 0 1 45.53 15.19l-61.33 184a62.8 62.8 0 0 1 13.68 80.4H232.88a63.34 63.34 0 0 1 51-95.59z"/></svg>',
                'condition' => $license_is_valid
            ],
            'cache' => [
                'url' => '?page=intucart-status&tab=cache',
                'label' => __('Cache', 'intufind'),
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 320 512"><path opacity=".4" d="M224 24.19a23.89 23.89 0 0 1-.79 6L180.61 160l-37.9 128H24a24 24 0 0 1-24-24 24.45 24.45 0 0 1 .21-3.2l32-240A24 24 0 0 1 56 0h144a24.09 24.09 0 0 1 24 24.19z"/><path d="M319.93 184.07a23.93 23.93 0 0 1-3.22 11.93l-176 304a24 24 0 0 1-44.1-17.5l46.1-194.5 37.9-128H296a24 24 0 0 1 23.93 24.07z"/></svg>'
            ],
            'mcp' => [
                'url' => '?page=intucart-status&tab=mcp',
                'label' => __('MCP', 'intufind'),
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path opacity=".4" d="M432 120a24 24 0 1 0-24-24 24 24 0 0 0 24 24zm0 272a24 24 0 1 0 24 24 24 24 0 0 0-24-24zm48-200H32a32 32 0 0 0-32 32v64a32 32 0 0 0 32 32h448a32 32 0 0 0 32-32v-64a32 32 0 0 0-32-32zm-112 88a24 24 0 1 1 24-24 24 24 0 0 1-24 24zm64 0a24 24 0 1 1 24-24 24 24 0 0 1-24 24z"/><path d="M456 256a24 24 0 1 0-24 24 24 24 0 0 0 24-24zm24-224H32A32 32 0 0 0 0 64v64a32 32 0 0 0 32 32h448a32 32 0 0 0 32-32V64a32 32 0 0 0-32-32zm-112 88a24 24 0 1 1 24-24 24 24 0 0 1-24 24zm64 0a24 24 0 1 1 24-24 24 24 0 0 1-24 24zm48 232H32a32 32 0 0 0-32 32v64a32 32 0 0 0 32 32h448a32 32 0 0 0 32-32v-64a32 32 0 0 0-32-32zm-112 88a24 24 0 1 1 24-24 24 24 0 0 1-24 24zm64 0a24 24 0 1 1 24-24 24 24 0 0 1-24 24z"/></svg>',
                'condition' => $has_mcp_domains
            ]
        ];

        // Render tabs
        Tabs::renderOuter($tabs, $current_tab);
        Tabs::openContent();

        switch ($current_tab) {
            case 'plugin':
                $this->renderPluginStatusTab();
                break;
            case 'index':
                $this->renderIndexStatusTab();
                break;
            case 'usage':
                // Only render usage tab if license is valid
                if ($license_is_valid) {
                    $this->renderUsageStatusTab();
                } else {
                    $this->renderPluginStatusTab();
                }
                break;
            case 'cache':
                $this->renderCacheStatusTab();
                break;
            case 'mcp':
                $this->renderMcpStatusTab();
                break;
            default:
                $this->renderPluginStatusTab();
                break;
        }

        Tabs::closeContent();
        Tabs::closeOuter();
    }

    /**
     * Render plugin status tab
     *
     * @return void
     */
    private function renderPluginStatusTab(): void
    {
        echo '<div class="intucart-header-section">';
        echo '<h2 class="intucart-main-header">' . esc_html__('Plugin Status', 'intufind') . '</h2>';
        echo '<p class="intucart-main-description">' . esc_html__('View plugin version, license status, and system connectivity information.', 'intufind') . '</p>';
        echo '</div>';

        // System status overview
        $this->renderSystemStatusOverview();

        // Recent errors (only show in debug mode)
        if (defined('WP_DEBUG') && WP_DEBUG) {
            $this->renderRecentErrors();
        }
    }

    /**
     * Render index status tab
     *
     * @return void
     */
    private function renderIndexStatusTab(): void
    {
        echo '<div class="intucart-header-section">';
        echo '<h2 class="intucart-main-header">' . esc_html__('Index & Sync', 'intufind') . '</h2>';
        echo '<p class="intucart-main-description">' . esc_html__('Monitor content synchronization status, index health, and scheduled background tasks.', 'intufind') . '</p>';
        echo '</div>';

        // Index & Sync Status - get fresh data without caching for real-time counts
        $health = $this->getSystemHealth(true);
        $this->renderIndexAndSyncStatusTable($health);

        // Show scheduled cron jobs for debugging
        $this->renderScheduledCronJobs();
    }

    /**
     * Render usage status tab
     *
     * @return void
     */
    private function renderUsageStatusTab(): void
    {
        echo '<div class="intucart-header-section">';
        echo '<h2 class="intucart-main-header">' . esc_html__('Usage & Limits', 'intufind') . '</h2>';
        echo '<p class="intucart-main-description">' . esc_html__('Track your feature usage against tier limits and monitor quota consumption.', 'intufind') . '</p>';
        echo '</div>';

        // Usage overview (combined with feature status)
        $this->renderUsageOverview();
    }

    /**
     * Render cache status tab
     *
     * @return void
     */
    private function renderCacheStatusTab(): void
    {
        echo '<div class="intucart-header-section">';
        echo '<h2 class="intucart-main-header">' . esc_html__('Cache Management', 'intufind') . '</h2>';
        echo '<p class="intucart-main-description">' . esc_html__('Monitor cache performance, hit rates, and manage cached data across different cache groups.', 'intufind') . '</p>';
        echo '</div>';

        // Cache statistics
        $this->renderCacheStats();
    }

    /**
     * Render MCP status tab
     *
     * @return void
     */
    private function renderMcpStatusTab(): void
    {
        echo '<div class="intucart-header-section">';
        echo '<h2 class="intucart-main-header">' . esc_html__('MCP Server', 'intufind') . '</h2>';
        echo '<p class="intucart-main-description">' . esc_html__('Monitor Model Context Protocol server status, available endpoints, and connection tests.', 'intufind') . '</p>';
        echo '</div>';

        $this->renderMcpServerStatus();
        $this->renderMcpEndpoints();
        $this->renderMcpConnectionTests();
    }

    /**
     * Render MCP server status section
     *
     * @return void
     */
    private function renderMcpServerStatus(): void
    {
        $status = $this->mcpServer->getStatus();

        echo '<div class="intucart-content-panel" style="margin-bottom: 24px;">';
        echo '<h3 class="intucart-panel-header">';
        echo '<span class="dashicons dashicons-admin-network"></span>';
        echo esc_html__('MCP Server Status', 'intufind');
        echo '</h3>';

        if ($status['status'] === 'active') {
            echo '<div class="intucart-notice intucart-notice-success"><p><strong>✓ ' . esc_html__('MCP Server is Active', 'intufind') . '</strong></p></div>';
        } else {
            echo '<div class="intucart-notice intucart-notice-error"><p><strong>✗ ' . esc_html__('MCP Server is Not Active', 'intufind') . '</strong></p></div>';
        }

        echo '<table class="widefat fixed striped">';
        echo '<tbody>';
        echo '<tr><th>' . esc_html__('Status', 'intufind') . '</th><td>' . esc_html($status['status']) . '</td></tr>';
        echo '<tr><th>' . esc_html__('Message', 'intufind') . '</th><td>' . esc_html($status['message']) . '</td></tr>';

        // Show available domains
        if (isset($status['domains']) && !empty($status['domains'])) {
            echo '<tr><th>' . esc_html__('Available Domains', 'intufind') . '</th><td>' . esc_html(implode(', ', $status['domains'])) . '</td></tr>';
        }

        echo '</tbody>';
        echo '</table>';
        echo '</div>'; // .intucart-content-panel
    }

    /**
     * Render MCP endpoints section
     *
     * @return void
     */
    private function renderMcpEndpoints(): void
    {
        $endpoints = $this->mcpServer->getEndpoints();

        echo '<div class="intucart-content-panel" style="margin-bottom: 24px;">';
        echo '<h3 class="intucart-panel-header">';
        echo '<span class="dashicons dashicons-rest-api"></span>';
        echo esc_html__('Available MCP Endpoints', 'intufind');
        echo '</h3>';

        echo '<table class="widefat fixed striped">';
        echo '<thead>';
        echo '<tr><th>' . esc_html__('Endpoint', 'intufind') . '</th><th>' . esc_html__('URL', 'intufind') . '</th></tr>';
        echo '</thead>';
        echo '<tbody>';

        foreach ($endpoints as $key => $url) {
            if (is_array($url)) {
                foreach ($url as $subKey => $subUrl) {
                    echo '<tr>';
                    echo '<td>' . esc_html($subKey) . '</td>';
                    echo '<td><code>' . esc_html($subUrl) . '</code></td>';
                    echo '</tr>';
                }
            } else {
                echo '<tr>';
                echo '<td>' . esc_html($key) . '</td>';
                echo '<td><code>' . esc_html($url) . '</code></td>';
                echo '</tr>';
            }
        }

        echo '</tbody>';
        echo '</table>';
        echo '</div>'; // .intucart-content-panel
    }

    /**
     * Render MCP connection tests section
     *
     * @return void
     */
    private function renderMcpConnectionTests(): void
    {
        $status = $this->mcpServer->getStatus();

        echo '<div class="intucart-content-panel">';
        echo '<h3 class="intucart-panel-header">';
        echo '<span class="dashicons dashicons-admin-tools"></span>';
        echo esc_html__('MCP Connection Tests', 'intufind');
        echo '</h3>';

        if ($status['status'] !== 'active') {
            echo '<p>' . esc_html__('MCP server must be active to run connection tests.', 'intufind') . '</p>';
            echo '</div>'; // .intucart-content-panel
            return;
        }

        // Test domain-based connections (new architecture)
        $domains = $status['domains'] ?? [];
        if (!empty($domains)) {
            echo '<h4 style="margin-top: 16px;">' . esc_html__('Domain Endpoints', 'intufind') . '</h4>';
            echo '<table class="widefat fixed striped">';
            echo '<thead>';
            echo '<tr><th>' . esc_html__('Domain', 'intufind') . '</th><th>' . esc_html__('Status', 'intufind') . '</th><th>' . esc_html__('Tools Available', 'intufind') . '</th><th>' . esc_html__('Tool Names', 'intufind') . '</th></tr>';
            echo '</thead>';
            echo '<tbody>';

            foreach ($domains as $domain) {
                $testResult = $this->mcpServer->testDomainConnection($domain);

                echo '<tr>';
                echo '<td>' . esc_html($domain) . '</td>';

                if ($testResult['success']) {
                    echo '<td><span style="color: green;">✓ ' . esc_html__('Connected', 'intufind') . '</span></td>';
                    echo '<td>' . esc_html($testResult['tools_available']) . ' ' . esc_html__('tools', 'intufind') . '</td>';
                    echo '<td><small>' . esc_html(implode(', ', $testResult['tools'] ?? [])) . '</small></td>';
                } else {
                    echo '<td><span style="color: red;">✗ ' . esc_html__('Failed', 'intufind') . '</span></td>';
                    echo '<td colspan="2">' . esc_html($testResult['error'] ?? __('Unknown error', 'intufind')) . '</td>';
                }

                echo '</tr>';
            }

            echo '</tbody>';
            echo '</table>';
        }

        echo '</div>'; // .intucart-content-panel
    }

    /**
     * Render system status overview
     *
     * @return void
     */
    private function renderSystemStatusOverview(): void
    {
        echo '<div class="intucart-content-panel" style="margin-bottom: 24px;">';
        echo '<h3 class="intucart-panel-header">';
        echo '<span class="dashicons dashicons-admin-plugins"></span>';
        echo esc_html__('Plugin Status', 'intufind');
        echo '</h3>';

        // Get plugin version
        $plugin_version = defined('INTUCART_VERSION') ? INTUCART_VERSION : __('Unknown', 'intufind');

        // Get system health and license validation (consistent with Admin.php approach)
        $license_key = get_option(Constants::LICENSE_KEY_OPTION, '');
        $license_status = __('Not Configured', 'intufind');
        $license_tier = __('N/A', 'intufind');
        $license_class = 'status-neutral';
        $license_is_valid = false;

        if (!empty($license_key) && $this->aiClientManager->hasApiKey()) {
            // Actually validate against cloud (consistent with Settings page)
            $validation = $this->license->validateLicense();
            $license_is_valid = $validation['valid'] ?? false;
            
            if ($license_is_valid) {
                $license_status = __('Active', 'intufind');
                $license_class = 'status-ok';
            } else {
                $license_status = __('Invalid', 'intufind');
                $license_class = 'status-error';
                // Clear stale caches for invalid key
                $this->usageService->invalidateUsageCache();
                $this->license->invalidateTierCache();
                $this->cacheService->delete('system_health_check', 'intucart_system');
                $this->systemHealthCache = null;
            }
        } elseif (!empty($license_key)) {
            $license_status = __('Invalid', 'intufind');
            $license_class = 'status-error';
            // Clear caches for invalid key
            $this->usageService->invalidateUsageCache();
            $this->license->invalidateTierCache();
            $this->cacheService->delete('system_health_check', 'intucart_system');
            $this->systemHealthCache = null;
        }

        // Get system health data
        $health = $this->getSystemHealth(!$license_is_valid); // Force fresh if license invalid

        // Extract tier information from health data if available
        if (isset($health['license']['tier'])) {
            $license_tier = ucfirst($health['license']['tier']);
        } elseif ($license_is_valid && $this->license !== null) {
            $local_tier = $this->license->getUserTier();
            $license_tier = ucfirst($local_tier);
        }

        // Format license tier with appropriate styling
        $tier_class = 'status-neutral';
        if (in_array($license_tier, ['Starter', 'Professional', 'Enterprise'], true)) {
            $tier_class = 'status-ok';
        }

        // Determine system status based on license state and health data
        $system_status = __('Offline', 'intufind');
        $system_class = 'status-error';
        $system_details = '';

        if (empty($license_key)) {
            $system_status = __('Not Configured', 'intufind');
            $system_class = 'status-neutral';
        } elseif (!$license_is_valid) {
            $system_status = __('License Inactive', 'intufind');
            $system_class = 'status-error';
        } else {
            // License is valid, check cloud connectivity
            switch ($health['status']) {
                case 'online':
                    $system_status = __('Online', 'intufind');
                    $system_class = 'status-ok';
                    break;
                case 'timeout':
                    $system_status = __('Connection Timeout', 'intufind');
                    $system_class = 'status-error';
                    break;
                case 'network_error':
                    $system_status = __('Network Error', 'intufind');
                    $system_class = 'status-error';
                    break;
                default:
                    $system_status = __('Connection Issues', 'intufind');
                    $system_class = 'status-error';
                    break;
            }
        }

        // Display plugin info
        echo '<div class="intucart-system-info">';
        echo '<table class="widefat">';
        echo '<tbody>';

        // Plugin info
        echo '<tr><th>' . esc_html__('Plugin Version', 'intufind') . '</th><td><div class="plugin-version">'
            . esc_html($plugin_version) . '</div></td></tr>';
        echo '<tr><th>' . esc_html__('System Status', 'intufind') . '</th><td><div class="status-indicator '
            . esc_attr($system_class) . '">' . esc_html($system_status) . esc_html($system_details) . '</div></td></tr>';
        echo '<tr><th>' . esc_html__('License Status', 'intufind') . '</th><td><div class="status-indicator '
            . esc_attr($license_class) . '">' . esc_html($license_status) . '</div></td></tr>';
        echo '<tr><th>' . esc_html__('License Tier', 'intufind') . '</th><td><div class="status-indicator '
            . esc_attr($tier_class) . '">' . esc_html($license_tier) . '</div></td></tr>';

        echo '</tbody>';
        echo '</table>';
        echo '</div>';

        echo '</div>'; // .intucart-content-panel
    }

    /**
     * Get comprehensive system status with real API connectivity test
     *
     * @param bool $skipCache Whether to skip reading from persistent cache and force fresh API call (always writes to cache) (default: false)
     * @return array Array with status, message, and details
     */
    private function getSystemHealth(bool $skipCache = false): array
    {
        // Always use 5-minute cache duration for writing
        $cacheSeconds = 300;

        // Always check class-level cache first to prevent multiple API calls during same request
        if ($this->systemHealthCache !== null) {
            return $this->systemHealthCache;
        }

        // Check persistent cache only if not skipping cache
        if (!$skipCache) {
            $cache_key = 'system_health_check';
            $cached_result = $this->cacheService->get($cache_key, 'intucart_system');

            // Ensure we handle both null and false from cache
            if ($cached_result !== null && $cached_result !== false && is_array($cached_result)) {
                $this->systemHealthCache = $cached_result;
                return $cached_result;
            }
        }

        $health = [
            'status' => 'error',
            'message' => 'Unknown',
            'details' => []
        ];

        // Check basic configuration first
        $license_key = get_option(Constants::LICENSE_KEY_OPTION, '');

        if (empty($license_key)) {
            $health['status'] = 'not_configured';
            $health['message'] = 'Not Configured';
            $health['details']['issue'] = 'No license key configured';

            // Always cache result for future requests
            $cache_key = 'system_health_check';
            $this->cacheService->set($cache_key, $health, 'intucart_system', $cacheSeconds);
            $this->systemHealthCache = $health;
            return $health;
        }

        // Test actual API connectivity using AI SDK
        try {
            $tenantService = $this->aiClientManager->getClient()->tenant();
            
            // Get subscription status (lightweight call)
            $statusResult = $tenantService->getStatus();
            
            // Get usage data with indexes (heavier call, but needed for display)
            $usageResult = $tenantService->getUsage();
            
            $isValid = $statusResult['subscription']['valid'] ?? false;
            $tenantStatus = $statusResult['status'] ?? 'unknown';
            
            if ($isValid && $tenantStatus === 'active') {
                $health['status'] = 'online';
                $health['message'] = 'Online';
                $health['details']['api_endpoint'] = $this->aiClientManager->getClient()->getConfig()->getApiEndpoint();
                $health['details']['license_valid'] = 'Yes';
                
                // Map new API response to expected format
                $health['license'] = [
                    'tier' => $statusResult['subscription']['tier'] ?? null,
                    'valid' => $isValid,
                    'expires' => $statusResult['subscription']['expires'] ?? null,
                ];
                $health['indexes'] = $usageResult['indexes'] ?? null;
                $health['data_summary'] = $usageResult['data_summary'] ?? null;

                // Extract and cache license information
                $tier = $statusResult['subscription']['tier'] ?? null;
                if ($tier) {
                    $this->cacheService->set('license_tier', $tier, 'intucart_license', 86400);
                    $health['details']['license_tier'] = ucfirst($tier);
                }

                // Cache the license validity
                $license_cache_key = 'license_valid_' . md5($license_key);
                $this->cacheService->set($license_cache_key, $isValid, 'intucart_license', 86400);

                // Debug logging for development
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    error_log('[Intucart Status] Cached license info from status check: tier=' . ($tier ?? 'unknown') . ', valid=' . ($isValid ? 'true' : 'false'));
                }

                // Always cache successful result for future requests
                $cache_key = 'system_health_check';
                $this->cacheService->set($cache_key, $health, 'intucart_system', $cacheSeconds);
                $this->systemHealthCache = $health;
            } else {
                // Status check returned but subscription is not valid
                $health['status'] = 'api_error';
                $health['message'] = __('Service Unavailable', 'intufind');
                $validationError = $statusResult['subscription']['validation']['error'] ?? null;
                $health['details']['issue'] = $validationError ?? __('Subscription validation failed', 'intufind');

                // Always cache failed result for shorter duration
                $cache_key = 'system_health_check';
                $failureCacheDuration = min($cacheSeconds, 120);
                $this->cacheService->set($cache_key, $health, 'intucart_system', $failureCacheDuration);
                $this->systemHealthCache = $health;
            }
        } catch (\Exception $e) {
            $health['status'] = 'error';
            $health['message'] = __('System Error', 'intufind');
            $health['details']['issue'] = sprintf(__('Unexpected error: %s', 'intufind'), $e->getMessage());

            $this->logger->error('System health check failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            // Always cache error for shorter duration
            $cache_key = 'system_health_check';
            $errorCacheDuration = min($cacheSeconds, 60);
            $this->cacheService->set($cache_key, $health, 'intucart_system', $errorCacheDuration);
            $this->systemHealthCache = $health;
        }

        return $health;
    }



    /**
     * Render cache statistics
     *
     * @return void
     */
    private function renderCacheStats(): void
    {
        echo '<div class="intucart-cache-stats intucart-content-panel">';
        echo '<h3 class="intucart-panel-header">';
        echo '<span class="dashicons dashicons-performance"></span>';
        echo esc_html__('Cache Statistics', 'intufind');
        echo '</h3>';

        // Get cache statistics from options
        $cache_hits = get_option(Constants::CACHE_HITS_OPTION, []);
        $cache_misses = get_option(Constants::CACHE_MISSES_OPTION, []);

        // Remove the license cache group if it exists
        if (isset($cache_hits['intucart_license'])) {
            unset($cache_hits['intucart_license']);
        }
        if (isset($cache_misses['intucart_license'])) {
            unset($cache_misses['intucart_license']);
        }

        $total_hits = array_sum($cache_hits);
        $total_misses = array_sum($cache_misses);
        $total_requests = $total_hits + $total_misses;
        $hit_rate = $total_requests > 0 ? round(($total_hits / $total_requests) * 100, 2) : 0;

        // Display cache statistics by group
        echo '<div class="cache-stats-by-group">';

        if (empty($cache_hits) && empty($cache_misses)) {
            echo '<div class="intucart-notice intucart-notice-info">';
            echo '<p>' . esc_html__('No cache statistics available yet. Statistics will appear here once the cache is used.', 'intufind') . '</p>';
            echo '</div>';
        } else {
            echo '<table class="wp-list-table widefat fixed striped">';
            echo '<thead><tr>';
            echo '<th>' . esc_html__('Cache Group', 'intufind') . '</th>';
            echo '<th>' . esc_html__('Hits', 'intufind') . '</th>';
            echo '<th>' . esc_html__('Misses', 'intufind') . '</th>';
            echo '<th>' . esc_html__('Hit Rate', 'intufind') . '</th>';
            echo '<th>' . esc_html__('Actions', 'intufind') . '</th>';
            echo '</tr></thead>';
            echo '<tbody>';

            // Add Total row at the top
            echo '<tr style="font-weight: bold; background-color: #f9f9f9;">';
            echo '<td>' . esc_html__('Total', 'intufind') . '</td>';
            echo '<td>' . number_format($total_hits) . '</td>';
            echo '<td>' . number_format($total_misses) . '</td>';
            echo '<td class="' . ($hit_rate < 40 ? 'cache-hit-rate-warning' : 'cache-hit-rate-good') . '">' . esc_html($hit_rate) . '%</td>';
            echo '<td>—</td>';
            echo '</tr>';

            // Combine all cache groups from both hits and misses
            $all_groups = array_unique(array_merge(array_keys($cache_hits), array_keys($cache_misses)));
            sort($all_groups);

            foreach ($all_groups as $group) {
                // Skip the license cache group
                if ($group === 'intucart_license') {
                    continue;
                }

                $group_hits = $cache_hits[$group] ?? 0;
                $group_misses = $cache_misses[$group] ?? 0;
                $group_total = $group_hits + $group_misses;
                $group_hit_rate = $group_total > 0 ? round(($group_hits / $group_total) * 100, 2) : 0;

                // Display a more readable group name by removing the prefix
                $display_name = str_replace('intucart_', '', $group);

                // Format specific cases for better readability
                if ($display_name === 'product_associations') {
                    $display_name = __('Product Associations', 'intufind');
                } else {
                    // For other cases, just capitalize the first letter
                    $display_name = ucfirst($display_name);
                }

                echo '<tr>';
                echo '<td>' . esc_html($display_name) . '</td>';
                echo '<td>' . number_format($group_hits) . '</td>';
                echo '<td>' . number_format($group_misses) . '</td>';
                echo '<td>' . esc_html($group_hit_rate) . '%</td>';
                echo '<td>';
                Button::secondary(__('Clear', 'intufind'), [
                    'size' => 'small',
                    'attributes' => [
                        'class' => 'clear-cache-group',
                        'data-group' => esc_attr($group)
                    ],
                    'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path opacity=".4" d="M53.2 467L32 96h384l-21.2 371a48 48 0 0 1-47.9 45H101.1a48 48 0 0 1-47.9-45z"/><path d="M0 80V48a16 16 0 0 1 16-16h120l9.4-18.7A23.72 23.72 0 0 1 166.8 0h114.3a24 24 0 0 1 21.5 13.3L312 32h120a16 16 0 0 1 16 16v32a16 16 0 0 1-16 16H16A16 16 0 0 1 0 80z"/></svg>'
                ]);
                echo '</td>';
                echo '</tr>';
            }

            echo '</tbody>';
            echo '</table>';
        }

        echo '</div>'; // .cache-stats-by-group

        // Add a button to clear all cache
        echo '<div class="cache-actions">';
        Button::danger(__('Clear All Cache', 'intufind'), [
            'attributes' => [
                'class' => 'clear-all-cache'
            ],
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path opacity=".4" d="M53.2 467L32 96h384l-21.2 371a48 48 0 0 1-47.9 45H101.1a48 48 0 0 1-47.9-45z"/><path d="M0 80V48a16 16 0 0 1 16-16h120l9.4-18.7A23.72 23.72 0 0 1 166.8 0h114.3a24 24 0 0 1 21.5 13.3L312 32h120a16 16 0 0 1 16 16v32a16 16 0 0 1-16 16H16A16 16 0 0 1 0 80z"/></svg>'
        ]);
        Button::secondary(__('Reset Statistics', 'intufind'), [
            'attributes' => [
                'class' => 'reset-cache-stats'
            ],
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path opacity=".4" d="M0 500V299.67a12 12 0 0 1 12-12h200.33a12 12 0 0 1 12 12v47.41a12 12 0 0 1-12.57 12l-101.87-4.88a176.07 176.07 0 0 0 317.25-56.94 12 12 0 0 1 11.67-9.26h49.09a12 12 0 0 1 11.8 14.18C478.07 417.08 377.19 504 256 504a247.43 247.43 0 0 1-188.76-87.17l4.13 82.57a12 12 0 0 1-12 12.6H12a12 12 0 0 1-12-12z"/><path d="M12.3 209.82C33.93 94.92 134.81 8 256 8a247.4 247.4 0 0 1 188.9 87.34l-4-82.77A12 12 0 0 1 452.92 0h47.41a12 12 0 0 1 12 12v200.33a12 12 0 0 1-12 12H300a12 12 0 0 1-12-12v-47.41a12 12 0 0 1 12.57-12l101.53 4.88a176.07 176.07 0 0 0-317.24 56.94A12 12 0 0 1 73.19 224H24.1a12 12 0 0 1-11.8-14.18z"/></svg>'
        ]);
        echo '</div>';

        echo '</div>'; // .intucart-cache-stats .intucart-content-panel
    }

    /**
     * Render recent errors
     *
     * @return void
     */
    private function renderRecentErrors(): void
    {
        echo '<div class="intucart-content-panel">';
        echo '<h3 class="intucart-panel-header">';
        echo '<span class="dashicons dashicons-warning"></span>';
        echo esc_html__('Recent Errors', 'intufind');
        echo '</h3>';
        echo '<div class="intucart-recent-errors">';

        global $wpdb;
        $table_name = $wpdb->prefix . 'intucart_logs';

        // Check if the logs table exists
        $table_exists = $wpdb->get_var($wpdb->prepare('SHOW TABLES LIKE %s', $table_name)) === $table_name;

        if (!$table_exists) {
            echo '<div class="intucart-notice intucart-notice-warning">';
            echo '<p>' . esc_html__('Logs table does not exist. Please deactivate and reactivate the plugin to create it.', 'intufind') . '</p>';
            echo '</div>';
            echo '</div>'; // .intucart-recent-errors
            echo '</div>'; // .intucart-content-panel
            return;
        }

        // Get recent errors
        $errors = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM $table_name WHERE level IN ('error', 'warning') ORDER BY created_at DESC LIMIT %d",
                10
            )
        );

        if (empty($errors)) {
            echo '<div class="intucart-notice intucart-notice-success">';
            echo '<p>✓ No recent errors recorded.</p>';
            echo '</div>';
            echo '</div>'; // .intucart-recent-errors
            echo '</div>'; // .intucart-content-panel
            return;
        }

        echo '<div class="error-logs-container">';
        echo '<table class="widefat">';
        echo '<thead>';
        echo '<tr>';
        echo '<th>Time</th>';
        echo '<th>Level</th>';
        echo '<th>Message</th>';
        echo '<th>Context</th>';
        echo '</tr>';
        echo '</thead>';
        echo '<tbody>';

        foreach ($errors as $error) {
            // Ensure context is a string before decoding, default to empty array if null or not a string
            $context_data = is_string($error->context) ? $error->context : '{}';
            $context = json_decode($context_data, true);
            $context_html = '';

            if (is_array($context)) {
                foreach ($context as $key => $value) {
                    // Ensure value is scalar before outputting
                    if (is_array($value) || is_object($value)) {
                        $value = json_encode($value);
                    }
                    $context_html .= '<div><strong>' . esc_html($key) . ':</strong> '
                        . esc_html((string)$value) . '</div>';
                }
            }

            echo '<tr>';
            echo '<td>' . esc_html(date('Y-m-d H:i:s', strtotime($error->created_at))) . '</td>';
            echo '<td><span class="log-level log-level-' . esc_attr($error->level) .
                '">' . esc_html($error->level) . '</span></td>';
            echo '<td>' . esc_html($error->message) . '</td>';
            echo '<td>' . wp_kses_post($context_html) . '</td>';
            echo '</tr>';
        }

        echo '</tbody>';
        echo '</table>';
        echo '</div>'; // .error-logs-container

        // Add a button to clear all logs
        echo '<div class="log-actions">';
        echo '<button class="button button-primary clear-all-logs">Clear All Logs</button>';
        echo '</div>';

        echo '</div>'; // .intucart-recent-errors
        echo '</div>'; // .intucart-content-panel
    }

    /**
     * Handle AJAX request to clear all logs
     *
     * @return void
     */
    public function handleAjaxClearLogs(): void
    {
        // Check nonce for security
        check_ajax_referer('intucart_status_nonce', 'nonce');

        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('You do not have permission to perform this action.', 'intufind')]);
            return;
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'intucart_logs';

        // Check if the logs table exists
        $table_exists = $wpdb->get_var($wpdb->prepare('SHOW TABLES LIKE %s', $table_name)) === $table_name;

        if (!$table_exists) {
            wp_send_json_error(['message' => __('Logs table does not exist.', 'intufind')]);
            return;
        }

        // Clear all logs
        $result = $wpdb->query("TRUNCATE TABLE $table_name");

        if ($result !== false) {
            wp_send_json_success(['message' => __('All logs have been cleared successfully.', 'intufind')]);
        } else {
            wp_send_json_error(['message' => __('Failed to clear logs.', 'intufind')]);
        }
    }

    /**
     * Get sync status label
     *
     * @param int $lastSync     Last sync timestamp
     * @param int $syncInterval Sync interval in seconds
     *
     * @return string
     */
    private function getSyncStatusLabel(int $lastSync, int $syncInterval): string
    {
        if ($lastSync === 0) {
            return '<span class="status-text-error" style="margin-right: 0;">Never Run</span>';
        }

        $now = time();
        $diff = $now - $lastSync;

        if ($diff > ($syncInterval * 2)) {
            return '<span class="status-text-error" style="margin-right: 0;">Overdue</span>';
        } elseif ($diff > $syncInterval) {
            return '<span class="status-text-warning" style="margin-right: 0;">Due Soon</span>';
        } else {
            return '<span class="status-text-ok" style="margin-right: 0;">Up to Date</span>';
        }
    }

    /**
     * Render index and sync status
     *
     * @param array|null $health_data Tenant data from API response
     * @return void
     */
    private function renderIndexAndSyncStatusTable(?array $health_data): void
    {
        echo '<div class="intucart-content-panel" style="margin-bottom: 24px;">';
        echo '<h3 class="intucart-panel-header">';
        echo '<span class="dashicons dashicons-database-view"></span>';
        echo 'Index & Sync Status';
        echo '</h3>';

        // Check if API key is configured to determine if sync buttons should be enabled
        $license_is_valid = $this->aiClientManager->hasApiKey();

        // Get sync timing information
        $nextPostSync = wp_next_scheduled(Constants::POST_SYNC_HOOK);
        $nextTaxonomySync = wp_next_scheduled(Constants::TAXONOMY_SYNC_HOOK);

        // Get taxonomy sync time (special handling)
        $lastTaxonomySync = get_option(Constants::LAST_TAXONOMY_SYNC_OPTION, 0);

        // Check for more recent individual taxonomy syncs (using syncable post types for consistency)
        $syncablePostTypesForTaxonomyCheck = $this->postTypeManager->getSyncablePostTypes();
        $availableTaxonomiesForSyncCheck = $this->taxonomyManager->getAllAvailableTaxonomyNames($syncablePostTypesForTaxonomyCheck);
        foreach ($availableTaxonomiesForSyncCheck as $taxonomy) {
            $individualSync = get_option("intucart_last_{$taxonomy}_sync", 0);
            if ($individualSync > $lastTaxonomySync) {
                $lastTaxonomySync = $individualSync;
            }
        }
        $syncInterval = get_option(Constants::SYNC_INTERVAL_OPTION, Constants::DEFAULT_SYNC_INTERVAL);

        echo '<table class="widefat">';
        echo '<thead><tr><th>Entity</th><th>Index Status</th><th>Cloud Documents <span class="dashicons dashicons-info-outline" style="font-size: 14px; line-height: inherit; width: 14px; height: 14px; color: #666; cursor: help; vertical-align: text-top;" data-popover="These counts reflect all documents across all sites using the same license key. If you\'re using the same license on multiple sites (staging, production, etc.), you may see higher counts than expected."></span></th><th>Last Sync</th><th>Next Scheduled</th><th>Actions</th></tr></thead>';
        echo '<tbody>';

        // Get all available post types and their enabled status
        $allPostTypes = $this->postTypeManager->getAvailablePostTypes();
        $syncablePostTypes = $this->postTypeManager->getSyncablePostTypes();
        $userPreferences = get_option('intucart_syncable_post_types', []);

        // Get all available taxonomies and their enabled status (filtered by syncable post types)
        $allTaxonomies = $this->taxonomyManager->getAllAvailableTaxonomyNames($syncablePostTypes);
        $syncableTaxonomies = $this->taxonomyManager->getSyncableTaxonomyNames();

        // Group post types logically (including disabled ones)
        $postTypeGroups = $this->groupPostTypes($allPostTypes);

        // TODO: Users sync is temporarily disabled - uncomment when ready to use
        // $entities[] = 'user';

        // Render each group with a header
        foreach ($postTypeGroups as $groupName => $postTypes) {
            // Group header
            echo '<tr class="post-type-group-header" style="background-color: #f1f1f1; font-weight: bold;">';
            echo '<td colspan="6" style="padding: 8px 12px; border-bottom: 1px solid #ddd;">' . esc_html($groupName) . '</td>';
            echo '</tr>';

            // Render each post type in this group
            foreach ($postTypes as $entity_name) {
                $isEnabled = in_array($entity_name, $syncablePostTypes, true);
                $this->renderEntityRow($entity_name, $health_data, $nextPostSync, $nextTaxonomySync, $lastTaxonomySync, $isEnabled, false, $license_is_valid);
            }
        }

        // Add taxonomies section
        if (!empty($allTaxonomies)) {
            // Taxonomies header
            echo '<tr class="post-type-group-header" style="background-color: #f1f1f1; font-weight: bold;">';
            echo '<td colspan="6" style="padding: 8px 12px; border-bottom: 1px solid #ddd;">Taxonomies</td>';
            echo '</tr>';

            // Render each taxonomy
            foreach ($allTaxonomies as $taxonomy) {
                $isEnabled = in_array($taxonomy, $syncableTaxonomies, true);
                $this->renderEntityRow($taxonomy, $health_data, $nextPostSync, $nextTaxonomySync, $lastTaxonomySync, $isEnabled, true, $license_is_valid);
            }
        }

        echo '</tbody>';
        echo '</table>';

        // Add helpful note about license status
        if (!$license_is_valid) {
            echo '<div class="intucart-notice intucart-notice-warning">';
            echo '<p><strong>⚠️ License Required</strong></p>';
            if (empty($license_key)) {
                echo '<p>No license key configured. Sync operations are disabled until you activate a valid license. ';
                echo '<a href="' . esc_url(admin_url('admin.php?page=intucart&tab=license')) . '">Configure License →</a></p>';
            } else {
                echo '<p>Your license is inactive or invalid. Sync operations are disabled until you activate a valid license. ';
                echo '<a href="' . esc_url(admin_url('admin.php?page=intucart&tab=license')) . '">Check License Status →</a></p>';
            }
            echo '</div>';
        }

        // Add helpful note about disabled content types if any exist
        $disabledPostTypesCount = count($allPostTypes) - count($syncablePostTypes);
        $disabledTaxonomiesCount = count($allTaxonomies) - count($syncableTaxonomies);
        $totalDisabledCount = $disabledPostTypesCount + $disabledTaxonomiesCount;
        
        if ($totalDisabledCount > 0) {
            echo '<div class="intucart-notice intucart-notice-info">';
            echo '<p><strong>Note:</strong> ';
            
            if ($disabledPostTypesCount > 0 && $disabledTaxonomiesCount > 0) {
                echo esc_html($disabledPostTypesCount) . ' post type(s) and ' . esc_html($disabledTaxonomiesCount) . ' taxonom(ies) are disabled for syncing.';
            } elseif ($disabledPostTypesCount > 0) {
                echo esc_html($disabledPostTypesCount) . ' post type(s) are disabled for syncing.';
            } else {
                echo esc_html($disabledTaxonomiesCount) . ' taxonom(ies) are disabled for syncing.';
            }
            
            echo ' You can enable them in <a href="' . esc_url(admin_url('admin.php?page=intucart&tab=indexing')) . '">Settings → Indexing → Content Types</a>.</p>';
            echo '</div>';
        }

        // Add sync settings summary
        $syncInterval = get_option(Constants::SYNC_INTERVAL_OPTION, Constants::DEFAULT_SYNC_INTERVAL);
        $syncHour = get_option(Constants::SYNC_HOUR_OPTION, Constants::DEFAULT_SYNC_HOUR);

        echo '<div class="sync-settings-summary">';
        echo '<p><strong>Sync Configuration:</strong> ';
        echo 'Every ' . esc_html((string)($syncInterval / 3600)) . ' hours at ' . esc_html(sprintf('%02d:00', $syncHour));
        echo '</p>';
        echo '</div>';

        echo '</div>'; // .intucart-content-panel
    }

    /**
     * Group post types into logical categories
     *
     * @param array $postTypes Array of post type names
     * @return array Grouped post types
     */
    private function groupPostTypes(array $postTypes): array
    {
        $groups = [
            'Core Content' => [],
            'Custom Content' => [],
            'System' => [] // System group for future system-level entities
        ];

        // Only add E-commerce group if WooCommerce is active AND recommendations are available
        $woocommerce_available = class_exists('WooCommerce') && $this->recommendationsAvailable;
        if ($woocommerce_available) {
            $groups['E-commerce'] = [];
        }

        foreach ($postTypes as $postType) {
            if (in_array($postType, ['post', 'page'], true)) {
                $groups['Core Content'][] = $postType;
            } elseif ($postType === 'product' && $woocommerce_available) {
                // Only include products in E-commerce group if WooCommerce is available
                $groups['E-commerce'][] = $postType;
            } else {
                // All other post types (including products when WooCommerce not available) are custom content
                $groups['Custom Content'][] = $postType;
            }
        }

        // Remove empty groups
        $groups = array_filter($groups, function ($group) {
            return !empty($group);
        });

        return $groups;
    }

    /**
     * Render a single entity row in the status table
     *
     * @param string $entity_name The entity name
     * @param array|null $health_data Health data from API
     * @param int|false $nextPostSync Next post sync timestamp
     * @param int|false $nextTaxonomySync Next taxonomy sync timestamp
     * @param int $lastTaxonomySync Last taxonomy sync timestamp
     * @param bool $isEnabled Whether this entity is enabled for syncing
     * @param bool $isTaxonomy Whether this entity is a taxonomy
     * @param bool $licenseIsValid Whether the license is valid
     * @return void
     */
    private function renderEntityRow(string $entity_name, ?array $health_data, $nextPostSync, $nextTaxonomySync, int $lastTaxonomySync, bool $isEnabled = true, bool $isTaxonomy = false, bool $licenseIsValid = true): void
    {
        // Get index data if available (API returns plural names)
        $index_data = null;
        
        // Determine correct index name based on entity type
        if ($isTaxonomy || $entity_name === 'taxonomy') {
            $plural_name = 'taxonomies'; // All taxonomies use the taxonomies index
        } elseif ($entity_name === 'product') {
            $plural_name = 'products'; // Products have their own index
        } elseif ($entity_name === 'user') {
            $plural_name = 'users'; // Users have their own index
        } else {
            // All post types (post, page, etc.) use the posts index
            $plural_name = 'posts';
        }

        if ($health_data && isset($health_data['indexes'][$plural_name])) {
            $index_data = $health_data['indexes'][$plural_name];
        }

        // Index status - check if post type is enabled first
        if (!$isEnabled) {
            $status_class = 'status-neutral';
            $status_text = 'Disabled';
        } elseif ($index_data) {
            $status_class = $index_data['exists'] ? 'status-ok' : 'status-error';
            $status_text = $index_data['exists'] ? 'Active' : 'Missing';

            // Get entity-specific document count from cloud
            if ($isTaxonomy || $entity_name === 'taxonomy') {
                // Taxonomies have their own index - check for specific taxonomy breakdown
                if (isset($index_data['by_type'][$entity_name])) {
                    // Show specific count for this taxonomy from cloud
                    $cloud_count = number_format($index_data['by_type'][$entity_name]);
                } elseif (isset($index_data['count'])) {
                    // If no by_type breakdown exists, show the total count for all individual taxonomies
                    // This is a fallback when the API doesn't provide per-taxonomy breakdowns
                    if ($entity_name === 'taxonomy') {
                        // For the generic "taxonomy" row, show total count
                        $cloud_count = number_format($index_data['count']);
                    } else {
                        // Check if cloud has breakdown but this taxonomy isn't in it
                        if (isset($index_data['by_type']) && is_array($index_data['by_type'])) {
                            // Cloud has breakdown but this taxonomy isn't in it, so it has 0 terms in cloud
                            $cloud_count = '0';
                        } else {
                            // No breakdown available from cloud, show local term count as fallback
                            $term_count = wp_count_terms(['taxonomy' => $entity_name, 'hide_empty' => false]);
                            if (!is_wp_error($term_count) && $term_count > 0) {
                                $cloud_count = $term_count . ' terms';
                            } else {
                                $cloud_count = '—';
                            }
                        }
                    }
                } else {
                    // No data available
                    $cloud_count = '0';
                }
                $doc_count = $cloud_count;
            } elseif ($entity_name === 'product') {
                // Products have their own index
                $cloud_count = isset($index_data['count']) ? number_format($index_data['count']) : 'N/A';
                $doc_count = $cloud_count;
            } else {
                // All post types (post, page, movie, etc.) share the posts index
                // Check if we have post type breakdown data from cloud (new schema: posts.by_type)
                if (isset($index_data['by_type'][$entity_name])) {
                    // Show specific count for this post type
                    $cloud_count = number_format($index_data['by_type'][$entity_name]);
                } else {
                    // If this post type doesn't exist in the breakdown, it means 0 documents
                    $cloud_count = '0';
                }
                $doc_count = $cloud_count;
            }
        } else {
            $status_class = 'status-neutral';
            $status_text = 'No Data';
            $doc_count = 'N/A';
        }

        // Override document count for disabled post types
        if (!$isEnabled) {
            $doc_count = '—';
        }

        // Determine sync info based on entity type
        $last_sync = '';
        $next_sync = '';
        $sync_button = '';

        if (!$isEnabled) {
            // Disabled entities
            $last_sync = '—';
            $next_sync = '—';
            $sync_button = '<span style="color: #666; font-style: italic;">Disabled</span>';
        } elseif (!$licenseIsValid) {
            // License not valid - show sync times but disable buttons
            if ($isTaxonomy || $entity_name === 'taxonomy') {
                if ($isTaxonomy) {
                    $individualSync = get_option("intucart_last_{$entity_name}_sync", 0);
                    $last_sync = $individualSync ? date('M j, H:i', $individualSync) : 'Never';
                } else {
                    $last_sync = $lastTaxonomySync ? date('M j, H:i', $lastTaxonomySync) : 'Never';
                }
                $next_sync = $nextTaxonomySync ? date('M j, H:i', $nextTaxonomySync) : 'Not Scheduled';
            } else {
                $lastSyncOption = "intucart_last_{$entity_name}_sync";
                $entityLastSync = get_option($lastSyncOption, 0);
                $last_sync = $entityLastSync ? date('M j, H:i', $entityLastSync) : 'Never';
                $next_sync = $nextPostSync ? date('M j, H:i', $nextPostSync) : 'Not Scheduled';
            }
            $sync_button = '<span style="color: #d63638; font-style: italic;" data-popover="License required to sync content. Please activate your license to enable sync operations." title="License required">License Required</span>';
        } elseif ($isTaxonomy || $entity_name === 'taxonomy') {
            // Special handling for taxonomies
            if ($isTaxonomy) {
                // For individual taxonomies, check their specific sync time
                $individualSync = get_option("intucart_last_{$entity_name}_sync", 0);
                $last_sync = $individualSync ? date('M j, H:i', $individualSync) : 'Never';
                
                // Get friendly taxonomy name for the button
                $taxonomyObj = get_taxonomy($entity_name);
                $taxonomyLabel = $taxonomyObj ? ($taxonomyObj->labels->name ?? $entity_name) : $entity_name;
                
                ob_start();
                Button::secondary('Sync', [
                    'size' => 'small',
                    'attributes' => [
                        'class' => 'sync-now-button',
                        'data-type' => 'taxonomies',
                        'data-taxonomy' => esc_attr($entity_name),
                        'data-taxonomy-label' => esc_attr($taxonomyLabel)
                    ],
                    'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path opacity=".4" d="M0 500V299.67a12 12 0 0 1 12-12h200.33a12 12 0 0 1 12 12v47.41a12 12 0 0 1-12.57 12l-101.87-4.88a176.07 176.07 0 0 0 317.25-56.94 12 12 0 0 1 11.67-9.26h49.09a12 12 0 0 1 11.8 14.18C478.07 417.08 377.19 504 256 504a247.43 247.43 0 0 1-188.76-87.17l4.13 82.57a12 12 0 0 1-12 12.6H12a12 12 0 0 1-12-12z"/><path d="M12.3 209.82C33.93 94.92 134.81 8 256 8a247.4 247.4 0 0 1 188.9 87.34l-4-82.77A12 12 0 0 1 452.92 0h47.41a12 12 0 0 1 12 12v200.33a12 12 0 0 1-12 12H300a12 12 0 0 1-12-12v-47.41a12 12 0 0 1 12.57-12l101.53 4.88a176.07 176.07 0 0 0-317.24 56.94A12 12 0 0 1 73.19 224H24.1a12 12 0 0 1-11.8-14.18z"/></svg>'
                ]);
                $sync_button = ob_get_clean();
            } else {
                // For the general "taxonomy" entity, use global taxonomy sync time
                $last_sync = $lastTaxonomySync ? date('M j, H:i', $lastTaxonomySync) : 'Never';
                ob_start();
                Button::secondary('Sync Taxonomies', [
                    'size' => 'small',
                    'attributes' => [
                        'class' => 'sync-now-button',
                        'data-type' => 'taxonomies'
                    ],
                    'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path opacity=".4" d="M0 500V299.67a12 12 0 0 1 12-12h200.33a12 12 0 0 1 12 12v47.41a12 12 0 0 1-12.57 12l-101.87-4.88a176.07 176.07 0 0 0 317.25-56.94 12 12 0 0 1 11.67-9.26h49.09a12 12 0 0 1 11.8 14.18C478.07 417.08 377.19 504 256 504a247.43 247.43 0 0 1-188.76-87.17l4.13 82.57a12 12 0 0 1-12 12.6H12a12 12 0 0 1-12-12z"/><path d="M12.3 209.82C33.93 94.92 134.81 8 256 8a247.4 247.4 0 0 1 188.9 87.34l-4-82.77A12 12 0 0 1 452.92 0h47.41a12 12 0 0 1 12 12v200.33a12 12 0 0 1-12 12H300a12 12 0 0 1-12-12v-47.41a12 12 0 0 1 12.57-12l101.53 4.88a176.07 176.07 0 0 0-317.24 56.94A12 12 0 0 1 73.19 224H24.1a12 12 0 0 1-11.8-14.18z"/></svg>'
                ]);
                $sync_button = ob_get_clean();
            }
            $next_sync = $nextTaxonomySync ? date('M j, H:i', $nextTaxonomySync) : 'Not Scheduled';
        } else {
            // Generic handling for all post types
            $lastSyncOption = "intucart_last_{$entity_name}_sync";
            $entityLastSync = get_option($lastSyncOption, 0);

            // Use the most recent sync time for this post type
            $last_sync = $entityLastSync ? date('M j, H:i', $entityLastSync) : 'Never';
            $next_sync = $nextPostSync ? date('M j, H:i', $nextPostSync) : 'Not Scheduled';
            ob_start();
            Button::secondary('Sync', [
                'size' => 'small',
                'attributes' => [
                    'class' => 'sync-now-button',
                    'data-type' => 'post_types',
                    'data-post-type' => esc_attr($entity_name)
                ],
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path opacity=".4" d="M0 500V299.67a12 12 0 0 1 12-12h200.33a12 12 0 0 1 12 12v47.41a12 12 0 0 1-12.57 12l-101.87-4.88a176.07 176.07 0 0 0 317.25-56.94 12 12 0 0 1 11.67-9.26h49.09a12 12 0 0 1 11.8 14.18C478.07 417.08 377.19 504 256 504a247.43 247.43 0 0 1-188.76-87.17l4.13 82.57a12 12 0 0 1-12 12.6H12a12 12 0 0 1-12-12z"/><path d="M12.3 209.82C33.93 94.92 134.81 8 256 8a247.4 247.4 0 0 1 188.9 87.34l-4-82.77A12 12 0 0 1 452.92 0h47.41a12 12 0 0 1 12 12v200.33a12 12 0 0 1-12 12H300a12 12 0 0 1-12-12v-47.41a12 12 0 0 1 12.57-12l101.53 4.88a176.07 176.07 0 0 0-317.24 56.94A12 12 0 0 1 73.19 224H24.1a12 12 0 0 1-11.8-14.18z"/></svg>'
            ]);
            $sync_button = ob_get_clean();
        }

        // Add visual styling for disabled rows
        $row_style = !$isEnabled ? ' style="opacity: 0.6; background-color: #f9f9f9;"' : '';
        
        echo '<tr' . $row_style . '>';
        
        // Display entity name with proper formatting
        $display_name = $entity_name;
        if ($isTaxonomy) {
            $taxonomyObj = get_taxonomy($entity_name);
            $display_name = $taxonomyObj ? ($taxonomyObj->labels->name ?? $entity_name) : $entity_name;
        } elseif ($entity_name === 'taxonomy') {
            $display_name = 'taxonomies';
        } else {
            $display_name = ucfirst($entity_name);
        }
        
        echo '<td>' . esc_html($display_name) . '</td>';
        echo '<td><span class="status-indicator ' . esc_attr($status_class) . '">' . esc_html($status_text) . '</span></td>';
        echo '<td>' . wp_kses_post($doc_count) . '</td>';
        echo '<td>' . wp_kses_post($last_sync) . '</td>';
        echo '<td>' . wp_kses_post($next_sync) . '</td>';
        echo '<td>';

        // Sync buttons - output without kses since Button component handles escaping
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
        echo $sync_button;
        echo '</td>';
        echo '</tr>';

        // Show index errors if any
        if ($index_data && isset($index_data['error'])) {
            echo '<tr><td colspan="6"><div class="index-error">' . esc_html($index_data['error']) . '</div></td></tr>';
        }
    }

    /**
     * Get local post count for specific post type
     *
     * @param string $post_type The post type (e.g., 'post', 'page', 'movie', etc.)
     * @return string Formatted count or 'N/A'
     */
    private function getLocalPostCount(string $post_type): string
    {
        global $wpdb;

        try {
            // Get count of posts for the specified post type
            $count = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = %s AND post_status IN ('publish', 'draft', 'private', 'pending', 'future')",
                $post_type
            ));

            return $count !== null ? number_format((int)$count) : 'N/A';
        } catch (\Exception $e) {
            return 'N/A';
        }
    }

    /**
     * Get the required tier for a feature from the cloud response
     *
     * @param array $summary The usage summary from the cloud service
     * @param string $featureKey The feature key (e.g., 'recommendations', 'search', etc.)
     * @return string Human-readable tier requirement message
     */
    private function getRequiredTierFromResponse(array $summary, string $featureKey): string
    {
        // Check if the cloud response includes requiredTier information
        if (isset($summary['features'][$featureKey]['requiredTier'])) {
            $requiredTier = $summary['features'][$featureKey]['requiredTier'];
            return ucfirst($requiredTier) . ' Plan Required';
        }

        // Fallback to hardcoded tier requirements if cloud doesn't provide the info
        return $this->getRequiredTierForFeature($featureKey);
    }

    /**
     * Get the required tier for a feature (fallback method)
     *
     * @param string $featureKey The feature key (e.g., 'recommendations', 'search', etc.)
     * @return string Human-readable tier requirement message
     */
    private function getRequiredTierForFeature(string $featureKey): string
    {
        // Define minimum tier requirements for each feature based on cloud tier service
        $tierRequirements = [
            'search' => null, // Available in all tiers including trial
            'chat' => null, // Available in all tiers including trial
            'recommendations' => null, // Available in all tiers including trial
            'products' => null, // Available in all tiers
            'posts' => null, // Available in all tiers
            'prompts' => null, // Available in all tiers
            'webhooks' => null, // Available in all tiers
            'webhook_deliveries' => null, // Available in all tiers
        ];

        return $tierRequirements[$featureKey] ?? 'Professional Plan Required';
    }

    /**
     * Render scheduled cron jobs section
     *
     * @return void
     */
    private function renderScheduledCronJobs(): void
    {
        echo '<div class="intucart-content-panel">';
        echo '<h3 class="intucart-panel-header">';
        echo '<span class="dashicons dashicons-clock"></span>';
        echo 'Scheduled Background Tasks';
        echo '</h3>';
        echo '<p style="margin-bottom: 15px; color: #666;">These tasks automatically index your content with ' . Constants::PLUGIN_SHORT_NAME . '\'s AI search and recommendations.</p>';

        $scheduled_jobs = $this->license->getScheduledCronJobs();

        if (empty($scheduled_jobs)) {
            echo '<div class="intucart-notice intucart-notice-info">';
            echo '<p><strong>No background tasks currently scheduled.</strong></p>';
            echo '<p>Tasks are automatically scheduled when you activate your license or manually trigger a sync.</p>';
            echo '</div>';
        } else {
            echo '<table class="widefat fixed striped">';
            echo '<thead>';
            echo '<tr><th style="width: 35%;">Task</th><th style="width: 20%;">Next Run</th><th style="width: 20%;">Time Until</th><th style="width: 25%;">Status</th></tr>';
            echo '</thead>';
            echo '<tbody>';

            foreach ($scheduled_jobs as $job) {
                // Make task names user-friendly
                $friendly_names = [
                    Constants::POST_SYNC_HOOK => class_exists('WooCommerce') && $this->recommendationsAvailable ?
                        '📄 Sync Posts, Pages & Products' : '📄 Sync Posts & Pages',
                    Constants::TAXONOMY_SYNC_HOOK => '🏷️ Sync Categories, Tags, etc.',
                    Constants::TRIGGER_FULL_SYNC_HOOK => '🔄 Full Content Sync',
                    'intucart_cleanup_initial_sync_notice' => '🧹 Cleanup Task',
                ];

                $friendly_name = $friendly_names[$job['hook']] ?? '⚙️ ' . ucwords(str_replace('_', ' ', str_replace('intucart_', '', $job['hook'])));

                echo '<tr>';
                echo '<td>';
                echo '<strong>' . esc_html($friendly_name) . '</strong><br>';
                echo '<small style="color: #666;"><code>' . esc_html($job['hook']) . '</code></small>';
                echo '</td>';

                echo '<td>' . esc_html($job['next_run_formatted']) . '</td>';

                // More user-friendly time display
                if ($job['time_until'] < 0) {
                    echo '<td style="color: #d63638;"><strong>Overdue</strong><br><small>' . esc_html($job['time_until_formatted']) . '</small></td>';
                } else {
                    $time_friendly = $this->formatTimeUntil($job['time_until']);
                    echo '<td style="color: #00a32a;">' . esc_html($time_friendly) . '</td>';
                }

                echo '<td>';
                if ($job['time_until'] < 0) {
                    echo '<span style="color: #d63638; font-weight: bold;">⚠️ Delayed</span><br>';
                    echo '<small style="color: #666;">WordPress cron may not be running properly</small>';
                } else {
                    echo '<span style="color: #00a32a; font-weight: bold;">✓ Scheduled</span><br>';
                    echo '<small style="color: #666;">Will run automatically</small>';
                }
                echo '</td>';
                echo '</tr>';
            }

            echo '</tbody>';
            echo '</table>';

            // Add helpful explanation
            echo '<div style="margin-top: 15px; padding: 12px; background: #f0f6fc; border: 1px solid #c3dcf2; border-radius: 4px;">';
            echo '<p style="margin: 0;"><strong>💡 How it works:</strong> These tasks run automatically in the background to keep your search, recommendations, and chatbot up-to-date. ';
            echo 'If a task shows as "Delayed", try visiting your website\'s front page to trigger WordPress cron, or contact your hosting provider about cron job configuration.</p>';
            echo '</div>';
        }

        // Collapsible debug info
        echo '<details style="margin-top: 20px;">';
        echo '<summary style="cursor: pointer; font-weight: bold; color: #666;">🔧 Technical Details</summary>';
        echo '<div style="margin-top: 10px; padding: 10px; background: #f9f9f9; border-radius: 4px;">';
        echo '<ul style="margin: 0; padding-left: 20px;">';
        echo '<li><strong>WordPress Cron URL:</strong> <code>' . esc_html(site_url('wp-cron.php')) . '</code></li>';
        echo '<li><strong>Current Server Time:</strong> ' . esc_html(date('Y-m-d H:i:s T', time())) . '</li>';
        $wp_cron_disabled = defined('DISABLE_WP_CRON') && constant('DISABLE_WP_CRON');
        echo '<li><strong>WordPress Cron Status:</strong> ' . ($wp_cron_disabled ?
        '<span style="color: #d63638;">Disabled</span> (may require server-level cron)' :
        '<span style="color: #00a32a;">Enabled</span>') . '</li>';
        echo '</ul>';
        if ($wp_cron_disabled) {
            echo '<p style="margin: 10px 0 0 0; color: #d63638;"><strong>Note:</strong> WordPress cron is disabled. Your hosting provider should have server-level cron jobs configured.</p>';
        }
        echo '</div>';
        echo '</details>';

        echo '</div>'; // .intucart-content-panel
    }

    /**
     * Format time until in a user-friendly way
     */
    private function formatTimeUntil(int $seconds): string
    {
        if ($seconds < 60) {
            return 'Less than 1 minute';
        } elseif ($seconds < 3600) {
            $minutes = round($seconds / 60);
            return $minutes . ' minute' . ($minutes == 1 ? '' : 's');
        } elseif ($seconds < 86400) {
            $hours = round($seconds / 3600);
            return $hours . ' hour' . ($hours == 1 ? '' : 's');
        } else {
            $days = round($seconds / 86400);
            return $days . ' day' . ($days == 1 ? '' : 's');
        }
    }

    /**
     * Render usage overview section
     *
     * @return void
     */
    private function renderUsageOverview(): void
    {
        echo '<div class="intucart-content-panel">';
        echo '<h3 class="intucart-panel-header">';
        echo '<span class="dashicons dashicons-chart-bar"></span>';
        echo 'Usage Overview';
        echo '</h3>';

        $summary = $this->usageService->getUsageSummary();

        if (!$summary['success']) {
            echo '<div class="intucart-notice intucart-notice-error">';
            echo '<p>Unable to retrieve usage data: ' . esc_html($summary['message'] ?? 'Unknown error') . '</p>';
            echo '</div>';
            echo '</div>'; // .intucart-content-panel
            return;
        }

        // Get feature status for remaining counts
        $featureStatus = $this->usageService->getFeatureStatus();

        // Show debug mode notice if active
        $isDebugMode = isset($summary['debug_mode']) && $summary['debug_mode'];
        if ($isDebugMode) {
            echo '<div class="intucart-notice intucart-notice-info">';
            echo '<p><strong>🐛 Debug Mode Active</strong> — All limits are bypassed and data shown below is for display purposes only.</p>';
            echo '</div>';
        }

        // Display tier and reset information
        echo '<div class="usage-tier-info">';
        echo '<table class="widefat">';
        echo '<tbody>';
        echo '<tr><th>Current Tier</th><td>' . esc_html(ucfirst($summary['tier'])) . '</td></tr>';

        if ($summary['resetTime']) {
            $resetDate = date('M j, Y', $summary['resetTime']);
            echo '<tr><th>Quota Reset</th><td>' . esc_html($resetDate) . '</td></tr>';
        }

        echo '</tbody>';
        echo '</table>';
        echo '</div>';

        // Display usage for each feature
        if (!empty($summary['features'])) {
            echo '<div class="usage-features">';
            echo '<h4>Feature Usage</h4>';
            echo '<table class="widefat fixed striped">';
            echo '<thead>';
            echo '<tr><th>Feature</th><th>Current <span class="dashicons dashicons-info-outline" style="font-size: 14px; line-height: inherit; width: 14px; height: 14px; color: #666; cursor: help; vertical-align: text-top;" data-popover="Usage counts are aggregated across all sites using the same license key. If you have multiple sites (staging, production, etc.), usage from all sites contributes to these totals."></span></th><th>Limit</th><th>Remaining</th><th>Percentage</th><th>Status</th></tr>';
            echo '</thead>';
            echo '<tbody>';

            foreach ($summary['features'] as $featureKey => $feature) {
                // Skip WooCommerce-specific features if WooCommerce is not active
                if (($featureKey === 'products' || $featureKey === 'recommendations') && !class_exists('WooCommerce')) {
                    continue;
                }
                
                // Special handling for webhooks - show count instead of usage
                if ($featureKey === 'webhooks') {
                    // For webhooks, the 'used' field contains the current webhook count
                    // and 'limit' contains the max webhooks allowed for this tier
                    $feature['name'] = 'Webhooks Configured';
                }

                // Determine status based on feature availability and limits
                $isFeatureDisabled = $feature['limit'] == 0;
                $isOverLimit = $feature['used'] >= $feature['limit'] && $feature['limit'] > 0;

                if ($isFeatureDisabled) {
                    $statusClass = 'status-neutral';
                    $statusText = $this->getRequiredTierFromResponse($summary, $featureKey);
                    $remaining = 'N/A';
                } elseif ($isOverLimit) {
                    $statusClass = 'status-error';
                    $statusText = 'Limit Exceeded';
                    $remaining = '0';
                } else {
                    $statusClass = 'status-ok';
                    $statusText = 'Available';

                    // Get remaining count from feature status using the same key
                    $remaining = 'N/A';
                    if (isset($featureStatus[$featureKey]['success']) && $featureStatus[$featureKey]['success']) {
                        if (isset($featureStatus[$featureKey]['usage']['remaining'])) {
                            $remainingCount = $featureStatus[$featureKey]['usage']['remaining'];
                            if ($isDebugMode && $remainingCount > 900000) {
                                $remaining = 'Unlimited';
                            } else {
                                $remaining = number_format($remainingCount);
                            }
                        }
                    }
                }

                // Color code percentage based on usage (only for enabled features)
                $percentageClass = '';
                if (!$isFeatureDisabled) {
                    if ($feature['percentage'] >= 90) {
                        $percentageClass = 'usage-critical';
                    } elseif ($feature['percentage'] >= 75) {
                        $percentageClass = 'usage-warning';
                    } else {
                        $percentageClass = 'usage-ok';
                    }
                } else {
                    $percentageClass = 'usage-disabled';
                }

                echo '<tr>';
                echo '<td>' . esc_html($feature['name']) . '</td>';
                echo '<td>' . number_format($feature['used']) . '</td>';
                echo '<td>' . ($isFeatureDisabled ? 'Not Included' : number_format($feature['limit'])) . '</td>';
                echo '<td>' . esc_html($remaining) . '</td>';
                echo '<td><span class="' . esc_attr($percentageClass) . '">' . esc_html($feature['percentage']) . '%</span></td>';
                echo '<td><span class="status-indicator ' . esc_attr($statusClass) . '">' . esc_html($statusText) . '</span></td>';
                echo '</tr>';
            }

            echo '</tbody>';
            echo '</table>';
            echo '</div>';
        }

        // Add refresh button and note
        echo '<div class="usage-actions">';
        Button::secondary('Refresh Usage Data', [
            'attributes' => [
                'class' => 'refresh-usage-data'
            ],
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path opacity=".4" d="M8 454.06V320a24 24 0 0 1 24-24h134.06c21.38 0 32.09 25.85 17 41l-41.75 41.75A166.82 166.82 0 0 0 256.16 424c77.41-.07 144.31-53.14 162.78-126.85a12 12 0 0 1 11.65-9.15h57.31a12 12 0 0 1 11.81 14.18C478.07 417.08 377.19 504 256 504a247.14 247.14 0 0 1-171.31-68.69L49 471c-15.15 15.15-41 4.44-41-16.94z"/><path d="M12.3 209.82C33.93 94.92 134.81 8 256 8a247.14 247.14 0 0 1 171.31 68.69L463 41c15.12-15.12 41-4.41 41 17v134a24 24 0 0 1-24 24H345.94c-21.38 0-32.09-25.85-17-41l41.75-41.75A166.8 166.8 0 0 0 255.85 88c-77.46.07-144.33 53.18-162.79 126.85A12 12 0 0 1 81.41 224H24.1a12 12 0 0 1-11.8-14.18z"/></svg>'
        ]);
        echo '</div>';

        echo '<div class="usage-note">';
        if ($isDebugMode) {
            echo '<p><em>Debug mode active - all limits bypassed. Real-time enforcement still occurs in cloud service.</em></p>';
        } else {
            echo '<p><em>Remaining counts refresh every 5 minutes. For real-time limits, the cloud service will return appropriate errors when features are used.</em></p>';
        }
        echo '</div>';

        echo '</div>'; // .intucart-content-panel
    }


}
