<?php

declare(strict_types=1);

namespace Intucart;

use DI\ContainerBuilder;
use DI\Container;
use Intucart\Database\Schema;
use Intucart\Services\Providers\Events\WooCommerceEventProvider;
use Intucart\Services\Managers\CronManager;
use Intucart\Services\Logger;
use Intucart\Admin\AdminController;
use Intucart\Services\Search;
use Intucart\Shortcodes\AssistantChatShortcode;
use Intucart\Shortcodes\SearchWidgetShortcode;
use Intucart\Pages\Settings\SearchWidgetSettings;
use Intucart\Integration\SearchOverride;
use Intucart\Integration\RecommendationsOverride;
use Intucart\Services\Licensing\License;
use Intucart\Services\CacheAdmin;
use Intucart\Services\Constants;
use Intucart\Base\SettingsLinks;
use Intucart\DI\CoreServicesProvider;
use Intucart\Base\Enqueue;
use Intucart\DI\FeatureServicesProvider;
use Intucart\DI\PresentationServicesProvider;
use Intucart\DI\AdminServicesProvider;
use Intucart\Services\Managers\PostTypeManager;
use Intucart\Services\Managers\TaxonomyManager;
use Intucart\Services\Mcp\McpServer;
use Intucart\Admin\SyncMetaBox;
use Intucart\Services\AIClientManager;
use Intucart\Admin\SyncAdmin;
use Intucart\Services\PageBuilderService;
use Intucart\Services\Cache\CacheService;

/**
 * Main Plugin class
 */
class Plugin
{
    private Container $container;
    private ?Logger $logger = null;

    public function __construct()
    {
        // Move container building to plugins_loaded hook
        add_action('plugins_loaded', [$this, 'initializeContainer']);
    }

    public function getContainer(): ?Container
    {
        return $this->container ?? null;
    }

    /**
     * Plugin activation hook
     *
     * @return void
     */
    public static function activate(): void
    {
        // WooCommerce optional - plugin can work without it
        // Create database tables
        Schema::createTables();

        // Set plugin version
        update_option(Constants::VERSION_OPTION, INTUCART_VERSION);

        // Initialize container for license activation
        $plugin = new self();
        $plugin->initializeContainer();

        // Check if container was successfully initialized
        if (!isset($plugin->container)) {
            error_log('[Intucart] [ERROR] Failed to initialize container during activation. ' .
                'Cron jobs will not be scheduled.');
            return;
        }

        // Get required services
        $license_key = get_option(Constants::LICENSE_KEY_OPTION, '');

        try {
            $license = $plugin->container->get(License::class);
            $aiClientManager = $plugin->container->get(AIClientManager::class);
        } catch (\Exception $e) {
             // Log error if container/services aren't available
            if ($plugin->logger) {
                $plugin->logger->error(
                    'Failed to get required services during activation',
                    ['error' => $e->getMessage()]
                );
            }
            return;
        }

        // Activate license *only* if the key exists and we don't have an API key yet
        if (!empty($license_key)) {
            try {
                // Check if we already have an API key (meaning license is already activated)
                if ($aiClientManager->hasApiKey()) {
                    if ($plugin->logger) {
                        $plugin->logger->info(
                            'Skipping license activation during plugin reactivation: API key already configured.'
                        );
                    }
                } else {
                    // Key exists but no API key yet, attempt activation
                    if ($plugin->logger) {
                        $plugin->logger->info('Attempting license activation during plugin reactivation.');
                    }
                    $activated = $license->activateLicense($license_key);

                    if (!$activated && $plugin->logger) {
                        $plugin->logger->error('Failed to activate license during plugin activation');
                    }
                }
            } catch (\Exception $e) {
                if ($plugin->logger) {
                    $plugin->logger->error('Exception during license check/activation', ['error' => $e->getMessage()]);
                }
            }
        }

        // Schedule cron jobs via CronManager
        try {
            $plugin->container->get(CronManager::class)->initialize();
        } catch (\Exception $e) {
            $plugin->logger->error('Failed to schedule cron jobs during activation', ['error' => $e->getMessage()]);
        }
    }

    public static function deactivate(): void
    {
        CronManager::deactivate();
    }

    /**
     * Plugin uninstall hook
     *
     * @return void
     */
    public static function uninstall(): void
    {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('Intucart uninstall started');
        }

        try {
            global $wpdb;

            // Drop custom tables using Schema class
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('Intucart uninstall: Attempting to drop tables using Schema::dropTables()');
            }

            Schema::dropTables();

            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('Intucart uninstall: All database tables dropped');
            }

            // Clean up cron jobs using CronManager
            CronManager::deactivate();

            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('Intucart uninstall: Cron jobs cleaned up');
            }

            // Delete live agent credentials from AWS SSM (before deleting local options)
            try {
                $licenseKey = get_option(Constants::LICENSE_KEY_OPTION, '');
                
                if (!empty($licenseKey)) {
                    // Create temporary services for the API call
                    require_once INTUCART_PLUGIN_DIR . 'includes/Services/Cache/CacheService.php';
                    require_once INTUCART_PLUGIN_DIR . 'includes/Services/AIClientManager.php';
                    
                    $cacheService = new CacheService();
                    $aiClientManager = new AIClientManager($cacheService);
                    $aiClientManager->deleteAllLiveAgentCredentials();
                } else {
                    if (defined('WP_DEBUG') && WP_DEBUG) {
                        error_log('Intucart uninstall: No license key, skipping credential deletion');
                    }
                }
            } catch (\Exception $e) {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    error_log('Intucart uninstall: Error during credential deletion: ' . $e->getMessage());
                }
                // Don't throw - allow uninstall to continue
            }

            // Delete all options
            $options = [
                // API Key settings
                Constants::LICENSE_KEY_OPTION,
                Constants::PUBLISHABLE_KEY_OPTION,
                Constants::WORKSPACE_ID_OPTION,
                Constants::TENANT_ID_OPTION,
                Constants::TIER_OPTION,
                Constants::LICENSE_NOTICE_DISMISSED_OPTION,

                // Search settings
                Constants::ENABLE_SEMANTIC_SEARCH_OPTION,
                Constants::ENABLE_MIXED_CONTENT_SEARCH_OPTION,
                Constants::ENABLE_ENHANCED_AI_SEARCH_OPTION,
                Constants::SEARCH_CACHE_DURATION_OPTION,

                // Recommendation settings
                Constants::RECOMMENDATION_HALF_LIFE_OPTION,
                Constants::ENABLE_RELATED_PRODUCTS_OPTION,
                Constants::ENABLE_UPSELLS_OPTION,
                Constants::ENABLE_CROSS_SELLS_OPTION,

                // Sync settings
                Constants::SYNC_INTERVAL_OPTION,
                Constants::SYNC_HOUR_OPTION,
                Constants::LAST_PRODUCT_SYNC_OPTION,
                Constants::LAST_POST_SYNC_OPTION,
                Constants::LAST_PAGE_SYNC_OPTION,
                Constants::LAST_TAXONOMY_SYNC_OPTION,

                // Stats and counters
                Constants::TOTAL_VECTOR_COUNT_OPTION,
                Constants::CACHE_HITS_OPTION,
                Constants::CACHE_MISSES_OPTION,

                // Notice dismissals
                Constants::LICENSE_NOTICE_DISMISSED_OPTION,

                // Version
                Constants::VERSION_OPTION,

                // Chat Assistant Settings (Core)
                Constants::OPTION_ENABLE_CHAT_ASSISTANT,
                Constants::OPTION_CHAT_MODEL,
                Constants::OPTION_CHAT_HEADER_TEXT,
                Constants::OPTION_CHAT_HEADER_ICON_ID,
                Constants::OPTION_CHAT_CUSTOM_CSS,
                Constants::OPTION_TOGGLE_ICON_ID,
                Constants::OPTION_ASSISTANT_WELCOME_MESSAGE,
                Constants::OPTION_CHAT_WIDGET_POSITION,
                Constants::OPTION_CHAT_PRIMARY_COLOR,
                Constants::OPTION_CHAT_CARD_LAYOUT,

                // Chat Header & Greeting
                Constants::OPTION_CHAT_HEADER_SUBTITLE,
                Constants::OPTION_CHAT_GREETING_ICON_ID,
                Constants::OPTION_CHAT_GREETING_SUBTEXT,

                // Chat Theme Settings
                Constants::OPTION_CHAT_COLOR_SCHEME,
                Constants::OPTION_CHAT_BORDER_RADIUS,
                Constants::OPTION_CHAT_ACCENT_COLOR,
                Constants::OPTION_CHAT_LINK_COLOR,
                Constants::OPTION_CHAT_ASSISTANT_BUBBLE_BG,
                Constants::OPTION_CHAT_HEADER_BACKGROUND,

                // Chat Button Colors
                Constants::OPTION_CHAT_SEND_BUTTON_COLOR,
                Constants::OPTION_CHAT_TRIGGER_BUTTON_COLOR,
                Constants::OPTION_CHAT_TRIGGER_BUTTON_HOVER_COLOR,
                Constants::OPTION_CHAT_TRIGGER_BUTTON_BORDER_COLOR,

                // Chat Widget Background
                Constants::OPTION_CHAT_WIDGET_BACKGROUND_MODE,
                Constants::OPTION_CHAT_WIDGET_BACKGROUND_CUSTOM,

                // Chat Typography
                Constants::OPTION_CHAT_FONT_FAMILY,
                Constants::OPTION_CHAT_CUSTOM_FONT_FAMILY,

                // Chat Shadow & Input
                Constants::OPTION_CHAT_WIDGET_SHADOW,
                Constants::OPTION_CHAT_INPUT_BACKGROUND,
                Constants::OPTION_CHAT_INPUT_BORDER,

                // Chat Widget Dimensions
                Constants::OPTION_CHAT_WIDGET_MAX_WIDTH,
                Constants::OPTION_CHAT_WIDGET_MAX_HEIGHT,

                // Dark Mode Color Variants
                Constants::OPTION_CHAT_DARK_PRIMARY_COLOR,
                Constants::OPTION_CHAT_DARK_ASSISTANT_BUBBLE_BG,
                Constants::OPTION_CHAT_DARK_HEADER_BACKGROUND,
                Constants::OPTION_CHAT_DARK_ACCENT_COLOR,
                Constants::OPTION_CHAT_DARK_SEND_BUTTON_COLOR,
                Constants::OPTION_CHAT_DARK_LINK_COLOR,
                Constants::OPTION_CHAT_DARK_TRIGGER_BUTTON_COLOR,
                Constants::OPTION_CHAT_DARK_TRIGGER_BUTTON_HOVER_COLOR,
                Constants::OPTION_CHAT_DARK_TRIGGER_BUTTON_BORDER_COLOR,
                Constants::OPTION_CHAT_DARK_INPUT_BACKGROUND,
                Constants::OPTION_CHAT_DARK_INPUT_BORDER,
                Constants::OPTION_CHAT_DARK_WIDGET_BACKGROUND_CUSTOM,

                // Message Actions & Timestamps
                Constants::OPTION_CHAT_SHOW_MESSAGE_ACTIONS,
                Constants::OPTION_CHAT_SHOW_USER_COPY_BUTTON,
                Constants::OPTION_CHAT_SHOW_MESSAGE_TIMESTAMPS,

                // Webhook Settings
                Constants::OPTION_WEBHOOKS_ENABLED,
                Constants::OPTION_WEBHOOK_ENDPOINTS,
                Constants::OPTION_WEBHOOK_GLOBAL_SECRET,
                Constants::OPTION_WEBHOOK_TIMEOUT,
                Constants::OPTION_WEBHOOK_RETRY_ATTEMPTS,
                Constants::OPTION_WEBHOOK_RETRY_DELAY,

                // Live Agent Settings
                Constants::OPTION_LIVE_AGENT_ENABLED,
                Constants::OPTION_LIVE_AGENT_PROVIDER,
                Constants::OPTION_LIVE_AGENT_CONFIGURED,
                Constants::OPTION_LIVE_AGENT_AUTO_HANDOFF,
                Constants::OPTION_LIVE_AGENT_MANUAL_HANDOFF,
                Constants::OPTION_LIVE_AGENT_BUSINESS_HOURS,
                Constants::OPTION_LIVE_AGENT_INACTIVITY_TIMEOUT,
                Constants::OPTION_LIVE_AGENT_ESCALATION_PROMPT,

                // Search Widget Settings
                Constants::OPTION_SEARCH_WIDGET_ENABLED,
                Constants::OPTION_SEARCH_WIDGET_PRODUCTS,
                Constants::OPTION_SEARCH_WIDGET_POSTS,
                Constants::OPTION_SEARCH_WIDGET_KEYBOARD_SHORTCUT,
                Constants::OPTION_SEARCH_WIDGET_SHOW_TRIGGER,
                Constants::OPTION_SEARCH_WIDGET_TRIGGER_POSITION,
                Constants::OPTION_SEARCH_WIDGET_PRIMARY_COLOR,
                Constants::OPTION_SEARCH_WIDGET_COLOR_SCHEME,
            ];

            foreach ($options as $option) {
                delete_option($option);
            }

            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('Intucart uninstall: Options deleted');
            }

            // Delete all transients
            $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '%_transient_intucart_%'");
            $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '%_transient_timeout_intucart_%'");

            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('Intucart uninstall: Transients deleted');
            }

            // Delete user metadata
            $wpdb->query("DELETE FROM {$wpdb->usermeta} WHERE meta_key LIKE 'intucart_%'");

            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('Intucart uninstall: User metadata deleted');
            }

            // Delete post metadata (including sync metadata)
            $wpdb->query("DELETE FROM {$wpdb->postmeta} WHERE meta_key LIKE 'intucart_%'");
            $wpdb->query("DELETE FROM {$wpdb->postmeta} WHERE meta_key LIKE '_intucart_%'");

            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('Intucart uninstall: Post metadata deleted');
            }

            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('Intucart uninstall: Cleanup completed successfully');
            }
        } catch (\Exception $e) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('Intucart uninstall error: ' . $e->getMessage());
                error_log('Stack trace: ' . $e->getTraceAsString());
            }
        }

        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('Intucart uninstall finished');
        }
    }

    public function initializeContainer(): void
    {
        try {
            // 1. Initialize temporary logger for early error tracking
            $this->initializeTemporaryLogger();

            // 2. Build and initialize the main container
            $this->buildMainContainer();

            // 3. Register WordPress hooks first
            $this->registerHooks();

            // 4. Initialize all services
            $this->initializeServices();
        } catch (\Exception $e) {
            $this->logInitializationError('Plugin initialization failed', $e);
        }
    }

    /**
     * Initialize temporary logger for early error tracking
     *
     * @throws \Exception If logger initialization fails
     */
    private function initializeTemporaryLogger(): void
    {
        $builder = new ContainerBuilder();
        $builder->addDefinitions([Logger::class => \DI\create()]);
        $tempContainer = $builder->build();
        $this->logger = $tempContainer->get(Logger::class);
    }

    /**
     * Build and initialize the main container
     *
     * @throws \Exception If container initialization fails
     */
    private function buildMainContainer(): void
    {
        $builder = new ContainerBuilder();
        $builder->addDefinitions($this->getContainerDefinitions());
        $this->container = $builder->build();

        // Re-initialize logger with full container
        $this->logger = $this->container->get(Logger::class);
    }

    /**
     * Log initialization errors with proper context
     *
     * @param string     $message Error message
     * @param \Throwable $e       Exception that occurred
     */
    private function logInitializationError(string $message, \Throwable $e): void
    {
        if ($this->logger) {
            $this->logger->error($message, [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        } else {
            error_log(Constants::PLUGIN_NAME . ": {$message} - " . $e->getMessage());
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('Stack trace: ' . $e->getTraceAsString());
            }
        }
    }

    private function getContainerDefinitions(): array
    {
        // Instantiate consolidated providers
        $coreProvider = new CoreServicesProvider();
        $featureProvider = new FeatureServicesProvider();
        $presentationProvider = new PresentationServicesProvider();
        $adminProvider = new AdminServicesProvider();

        // Combine definitions from all providers
        $definitions = array_merge(
            $coreProvider->getDefinitions(),
            $featureProvider->getDefinitions(),
            $presentationProvider->getDefinitions(),
            $adminProvider->getDefinitions()
        );

        // Add definitions that need the Plugin context or are simpler here
        $definitions[Container::class] = function () {
            return $this->container;
        };

        return $definitions;
    }

    /**
     * Initialize all plugin services
     *
     * @return void
     */
    private function initializeServices(): void
    {
        // Initialize Enqueue service to register hooks
        $this->container->get(Enqueue::class)->register();

        // Initialize settings links
        $settingsLinks = new SettingsLinks();
        $settingsLinks->initialize();

        // Initialize Search service
        $this->container->get(Search::class)->initialize();

        // Initialize Cache Admin service
        $this->container->get(CacheAdmin::class)->initialize();

        // Initialize Page Builder Service for cache busting
        $this->container->get(PageBuilderService::class)->initialize();

        // Initialize core managers with their own batch processing
        $this->container->get(PostTypeManager::class)->initialize();
        $this->container->get(TaxonomyManager::class)->initialize();

        // Initialize Admin pages
        $this->container->get(AdminController::class)->initialize();

        // Initialize Plugin Updater for self-hosted updates
        $this->container->get(\Intucart\Services\Licensing\PluginUpdater::class);

        // Initialize Sync Meta Box (admin UI for sync control)
        $this->container->get(SyncMetaBox::class)->initialize();

        // Initialize Sync Admin (list table columns for sync control)
        $this->container->get(SyncAdmin::class)->initialize();

        // Initialize SearchOverride for all WordPress searches
        try {
            $this->container->get(SearchOverride::class)->initialize();
        } catch (\Exception $e) {
            $this->logger->error(
                'Failed to initialize SearchOverride',
                ['error' => $e->getMessage()]
            );
        }

        // Initialize WooCommerce-specific providers if available
        if (class_exists('WooCommerce')) {
            try {
                $this->container->get(WooCommerceEventProvider::class)->initialize();
                $this->container->get(RecommendationsOverride::class)->initialize();
            } catch (\Exception $e) {
                $this->logger->error(
                    'Failed to initialize WooCommerce provider',
                    ['error' => $e->getMessage()]
                );
            }
        } else {
            $this->logger->info(
                'WooCommerce not available, skipping WooCommerce-specific providers'
            );
        }

        // Initialize shortcodes
        $this->container->get(AssistantChatShortcode::class)->initialize();
        $this->container->get(SearchWidgetShortcode::class)->initialize();
        
        // Register search widget settings (needs to happen on admin_init)
        add_action('admin_init', function () {
            $this->container->get(SearchWidgetSettings::class)->registerSettings();
        });

        // Initialize MCP server
        try {
            $this->container->get(McpServer::class);
        } catch (\Exception $e) {
            $this->logger->error(
                'Failed to initialize MCP server',
                ['error' => $e->getMessage()]
            );
        }
    }

    /**
     * Register hooks for WordPress actions and filters
     *
     * @return void
     */
    private function registerHooks(): void
    {
        try {
            // Initialize cron jobs
            add_action(
                'init',
                function () {
                    if (isset($this->container)) {
                        try {
                            $this->container->get(CronManager::class)->initialize();
                        } catch (\Exception $e) {
                            $this->logger->error(
                                'Failed to initialize cron jobs during init',
                                ['error' => $e->getMessage()]
                            );
                        }
                    } else {
                        error_log('Cannot initialize cron jobs: container not initialized');
                    }
                },
                1
            );

            // Legacy batch completion hook removed - now handled by individual managers
        } catch (\Exception $e) {
            $this->logger->error(
                'Failed to register plugin hooks',
                ['error' => $e->getMessage()]
            );
        }
    }
}
